﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using System;
using System.Runtime.InteropServices;

namespace App.Utility
{
    public static class CommandLineUtility
    {
        // 実行する
        public static int Execute(string filename, string arguments)
        {
            string stdOutput;
            string stdError;

            return Execute(filename, arguments, out stdOutput, out stdError);
        }

        // 実行する
        // 標準出力と標準エラーを返します
        public static int Execute(string filename, string arguments, out string stdOutput, out string stdError)
        {
            // 実行する
            using (var process = Process.Start(
                new ProcessStartInfo()
                {
                    FileName = filename,
                    Arguments = arguments,
                    CreateNoWindow = true,
                    UseShellExecute = false,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true
                }
            ))
            {
                // 実行が終わるまで待つ
                process.WaitForExit();

                // 結果を保存する
                stdOutput = process.StandardOutput.ReadToEnd();
                stdError  = process.StandardError.ReadToEnd();

                return process.ExitCode;
            }
        }

        /// <summary>
        /// コマンドラインの入力を引数に分ける
        /// </summary>
        public static string[] SplitArguments(string commandLine)
        {
            if (commandLine == null)
            {
                return new string[0];
            }

            int argc;

            // コマンド部分が抜けているのでダミーのコマンド c をつけておく
            var argv = Win32.NativeMethods.CommandLineToArgvW("c " + commandLine, out argc);

            var arguments = new string[argc - 1];
            try
            {
                // ダミーのコマンドは飛ばす
                for (int i = 1; i < argc; i++)
                {
                    // 文字列へのポインタを取得
                    var arg = Marshal.ReadIntPtr(argv, i * IntPtr.Size);

                    // string へ変換
                    arguments[i - 1] = Marshal.PtrToStringUni(arg);
                }
            }
            finally
            {
                // CommandLineToArgvW で確保されたメモリは呼び出し元が解放する
                Marshal.FreeHGlobal(argv);
            }

            return arguments;
        }
    }
}
