﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;

namespace App.PropertyEdit
{
    public partial class ObjectPropertyPanel : UIUserControl
    {
        /// <summary>
        /// オーナー。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public ObjectPropertyDialog Owner{ get; set; }

        /// <summary>
        /// ターゲット。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public GuiObjectGroup Targets
        {
            get { return Owner != null ? Owner.Targets: null; }
        }

        /// <summary>
        /// 左側領域幅。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public int LeftWidth
        {
            get { return spcClient.SplitterDistance; }
            set { spcClient.SplitterDistance = value; }
        }

        /// <summary>
        /// 左領域幅を強制的に設定
        /// </summary>
        public void SetLeftWidthForcibly(int value)
        {
            // 必要ならパネルの幅を修正
            if (spcClient.Width < value + spcClient.Panel2MinSize)
            {
                var diff = value + spcClient.Panel2MinSize - spcClient.Width;
                Width += diff;
            }

            spcClient.SplitterDistance = value;
        }
        /// <summary>
        /// カテゴリビュー。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public CategoryView CategoryView
        {
            get { return tvwCategory; }
        }

        // プロパティページリスト
        private readonly List<ObjectPropertyPage> _propertyPages = new List<ObjectPropertyPage>();

        // カテゴリ項目リスト
        //private readonly Dictionary<PropertyPageID, PropertyCategoryNode> categoryItems_ = new Dictionary<PropertyPageID, PropertyCategoryNode>();

        // アクティブページ
        public ObjectPropertyPage ActivePage{ get; private set; }

        // アクティブページＩＤ
        public PropertyPageID ActivePageID{	get{	return (ActivePage != null) ? ActivePage.PageID : PropertyPageID.Null;	}	}

        public ObjectPropertyPanel()
        {
            InitializeComponent();
        }

//		// イベントハンドラ保存用
//		private DocumentPropertyChangedEventHandler _docPropertyChangedHandler;

        /// <summary>
        /// フォーム初期化。
        /// </summary>
        public void InitializeForm()
        {
            // 外部イベント
//			_docPropertyChangedHandler = Event_Document_PropertyChanged;
            {
                // 登録
//				App.AppContext.PropertyChanged += _docPropertyChangedHandler;
                App.AppContext.PropertyChanged += Event_Document_PropertyChanged;

                // 削除（Disposedで行う）
                Disposed += delegate
                {
//					App.AppContext.PropertyChanged -= _docPropertyChangedHandler;
                    App.AppContext.PropertyChanged -= Event_Document_PropertyChanged;
                };
            }

            // 編集カラーの初期化
            InitializeEditColor();

            // フォーム初期化
            InitializeFormInternal();
        }

        /// <summary>
        /// フォーム初期化（内部処理用）。
        /// </summary>
        protected virtual void InitializeFormInternal()
        {
            // 派生クラスで実装する
            // 抽象にしないのはフォームデザイナでインスタンス化させる為
        }

        /// <summary>
        /// 初期ページ設定。
        /// </summary>
        public void SetInitialPage(ObjectPropertyPanel panel)
        {
            Debug.Assert(GetType().Equals(panel.GetType()));

            // 選択カテゴリがなければ何もしない
            if (panel.CategoryView.SelectedNode == null)
            {
                return;
            }

            // 指定されたパネルと同じページにする
            foreach (PropertyCategoryNode node in CategoryView.Nodes)
            {
                PropertyCategoryNode result = node.Search((PropertyCategoryNode)panel.CategoryView.SelectedNode);
                if (result != null)
                {
                    CategoryView.SelectedNode = result;
                    return;
                }
            }
        }

        /// <summary>
        /// フォーム更新。
        /// </summary>
        public void UpdateForm(bool updateActivePage, bool targetOrPageChanged)
        {
            // 自身の更新
            using (var block = new UIControlEventSuppressBlock())
            {
                // 編集カラー・コメントの更新
                UpdateEditColor();

                // 派生クラスの更新処理
                UpdateFormInternal();
            }

            // アクティブページ更新
            if (updateActivePage)
            {
                if (ActivePage != null)
                {
                    ActivePage.UpdateForm(targetOrPageChanged, false);
                    Owner.HelpPageKey = ActivePage.HelpKey;
                }
                else
                {
                    Owner.HelpPageKey = HelpUtility.PageKey.p_property_window;
                }
            }

            foreach (var node in TreeViewUtility.AllNodes(CategoryView).OfType<PropertyCategoryNode>())
            {
                node.ShowModifiedMark = node.UpdateModified != null && node.UpdateModified();
                node.ValidPage = node.IsValidPage == null || node.IsValidPage();
            }

            CategoryView.Invalidate();
        }

        /// <summary>
        /// フォーム更新（内部処理用）。
        /// </summary>
        protected virtual void UpdateFormInternal()
        {
            // 必要なら派生クラスで実装する
            // 抽象にしないのはフォームデザイナでインスタンス化させる為
        }

        /// <summary>
        /// カテゴリ登録。
        /// </summary>
        protected PropertyCategoryNode RegisterCategory(
            int							imageIndex,
            string						text,
            PropertyPageID				pageID,
            CreatePropertyPageHandler	createPageHandler,
            object						createPageArg = null,
            Func<bool>                  updateModifiedFunc = null,
            Func<bool>                  isValidPageFunc = null
        )
        {
            Debug.Assert(pageID != PropertyPageID.Null);
            Debug.Assert(createPageHandler != null);

            // 項目作成
            var item = new PropertyCategoryNode(text, imageIndex, pageID, createPageHandler, createPageArg)
            {
                UpdateModified = updateModifiedFunc,
                IsValidPage = isValidPageFunc
            };
            return item;
        }

        public virtual void SetDefaultCategory()
        {
            if (tvwCategory.Nodes.Count > 0)
            {
                tvwCategory.SelectedNode = tvwCategory.Nodes[0];
            }
        }

        private void tvwCategory_AfterSelect(object sender, TreeViewEventArgs e)
        {
            DebugConsole.WriteLine("AfterSelect");

            SelectCategoryNode((PropertyCategoryNode)e.Node);
        }

        public bool CreateCategoryNode(PropertyCategoryNode node)
        {
            var isPageCreated = false;

            var page = node.PropertyPage;
            {
                if (page == null)
                {
                    using (WaitCursor wait = new WaitCursor())
                    {
                        page = node.CreatePropertyPage();
                        page.Owner = this;
                        page.Parent = pnlPage;
                        page.Visible = false;
                        page.Dock = DockStyle.Fill;
                        page.InitializeForm();

                        // ヘルプの設定
                        //Owner.BindHelp(page, page.HelpPath);
                        DialogUtility.SetHelp(page, page.HelpKey);

                        _propertyPages.Add(page);

                        isPageCreated = true;
                    }
                }
            }

            return isPageCreated;
        }

        private void SelectCategoryNode(PropertyCategoryNode node)
        {
            using (var block = new ObjectPropertyDialog.InternallyChangingSize())
            {
                // ページ未作成なら作成する
                var isPageCreated = CreateCategoryNode(node);

                ObjectPropertyPage   page = node.PropertyPage;

                page.CategoryNode = node;

                var oldPage = ActivePage;
                if (oldPage != null)
                {
                    oldPage.BeforePageDeactivated();
                }

                // 表示サイズ変更
                bool updated = false;
                if (oldPage != page)
                {
                    if (oldPage == null || oldPage.ChangeDockStyle)
                    {
                        ActivePage = null;
                        if (ApplicationConfig.Setting.PropertyEdit.OptimizePageSize)
                        {
                            OptimizePageSize(page);
                        }
                        page.UpdateForm(true, isPageCreated);
                        updated = true;
                    }
                }
                else
                {
                    page.UpdateForm(true, isPageCreated);
                    updated = true;
                }

                // 表示するヘルプページを設定
                Owner.HelpPageKey = page.HelpKey;

                // アクティブページ切り替え
                if (oldPage == null)
                {
                    ActivePage = page;
                    ActivePage.Visible = true;
                }
                else if (oldPage != page)
                {
                    // ちらつき防止のため
                    if (oldPage.PageID == page.PageID)
                    {
                        if (!updated)
                        {
                            ActivePage = null;
                            if (ApplicationConfig.Setting.PropertyEdit.OptimizePageSize)
                            {
                                OptimizePageSize(page);
                            }
                            page.UpdateForm(true, isPageCreated);
                        }
                        page.Visible = true;
                        oldPage.Visible = false;
                        ActivePage = page;
                    }
                    else
                    {
                        oldPage.Visible = false;
                        if (!updated)
                        {
                            ActivePage = null;
                            if (ApplicationConfig.Setting.PropertyEdit.OptimizePageSize)
                            {
                                OptimizePageSize(page);
                            }
                            page.UpdateForm(true, isPageCreated);
                        }
                        ActivePage = page;
                        ActivePage.Visible = true;
                    }
                }

                if (ActivePage != null)
                {
                    if (ActivePage.HideParentScrollBar)
                    {
                        pnlPage.AutoScroll = false;
                    }
                    else if (ActivePage.ShowParentScrollBar)
                    {
                        pnlPage.AutoScroll = !ActivePage.AutoScroll;
                    }
                    else
                    {
                        pnlPage.AutoScroll = true;
                    }


                    if (this.IsHandleCreated)
                    {
                        // 最小化されていると正しく動作しないので後回し
                        BeginInvoke((Action)(() =>
                            {
                                ActivePage.AfterPageActiveted(oldPage);
                            }));
                    }
                    else
                    {
                        ActivePage.AfterPageActiveted(oldPage);
                    }
                }

                if (ApplicationConfig.Setting.PropertyEdit.OptimizePageSize == false)
                {
                    UpdateActivePageDockStyle();
                }

                // ダイアログのコピー貼り付けボタン更新
                Owner.UpdateCopyPasteButton(this);

                OnCategoryAfterSelect();

                // パネルのみ更新
                UpdateForm(false, false);

                if (isPageCreated)
                {
                    ActivePage.UpdateFormOnPageCreated();
                }
            }
        }

        /// <summary>
        /// ページサイズを最適化。
        /// </summary>
        public void OptimizePageSize(ObjectPropertyPage page)
        {
            if (page != null)
            {
                //page.Dock = DockStyle.Fill;

                // サイズ変更ページの場合かつデフォルトが設定されているときはデフォルトを使用
                Size defaultSize = page.DefaultPageSize;

                // TODO: 定数決めうちをどうするか
                int x = defaultSize.Width + 2;
                int y = defaultSize.Height + 2;
                for (Control parent = page; parent != null && parent != Owner; parent = parent.Parent)
                {
                    x += parent.Location.X;
                    y += parent.Location.Y;
                }

                // ダイアログ枠部分の考慮
                // 最小化されていると ClientSize から計算できないので事前計算したサイズを使用する
                Size size = new Size(TheApp.MainFrame.DialogFrameWidth + x, TheApp.MainFrame.DialogFrameHeight + y);
                if (Owner.Size != size)
                {
                    Owner.Size = size;
                }
            }
            else
            {
                // ページのデフォルトサイズが360なので、変更するときはここも変更する必要あり
                int x = 360 + 2;
                for (Control parent = spcClient.Panel2; parent != null && parent != Owner; parent = parent.Parent)
                {
                    x += parent.Location.X;
                }

                Size size = new Size(TheApp.MainFrame.DialogFrameWidth + x, Owner.MinimumSize.Height);
                if (Owner.Size != size)
                {
                    Owner.Size = size;
                }
            }
        }

        /// <summary>
        /// アクティブページのドッキングスタイルを更新。
        /// </summary>
        public void UpdateActivePageDockStyle()
        {
            if (ActivePage == null) { return; }

            if (!ActivePage.ChangeDockStyle)
            {
                return;
            }

            // TODO: 定数決めうちを回避する
            int x = Owner.ClientSize.Width - 6;
            int y = Owner.ClientSize.Height - 6;

            // 親コントロールを遡る
            for (Control parent = ActivePage; parent != null && parent != Owner; parent = parent.Parent)
            {
                x -= parent.Location.X;
                y -= parent.Location.Y;
            }

            DockStyle style = DockStyle.Fill;

            if (ActivePage.ShowParentScrollBar)
            {
                if (ActivePage.DefaultPageSize.Width > x)
                {
                    y -= SystemInformation.HorizontalScrollBarHeight;
                }
                if (ActivePage.DefaultPageSize.Height > y)
                {
                    x -= SystemInformation.VerticalScrollBarWidth;
                }
                if (ActivePage.DefaultPageSize.Width > x)
                {
                    style = DockStyle.Left;
                }
                if (ActivePage.DefaultPageSize.Height > y)
                {
                    style = (style == DockStyle.Fill) ? DockStyle.Top : DockStyle.None;
                }
            }

            ActivePage.Dock = style;
        }

        //---------------------------------------------------------------------
        // ドキュメントプロパティ変更
        protected virtual void Event_Document_PropertyChanged(object sender, IEnumerable<DocumentPropertyChangedArgs> e)
        {
            // 表示中なら
            if (Visible)
            {
                // パネル＋表示ページを更新
                UpdateForm(true, false);

                // カラーピッカーを更新
                symbolColorAdapter_.NotifyUpdate();
            }
        }



        private void ecmEditColorMark_MouseClick(object sender, MouseEventArgs e)
        {
            ColorPickerDialog.StartConnection(symbolColorAdapter_);
        }

        private void tbxComment_ValueChanged(object sender, EventArgs e)
        {
            var editables = Targets.Objects.Where(x => x.Editable);
            TheApp.CommandManager.Execute(CreateEditCommentEditCommand(new GuiObjectGroup(editables), tbxComment.Value));
        }

        private void tbxLabel_ValueChanged(object sender, EventArgs e)
        {
            var editables = Targets.Objects.Where(x => x.Editable);
            TheApp.CommandManager.Execute(CreateEditLabelEditCommand(new GuiObjectGroup(editables), tbxLabel.Value));
        }

        protected virtual void OnCategoryAfterSelect()
        {
            ;
        }

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public bool CanCopy()
        {
            return (ActivePage != null) ? ActivePage.CanCopy() : false;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public object Copy(ref object copyObjectInfo)
        {
            Debug.Assert(CanCopy());
            return (ActivePage != null) ? ActivePage.Copy(ref copyObjectInfo) : null;
        }

        /// <summary>
        /// ペーストが有効か。
        /// </summary>
        public bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            Debug.Assert(ActivePage != null);
            return ActivePage.CanPaste(copiedObjectInfo, copiedObject);
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public void Paste(object pasteObject)
        {
            Debug.Assert((ActivePage != null) && (pasteObject != null));
            ActivePage.Paste(pasteObject);
        }
        #endregion

        public void SetCategoryMinimumSize()
        {
            // 最小化しているときは spcClient で例外がおきるので何もしない。
            if (Owner.WindowState == FormWindowState.Minimized)
            {
                return;
            }

            int width = 0;
            {
                using(var g = CategoryView.CreateGraphics())
                {
                    foreach (TreeNode node in CategoryView.Nodes)
                    {
                        width = Math.Max(width, MakeMaxNodeWidth(node, 0, g, CategoryView.Font));
                    }
                }
            }

            width += 4 + ModifiedMark.StarMark.Width;

            int oldWidth = CategoryView.Width;

            spcClient.Panel1MinSize = width;

            // サイズが変化したらその分ダイアログのサイズを変える。
            int diff = CategoryView.Width - oldWidth;
            if (diff != 0)
            {
                Owner.Width += diff;
            }
        }

        private int MakeMaxNodeWidth(TreeNode node, int depth, Graphics g, Font font)
        {
            const int xBase		= 24;	// 開始値
            const int xWidth	= 19;	// depthごとの幅

            int width = 0;
            {
                Debug.Assert(node is TreeNode);

                var cNode = node as TreeNode;

                int textWidth    = TextRenderer.MeasureText(g, cNode.Text, font).Width;
                int currentWidth = textWidth + xBase + xWidth * depth;

                width = Math.Max(width, currentWidth);
            }

            foreach (TreeNode childWode in node.Nodes)
            {
                width = Math.Max(width, MakeMaxNodeWidth(childWode, depth + 1, g, font));
            }

            return width;
        }

        public virtual HelpUtility.PageKey HelpKey
        {
            get
            {
                return HelpUtility.PageKey.p_property_window;
            }
        }
    }

    #region CreatePropertyPageHandler
    /// <summary>
    /// プロパティページ作成ハンドラデリゲート。
    /// </summary>
    public delegate ObjectPropertyPage CreatePropertyPageHandler(object arg);
    #endregion
}
