﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Data;
using App.Utility;

namespace App.PropertyEdit
{
    public static class CurveEditorUtility
    {
        public const double ScaleMinX = 0.001;
        public const double ScaleMinY = 0.00001;
        public const double ScaleMax = 50000.0;
        public static double ClampScaleX(double src)
        {
            return Math.Min(Math.Max(src, ScaleMinX), ScaleMax);
        }
        public static double ClampScaleY(double src)
        {
            return Math.Min(Math.Max(src, ScaleMinY), ScaleMax);
        }

        public delegate bool CollectNodePredicateDelegate(TreeNode node);

        // 指定ツリーノード以下に割り当てられているカーブを収集します。
        public static List<TreeNode> CollectNode(TreeNode node, CollectNodePredicateDelegate collectNodePredicate = null, bool isIncludeNullNode = false)
        {
            List<TreeNode> curves = new List<TreeNode>();
            {
                if (isIncludeNullNode || (node.Tag != null))
                {
                    if ((collectNodePredicate == null) || collectNodePredicate(node))
                    {
                        curves.Add(node);
                    }
                }

                foreach (TreeNode childNode in node.Nodes)
                {
                    curves.AddRange(CollectNode(childNode, collectNodePredicate, isIncludeNullNode));
                }
            }
            return curves;
        }

        public static bool MakeMinMaxFrame(IAnimationCurve curve, out float min, out float max)
        {
            var keyFrames = curve.KeyFrames;
            if (keyFrames.Any())
            {
                min = keyFrames.First().Frame;
                max = keyFrames.Last().Frame;
                return true;
            }
            else
            {
                min = 0.0f;
                max = 0.0f;
                return false;
            }
        }

        public static bool MakeMinMaxValue(IAnimationCurve curve, out float min, out float max)
        {
            var keyFrames = curve.KeyFrames;
            if (keyFrames.Any())
            {
                min =  float.MaxValue;
                max = -float.MaxValue;

                var curveType = curve.CurveInterpolationType;

                for(int i = 0;i != keyFrames.Count();++ i)
                {
                    var key = keyFrames[i];

                    switch(curveType)
                    {
                        case InterpolationType.Hermite:
                        {
                            if (i == (keyFrames.Count() - 1))
                            {
                                min = Math.Min(min, key.Value);
                                max = Math.Max(max, key.Value);
                            }
                            else
                            {
                                var left  = key;
                                var right = keyFrames[i + 1];

                                float tempMin, tempMax;
                                MathUtility.HermiteMinMax(left, right, out tempMin, out tempMax);

                                min = Math.Min(min, tempMin);
                                max = Math.Max(max, tempMax);
                            }

                            break;
                        }

                        case InterpolationType.Linear:
                        case InterpolationType.Step:
                        {
                            min = Math.Min(min, key.Value);
                            max = Math.Max(max, key.Value);

                            break;
                        }
                    }
                }

                return true;
            }
            else
            {
                min = 0.0f;
                max = 0.0f;
                return false;
            }
        }

        private static bool IsInSelection(PointF p, AnimationCurveSelectionInfo info)
        {
            return (p.X >= info.SelectionLeft && p.X <= info.SelectionRight
                    && p.Y >= info.SelectionTop && p.Y <= info.SelectionBottom);
        }

        // bezier 対応版
        public static bool IsHit(IAnimationCurve curve, AnimationCurveSelectionInfo info)
        {
            if (curve.KeyFrames.Count == 0)
            {
                return false;
            }
            var oldDrawCurvePosition = info.Painter.OldDrawCurvePositions.FirstOrDefault(c => c.Curve == curve);

            if (oldDrawCurvePosition == null) return false;
            if (oldDrawCurvePosition.Positions.Count < 4) return false;

            var left = info.SelectionLeft;
            var top = info.SelectionTop;
            var right = info.SelectionRight;
            var bottom = info.SelectionBottom;

            for (var i = 0; i < oldDrawCurvePosition.Positions.Count ; i += 4) // bezier
            {
                var p0 = oldDrawCurvePosition.Positions[i];
                var c0 = oldDrawCurvePosition.Positions[i + 1];
                var c1 = oldDrawCurvePosition.Positions[i + 2];
                var p1 = oldDrawCurvePosition.Positions[i + 3];

                // 二点ともX範囲外
                if (((p0.X < left) && (p1.X < left)) ||
                    ((p0.X > right) && (p1.X > right)))
                //((p0.Y < top   ) && (p1.Y < top   )) ||
                //((p0.Y > bottom) && (p1.Y > bottom)))
                {
                    continue;
                }

                // 二点とも選択範囲内
                if (IsInSelection(p0, info) && IsInSelection(p1, info))
                {
                    return true;
                }


                if (p0.X == p1.X)
                {
                    if ((p0.X >= left) && (p0.X <= right))
                    {
                        return true;
                    }
                }
                else
                {
                    // bezierからエルミートに変換
                    // 制御点からスロープを計算
                    var ep0 = new PointF((float)info.View.GetXInCurvePlane((int)p0.X), (float)info.View.GetYInCurvePlane((int)p0.Y));
                    var ep1 = new PointF((float)info.View.GetXInCurvePlane((int)p1.X), (float)info.View.GetYInCurvePlane((int)p1.Y));
                    var ec0 = new PointF((float)info.View.GetXInCurvePlane((int)c0.X), (float)info.View.GetYInCurvePlane((int)c0.Y));
                    var ec1 = new PointF((float)info.View.GetXInCurvePlane((int)c1.X), (float)info.View.GetYInCurvePlane((int)c1.Y));

                    var key0 = new KeyFrame { Frame = ep0.X, Value = ep0.Y, OutSlope = (ec0.Y - ep0.Y)/(ec0.X - ep0.X)};
                    var key1 = new KeyFrame { Frame = ep1.X, Value = ep1.Y, InSlope = (ec1.Y - ep1.Y) / (ec1.X - ep1.X) };

                    // 左右の値を求める(選択範囲でクランプ)
                    var leftValue = MathUtility.HermiteInterpolation((float)(info.MinFrame > key0.Frame? info.MinFrame : key0.Frame), key0, key1);
                    var rightValue = MathUtility.HermiteInterpolation((float)(info.MaxFrame < key1.Frame ? info.MaxFrame : key1.Frame), key0, key1);

                    // 左右の値どちらかが範囲内ならtrue
                    if (leftValue >= info.MinValue && leftValue <= info.MaxValue)
                    {
                        return true;
                    }
                    if (rightValue >= info.MinValue && rightValue <= info.MaxValue)
                    {
                        return true;
                    }
                    // 左右の値が範囲外で、上下反対方向に有る場合は範囲内を通るのでtrue
                    if (leftValue < info.MinValue && rightValue > info.MaxValue)
                    {
                        return true;
                    }
                    if (rightValue < info.MinValue && leftValue > info.MaxValue)
                    {
                        return true;
                    }
                    // 左右の値が範囲外で、上下同じ方向に有りMinMax値が選択範囲を交差する場合はtrue
                    KeyValuePair<float, float> min, max;
                    MathUtility.HermiteMinMax(key0, key1, out min, out max, (float)info.MinFrame, (float)info.MaxFrame);
                    if (leftValue < info.MinValue && rightValue < info.MinValue
                        && max.Key > key0.Frame && max.Key < key1.Frame && max.Value > info.MinValue)
                    {
                        return true;
                    }
                    if (leftValue > info.MaxValue && rightValue > info.MaxValue
                        && min.Key > key0.Frame && min.Key < key1.Frame && min.Value < info.MaxValue)
                    {
                        return true;
                    }

                }
            }

            return false;
        }

        public static bool IsHit(KeyFrame key, AnimationCurveSelectionInfo info)
        {
            double frame = key.Frame;
            double value = key.Value;

            if (info.MinFrame <= frame && frame <= info.MaxFrame &&
                info.MinValue <= value && value <= info.MaxValue)
            {
                return true;
            }

            return false;
        }

        public static bool IsHitScreen(float p0x, float p0y, float p1x, float p1y, AnimationCurveSelectionInfo info)
        {
            return IsHit(p0x, p0y, p1x, p1y, info.SelectionLeft, info.SelectionTop, info.SelectionRight, info.SelectionBottom);
        }

        public static bool IsHit(float p0x, float p0y, float p1x, float p1y, float left, float top, float right, float bottom)
        {
            if (((p0x < left) && (p1x < left)) ||
                                ((p0x > right) && (p1x > right)) ||
                                ((p0y < top) && (p1y < top)) ||
                                ((p0y > bottom) && (p1y > bottom)))
            {
                return false;
            }
            else
            {
                if (p0x == p1x)
                {
                    if ((p0x >= left) && (p0x <= right))
                    {
                        return true;
                    }
                }
                else
                {
                    // y = a * x + b

                    float dx = p1x - p0x;
                    float dy = p1y - p0y;

                    float a = dy / dx;
                    float b = (0.0f - p0x) * a + p0y;

                    float r0x = left;
                    float r0y = top;
                    float r1x = right;
                    float r1y = top;
                    float r2x = left;
                    float r2y = bottom;
                    float r3x = right;
                    float r3y = bottom;

                    if (((r0y > a * r0x + b) && (r1y > a * r1x + b) && (r2y > a * r2x + b) && (r3y > a * r3x + b)) ||
                        ((r0y < a * r0x + b) && (r1y < a * r1x + b) && (r2y < a * r2x + b) && (r3y < a * r3x + b)))
                    {
                        return false;
                    }
                    else
                    {
                        return true;
                    }
                }
            }

            return false;
        }
    }
}
