﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.Data;
using ConfigCommon;

namespace App.PropertyEdit
{
    public class CurveEditorCopyData
    {
        private class Data
        {
            public IAnimationCurve	Curve{		get; set; }
            public int				CurveIndex{	get; set; }
            public KeyFrame			Key{		get; set; }
            public InterpolationType Interpolation { get; set; }
        };

        private readonly List<Data>	data_ = new List<Data>();

        public void CopyKeys(List<IAnimationCurve> sourceCurves)
        {
            data_.Clear();

            sourceCurves.ForEach(x => x.SortByFrame());

            int curveIndex = 0;

            foreach (var curve in sourceCurves)
            {
                bool hasSelectedKey = false;

                foreach (var key in curve.KeyFrames)
                {
                    if (key.Selected)
                    {
                        data_.Add(
                            new Data()
                            {
                                Curve = curve,
                                CurveIndex	= curveIndex,
                                Key			= key.Clone(),
                                Interpolation = curve.CurveInterpolationType,
                            }
                        );

                        hasSelectedKey = true;
                    }
                }

                if (hasSelectedKey)
                {
                    ++ curveIndex;
                }
            }
        }

        private ICommand CreatePasteKeysCommand(
            GuiObjectGroup targetGroup,
            GuiObjectID targetGuiObjectID,
            IAnimationCurve target,
            int targetIndex,
            float currentFrame,
            ConfigCommon.PasteMethodType pasteMethod,
            ConfigCommon.PastePositionType pastePosition)
        {
            var data =
                (from d in data_
                 where d.CurveIndex == targetIndex
                 select d).ToList();

            if (data.Any() == false)
            {
                return null;
            }

            Debug.Assert(data.Any());

            var keys = new List<KeyFrame>();
            {
                if (target.KeyFrames != null)
                {
                    target.KeyFrames.ForEach(x => keys.Add(x.Clone()));
                }
            }

            int index = 0;
            float minFrame = +float.MaxValue;
            float maxFrame = -float.MaxValue;
            {
                data.ForEach(
                    x =>
                    {
                        minFrame = Math.Min(minFrame, x.Key.Frame);
                        maxFrame = Math.Max(maxFrame, x.Key.Frame);
                        x.Key.Index = index++;
                    }
                );

                float offsetFrame = currentFrame - data[0].Key.Frame;
                minFrame += offsetFrame;
                maxFrame += offsetFrame;
            }
            float frameLength = maxFrame - minFrame;

            switch(pasteMethod)
            {
                case ConfigCommon.PasteMethodType.Insert:
                {
                    keys.ForEach(
                        x =>
                        {
                            if (x.Frame >= minFrame)
                            {
                                x.Frame += frameLength;
                                x.Index = index++;
                            }
                        }
                    );

                    break;
                }

                case ConfigCommon.PasteMethodType.Displacement:
                {
                    keys.RemoveAll(x => (x.Frame >= minFrame) && (x.Frame <= maxFrame));
                    break;
                }

                case ConfigCommon.PasteMethodType.Merge:
                {
                    keys.ForEach(
                        x =>
                        {
                            x.Index = index++;
                        }
                    );
                    break;
                }
            }

            InterpolationType? interpolation = null;
            {
                float offsetFrame = currentFrame - data[0].Key.Frame;
                var anim = targetGroup.GetObjects(targetGuiObjectID).FirstOrDefault();
                var curveAnimtarget = target.GetAnimTarget(anim);

//				float offsetValue = target.KeyFrames.Any() ? target.GetValue(currentFrame, curveAnimtarget.pre_wrap, curveAnimtarget.post_wrap) - data[0].Key.Value : 0;
                float offsetValue = 0.0f;
                {
                    switch(pastePosition)
                    {
                        case ConfigCommon.PastePositionType.Relative:
                        {
                            offsetValue = target.KeyFrames.Any() ? target.GetValue(currentFrame, curveAnimtarget.pre_wrap, curveAnimtarget.post_wrap) - data[0].Key.Value : 0;
                            break;
                        }

                        case ConfigCommon.PastePositionType.Absolute:
                        {
                            offsetValue = 0.0f;
                            break;
                        }

                        default:
                        {
                            Debug.Assert(false);
                            break;
                        }
                    }
                }

                data.ForEach(
                    x =>
                    {
                        var key = x.Key.Clone();

                        key.Frame += offsetFrame;
                        key.Value += offsetValue;

                        // 整数のときは丸める
                        switch (target.CurvePrimitiveType)
                        {
                            case PrimitiveTypeKind.Bool:
                                key.Value = (key.Value >= 0.5f) ? 1.0f : 0.0f;
                                break;
                            case PrimitiveTypeKind.Int:
                                key.Value =(float)Math.Round(key.Value, MidpointRounding.AwayFromZero);
                                break;
                            case PrimitiveTypeKind.Uint:
                                key.Value =(float)Math.Max(Math.Round(key.Value, MidpointRounding.AwayFromZero), 0);
                                break;
                        }

                        keys.Add(key);
                    }
                );

                if (data.Count >= 2 &&
                    curveAnimtarget.CurveInterpolationType != data[0].Interpolation &&
                    curveAnimtarget.KeyFrames.Count <= 1 &&
                    (data[0].Interpolation == InterpolationType.Step ||
                    target.CurvePrimitiveType == PrimitiveTypeKind.Float))
                {
                    interpolation = data[0].Interpolation;
                }
            }

            return
                AnimationCurveEditCommand.Create(
                    targetGroup,
                    targetGuiObjectID,
                    target,
                    keys,
                    interpolation
                );
        }

        public EditCommandSet CreatePasteKeysCommand(
            GuiObjectGroup targetGroup,
            GuiObjectID targetGuiObjectID,
            List<IAnimationCurve> targets,
            CurveView curveView,
            int mouseX,
            int mouseY,
            float currentFrame,
            ConfigCommon.PasteMethodType pasteMethod,
            ConfigCommon.PastePositionType pastePosition)
        {
            var commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);

            if (data_.Any())
            {
                int targetIndex = 0;

                foreach (var target in targets)
                {
                    var command = CreatePasteKeysCommand(
                        targetGroup,
                        targetGuiObjectID,
                        target,
                        targetIndex,
                        currentFrame,
                        pasteMethod,
                        pastePosition
                    );

                    if (command != null)
                    {
                        commandSet.Add(command);
                    }

                    ++ targetIndex;
                }
            }

            return commandSet;
        }
    }
}
