﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public static class AnimationCurveEditCommand
    {
        public static ICommand Create(GuiObjectGroup targetGroup, GuiObjectID targetGuiObjectID, IAnimationCurve curve, List<KeyFrame> keys, InterpolationType? type = null, bool notReloadFirst = false)
        {
            KeyFramesUtility.SortByFrame(keys);

            // TODO: 削除
            // 作成 Undo 対象外!
            foreach (var target in targetGroup.GetObjects(targetGuiObjectID))
            {
                if (curve.GetAnimTarget(target) == null)
                {
                    Debug.Assert(false);
                    curve.CreateAnimTarget(target);
                }
            }

            var commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(AnimationMessageFilter);
            if (type.HasValue)
            {
                commandSet.Add(
                    CreateInterpolationTypeEditCommand(
                        targetGroup,
                        targetGuiObjectID,
                        curve,
                        type.Value,
                        notReloadFirst));
            }

            commandSet.Add(CreateKeyFramesEditCommand(targetGroup, targetGuiObjectID, curve, keys, notReloadFirst));
            commandSet.Add(new LazyCommand(() => CreateQuantizeEditCommand(targetGroup, targetGuiObjectID, curve)));

            return commandSet;
        }

        public static void AnimationMessageFilter(List<Viewer.BaseMessage> messages)
        {
            var reloadAnimations = new HashSet<AnimationDocument>();

            // リロードの位置
            for (int i = messages.Count - 1; i >= 0 ; i--)
            {
                var reload = messages[i] as Viewer.LoadOrReloadAnimation;
                if (reload != null)
                {
                    if (!reloadAnimations.Contains(reload.OriginalTarget))
                    {
                        reloadAnimations.Add(reload.OriginalTarget);
                        messages[i] = (new Viewer.LoadOrReloadAnimation(reload.OriginalTarget));
                    }
                    else
                    {
                        messages[i] = null;
                    }
                }
            }

            for (int i = 0; i < messages.Count; i++)
            {
                var edit = messages[i] as Viewer.EditCurve;
                if (edit != null)
                {
                    if (reloadAnimations.Contains(edit.document_))
                    {
                        messages[i] = null;
                    }

                    if (messages[i] != null)
                    {
                        for (int j = i + 1; j < messages.Count; j++)
                        {
                            var following = messages[j] as Viewer.EditCurve;
                            if (following != null &&
                                edit.document_ == following.document_ &&
                                edit.parentIndex_ == following.parentIndex_ &&
                                edit.curveIndex_ == following.curveIndex_ &&
                                edit.curveComponentIndex_ == following.curveComponentIndex_)
                            {
                                messages[i] = null;
                            }
                        }
                    }
                }
            }

            var tmp = messages.Where(x => x != null).ToArray();
            messages.Clear();
            messages.AddRange(tmp);
        }

        public static ICommand CreateInterpolationTypeEditCommand(GuiObjectGroup targetGroup, GuiObjectID targetGuiObjectID, IAnimationCurve curve, InterpolationType type, bool notReloadFirst = false)
        {
            return new GeneralGroupValueEditCommand<InterpolationType>(
                targetGroup,
                targetGuiObjectID,
                type,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var dst = curve.GetAnimTarget(target);
                    var oldInterpolationType = dst.CurveInterpolationType;
                    swap = dst.CurveInterpolationType;
                    dst.CurveInterpolationType = (InterpolationType)data;

                    // Viewerへ再転送
                    if (oldInterpolationType != dst.CurveInterpolationType && !notReloadFirst)
                    {
                        var anim = target.OwnerDocument as AnimationDocument;

                        Viewer.LoadOrReloadAnimation.Send(anim);
                    }
                },
                postEditDelegate: (s, e) => { notReloadFirst = false; }
            );
        }

        public static ICommand CreateKeyFramesEditCommand(GuiObjectGroup targetGroup, GuiObjectID targetGuiObjectID, IAnimationCurve curve, List<KeyFrame> keys, bool notReloadFirst = false)
        {
            return new GeneralGroupReferenceEditCommand<List<KeyFrame>>(
                targetGroup,
                targetGuiObjectID,
                ObjectUtility.MultipleClone(keys, targetGroup.GetObjects(targetGuiObjectID).Count),
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var paramAnim = curve.GetAnimTarget(target);
                    CurveExportType oldType = paramAnim.ExportType;
                    swap = paramAnim.KeyFrames;
                    paramAnim.KeyFrames = (List<KeyFrame>)data;

                    curve.UpdateIsModified(target);

                    paramAnim.IsFileOutputable = MakeIsFileOutputable(target, curve);

                    // 編集後の出力形式に合わせて転送を変える
                    CurveExportType type = paramAnim.ExportType;
                    if ((oldType == CurveExportType.Ignored || type == CurveExportType.Ignored || type == CurveExportType.Dependent) && !notReloadFirst)
                    {
                        var anim = target.OwnerDocument as AnimationDocument;

                        // 不正アニメーションチェック
                        anim.CheckAndDisConnect();

                        Viewer.LoadOrReloadAnimation.Send(anim);
                    }
                    else
                    {
                        var anim = target.OwnerDocument as AnimationDocument;

                        // 不正アニメーションチェック
                        anim.CheckAndDisConnect();
                        CurveEditorPanel.SendViewerEditCurve(target, curve, type);
                    }
                },
                postEditDelegate: (s, e) => { notReloadFirst = false;  });
        }

        private class QuantizeAnalysisData
        {
            public AnimTarget target;
            public AnimationDocument.QuantizationInfo info;
        }

        public static EditCommand CreateQuantizeAnalyseAllCommand(GuiObject activeTarget, bool reload)
        {
            Debug.Assert(activeTarget is IQuantizeAnalysis);
            var results = (activeTarget as IQuantizeAnalysis).QuantizeAnalyse().Select(x =>
                {
                    var info = new AnimationDocument.QuantizationInfo();
                    if (x.Item2 != null)
                    {
                        info.frame_type = x.Item2.frameType;
                        info.key_type = x.Item2.keyType;
                        info.offset = x.Item2.offset;
                        info.scale = x.Item2.scale;
                    }
                    return new QuantizeAnalysisData { target = x.Item1, info = info };
                }).ToArray();
            return new GeneralGroupValueEditCommand<int>(
                new GuiObjectGroup(activeTarget),
                null,
                0,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    foreach (var result in results)
                    {
                        var tmp = result.target.QuantizationInfo;
                        result.target.QuantizationInfo = result.info;
                        result.info = tmp;
                    }

                    UpdateIsModifiedAnimTargetAll(target);

                    if (reload && target.OwnerDocument is AnimationDocument)
                    {
                        Viewer.LoadOrReloadAnimation.Send((AnimationDocument)target.OwnerDocument);
                    }
                });
        }

        public static void UpdateIsModifiedAnimTargetAll(App.Data.GuiObject obj)
        {
            var objId = obj.ObjectID;
            switch (objId)
            {
                case GuiObjectID.CameraAnimation:
                    {
                        var anim = obj as CameraAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }
                case GuiObjectID.FogAnimation:
                    {
                        var anim = obj as FogAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }
                case GuiObjectID.LightAnimation:
                    {
                        var anim = obj as LightAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }

                case GuiObjectID.BoneVisibilityAnimation:
                    {
                        var anim = obj as BoneVisibilityAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }
                case GuiObjectID.MaterialVisibilityAnimation:
                    {
                        var anim = obj as MaterialVisibilityAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }
                case GuiObjectID.TexturePatternAnimation:
                    {
                        var anim = obj as TexturePatternAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }
                case GuiObjectID.ShapeAnimation:
                    {
                        var anim = obj as ShapeAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }
                case GuiObjectID.SkeletalAnimation:
                    {
                        var anim = obj as SkeletalAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                    {
                        var anim = obj as ShaderParameterAnimation;
                        if (anim != null)
                        {
                            anim.UpdateIsModifiedAnimTargetAll();
                        }
                        break;
                    }
            }
        }

        public static bool MakeIsFileOutputable(GuiObject target, IAnimationCurve curve)
        {
            try
            {
                AnimationCurveEditCommand.MakeQuantizationAnalysis(target, target.ObjectID, curve, false);
            }
            catch(Exception)
            {
                return false;
            }

            return true;
        }

        public static EditCommand CreateQuantizeEditCommand(GuiObjectGroup targetGroup, GuiObjectID targetGuiObjectID, IAnimationCurve curve)
        {
            var results = new List<AnimationDocument.QuantizationInfo>();
            foreach (var target in targetGroup.GetObjects(targetGuiObjectID))
            {

                var result = MakeQuantizationAnalysis(target, targetGuiObjectID, curve);
                if (result != null)
                {
                    results.Add(new AnimationDocument.QuantizationInfo()
                    {
                        frame_type = result.frameType,
                        key_type = result.keyType,
                        offset = result.offset,
                        scale = result.scale,
                    });
                }
                else
                {
                    results.Add(curve.GetAnimTarget(target).QuantizationInfo);
                }
            }

            return new GeneralGroupReferenceEditCommand<AnimationDocument.QuantizationInfo>(
                targetGroup,
                targetGuiObjectID,
                results,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var paramAnim = curve.GetAnimTarget(target);
                    swap = paramAnim.QuantizationInfo;
                    paramAnim.QuantizationInfo = (AnimationDocument.QuantizationInfo)data;
                    curve.UpdateIsModified(target);
                    CurveEditorPanel.SendViewerEditCurve(target, curve, paramAnim.ExportType);
                });
        }

        public static TexturePatternAnimation.PatternAnimTarget CreateTemporaryPatternAnimTarget(string sampler_name)
        {
            return
                new TexturePatternAnimation.PatternAnimTarget()
                {
                    sampler_name	= sampler_name,
                    hint			= ""
                };
        }

        public static IfQuantizationAnalysisResult MakeQuantizationAnalysis(GuiObject target, GuiObjectID targetGuiObjectID, IAnimationCurve srcCurve, bool isIgnoreException = true)
        {
            IfQuantizationAnalysisResult result = new IfQuantizationAnalysisResult();
            {
                IfAnimCurve.CurveType curveType;
                {
                    switch (srcCurve.CurvePrimitiveType)
                    {
                        case PrimitiveTypeKind.Bool:
                            curveType = IfAnimCurve.CurveType.Bool;
                            break;
                        case PrimitiveTypeKind.Int:
                        case PrimitiveTypeKind.Uint:
                            curveType = IfAnimCurve.CurveType.Int;
                            break;
                        case PrimitiveTypeKind.Float:
                            curveType = IfAnimCurve.CurveType.Float;
                            break;
                        default:
                            throw new NotImplementedException();
                    }
                }

                var ifAnimCurve = new IfAnimCurve()
                {
                    curveType = curveType,
                };

                string parentName			= srcCurve.ParentName;
                string curveName			= srcCurve.Name;

                try
                {
                    // 注意：	このコードを参考に ***Animation.IsFileOutputable を実装しています。
                    //			修正したら確認する

                    switch(targetGuiObjectID)
                    {
                        case GuiObjectID.ShaderParameterAnimation:
                        case GuiObjectID.ColorAnimation:
                        case GuiObjectID.TextureSrtAnimation:
                        {
                            var anim = target as ShaderParameterAnimation;
                            var animTarget = (ShaderParameterAnimation.ParamAnimTarget)srcCurve.GetAnimTarget(target);

                            G3dStream stream;
                            var anim_target = animTarget.ConvertTo_param_anim_targetType(out stream);
                            if (anim_target.Curve != null)
                            {
                                ifAnimCurve.BuildIfAnimCurve(anim_target.Curve, srcCurve.IsRotate, stream);

                                result = IfShaderParamAnimQuantizationAnalysis.Analyse(
                                    anim.Data.shader_param_anim_info,
                                    ((ShaderParameterAnimationCurveTreeNodeInfo)srcCurve).UniformVar.type,
                                    animTarget.component_index,
                                    ifAnimCurve,
                                    true
                                );
                            }

                            break;
                        }

                        case GuiObjectID.MaterialVisibilityAnimation:
                        {
                            var anim	= target as MaterialVisibilityAnimation;
                            anim.UpdateData();

                            var matVisMatAnimTarget =
                                (anim.Data.mat_vis_mat_anim_array != null) ?
                                    anim.Data.mat_vis_mat_anim_array.mat_vis_mat_anim.
                                    Where(x => x.mat_name == parentName).
                                    FirstOrDefault() :
                                    null;

                            if (matVisMatAnimTarget != null)
                            {
                                if (matVisMatAnimTarget.step_curve != null)
                                {
                                    ifAnimCurve.BuildIfAnimCurve(matVisMatAnimTarget.step_curve, srcCurve.IsRotate, anim.BinaryStreams);

                                    result = IfMatVisibilityAnimQuantizationAnalysis.Analyse(
                                        anim.Data.mat_visibility_anim_info,
                                        ifAnimCurve,
                                        true
                                    );
                                }
                            }

                            break;
                        }

                        case GuiObjectID.BoneVisibilityAnimation:
                        {
                            var anim	= target as BoneVisibilityAnimation;
                            anim.UpdateData();

                            var boneVisBoneAnimTarget =
                                (anim.Data.bone_vis_bone_anim_array != null) ?
                                    anim.Data.bone_vis_bone_anim_array.bone_vis_bone_anim.
                                    Where(x => x.bone_name == parentName).
                                    FirstOrDefault() :
                                    null;

                            if (boneVisBoneAnimTarget != null)
                            {
                                if (boneVisBoneAnimTarget.step_curve != null)
                                {
                                    ifAnimCurve.BuildIfAnimCurve(boneVisBoneAnimTarget.step_curve, srcCurve.IsRotate, anim.BinaryStreams);

                                    result = IfBoneVisibilityAnimQuantizationAnalysis.Analyse(
                                        anim.Data.bone_visibility_anim_info,
                                        ifAnimCurve,
                                        true
                                    );
                                }
                            }

                            break;
                        }

                        case GuiObjectID.TexturePatternAnimation:
                        {
                            var anim	= target as TexturePatternAnimation;
                            anim.UpdateData();

                            if (anim.Data.tex_pattern_mat_anim_array == null)
                            {
                                break;
                            }

                            var tex_pattern_mat_anim = anim.Data.tex_pattern_mat_anim_array.tex_pattern_mat_anim.FirstOrDefault(x => x.mat_name == parentName);
                            if (tex_pattern_mat_anim == null || tex_pattern_mat_anim.pattern_anim_target == null)
                            {
                                break;
                            }

                            var texPatternMatAnimTarget = tex_pattern_mat_anim.pattern_anim_target.FirstOrDefault(x => x.sampler_name == curveName);
                            if (texPatternMatAnimTarget == null)
                            {
                                break;
                            }

                            if (texPatternMatAnimTarget.step_curve != null)
                            {
                                ifAnimCurve.BuildIfAnimCurve(texPatternMatAnimTarget.step_curve, srcCurve.IsRotate, anim.BinaryStreams);

                                result = IfTexPatternAnimQuantizationAnalysis.Analyse(
                                    anim.Data.tex_pattern_anim_info,
                                    ifAnimCurve,
                                    true
                                );
                            }

                            break;
                        }
                        case GuiObjectID.CameraAnimation:
                        {
                            var anim = target as CameraAnimation;
                            var animTarget = (CameraAnimationTarget)srcCurve.GetAnimTarget(target);

                            G3dStream stream;
                            camera_anim_targetType anim_target = animTarget.ConvertTo_camera_animType(out stream);
                            if (anim_target.Curve != null)
                            {
                                ifAnimCurve.BuildIfAnimCurve(anim_target.Curve, srcCurve.IsRotate, stream);

                                result = IfSceneAnimQuantizationAnalysis.Analyse(
                                    anim.Owner.Data.scene_anim_info,
                                    animTarget.targetType,
                                    ifAnimCurve,
                                    true
                                );
                            }

                            break;
                        }
                        case GuiObjectID.LightAnimation:
                        {
                            var anim = target as LightAnimation;
                            var animTarget = (LightAnimationTarget)srcCurve.GetAnimTarget(target);

                            G3dStream stream;
                            light_anim_targetType anim_target = animTarget.ConvertTo_light_animType(out stream, anim.Data.type);
                            if (anim_target.Curve != null)
                            {
                                ifAnimCurve.BuildIfAnimCurve(anim_target.Curve, srcCurve.IsRotate, stream);

                                result = IfSceneAnimQuantizationAnalysis.Analyse(
                                    anim.Owner.Data.scene_anim_info,
                                    animTarget.targetType,
                                    ifAnimCurve,
                                    true
                                );
                            }

                            break;
                        }
                        case GuiObjectID.FogAnimation:
                        {
                            var anim = target as FogAnimation;
                            var animTarget = (FogAnimationTarget)srcCurve.GetAnimTarget(target);

                            G3dStream stream;
                            fog_anim_targetType anim_target = animTarget.ConvertTo_fog_animType(out stream);
                            if (anim_target.Curve != null)
                            {
                                ifAnimCurve.BuildIfAnimCurve(anim_target.Curve, srcCurve.IsRotate, stream);

                                result = IfSceneAnimQuantizationAnalysis.Analyse(
                                    anim.Owner.Data.scene_anim_info,
                                    animTarget.targetType,
                                    ifAnimCurve,
                                    true
                                );
                            }

                            break;
                        }
                        case GuiObjectID.ShapeAnimation:
                        {
                            var anim = target as ShapeAnimation;
                            var animTarget = (ShapeAnimation.ShapeAnim)srcCurve.GetAnimTarget(target);

                            G3dStream stream;
                            shape_anim_targetType anim_target = animTarget.ConvertTo_shape_anim_target(out stream);
                            if (anim_target.Curve != null)
                            {
                                ifAnimCurve.BuildIfAnimCurve(anim_target.Curve, srcCurve.IsRotate, stream);

                                result = IfShapeAnimQuantizationAnalysis.Analyse(
                                    anim.Data.shape_anim_info,
                                    ifAnimCurve,
                                    true
                                );
                            }

                            break;
                        }
                        case GuiObjectID.MaterialAnimation:
                        {
                            var anim = target as MaterialAnimation;
                            var animTarget = srcCurve.GetAnimTarget(target);
                            var paramAnimTarget = animTarget as ShaderParameterAnimation.ParamAnimTarget;
                            if (paramAnimTarget != null)
                            {
                                G3dStream stream;
                                var anim_target = paramAnimTarget.ConvertTo_param_anim_targetType(out stream);
                                if (anim_target.Curve != null)
                                {
                                    ifAnimCurve.BuildIfAnimCurve(anim_target.Curve, srcCurve.IsRotate, stream);

                                    result = IfMaterialAnimQuantizationAnalysis.AnalyseShaderParameterCurve(
                                        anim.Data.material_anim_info,
                                        ((ShaderParameterAnimationCurveTreeNodeInfo2)srcCurve).UniformVar.type,
                                        paramAnimTarget.component_index,
                                        ifAnimCurve,
                                        true
                                    );
                                }
                                break;
                            }
                            var patternAnimTarget = animTarget as TexturePatternAnimation.PatternAnimTarget;
                            if (patternAnimTarget != null)
                            {
                                G3dStream stream;
                                var anim_target = patternAnimTarget.ConvertTo_pattern_anim_targetType(out stream);
                                if (anim_target.Curve != null)
                                {
                                    ifAnimCurve.BuildIfAnimCurve(anim_target.Curve, srcCurve.IsRotate, stream);

                                    result = IfMaterialAnimQuantizationAnalysis.AnalyseTexPatternCurve(
                                        anim.Data.material_anim_info,
                                        ifAnimCurve,
                                        true);
                                }
                                break;
                            }
                            var visibilityAnimTarget = animTarget as MaterialVisibilityAnimation.MaterialVisibilityMatAnim;
                            if (visibilityAnimTarget != null)
                            {
                                G3dStream stream;
                                step_curveType curve;
                                visibilityAnimTarget.Convert(out curve, out stream, false);
                                if (curve != null)
                                {
                                    ifAnimCurve.BuildIfAnimCurve(curve, srcCurve.IsRotate, stream);

                                    result = IfMaterialAnimQuantizationAnalysis.AnalyseMaterialVisibilityCurve(
                                        anim.Data.material_anim_info,
                                        ifAnimCurve,
                                        true);
                                }
                                break;
                            }

                            Debug.Assert(false);
                            break;
                        }
                        default:
                        Debug.Assert(false);
                        break;
                    }
                }
                catch(Exception e)
                {
                    // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1297
                    // nw.g3d.iflib.IfAnimCurveUtility.EvalQuantizeFrame() で例外になることがある。

                    DebugConsole.WriteLine("Exception Catch : AnimationCurveEditCommand.MakeQuantizationAnalysis() : {0}", e.Message);

                    if (isIgnoreException)
                    {
                        // 無視する
                        ;
                    }
                    else
                    {
                        // 再送する
                        throw;
                    }
                }
            }

            // アニメーション量子化が無効の場合は全てfloatに
            if (ApplicationConfig.DefaultValue.DisableAnimationQuantize)
            {
                result.keyType = curve_key_typeType.key32;
                result.offset = 0.0f;
                result.scale = 1.0f;
            }

            return result;
        }
    }
}
