﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Runtime.InteropServices;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class TextureGeneralPage : TexturePropertyPage
    {
        static private readonly texture_info_quantize_typeType[] supportedFormats_ =
        {
            texture_info_quantize_typeType.unorm_4_4,
            texture_info_quantize_typeType.unorm_5_6_5,
            texture_info_quantize_typeType.unorm_5_5_5_1,
            texture_info_quantize_typeType.unorm_4_4_4_4,
            texture_info_quantize_typeType.unorm_8,
            texture_info_quantize_typeType.uint_8,
            texture_info_quantize_typeType.unorm_8_8,
            texture_info_quantize_typeType.uint_8_8,
            texture_info_quantize_typeType.unorm_8_8_8_8,
            texture_info_quantize_typeType.uint_8_8_8_8,
            texture_info_quantize_typeType.snorm_8,
            texture_info_quantize_typeType.sint_8,
            texture_info_quantize_typeType.snorm_8_8,
            texture_info_quantize_typeType.sint_8_8,
            texture_info_quantize_typeType.snorm_8_8_8_8,
            texture_info_quantize_typeType.sint_8_8_8_8,
            texture_info_quantize_typeType.srgb_8_8_8_8,
            texture_info_quantize_typeType.unorm_bc1,
            texture_info_quantize_typeType.unorm_bc2,
            texture_info_quantize_typeType.unorm_bc3,
            texture_info_quantize_typeType.unorm_bc4,
            texture_info_quantize_typeType.unorm_bc5,
            texture_info_quantize_typeType.snorm_bc4,
            texture_info_quantize_typeType.snorm_bc5,
            texture_info_quantize_typeType.srgb_bc1,
            texture_info_quantize_typeType.srgb_bc2,
            texture_info_quantize_typeType.srgb_bc3,
            //
            texture_info_quantize_typeType.float_16,
            texture_info_quantize_typeType.float_32,
            texture_info_quantize_typeType.float_16_16,
            texture_info_quantize_typeType.float_32_32,
            texture_info_quantize_typeType.float_11_11_10,
            texture_info_quantize_typeType.float_16_16_16_16,
            texture_info_quantize_typeType.float_32_32_32_32
        };

        static private readonly HashSet<texture_info_quantize_typeType> supportedFloatFormats_ = new HashSet<texture_info_quantize_typeType>()
        {
            texture_info_quantize_typeType.float_16,
            texture_info_quantize_typeType.float_32,
            texture_info_quantize_typeType.float_16_16,
            texture_info_quantize_typeType.float_32_32,
            texture_info_quantize_typeType.float_11_11_10,
            texture_info_quantize_typeType.float_16_16_16_16,
            texture_info_quantize_typeType.float_32_32_32_32,
        };

        static private readonly HashSet<texture_info_quantize_typeType> supportedCompressFormats_ = new HashSet<texture_info_quantize_typeType>()
        {
            texture_info_quantize_typeType.unorm_bc1,
            texture_info_quantize_typeType.unorm_bc2,
            texture_info_quantize_typeType.unorm_bc3,
            texture_info_quantize_typeType.unorm_bc4,
            texture_info_quantize_typeType.unorm_bc5,
            texture_info_quantize_typeType.snorm_bc4,
            texture_info_quantize_typeType.snorm_bc5,
            texture_info_quantize_typeType.srgb_bc1,
            texture_info_quantize_typeType.srgb_bc2,
            texture_info_quantize_typeType.srgb_bc3,
        };

        public TextureGeneralPage() :
            base(PropertyPageID.TextureGeneral)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_texture_property_window_general_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new TextureGeneralPage();
        }

        protected override void InitializeFormInternal()
        {
            using(var ub = new UpdateBlock(cmbFormat))
            {
                var allFormats =
                    Enum.GetValues(typeof(texture_info_quantize_typeType)).Cast<texture_info_quantize_typeType>();
                foreach (var format in allFormats)
                {
                    cmbFormat.AddItem(UIText.EnumValue(format), format);
                }
            }

            using(var ub = new UpdateBlock(cmbPresetFormat))
            {
                foreach(var preset in ApplicationConfig.Preset.TexturePresets)
                {
                    cmbPresetFormat.AddItem(preset.Name, preset.quantize_type);
                }
            }

            using(var ub = new UpdateBlock(cmbMipmapGenFilter))
            {
                cmbMipmapGenFilter.AddItem(Strings.MipmapGenFilter_Point,	"point");
                cmbMipmapGenFilter.AddItem(Strings.MipmapGenFilter_Linear,	"linear");
                cmbMipmapGenFilter.AddItem(Strings.MipmapGenFilter_Cubic,	"cubic");
            }

            // 編集マーク表示・非表示
            {
                // 通常は編集不可
                var canEditTextureParams = ConfigData.ApplicationConfig.Setting.Debug != null &&
                    ConfigData.ApplicationConfig.Setting.Debug.ConvertTextureFormat;

                foreach (var modifiedMarkLabel in Controls.OfType<UIModifiedMarkLabel>())
                {
                    modifiedMarkLabel.IsMarkShow = canEditTextureParams;
                }

                modifiedMarkStream.Visible = canEditTextureParams;

                // 編集系コントロールを非表示にしてラベル系に変更する
                if (canEditTextureParams == false)
                {
                    cbxAllFormat.Visible	= false;
                    lblAllFormat.Visible	= true;
                    lblAllFormat.Location	= cbxAllFormat.Location;
                    lblAllFormat.Text		= cbxAllFormat.Text;

                    cbxWeightedCompress.Visible		= false;
                    lblWeightedCompress.Visible		= true;
                    lblWeightedCompress.Location	= cbxWeightedCompress.Location;
                    lblWeightedCompress.Text		= cbxWeightedCompress.Text + " :";
                    ltbWeightedCompress.Visible		= true;
                    ltbWeightedCompress.TabIndex	= cbxWeightedCompress.TabIndex;

                    ccsButton.Visible	= false;
                    ccsPanel.Visible	= true;
                    ccsPanel.Location	= ccsButton.Location;
                    ccsPanel.Width		= ccsButton.Width;

                    cmbPresetFormat.Visible		= false;
                    ltbPresetFormat.Visible		= true;
                    ltbPresetFormat.Location	= cmbPresetFormat.Location;
                    ltbPresetFormat.Width		= cmbPresetFormat.Width;
                    ltbPresetFormat.TabIndex	= cmbPresetFormat.TabIndex;

                    cmbFormat.Visible			= false;
                    ltbFormat.Visible			= true;
                    ltbFormat.Location			= cmbFormat.Location;
                    ltbFormat.Width				= cmbFormat.Width;
                    ltbFormat.TabIndex			= cmbFormat.TabIndex;

                    cmbMipmapGenFilter.Visible	= false;
                    ltbMipmapGenFilter.Visible	= true;
                    ltbMipmapGenFilter.Location	= cmbMipmapGenFilter.Location;
                    ltbMipmapGenFilter.Width	= cmbMipmapGenFilter.Width;
                    ltbMipmapGenFilter.TabIndex	= cmbMipmapGenFilter.TabIndex;
                }
            }
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            modifiedMarkStream.Visible	= ActiveTarget.TextureStreamChanged;

            ltbFullPath.Text			= ActiveTarget.FilePath;

            lblHint.IsModified			= ActiveTarget.IsValueChanged(x => x.hint);
            ltbHint.Text				= ActiveTarget.Data.texture_info.hint;

            lblDimension.IsModified		= ActiveTarget.IsValueChanged(x => x.dimension);
            ltbDimension.Text			= UIText.EnumValue(ActiveTarget.Data.texture_info.dimension);

            lblDccPreset.IsModified		= ActiveTarget.IsValueChanged(x => x.dcc_preset);
            ltbDccPreset.Text			= ActiveTarget.Data.texture_info.dcc_preset;

            modifiedMarkFormat.Visible 			= ActiveTarget.IsValueChanged(x => x.quantize_type);
            cmbFormat.SelectedItemData			= ActiveTarget.Data.texture_info.quantize_type;
            ltbFormat.Text						= cmbFormat.Text;

            lblMipmapGenFilter.IsModified		= ActiveTarget.IsValueChanged(x => x.mip_gen_filter);
            cmbMipmapGenFilter.SelectedItemData	= ActiveTarget.Data.texture_info.mip_gen_filter;
            ltbMipmapGenFilter.Text				= cmbMipmapGenFilter.Text;

            cmbPresetFormat.SelectedItemData	= ActiveTarget.Data.texture_info.quantize_type;
            ltbPresetFormat.Text				= cmbPresetFormat.Text;

            lblComponentSelector.IsModified		= ActiveTarget.IsSequenceValueChanged(x => x.comp_sel);
            ccsButton.Value						= ActiveTarget.Data.texture_info.comp_sel;
            ccsPanel.Value						= ActiveTarget.Data.texture_info.comp_sel;

            cbxWeightedCompress.IsModified		= ActiveTarget.IsValueChanged(x => x.weighted_compress);
            cbxWeightedCompress.Checked			= ActiveTarget.Data.texture_info.weighted_compress;
            ltbWeightedCompress.Text			= UIText.FlagEnableDisable(ActiveTarget.Data.texture_info.weighted_compress);

            lblWidth.IsModified					= ActiveTarget.IsValueChanged(x => x.width);
            ltbWidth.Text						= ActiveTarget.Data.texture_info.width.ToString();

            lblHeight.IsModified				= ActiveTarget.IsValueChanged(x => x.height);
            ltbHeight.Text						= ActiveTarget.Data.texture_info.height.ToString();

            lblDepth.IsModified					= ActiveTarget.IsValueChanged(x => x.depth);
            ltbDepth.Text						= ActiveTarget.Data.texture_info.depth.ToString();

            lblVramSize.IsModified				= ActiveTarget.IsValueChanged(x => x.size);
            ltbVramSize.Text					= DataSize.XBytesText(ActiveTarget.Data.texture_info.size);

            lblInitialSwizzle.IsModified		= ActiveTarget.IsValueChanged(x => x.initial_swizzle);
            ltbInitialSwizzle.Text				= ActiveTarget.Data.texture_info.initial_swizzle.ToString();

            lblOriginalImageHash.IsModified     = ActiveTarget.IsValueChanged(x => x.original_image_hash);
            ltbOriginalImageHash.Text           = ActiveTarget.Data.texture_info.original_image_hash;

            lblMipmap.IsModified				= ActiveTarget.IsValueChanged(x => x.mip_level);
            ltbMipmap.Text						= UIText.MipLevel(ActiveTarget.Data.texture_info.mip_level);

            lblMipmapMinSize.IsModified			= ActiveTarget.IsValueChanged(x => UIText.MipmapMinSize(x));
            ltbMipmapMinSize.Text				= UIText.MipmapMinSize(ActiveTarget.Data.texture_info);

            lblGammaCorrection.IsModified		= ActiveTarget.IsArrayChanged(x => x.linear);
            rgbaGammaCorrection.Values			= ActiveTarget.Data.texture_info.linear;

            tvpImage.Target		= ActiveTarget;

            cbxWeightedCompress.Enabled	= (ActiveTarget.Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_bc1) ||
                                          (ActiveTarget.Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_bc2) ||
                                          (ActiveTarget.Data.texture_info.quantize_type == texture_info_quantize_typeType.unorm_bc3) ||
                                          (ActiveTarget.Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_bc1) ||
                                          (ActiveTarget.Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_bc2) ||
                                          (ActiveTarget.Data.texture_info.quantize_type == texture_info_quantize_typeType.srgb_bc3);

            lblPresetFormat.Enabled		= cbxAllFormat.Checked == false;
            cmbPresetFormat.Enabled		= cbxAllFormat.Checked == false;
            cmbFormat.Enabled			= cbxAllFormat.Checked;
        }

        public static bool IsModified(Texture activeTarget)
        {
            return activeTarget != null &&
                (activeTarget.TextureStreamChanged ||
                activeTarget.IsValueChanged(x => x.hint) ||
                activeTarget.IsValueChanged(x => x.dimension) ||
                activeTarget.IsValueChanged(x => x.dcc_preset) ||
                activeTarget.IsValueChanged(x => x.quantize_type) ||
                activeTarget.IsValueChanged(x => x.mip_gen_filter) ||
                activeTarget.IsSequenceValueChanged(x => x.comp_sel) ||
                activeTarget.IsValueChanged(x => x.weighted_compress) ||
                activeTarget.IsValueChanged(x => x.width) ||
                activeTarget.IsValueChanged(x => x.height) ||
                activeTarget.IsValueChanged(x => x.depth) ||
                activeTarget.IsValueChanged(x => x.size) ||
                activeTarget.IsValueChanged(x => x.initial_swizzle) ||
                activeTarget.IsValueChanged(x => x.mip_level) ||
                activeTarget.IsValueChanged(UIText.MipmapMinSize) ||
                activeTarget.IsArrayChanged(x => x.linear));
        }

        private void ccsButton_PopupClose(object sender, EventArgs e)
        {
            Debug.Assert(sender is ColorComponentSelectButton);

            var ccsb = sender as ColorComponentSelectButton;
            if (ccsb.IsChangeValue)
            {
                using (var wc = new WaitCursor())
                {
                    var commandSet	= new EditCommandSet();
                    {
                        commandSet.Add(CreateEditCommand_comp_sel(Targets, ccsb.Value));
                        commandSet.Add(CreateEditCommand_Reload(Targets));
                    }

                    TheApp.CommandManager.Execute(commandSet);
                }
            }
        }

        private void cbxWeightedCompress_CheckedChanged(object sender, EventArgs e)
        {
            bool weighted_compress = ((UICheckBox)sender).Checked;

            using (var wc = new WaitCursor())
            {
                var commandSet	= new EditCommandSet();
                {
                    commandSet.Add(CreateEditCommand_weighted_compress(Targets, weighted_compress));
                    commandSet.Add(
                        GroupEditCommand_ConvertTexture(
                            Targets,
                            ActiveTarget.Data.texture_info.quantize_type,
                            weighted_compress,
                            ActiveTarget.Data.texture_info.mip_gen_filter
                        )
                    );
                }

                TheApp.CommandManager.Execute(commandSet);
            }
        }

        private void cmbFormat_SelectedIndexChanged(object sender, EventArgs e)
        {
            var targetTextures		= Targets.Objects.OfType<Texture>();
            var currentQuantizes	= targetTextures.Select(x => x.Data.texture_info.quantize_type);
            var nextQuantizeType	= (texture_info_quantize_typeType)cmbFormat.SelectedItemData;

            var isNextFloat = supportedFloatFormats_.Contains(nextQuantizeType);

            // サポートしているフォーマットにのみ変更できる
            if (!supportedFormats_.Contains(nextQuantizeType))
            {
                // フォームを元の状態にもどす
                UpdateForm(false, false);
            }
            // float <-> int のコンバートはできない
            else if (currentQuantizes.Any(x => supportedFloatFormats_.Contains(x) != isNextFloat))
            {
                UIMessageBox.Warning(
                    isNextFloat ?
                        Strings.Texture_CannotConvertIntToFloat :
                        Strings.Texture_CannotConvertFloatToInt
                );

                // フォームを元の状態にもどす
                UpdateForm(false, false);
            }
            // 1D -> 圧縮 のコンバートはできない
            else
            if (targetTextures.Any(x => x.Is1d) && supportedCompressFormats_.Contains(nextQuantizeType))
            {
                UIMessageBox.Warning(Strings.Texture_CannotConvert1dToCompress);

                // フォームを元の状態にもどす
                UpdateForm(false, false);
            }
            else
            {
                using (var wc = new WaitCursor())
                {
                    var commandSet = new EditCommandSet();
                    {
                        commandSet.Add(GroupEditCommand_ConvertTexture(
                            Targets,
                            (texture_info_quantize_typeType)cmbFormat.SelectedItemData,
                            ActiveTarget.Data.texture_info.weighted_compress,
                            ActiveTarget.Data.texture_info.mip_gen_filter
                        ));
                    }

                    TheApp.CommandManager.Execute(commandSet);
                }
            }
        }

        private void cbxAllFormat_CheckedChanged(object sender, EventArgs e)
        {
            UpdateForm(false, false);
        }

        private void cmbPresetFormat_SelectedIndexChanged(object sender, EventArgs e)
        {
            var targetTextures		= Targets.Objects.OfType<Texture>();
            var nextQuantizeType	= (texture_info_quantize_typeType)cmbPresetFormat.SelectedItemData;

            // float <-> int のコンバートはできない
            var isNextFloat = supportedFloatFormats_.Contains(nextQuantizeType);
            if (targetTextures.Any(x => supportedFloatFormats_.Contains(x.Data.texture_info.quantize_type) != isNextFloat))
            {
                UIMessageBox.Warning(
                    isNextFloat ?
                        Strings.Texture_CannotConvertIntToFloat :
                        Strings.Texture_CannotConvertFloatToInt
                );

                // フォームを元の状態にもどす
                UpdateForm(false, false);
            }
            // 1D -> 圧縮 のコンバートはできない
            else
            if (targetTextures.Any(x => x.Is1d) && supportedCompressFormats_.Contains(nextQuantizeType))
            {
                UIMessageBox.Warning(Strings.Texture_CannotConvert1dToCompress);

                // フォームを元の状態にもどす
                UpdateForm(false, false);
            }
            else
            {
                using (var wc = new WaitCursor())
                {
                    var commandSet = new EditCommandSet();
                    {
                        commandSet.Add(GroupEditCommand_ConvertTexture(
                            Targets,
                            (texture_info_quantize_typeType)cmbPresetFormat.SelectedItemData,
                            ActiveTarget.Data.texture_info.weighted_compress,
                            ActiveTarget.Data.texture_info.mip_gen_filter
                        ));
                    }

                    TheApp.CommandManager.Execute(commandSet);
                }
            }
        }

        private void cmbMipmapGenFilter_SelectedIndexChanged(object sender, EventArgs e)
        {
            using (var wc = new WaitCursor())
            {
                var commandSet = new EditCommandSet();
                {
                    commandSet.Add(
                        GroupEditCommand_ConvertTexture(
                            Targets,
                            ActiveTarget.Data.texture_info.quantize_type,
                            ActiveTarget.Data.texture_info.weighted_compress,
                            (string)cmbMipmapGenFilter.SelectedItemData
                        )
                    );
                }

                TheApp.CommandManager.Execute(commandSet);
            }
        }

        public sealed class TextureEditData : IDisposable
        {
            public textureType texture;

            public List<G3dStream> streams {
                get{
                    if (stream_ != null) {
                        return stream_;
                    }
                    else if (cache != null)
                    {
                        return cache.GetStreams();
                    }
                    return null;
                }
                set { stream_ = value; }
            }

            private List<G3dStream> stream_;
            public void WriteToFile()
            {
                cache = new G3dStreamCache(stream_);
                stream_ = null;
            }
            G3dStreamCache cache = null;
            public void Dispose()
            {
                if (cache != null)
                {
                    cache.Dispose();
                    cache = null;
                }
                stream_ = null;
            }
        }

        static private ICommand GroupEditCommand_ConvertTexture(
            GuiObjectGroup targets,
            texture_info_quantize_typeType format,
            bool weighted_compress,
            string mip_gen_filter
        )
        {
            var editDataList = new List<TextureEditData>();
            {
                foreach(Texture texture in targets.GetObjects(GuiObjectID.Texture))
                {
                    var originalPaths = texture.Data.original_image_array.original_image.Select(x => x.original_path).ToArray();

                    // 一旦ファイルに書きだす
                    using(var currentTempFileName = TemporaryFileUtility.MakeDisposableFileName(".current" + G3dPath.TextureBinaryExtension))
                    {
                        (new DocumentSaver()).WriteDocument(texture, currentTempFileName.Path, false);

                        // コンバート後のファイル名 -- 実際にはファイルにか書き出されない。コンバート後のデータはメモリの中にある
                        string convertedFileName = "converted" + G3dPath.TextureBinaryExtension;

                        // ファイルとフォーマットの指定
                        {
                            TexcvtrManager.Clear();

                            bool isSuccess = TexcvtrManager.ReadInputFile(
                                new[]{ currentTempFileName.Path, null },
                                new[]
                                    {
                                        string.Format("--output=\"{0}\"", convertedFileName),
                                        string.Format("--format=\"{0}\"", format),
                                        string.Format("--weighted-compress={0}", weighted_compress ? "true" : "false"),
                                        string.Format("--mip-gen-filter={0}", mip_gen_filter),
                                        null
                                    }
                            );
                            Debug.Assert(isSuccess);
                        }

                        // コンバートする
                        var convertedData = IntPtr.Zero;
                        int convertedDataSize = 0;
                        {
                            bool isSuccess = TexcvtrManager.ConvertToData(ref convertedData, ref convertedDataSize);
                            Debug.Assert(isSuccess);
                        }

                        // マネージドなところに持ってくる
                        var convertedDataArray = new byte[convertedDataSize];
                        Marshal.Copy(convertedData, convertedDataArray, 0, convertedDataSize);

                        TexcvtrManager.Clear();

                        var streams = new List<G3dStream>();
                        var nw4f_3dif = IfBinaryReadUtility.Read(streams, convertedDataArray, DocumentManager.XsdBasePath);

                        Debug.Assert(nw4f_3dif.Item is textureType);

                        var textureEditData = new TextureEditData()
                        {
                            texture = (textureType)nw4f_3dif.Item,
                            streams = streams,
                        };

                        // オリジナルの数はコンバート前・後は同じはず。
                        Debug.Assert(originalPaths.Count() == textureEditData.texture.original_image_array.original_image.Count());

                        // コンバートするとオリジナルパスが変わってしまうので戻す
                        int index = 0;
                        foreach(var originalPath in originalPaths)
                        {
                            textureEditData.texture.original_image_array.original_image[index].original_path = originalPath;
                            ++ index;
                        }

                        // 編集情報は変えない
                        editDataList.Add(textureEditData);
                    }
                }
            }

            return CreateTextureChangeCommand(targets, editDataList);
        }

        static public EditCommand CreateTextureChangeCommand(GuiObjectGroup targets, List<TextureEditData> textureData)
        {
            var textures = targets.Objects.OfType<Texture>().ToArray();

            var models = MakeInverseReferenceModels(textures).ToArray();
            var anims  = MakeInverseReferenceAnims( textures).ToArray();

            return new GeneralGroupReferenceEditCommand<TextureEditData>(
                targets,
                GuiObjectID.Texture,
                textureData,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    Debug.Assert(target is Texture);
                    var texture = target as Texture;
                    var Data = (TextureEditData)data;
                    swap = new TextureEditData()
                    {
                        texture = texture.Data,
                        streams = texture.BinaryStreams,
                    };
                    ((TextureEditData)swap).WriteToFile();

                    lock (texture.creatingImagesLock_)
                    {
                        texture.Data = (textureType)Data.texture;
                        texture.BinaryStreams = Data.streams;
                        texture.TextureStreamChanged = 	!texture.IsSameStreamToSavedStream(texture.BinaryStreams[texture.Data.texture_info.stream_index]);
                    }
                    Data.Dispose();
                },
                createEventArgsDelegate: (target, command) => new FileView.FileTreeView.TextureImageChangedArgs(target, command),
                postEditDelegate: (x, y) =>
                    {
                        foreach (var model in models)
                        {
                            Viewer.LoadOrReloadModel.Send(model);
                        }

                        foreach (var anim in anims)
                        {
                            Viewer.LoadOrReloadAnimation.Send(anim);
                        }
                    }
            );
        }
        static public EditCommand CreateTextureDataChangeCommand(GuiObjectGroup targets, textureType textureData)
        {
            int objectCount = targets.GetObjects(GuiObjectID.Texture).Count;

            return
                CreateTextureDataChangeCommand(
                    targets,
                    ObjectUtility.MultipleClone(textureData, objectCount)
                );
        }

        static public EditCommand CreateTextureDataChangeCommand(GuiObjectGroup targets, IEnumerable<textureType> textureData)
        {
            return
                new GeneralGroupReferenceEditCommand<textureType>(
                    targets,
                    GuiObjectID.Texture,
                    textureData,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is Texture);
                        var texture = target as Texture;

                        commentType oldComment = texture.Data.comment;

                        swap = texture.Data;
                        texture.Data = (textureType)data;

                        // 上書きされる場合でもコメントは維持するので以前の状態に戻す
                        texture.Data.comment = oldComment;
                        texture.MakeComment(texture.Data.comment);
                    }
            );
        }

        static List<G3dStream> CopyG3dStreams(List<G3dStream> streams)
        {
            // ObjectUtility.MultipleClone() でクローンできないので直に作る
            var cloneStreams = new List<G3dStream>();
            {
                int index = 0;
                foreach (var stream in streams)
                {
                    //cloneStreams.Add(new G3dStream(stream.GetText(index)));
                    cloneStreams.Add(stream);
                    ++index;
                }
            }
            return cloneStreams;
        }

        static public EditCommand CreateTextureStreamChangeCommand(GuiObjectGroup targets, List<G3dStream> streams)
        {
            var src = new List<List<G3dStream>>();
            {
                int objectCount = targets.GetObjects(GuiObjectID.Texture).Count;

                for (int i = 0;i != objectCount;++ i)
                {
                    src.Add(CopyG3dStreams(streams));
                }
            }

            return CreateTextureStreamChangeCommand(targets, src);
        }

        static public EditCommand CreateTextureStreamChangeCommand(GuiObjectGroup targets, IEnumerable<List<G3dStream>> streamsList)
        {
            return
                new GeneralGroupReferenceEditCommand<List<G3dStream>>(
                    targets,
                    GuiObjectID.Texture,
                    streamsList,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        Debug.Assert(target is Texture);
                        var texture = target as Texture;

                        swap = texture.BinaryStreams;
                        texture.BinaryStreams = (List<G3dStream>)data;
                    },
                    createEventArgsDelegate: (target, command) => new FileView.FileTreeView.TextureImageChangedArgs(target, command)
            );
        }

        static private GroupEditCommand CreateEditCommand_comp_sel(GuiObjectGroup targets, texture_info_comp_selValue[] comp_sel)
        {
            return
                new GeneralGroupReferenceEditCommand<texture_info_comp_selValue[]>(
                    targets,
                    GuiObjectID.Texture,
                    ObjectUtility.MultipleClone(comp_sel, targets.GetObjects(GuiObjectID.Texture).Count),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var texture = target as Texture;

                        swap = ObjectUtility.Clone(texture.Data.texture_info.comp_sel);
                        texture.Data.texture_info.comp_sel = (texture_info_comp_selValue[])data;
                        texture.ClearCache();
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                    },
                    createEventArgsDelegate : (target, command) => new FileView.FileTreeView.TextureImageChangedArgs(target, command)
                );
        }

        static private GroupEditCommand CreateEditCommand_weighted_compress(GuiObjectGroup targets, bool weighted_compress)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.Texture,
                    weighted_compress,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var texture = target as Texture;

                        swap = texture.Data.texture_info.weighted_compress;
                        texture.Data.texture_info.weighted_compress = (bool)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        ;	// リロードはまとめて送れないのでEditDelegateで送る
                    }
                );
        }

        static private GroupEditCommand CreateEditCommand_Reload(GuiObjectGroup targets)
        {
            var textures = targets.Objects.OfType<Texture>().ToArray();

            var models = MakeInverseReferenceModels(textures).ToArray();
            var anims  = MakeInverseReferenceAnims( textures).ToArray();

            return new GeneralGroupValueEditCommand<int>(
                DummyObject.TheDummyObjectGroup,
                GuiObjectID.DummyObject,
                0,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    foreach (var model in models)
                    {
                        Viewer.LoadOrReloadModel.Send(model);
                    }

                    foreach (var anim in anims)
                    {
                        Viewer.LoadOrReloadAnimation.Send(anim);
                    }
                }
            );
        }

        static public IEnumerable<Model> MakeInverseReferenceModels(IEnumerable<Texture> targets)
        {
            var models = new List<Model>();
            {
                foreach(var model in DocumentManager.Models)
                {
                    foreach(var texture in targets)
                    {
                        if (model.ReferenceDocuments.Any(x => x == texture))
                        {
                            models.Add(model);
                            break;
                        }
                    }
                }
            }

            return models;
        }

        static public IEnumerable<AnimationDocument> MakeInverseReferenceAnims(IEnumerable<Texture> targets)
        {
#if false
            var anims = DocumentManager.Animations.Where(x => x.ReferenceDocuments.Intersect(textures).Any()).ToArray();
#else
            var anims = new List<AnimationDocument>();
            {
                foreach(var anim in DocumentManager.Animations)
                {
                    foreach(var texture in targets)
                    {
                        if (anim.ReferenceDocuments.Any(x => x == texture))
                        {
                            anims.Add(anim);
                            break;
                        }
                    }
                }
            }
#endif

            return anims;
        }

        #region コピー＆ペースト
        private class CopyData
        {
 			public texture_info_comp_selValue[]		comp_sel			{ get; set; }
            public texture_info_quantize_typeType	quantize_type		{ get; set; }
 			public bool								weighted_compress	{ get; set; }
 			public string							mip_gen_filter		{ get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            // 不可
            return false;
        }

        public static bool CanCopy(Texture ActiveTarget)
        {
            // 不可
            return false;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(Texture target)
        {
            return
                new CopyData()
                {
 					comp_sel			= ObjectUtility.Clone(target.Data.texture_info.comp_sel),
                    quantize_type		= target.Data.texture_info.quantize_type,
     				weighted_compress	= target.Data.texture_info.weighted_compress,
     				mip_gen_filter		= target.Data.texture_info.mip_gen_filter,
                };
        }

        /// <summary>
        /// ペースト可能か。
        /// </summary>
        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return CanPaste(Targets, copiedObjectInfo, copiedObject);
        }

        /// <summary>
        /// ペースト可能か
        /// </summary>
        public static bool CanPaste(GuiObjectGroup targets, object copiedObjectInfo, object copiedObject)
        {
            var copyData = (CopyData)copiedObject;
            return targets.Objects.OfType<Texture>().All(
                x => supportedFloatFormats_.Contains(x.Data.texture_info.quantize_type) == supportedFloatFormats_.Contains(copyData.quantize_type));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(GroupEditCommand_ConvertTexture(targets, copyData.quantize_type, copyData.weighted_compress, copyData.mip_gen_filter));
                commandSet.Add(CreateEditCommand_comp_sel(targets, copyData.comp_sel));
            }

            return commandSet.Execute();
        }
        #endregion
    }
}
