﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class SkeletalAnimationCurveEditPage : SkeletalAnimationPropertyPage, IAnimationEditPage
    {
        public SkeletalAnimationCurveEditPage() :
            base(PropertyPageID.SkeletalAnimationCurveEdit)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_curve_editor;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new SkeletalAnimationCurveEditPage();
        }

        protected override void InitializeFormInternal()
        {
            var animationObjectPropertyPanel = Owner as AnimationObjectPropertyPanel;
            Debug.Assert(animationObjectPropertyPanel != null);

            CurveEditorPanel.Initialize(animationObjectPropertyPanel, ActiveTarget.ObjectID);

            CurveEditorPanel.UpdateTreeView       += (s, e) => UpdateTreeView();
            CurveEditorPanel.ChangeSelectedCurves += (s, e) => UpdateSelected();
            CurveEditorPanel.ChangeVisibledCurves += (s, e) => UpdateVisibled();
            CurveEditorPanel.GetFrameCount        = () => ActiveTarget.Data.skeletal_anim_info.frame_count;
            CurveEditorPanel.SetInterpolationTypeButtons(false, false, false);
        }

        public override Size DefaultPageSize
        {
            get
            {
                return new Size(
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width),
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height));
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if ((Owner != null) && (Owner.ActivePage == this) && !ObjectPropertyDialog.InternallyChangingSize.IsChanging &&
                Owner.Owner.WindowState == System.Windows.Forms.FormWindowState.Normal)
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width = Width;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height = Height;
            }

            base.OnSizeChanged(e);
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            CurveEditorPanel.TargetGroup = Targets;

            CurveEditorPanel.UpdateForm(updateFormInfo);
        }

        protected override void UpdateFormOnPageCreatedInternal()
        {
            CurveEditorPanel.UpdateFormOnPageCreatedInternal();
        }

        public override void BeforePageDeactivated()
        {
            CurveEditorPanel.BeforePageDeactivated();
        }

        public override void AfterPageActiveted(ObjectPropertyPage oldPage)
        {
            CurveEditorPanel.AfterPageActivated();
        }

        private void UpdateSelected()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private void UpdateVisibled()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private Dictionary<string, bool> visibledNodes_ = new Dictionary<string, bool>();
        private readonly Dictionary<string, bool> isExpandedNodes_ = new Dictionary<string, bool>();

        private void UpdateTreeView()
        {
            using (var sb = new UIControlEventSuppressBlock())
            {
                // ノードの状態を保存しておく
                if (CurveEditorPanel.CurveTreeView.Nodes.Count > 0)
                {
                    var root = (CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0];
                    foreach (var tuple in CurveTreeView.NodesAndFullPath(root, root.id))
                    {
                        // チェック状態
                        visibledNodes_[tuple.Item2] = tuple.Item1.Checked;

                        // 開閉状態
                        isExpandedNodes_[tuple.Item2] = tuple.Item1.Nodes.Count == 0 || tuple.Item1.IsExpanded;
                    }
                }

                var bones = new HashSet<string>((from model in DocumentManager.Models
                                                 where model.AllAnimations.Contains(ActiveTarget)
                                                 from bone in model.Bones
                                                 select bone.Name).Distinct());
                var isAllShowNode = (Owner as AnimationObjectPropertyPanel).IsAllShowNode;


                var rootNode = new CurveTreeInfo()
                {
                    Text = ActiveTarget.Name,
                    Id = ActiveTarget.Name,
                    ShowEmptyNode = true,
                };


                var isSortByAlphabetical = (Owner as AnimationObjectPropertyPanel).IsSortByAlphabetical;
                var anims = isSortByAlphabetical
                    ? ActiveTarget.BoneAnims.OrderBy(x => x.bone_name).ToList()
                    : ActiveTarget.BoneAnims;
                foreach (var boneAnim in anims)
                {
                    var groups = (from target in boneAnim.BoneAnimTargets.OrderBy(x => x.target)
                                  group target by SceneAnimation.Group(target.target)).ToArray();

                    var boneName = boneAnim.bone_name;
                    var animNodes= new List<CurveTreeInfo>();

#if true
                    foreach (var group in groups)
                    {
                        var targetNodes = new List<CurveTreeInfo>();
                        foreach (var boneAnimTarget in group)
                        {
                            var target = boneAnimTarget.target;

                            var targetNode = new CurveTreeInfo()
                            {
                                Id = SceneAnimation.Subscript(group.Key, target.ToString()),
                                Text = SceneAnimation.Subscript(group.Key, target.ToString()),
                                AnimationCurve = new SkeletalAnimationCurveTreeNodeInfo(ActiveTarget, boneName, target),
                                NonEditableKind = AnimationDocument.NonEditableKind.SkeletalAnimation_NotEditable,
                                IsModified = boneAnimTarget.IsModified,
                                ShowEmptyNode = isAllShowNode,
                                IsBound = false,
                            };

                            targetNodes.Add(targetNode);
                        }
                        var groupNode = new CurveTreeInfo()
                        {
                            Text = group.Key,
                            Id = group.Key,
                            Nodes = targetNodes,
                            ShowEmptyNode = isAllShowNode,
                        };
                        animNodes.Add(groupNode);



                    }

#else
                    foreach (var anim in boneAnim.BoneAnimTargets)
                    {
                        var animNode = new CurveTreeInfo()
                        {
                            Text = anim.target.ToString(),
                            Id = anim.target.ToString(),
                            AnimationCurve = new SkeletalAnimationCurveTreeNodeInfo(ActiveTarget, boneName, anim.target),
                            ShowEmptyNode = isAllShowNode,
                            //NonEditableKind = AnimationDocument.NonEditableKind.BoneVisibility_NotFoundBone,
                            IsBound = false,
                        };
                        animNodes.Add(animNode);
                    }

#endif
                    var boneNode = new CurveTreeInfo()
                    {
                        Text = boneName,
                        Id = boneName,
                        IsBound = bones.Contains(boneName),
                        ShowEmptyNode = isAllShowNode,
                        Nodes = animNodes,
                    };

                    rootNode.Nodes.Add(boneNode);

                }

                rootNode.TrimInvisibleNodes();

                // 文字列によるフィルター
                CurveTreeInfo.FilterCurveTreeInfoRoot(rootNode, (Owner as AnimationObjectPropertyPanel).SearchTexts);

                var nodes = rootNode.ConvertToTreeNode();
                if (nodes.IsSameStructure(CurveEditorPanel.CurveTreeView.Nodes) == false)
                {
                    if ((CurveEditorPanel.UpdateFormInfo == null) || (CurveEditorPanel.UpdateFormInfo.TargetOrPageChanged == false))
                    {
                        // つくり直す


                        foreach (var tuple in CurveTreeView.NodesAndFullPath((CurveTreeNode)nodes, nodes.id))
                        {
                            // チェック状態を設定する
                            bool isChecked = false;
                            if (visibledNodes_.TryGetValue(tuple.Item2, out isChecked))
                            {
                                if (isChecked != tuple.Item1.Checked)
                                {
                                    tuple.Item1.Checked = isChecked;
                                }
                            }

                            bool isExpanded = false;
                            if (isExpandedNodes_.TryGetValue(tuple.Item2, out isExpanded))
                            {
                                if (isExpanded)
                                {
                                    if (!tuple.Item1.IsExpanded)
                                    {
                                        tuple.Item1.Expand();
                                    }
                                }
                                else
                                {
                                    if (tuple.Item1.IsExpanded)
                                    {
                                        tuple.Item1.Collapse();
                                    }
                                }
                            }
                        }

                        CurveEditorPanel.CurveTreeView.Nodes.Clear();
                        CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);
                    }
                    else
                    {
                        nodes.ExpandAll();

                        // つくり直す
                        CurveEditorPanel.CurveTreeView.Nodes.Clear();
                        CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);

                        // 新規に開いたときは全表示にする
                        CurveEditorPanel.VisibleAllNode();
                    }
                }
                else
                {
                    ((CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0]).CopyInfo(nodes);
                }
            }
        }
#if false
        #region コピー＆ペースト
        private class CopyData
        {
            public int frame_count { get; set; }
            public bool loop { get; set; }
            public List<BoneVisibilityAnimation.BoneVisibilityBoneAnim>	BoneVisibilityBoneAnims{ get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(BoneVisibilityAnimation target)
        {
            return
                new CopyData()
                {
                    frame_count = target.Data.bone_visibility_anim_info.frame_count,
                    loop = target.Data.bone_visibility_anim_info.loop,
                    BoneVisibilityBoneAnims = ObjectUtility.Clone(target.BoneVisibilityBoneAnims)
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
            using (var block = new Viewer.ViewerDrawSuppressBlock(AnimationCurveEditCommand.AnimationMessageFilter))
            {
                var copyData = (CopyData)pasteObject;
                commandSet.Add(BoneVisibilityAnimationGeneralPage.CreateEditCommand_frame_count(targets, copyData.frame_count).Execute());
                commandSet.Add(BoneVisibilityAnimationGeneralPage.CreateEditCommand_loop(targets, copyData.loop).Execute());
                commandSet.Add(
                    new GeneralGroupReferenceEditCommand<List<BoneVisibilityAnimation.BoneVisibilityBoneAnim>>(
                        targets,
                        GuiObjectID.BoneVisibilityAnimation,
                        ObjectUtility.MultipleClone(copyData.BoneVisibilityBoneAnims, targets.Objects.Count),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            Debug.Assert(target is BoneVisibilityAnimation);
                            var boneVisAnim = target as BoneVisibilityAnimation;

                            swap = ObjectUtility.Clone(boneVisAnim.BoneVisibilityBoneAnims);
                            boneVisAnim.BoneVisibilityBoneAnims.Clear();
                            boneVisAnim.BoneVisibilityBoneAnims.AddRange(data as List<BoneVisibilityAnimation.BoneVisibilityBoneAnim>);

                            boneVisAnim.UpdateIsModifiedAnimTargetAll();
                            Viewer.LoadAnimation2.Send(boneVisAnim);
                        }
                    ).Execute()
                );

                var active = (BoneVisibilityAnimation)targets.Active;
                var command = active.CreateUpdateBindCommand();
                if (command != null)
                {
                    commandSet.Add(command.Execute());
                }
            }

            commandSet.Reverse();
            return commandSet;
        }
        #endregion
#endif
    }
}
