﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Data;

namespace App.PropertyEdit
{

    public class ShapeAnimationCurveTreeNodeInfo : IAnimationCurve
    {
        public bool IsEditable { get; set; }
        public AnimationDocument.NonEditableKind NonEditableKind { get; set; }
        public object NonEditableKindDisplayAux { get; set; }
        public bool IsColorCurve { get { return false; } }
        public int ColorComponentIndex { get { return -1; } }

        public AnimTarget GetAnimTarget(GuiObject editTarget)
        {
            Debug.Assert(editTarget is ShapeAnimation);
            var animation = (ShapeAnimation)editTarget;
            return animation.GetAnimTarget(animation.VertexShapeAnims, shapeName, keyShapeName);
        }

        public AnimTarget CreateAnimTarget(GuiObject editTarget)
        {
            var animation = (ShapeAnimation)editTarget;
            var shapeAnim = new ShapeAnimation.ShapeAnim()
            {
                key_shape_name = keyShapeName,
            };

            var vertexShapeAnim = animation.VertexShapeAnims.First(x => x.shape_name == shapeName);
            vertexShapeAnim.ShapeAnims.Add(shapeAnim);

            return shapeAnim;
        }

        public void UpdateIsModified(GuiObject editTarget)
        {
            var animation = (ShapeAnimation)editTarget;
            var animTarget = GetAnimTarget(animation);
            var savedTarget = animation.GetAnimTarget(animation.savedVertexShapeAnims, shapeName, keyShapeName);
            animation.UpdateIsModifiedAnimTarget(
                (ShapeAnimation.ShapeAnim)animTarget,
                savedTarget);
        }

        public AnimTarget CreateTemporaryTarget(GuiObject editTarget)
        {
            var source = (ShapeAnimation.ShapeAnim)GetAnimTarget(editTarget);
            return new ShapeAnimation.ShapeAnim()
            {
                key_shape_name = source.key_shape_name,
            };
        }

        public void SetAnimTarget(GuiObject editTarget, AnimTarget animTarget)
        {
            var animation = (ShapeAnimation)editTarget;
            var vertexShapeAnim = animation.VertexShapeAnims.First(x => x.shape_name == shapeName);
            var index = vertexShapeAnim.ShapeAnims.FindIndex(x => x.key_shape_name == keyShapeName);
            if (index == -1)
            {
                vertexShapeAnim.ShapeAnims.Add((ShapeAnimation.ShapeAnim)animTarget);
            }
            else
            {
                vertexShapeAnim.ShapeAnims[index] = (ShapeAnimation.ShapeAnim)animTarget;
            }
        }

        public float GetDefaultValue(GuiObject editTarget)
        {
            return GetAnimTarget(editTarget).GetBaseValue();
        }

        private readonly ShapeAnimation	target;
        private readonly string shapeName;
        private readonly string keyShapeName;

        public ShapeAnimationCurveTreeNodeInfo(ShapeAnimation target, string shapeName, string keyShapeName)
        {
            this.target		= target;
            this.shapeName = shapeName;
            this.keyShapeName = keyShapeName;
        }

        public List<KeyFrame> KeyFrames
        {
            get
            {
                return
                    (paramAnim != null) ?
                        paramAnim.KeyFrames :
                        new List<KeyFrame>();
            }
        }

        public ShapeAnimation.ShapeAnim paramAnim { get { return target.GetAnimTarget(target.VertexShapeAnims, shapeName, keyShapeName); } }
        public InterpolationType CurveInterpolationType
        {
            get
            {
                return paramAnim != null ? paramAnim.CurveInterpolationType: InterpolationType.Linear;
            }

            set
            {
                ;   // 無視
            }
        }

        public string ParentName{		get { return shapeName;			} }
        public string Name{				get { return keyShapeName;		} }
        public string FullPath{			get { return Name;				} }
        public int ComponentIndex{		get { return 0;					} }
        public bool IsRotate{			get { return false;				} }
        public float? MinClampValue{	get { return 0.0f;				} }
        public float? MaxClampValue{	get { return 1.0f;				} }
        public float? MinFitValue{		get { return 0.0f;				} }
        public float? MaxFitValue{		get { return 1.0f;				} }
        public PrimitiveTypeKind CurvePrimitiveType { get { return PrimitiveTypeKind.Float; } }

        public Color CurveColor { get { return CurveView.MakeColorFromString(Name + ParentName); } }

        public bool IsSame(IAnimationCurve curve)
        {
            var shapeCurve = (ShapeAnimationCurveTreeNodeInfo)curve;
            return shapeCurve.target == target &&
                shapeCurve.shapeName == shapeName &&
                shapeCurve.keyShapeName == keyShapeName;
        }

        public void IntermediateFileIndices(ShapeAnimation target, out int vertexShapeAnimIndex, out int shapeAnimTargetIndex)
        {
            vertexShapeAnimIndex = 0;
            shapeAnimTargetIndex = 0;
            foreach (var vertexShapeAnim in target.VertexShapeAnims)
            {
                if (vertexShapeAnim.shape_name == shapeName)
                {
                    foreach (var shapeAnimTarget in vertexShapeAnim.ShapeAnims)
                    {
                        if (shapeAnimTarget.key_shape_name == keyShapeName)
                        {
                            return;
                        }
                        switch (shapeAnimTarget.ExportType)
                        {
                            case CurveExportType.Curve:
                            case CurveExportType.Constant:
                                shapeAnimTargetIndex++;
                                break;
                        }
                    }
                }
                else
                {
                    if (vertexShapeAnim.ShapeAnims.Any(x => x.ExportType != CurveExportType.Ignored && x.ExportType != CurveExportType.Dependent))
                    {
                        vertexShapeAnimIndex++;
                    }
                }
            }
        }
    }
}
