﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text.RegularExpressions;
using App.Command;
using App.Controls;
using App.Data;
using App.Properties;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    using System.Windows.Forms;

    public partial class ShaderParameterAnimationCurveEditPage : ShaderParameterAnimationPropertyPage, IAnimationEditPage
    {
        public ShaderParameterAnimationCurveEditPage() :
            base(PropertyPageID.ShaderParameterAnimationCurveEdit)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_curve_editor;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new ShaderParameterAnimationCurveEditPage();
        }

        protected override void InitializeFormInternal()
        {
            AnimationObjectPropertyPanel animationObjectPropertyPanel = Owner as AnimationObjectPropertyPanel;
            Debug.Assert(animationObjectPropertyPanel != null);

            CurveEditorPanel.Initialize(animationObjectPropertyPanel, ActiveTarget.ObjectID);

            CurveEditorPanel.UpdateTreeView += (s, e) => UpdateTreeView();
            //CurveEditorPanel.TreeViewAfterCheck   += (s, e) => UpdateTreeViewAfterCheck();
            CurveEditorPanel.ChangeSelectedCurves += (s, e) => UpdateSelected();
            CurveEditorPanel.ChangeVisibledCurves += (s, e) => UpdateVisibled();
            CurveEditorPanel.GetFrameCount = () => ActiveTarget.Data.shader_param_anim_info.frame_count;

            // 選択オブジェクトが変更になったら更新する
            {
                EventHandler SelectedTargetChanged = (s, e) =>
                {
                    if (Targets.Active is ShaderParameterAnimation && (Owner as AnimationObjectPropertyPanel).IsOnlyTargetShowNode)
                    {
                        UpdateFormInternal(new UpdateFormInfo()
                        {
                            IsPageCreated = false,
                            TargetOrPageChanged = false,
                        });
                    }
                };

                App.AppContext.SelectedTargetChanged += SelectedTargetChanged;
                Disposed += (s, a) => App.AppContext.SelectedTargetChanged -= SelectedTargetChanged;
            }
        }

        public override Size DefaultPageSize
        {
            get
            {
                return new Size(
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width),
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height));
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if ((Owner != null) && (Owner.ActivePage == this) && !ObjectPropertyDialog.InternallyChangingSize.IsChanging &&
                Owner.Owner.WindowState == System.Windows.Forms.FormWindowState.Normal)
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width = Width;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height = Height;
            }

            base.OnSizeChanged(e);
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            CurveEditorPanel.TargetGroup = Targets;

            CurveEditorPanel.UpdateForm(updateFormInfo);
        }

        protected override void UpdateFormOnPageCreatedInternal()
        {
            CurveEditorPanel.UpdateFormOnPageCreatedInternal();
        }

        public override void BeforePageDeactivated()
        {
            CurveEditorPanel.BeforePageDeactivated();
        }

        public override void AfterPageActiveted(ObjectPropertyPage oldPage)
        {
            CurveEditorPanel.AfterPageActivated();
        }

        private void UpdateSelected()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private void UpdateVisibled()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private string MakePathFromIdRecusive(CurveTreeNode node)
        {
            Debug.Assert(node != null);
            var parent = node.Parent as CurveTreeNode;
            return parent == null
                ? node.id
                : this.MakePathFromIdRecusive(parent) + "\\" + node.id;
        }

        private Dictionary<string, bool> visibledNodes_ = new Dictionary<string, bool>();
        private readonly Dictionary<string, bool> isExpandedNodes_ = new Dictionary<string, bool>();

        private class UniformOrGroup
        {
            public CurveTreeInfo info;
            public uniform_varType uniform;
            public groupType group;
            public string Group()
            {
                return uniform != null ? uniform.Group() : group.Group();
            }
            public int Order()
            {
                return uniform != null ? uniform.Order() : group.Order();
            }
        }

        private void UpdateTreeView()
        {
            using (var sb = new UIControlEventSuppressBlock())
            {
                // ノードの状態を保存しておく
                {
                    if (CurveEditorPanel.CurveTreeView.Nodes.Count > 0)
                    {
                        var root = (CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0];
                        foreach (var tuple in CurveTreeView.NodesAndFullPath(root, root.id))
                        {
                            // チェック状態
                            visibledNodes_[tuple.Item2] = tuple.Item1.Checked;

                            // 開閉状態
                            isExpandedNodes_[tuple.Item2] = tuple.Item1.Nodes.Count == 0 || tuple.Item1.IsExpanded;
                        }
                    }
                }

                int oldScrollX = CurveEditorPanel.CurveTreeView.ScrollX;
                int oldScrollY = CurveEditorPanel.CurveTreeView.ScrollY;

                bool isAllShowNode = (Owner as AnimationObjectPropertyPanel).IsAllShowNode;
                bool isShaderParamAnimGroupCondShow = (Owner as AnimationObjectPropertyPanel).IsShaderParamAnimGroupCondShow;
                bool isOnlyTargetShowNode = (Owner as AnimationObjectPropertyPanel).IsOnlyTargetShowNode;

                // 現在選択されているマテリアル
                var selectedMaterials = App.AppContext.SelectedTarget.GetObjects(GuiObjectID.Material).Select(x => x as Material);

                // つくり直す
                var rootNode = new CurveTreeInfo()
                {
                    ShowEmptyNode = true,
                    Text = ActiveTarget.Name,
                    Id = ActiveTarget.Name,
                    Category = CurveTreeNodeCategory.ShaderParamAnim,
                };

                {
                    // マテリアルに付随するシェーダーパラメーター
                    foreach (var shaderParamMatAnim in ActiveTarget.ShaderParamAnims)
                    {
                        string materialName = shaderParamMatAnim.mat_name;

                        // 未選択されているマテリアルは無視する
                        if (isOnlyTargetShowNode && selectedMaterials.Any(x => (x.Name == materialName) && x.Referrers.Any(y => y.AllAnimations.Any(z => z.Name == ActiveTarget.Name))) == false)
                        {
                            continue;
                        }

                        var materialNode = new CurveTreeInfo()
                        {
                            ShowEmptyNode = true,
                            Text = materialName,
                            Id = materialName,
                            Category = CurveTreeNodeCategory.ShaderParamMatAnim,
                        };

                        var boundMaterials = ActiveTarget.GetBoundMaterials(shaderParamMatAnim.mat_name);
                        var boundMaterial = boundMaterials.FirstOrDefault(x => x.Name == materialName);
                        Definition.ShadingModelTable table = Definition.GetTable(
                            boundMaterial != null ? boundMaterial.MaterialShaderAssign.Definition : null);
                        var uniformNodes = new List<UniformOrGroup>();

                        VisibleInfo visibleInfo;
                        if (boundMaterial != null && boundMaterial.MaterialShaderAssign.Definition != null)
                        {
                            visibleInfo = new VisibleInfo(boundMaterial);
                        }
                        else
                        {
                            visibleInfo = null;
                        }

                        if (boundMaterial?.MaterialShaderAssign?.ShadingModel != null)
                        {
                            foreach (var group in boundMaterial.MaterialShaderAssign.ShadingModel.Groups())
                            {

                                uniformNodes.Add(new UniformOrGroup()
                                {
                                    group = group,
                                });
                            }
                        }

                        var customUIs = boundMaterial?.GetAllAncestorMaterials().Where(x => x != null).ToArray();
                        var customUIUniforms = customUIs?.Select(x => x.CustomUI.uniform_vars).ToArray();
                        var customUIGroups = customUIs?.Select(x => x.CustomUI.groups).ToArray();
                        HashSet<string> dummy = new HashSet<string>();
                        foreach (var tuple in ActiveTarget.GetParamAnimState(shaderParamMatAnim))
                        {
                            ShaderParameterAnimation.ParamAnim paramAnim = tuple.Item1;
                            uniform_varType uniform_var = tuple.Item2;
                            AnimationDocument.NonEditableKind nonEditableKind = tuple.Item3;
                            var isBound = false;

                            switch (nonEditableKind)
                            {
                                case AnimationDocument.NonEditableKind.ShaderParamAnim_Invisible:
                                    // 常に表示しない
                                    continue;
                                case AnimationDocument.NonEditableKind.ShaderParamAnim_InvalidType:
                                    if (paramAnim.ParamAnimTargets.All(x => x.ExportType == CurveExportType.Ignored || x.ExportType == CurveExportType.Dependent))
                                    {
                                        // キーがない場合は表示しない
                                        continue;
                                    }
                                    break;
                                case AnimationDocument.NonEditableKind.Editable:
                                case AnimationDocument.NonEditableKind.ShaderParamAnim_NotFoundMaterial:
                                    isBound = true;
                                    break;
                            }

                            // 無い場合は作成
                            if (uniform_var == null)
                            {
                                uniform_var = new uniform_varType()
                                {
                                    type = paramAnim.type,
                                    id = paramAnim.id,
                                    hint = string.Empty,
                                    @default = null,
                                };
                            }

                            var labelHelper = new ShaderParamControls.LabelHelper();
                            var isParentLabel = false;
                            string customLabel = null;
                            if (boundMaterial != null)
                            {
                                customLabel = ShaderParamControls.ShaderParamControlGroup.ResolveCustomLabel(customUIUniforms, paramAnim.id,
                                    boundMaterial.CustomUI.uniform_vars, out isParentLabel);
                            }
                            labelHelper.SetLabel(paramAnim.id, uniform_var.Label(), customLabel, boundMaterial != null ? boundMaterial.CustomUI : CustomUI.DummyCustomUI, table, x => false, dummy, false, false, isParentLabel);
                            var text = labelHelper.GetLabelString(false);
                            if (!MaterialShaderPage.IsMatchFilter(text))
                            {
                                continue;
                            }

                            var paramNode = new CurveTreeInfo()
                            {
                                Text = text,
                                Id = paramAnim.id,
                                Category = CurveTreeNodeCategory.ParamAnim,
                            };

                            string displayAux = tuple.Item4;

                            AddElementNodes(paramNode, uniform_var, materialName);
                            // マテリアルが見つからないときは、カーブが存在するパラメータのみ表示
                            // (カーブが存在しないパラメータは保存時に削除されるため)
                            if (boundMaterial == null)
                            {
                                if (uniform_var.type == shader_param_typeType.texsrt)
                                {
                                    if (paramNode.Descendants().All(x => x.AnimationCurve == null || !x.AnimationCurve.HasKeyFrame() || x.Id == @"Mode"))
                                    {
                                        continue;
                                    }
                                }
                                else if (paramNode.Descendants().All(x => x.AnimationCurve == null || !x.AnimationCurve.HasKeyFrame()))
                                {
                                    continue;
                                }
                            }
                            uniformNodes.Add(new UniformOrGroup()
                            {
                                info = paramNode,
                                uniform = uniform_var
                            });
                            foreach (var node in paramNode.Descendants().Where(x => x.AnimationCurve != null))
                            {
                                node.IsBound = isBound;
                                node.NonEditableKind = nonEditableKind;
                                node.NonEditableKindDisplayAux = displayAux;
                                node.ShowEmptyNode = isAllShowNode && (isShaderParamAnimGroupCondShow || isBound);
                                ((ShaderParameterAnimationCurveTreeNodeInfo)node.AnimationCurve).UniformVarLabel = text;
                            }
                        }

                        Dictionary<string, groupType> groupDict = new Dictionary<string, groupType>();
                        Func<string, groupType> findGroupFromName = (groupName) =>
                        {
                            groupType group = null;
                            if (groupDict.TryGetValue(groupName, out group) == false)
                            {
                                group =
                                    (from material in boundMaterials
                                     let definition = material.MaterialShaderAssign.Definition
                                     where definition != null
                                     from g in definition.Data.Groups()
                                     where g.name == groupName
                                     select g).FirstOrDefault();

                                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1524
                                if (group == null)
                                {
                                    group = new groupType()
                                    {
                                        // 最低限のプロパティーのみ初期化する
                                        ui_order = new ui_orderType() { value = 0 },
                                        index = int.MinValue,
                                        name = groupName
                                    };
                                }

                                groupDict[groupName] = group;
                            }

                            return group;
                        };

                        var groupGroups = (uniformNodes.GroupBy(pair => pair.Group())).ToDictionary(x => x.Key);
                        // ページを無視したときもページの順序は考慮する
                        var pageOrders =
                            boundMaterial?.MaterialShaderAssign?.ShadingModel?.Pages()
                                .Select((item, index) => new { item, index })
                                .ToDictionary(x => x.item.name, x => new Tuple<int, int>(x.item.Order(), x.index)) ??
                            new Dictionary<string, Tuple<int, int>>();

                        IGrouping<string, UniformOrGroup> roots;
                        if (groupGroups.TryGetValue(string.Empty, out roots))
                        {
                            var orderedRoots = from p in roots
                                               let isParam = p.uniform != null
                                               let groupName = !isParam ? p.@group.name : ""
                                               let gt = groupName != "" ? findGroupFromName(groupName) : null
                                               let pageOrder = gt != null && pageOrders.ContainsKey(gt.page_name) ? pageOrders[gt.page_name] : new Tuple<int, int>(int.MinValue, int.MinValue)
                                               let order = gt?.Order() ?? 0
                                               let index = gt?.index ?? int.MinValue
                                               orderby isParam, pageOrder, order, index, groupName
                                               select p;

                            Predicate<string> visible = x => !string.IsNullOrEmpty(findGroupFromName(x).Label());

                            var groupToText = new Dictionary<string, string>();
                            Func<string, CurveTreeInfo> createGroup = name =>
                            {
                                var pairs = (from material in boundMaterials
                                             let definition = material.MaterialShaderAssign.Definition
                                             where definition != null
                                             from g in definition.Data.Groups()
                                             where g.name == name
                                             select new { material, g }).ToArray();

                                var labels = pairs.Select(x => x.g.Label()).Where(x => !string.IsNullOrEmpty(x)).Distinct().ToArray();

                                var info = new CurveTreeInfo()
                                {
                                    Id = name,
                                    Category = CurveTreeNodeCategory.ShaderParamGroup,
                                };

                                var labelHelper = new ShaderParamControls.LabelHelper();
                                var isParentLabel = false;
                                string customLabel = null;
                                if (boundMaterial != null)
                                {
                                    customLabel = ShaderParamControls.ShaderParamControlGroup.ResolveCustomLabel(customUIGroups, name, boundMaterial.CustomUI.groups, out isParentLabel);
                                }
                                labelHelper.SetLabel(name, labels.Length == 1 ? labels[0] : null, customLabel, boundMaterial != null ? boundMaterial.CustomUI : CustomUI.DummyCustomUI, table, x => false, dummy, false, false, isParentLabel);
                                string text = labelHelper.GetLabelString(false);
                                info.Text = text;
                                groupToText[name] = text;

                                string condition;
                                if (visibleInfo != null && !visibleInfo.IsConditionEvalSuccess(name, out condition))
                                {
                                    info.Image = Resources.Merge_MergeTargetSelectWarning;
                                    info.ImageToolTipString = Strings.ShaderParamControl_FailedConditionEval + "\n--\n" + condition;
                                }

                                return info;
                            };

                            foreach (var root in orderedRoots)
                            {
                                if (root.uniform != null)
                                {
                                    materialNode.Nodes.Add(root.info);
                                }
                                else
                                {
                                    IGrouping<string, UniformOrGroup> rootGroup;
                                    if (groupGroups.TryGetValue(root.group.name, out rootGroup))
                                    {
                                        materialNode.Nodes.AddRange(CreateCurveTreeInfo(rootGroup, groupGroups, findGroupFromName, createGroup, x => x == root.group.name || visible(x)));
                                    }
                                }
                            }

                            foreach (var node in materialNode.Descendants().Where(x => x.AnimationCurve != null))
                            {
                                string text;
                                if (node.NonEditableKindDisplayAux is string && groupToText.TryGetValue((string)node.NonEditableKindDisplayAux, out text))
                                {
                                    node.NonEditableKindDisplayAux = text;
                                }
                            }
                        }

                        if (materialNode.Nodes.Count > 0)
                        {
                            rootNode.Nodes.Add(materialNode);
                        }

                        SetGroupPath(materialNode, "");
                    }

                    rootNode.TrimInvisibleNodes();

                    // 文字列によるフィルター
                    CurveTreeInfo.FilterCurveTreeInfoRoot(rootNode, (Owner as AnimationObjectPropertyPanel).SearchTexts);

                    // TODO: 不要なら削除
                    foreach (var node in rootNode.Descendants())
                    {
                        if (node.AnimationCurve != null)
                        {
                            if (node.AnimationCurve.GetAnimTarget(ActiveTarget) == null)
                            {
                                Debug.Assert(false); // ここにはもう来ない？
                                node.AnimationCurve.CreateAnimTarget(ActiveTarget);
                            }
                        }
                    }
                }

                using (var lw = new LockWindowUpdate(CurveEditorPanel.CurveTreeView))
                {
                    var nodes = rootNode.ConvertToTreeNode();
                    if (nodes.IsSameStructure(CurveEditorPanel.CurveTreeView.Nodes) == false)
                    {
                        if ((CurveEditorPanel.UpdateFormInfo == null) || (CurveEditorPanel.UpdateFormInfo.TargetOrPageChanged == false))
                        {
                            // つくり直す


                            foreach (var tuple in CurveTreeView.NodesAndFullPath((CurveTreeNode)nodes, nodes.id))
                            {
                                // チェック状態を設定する
                                bool isChecked = false;
                                if (visibledNodes_.TryGetValue(tuple.Item2, out isChecked))
                                {
                                    if (isChecked != tuple.Item1.Checked)
                                    {
                                        tuple.Item1.Checked = isChecked;
                                    }
                                }

                                bool isExpanded = false;
                                if (isExpandedNodes_.TryGetValue(tuple.Item2, out isExpanded))
                                {
                                    if (isExpanded)
                                    {
                                        if (!tuple.Item1.IsExpanded)
                                        {
                                            tuple.Item1.Expand();
                                        }
                                    }
                                    else
                                    {
                                        if (tuple.Item1.IsExpanded)
                                        {
                                            tuple.Item1.Collapse();
                                        }
                                    }
                                }
                            }

                            CurveEditorPanel.CurveTreeView.Nodes.Clear();
                            CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);
                        }
                        else
                        {
                            foreach (var node in CurveTreeView.TreeNodes(nodes, x => true))
                            {
                                switch (node.Info.Category)
                                {
                                    case CurveTreeNodeCategory.ParamAnim:
                                    case CurveTreeNodeCategory.ShaderParamAnimTarget:
                                    case CurveTreeNodeCategory.ShaderParamGroup:
                                        break;
                                    default:
                                        node.Expand();
                                        break;
                                }
                            }

                            // つくり直す
                            CurveEditorPanel.CurveTreeView.Nodes.Clear();
                            CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);

                            // 新規に開いたときは全表示にする
                            CurveEditorPanel.VisibleAllNode();
                        }
                    }
                    else
                    {
                        ((CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0]).CopyInfo(nodes);
                    }

                    CurveEditorPanel.CurveTreeView.ScrollX = oldScrollX;
                    CurveEditorPanel.CurveTreeView.ScrollY = oldScrollY;
                }
            }
        }

        public void SetGroupPath(CurveTreeInfo node, string groupLabel)
        {
            if (node.Category == CurveTreeNodeCategory.ShaderParamGroup)
            {
                if (string.IsNullOrEmpty(groupLabel))
                {
                    groupLabel = node.Text;
                }
                else
                {
                    groupLabel += " / " + node.Text;
                }
            }
            else if (node.AnimationCurve != null)
            {
                ((ShaderParameterAnimationCurveTreeNodeInfo)node.AnimationCurve).GroupLabel = groupLabel;
            }

            foreach (var child in node.Nodes)
            {
                SetGroupPath(child, groupLabel);
            }
        }

        private IEnumerable<CurveTreeInfo> CreateCurveTreeInfo(
            IGrouping<string, UniformOrGroup> grouping,
            Dictionary<string, IGrouping<string, UniformOrGroup>> groupOrUniforms,
            Func<string, groupType> nameToGroup,
            Func<string, CurveTreeInfo> createGroupInfo,
            Predicate<string> visible)
        {
            var name = grouping.Key;

            var info = grouping.SelectMany(x =>
            {
                if (x.uniform != null)
                {
                    return Enumerable.Repeat(x.info, 1);
                }

                IGrouping<string, UniformOrGroup> child;
                if (groupOrUniforms.TryGetValue(x.group.name, out child))
                {
                    return CreateCurveTreeInfo(child, groupOrUniforms, nameToGroup, createGroupInfo, visible);
                }

                return Enumerable.Empty<CurveTreeInfo>();
            });

            if (visible(name))
            {
                var curveTreeInfo = createGroupInfo(name);
                curveTreeInfo.Nodes.AddRange(info);
                return Enumerable.Repeat(curveTreeInfo, 1);
            }

            return info;
        }



        // フルパス文字列からTexSRTモード部分を取り除いた文字列を作る
        private static readonly Regex texSrtModea_ =
            new Regex(
                "(" + UIText.TextureSrtMode(0.0f) + ")|(" + UIText.TextureSrtMode(1.0f) + ")|(" + UIText.TextureSrtMode(2.0f) + ")",
                RegexOptions.Compiled
            );

        private static string MakeFullPathWithoutMode(string src)
        {
            return texSrtModea_.Replace(src, string.Empty);
        }

        private void AddElementNodes(CurveTreeInfo parent, uniform_varType param, string materialName)
        {
            //var paramAnim = ((ShaderParameterAnimationCurveTreeNodeInfo)parent.AnimationCurve).GetAnimParam(ActiveTarget);
            parent.IsModified = false;
            var shaderParamAnim = ActiveTarget.ShaderParamAnims[materialName];
            if (shaderParamAnim != null)
            {
                var paramAnim = shaderParamAnim.ParamAnims[parent.Id];
                parent.IsModified = (paramAnim != null && paramAnim.IsModified);
            }

            var isAddedColorNodes = false;

            if ((param.ui_item != null) && (param.ui_item.value == ui_item_valueType.color))
            {
                isAddedColorNodes = AddElementNodes_color(parent, param, materialName);
            }

            if (isAddedColorNodes == false)
            {
                switch (param.type)
                {
                    case shader_param_typeType.srt3d:
                        AddElementNodes_srt3d(parent, param, materialName);
                        break;
                    case shader_param_typeType.srt2d:
                        AddElementNodes_texcoord(parent, param, materialName);
                        break;
                    case shader_param_typeType.texsrt:
                        AddElementNodes_texcoord(parent, param, materialName);

                        // parentの名前はModeに応じて変える
                        {
                            KeyFrame keyframe = null;
                            var children = parent.Nodes;
                            if (children != null && children.Any())
                            {
                                // Modeの値を取得
                                var child = children[0];
                                keyframe = child.AnimationCurve.KeyFrames.FirstOrDefault();
                            }

                            float mode = 0.0f;
                            if (keyframe != null)
                            {
                                mode = keyframe.Value;
                            }
                            string modeStr = " [" + UIText.TextureSrtMode(mode) + "]";
                            parent.Text += modeStr;
                            parent.IsModified |= children[0].IsModified;
                        }
                        break;
                    default:
                        AddElementNodes_edit(parent, param, materialName);
                        break;
                }
            }
        }

        private bool AddElementNodes_color(CurveTreeInfo parent, uniform_varType param, string materialName)
        {
            switch (param.type)
            {
                case shader_param_typeType.float3:
                    {
                        CurveTreeInfo nodeR = CreateTreeNode("R", ActiveTarget, param, materialName);
                        CurveTreeInfo nodeG = CreateTreeNode("G", ActiveTarget, param, materialName);
                        CurveTreeInfo nodeB = CreateTreeNode("B", ActiveTarget, param, materialName);

                        nodeR.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeG.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeB.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;

                        parent.Nodes.Add(nodeR);
                        parent.Nodes.Add(nodeG);
                        parent.Nodes.Add(nodeB);

                        return true;
                    }

                case shader_param_typeType.float4:
                    {
                        CurveTreeInfo nodeR = CreateTreeNode("R", ActiveTarget, param, materialName);
                        CurveTreeInfo nodeG = CreateTreeNode("G", ActiveTarget, param, materialName);
                        CurveTreeInfo nodeB = CreateTreeNode("B", ActiveTarget, param, materialName);
                        CurveTreeInfo nodeA = CreateTreeNode("A", ActiveTarget, param, materialName);

                        nodeR.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeG.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeB.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeA.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;

                        parent.Nodes.Add(nodeR);
                        parent.Nodes.Add(nodeG);
                        parent.Nodes.Add(nodeB);
                        parent.Nodes.Add(nodeA);

                        return true;
                    }
            }

            return false;
        }

        private void AddElementNodes_texcoord(CurveTreeInfo parent, uniform_varType param, string materialName)
        {
            Debug.Assert(
                ShaderParameterAnimationCurveTreeNodeInfo.IsTextureSrt(param) ||
                (param.type == shader_param_typeType.float4),
                string.Format("未実装 -- {0}", param.type)
            );

            for (int i = 0; i < (param.type == shader_param_typeType.srt2d ? 5 : 6); i++)
            {
                parent.Nodes.Add(CreateTreeNode_texcoord(i, ActiveTarget, param, materialName));
            }
        }

        private static readonly Color[] colorPreset_tex3d = new[] { Color.Red, Color.Green, Color.Blue,
                                                           Color.Red, Color.Green, Color.Blue,
                                                           Color.Red, Color.Green, Color.Blue };
        private void AddElementNodes_srt3d(CurveTreeInfo parent, uniform_varType param, string materialName)
        {

            int componentIndex = 0;
            for (int i = 0; i < 3; i++)
            {
                string caption0 = i == 0 ? "Scale " : i == 1 ? "Rotate " : "Translate ";
                for (int j = 0; j < 3; j++)
                {
                    string caption1 = j == 0 ? "U" : j == 1 ? "V" : "W";
                    var animationCurve = new ShaderParameterAnimationCurveTreeNodeInfo(ActiveTarget, param, materialName, componentIndex);
                    var animTarget = animationCurve.GetAnimTarget(ActiveTarget);
                    parent.Nodes.Add(new CurveTreeInfo()
                    {
                        Text = caption0 + caption1,
                        Id = Text,
                        ForeColor = colorPreset_tex3d[componentIndex],
                        Category = CurveTreeNodeCategory.ShaderParamAnimTarget,
                        AnimationCurve = animationCurve,
                        IsModified = animTarget == null ? false : animTarget.IsModified,
                    });
                    componentIndex++;
                }
            }
        }


        private void AddElementNodes_edit(CurveTreeInfo parent, uniform_varType param, string materialName)
        {
            switch (param.type)
            {
                case shader_param_typeType.@bool:
                case shader_param_typeType.@int:
                case shader_param_typeType.@uint:
                case shader_param_typeType.@float:
                    {
                        parent.AnimationCurve = new ShaderParameterAnimationCurveTreeNodeInfo(ActiveTarget, param, materialName, -1);
                        var animTarget = parent.AnimationCurve.GetAnimTarget(ActiveTarget);
                        parent.IsModified |= (animTarget != null && animTarget.IsModified);
                        break;
                    }

                case shader_param_typeType.bool2:
                case shader_param_typeType.int2:
                case shader_param_typeType.uint2:
                case shader_param_typeType.float2:
                    {
                        var nodeX = CreateTreeNode("X", ActiveTarget, param, materialName);
                        var nodeY = CreateTreeNode("Y", ActiveTarget, param, materialName);

                        nodeX.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeY.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;

                        parent.Nodes.Add(nodeX);
                        parent.Nodes.Add(nodeY);
                        break;
                    }

                case shader_param_typeType.bool3:
                case shader_param_typeType.int3:
                case shader_param_typeType.uint3:
                case shader_param_typeType.float3:
                    {
                        var nodeX = CreateTreeNode("X", ActiveTarget, param, materialName);
                        var nodeY = CreateTreeNode("Y", ActiveTarget, param, materialName);
                        var nodeZ = CreateTreeNode("Z", ActiveTarget, param, materialName);

                        nodeX.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeY.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeZ.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;

                        parent.Nodes.Add(nodeX);
                        parent.Nodes.Add(nodeY);
                        parent.Nodes.Add(nodeZ);

                        break;
                    }

                case shader_param_typeType.bool4:
                case shader_param_typeType.int4:
                case shader_param_typeType.uint4:
                case shader_param_typeType.float4:
                    {
                        var nodeX = CreateTreeNode("X", ActiveTarget, param, materialName);
                        var nodeY = CreateTreeNode("Y", ActiveTarget, param, materialName);
                        var nodeZ = CreateTreeNode("Z", ActiveTarget, param, materialName);
                        var nodeW = CreateTreeNode("W", ActiveTarget, param, materialName);

                        nodeX.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeY.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeZ.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;
                        nodeW.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;

                        parent.Nodes.Add(nodeX);
                        parent.Nodes.Add(nodeY);
                        parent.Nodes.Add(nodeZ);
                        parent.Nodes.Add(nodeW);

                        break;
                    }

                case shader_param_typeType.float2x2:
                case shader_param_typeType.float2x3:
                case shader_param_typeType.float2x4:
                case shader_param_typeType.float3x2:
                case shader_param_typeType.float3x3:
                case shader_param_typeType.float3x4:
                case shader_param_typeType.float4x2:
                case shader_param_typeType.float4x3:
                case shader_param_typeType.float4x4:
                    {
                        AddElementNodes_edit_matrix(parent, param, materialName);
                        break;
                    }
            }
        }

        private void AddElementNodes_edit_matrix(CurveTreeInfo parent, uniform_varType param, string materialName)
        {
            {
                int row = 0;
                int column = 0;

                switch (param.type)
                {
                    case shader_param_typeType.float2x2: row = 2; column = 2; break;
                    case shader_param_typeType.float2x3: row = 2; column = 3; break;
                    case shader_param_typeType.float2x4: row = 2; column = 4; break;
                    case shader_param_typeType.float3x2: row = 3; column = 2; break;
                    case shader_param_typeType.float3x3: row = 3; column = 3; break;
                    case shader_param_typeType.float3x4: row = 3; column = 4; break;
                    case shader_param_typeType.float4x2: row = 4; column = 2; break;
                    case shader_param_typeType.float4x3: row = 4; column = 3; break;
                    case shader_param_typeType.float4x4: row = 4; column = 4; break;
                    default: Debug.Assert(false); break;
                }

                for (int i = 0; i != row; ++i)
                {
                    for (int j = 0; j != column; ++j)
                    {
                        int componentIndex = i * column + j;

                        var node = CreateTreeNode_matrix(string.Format("{0}{1}", i, j), ActiveTarget, param, materialName, componentIndex, colorPreset_matrix[column]);

                        node.Category = CurveTreeNodeCategory.ShaderParamAnimTarget;

                        parent.Nodes.Add(node);
                    }
                }
            }
        }

        private static readonly Color[] colorPreset_vector4 = new[] { Color.Red, Color.Green, Color.Blue, Color.Black };
        private static CurveTreeInfo CreateTreeNode(string caption, ShaderParameterAnimation target, uniform_varType param, string materialName)
        {
            int componentIndex = 0;
            {
                switch (caption)
                {
                    case "R":
                    case "X": componentIndex = 0; break;
                    case "G":
                    case "Y": componentIndex = 1; break;
                    case "B":
                    case "Z": componentIndex = 2; break;
                    case "A":
                    case "W": componentIndex = 3; break;
                    default: componentIndex = 3; break;		// 行列
                }
            }
            var animationCurve = new ShaderParameterAnimationCurveTreeNodeInfo(target, param, materialName, componentIndex);
            var animTarget = animationCurve.GetAnimTarget(target);
            return
                new CurveTreeInfo()
                {
                    Text = caption,
                    Id = componentIndex.ToString(),
                    ForeColor = colorPreset_vector4[componentIndex],
                    AnimationCurve = animationCurve,
                    IsModified = animTarget == null ? false : animTarget.IsModified,
                };
        }

        private static readonly Color[] colorPreset_matrix2 = new[] { Color.Red, Color.Green };
        private static readonly Color[] colorPreset_matrix3 = new[] { Color.Red, Color.Green, Color.Blue };
        private static readonly Color[] colorPreset_matrix4 = new[] { Color.Red, Color.Green, Color.Blue, Color.Black };

        private static readonly Color[][] colorPreset_matrix = new Color[][] { null, null, colorPreset_matrix2, colorPreset_matrix3, colorPreset_matrix4 };

        private static CurveTreeInfo CreateTreeNode_matrix(string caption, ShaderParameterAnimation target, uniform_varType param, string materialName, int componentIndex, Color[] colorPreset)
        {
            var animationCurve = new ShaderParameterAnimationCurveTreeNodeInfo(target, param, materialName, componentIndex);
            var animTarget = animationCurve.GetAnimTarget(target);
            return
                new CurveTreeInfo()
                {
                    Text = caption,
                    Id = caption,
                    ForeColor = colorPreset[componentIndex % colorPreset.Length],
                    AnimationCurve = animationCurve,
                    IsModified = animTarget == null ? false : animTarget.IsModified,
                };
        }

        private static readonly Color[] colorPreset_texcoord = new[] { Color.Black, Color.Red, Color.Green, Color.Black, Color.Red, Color.Green };
        private static CurveTreeInfo CreateTreeNode_texcoord(int componentIndex, ShaderParameterAnimation target, uniform_varType param, string materialName)
        {
            var animationCurve = new ShaderParameterAnimationCurveTreeNodeInfo(target, param, materialName, componentIndex);
            var animTarget = animationCurve.GetAnimTarget(target);
            return
                    new CurveTreeInfo()
                    {
                        Text = ShaderParameterAnimationCurveTreeNodeInfo.texcoordComponentNames_[componentIndex + (param.type == shader_param_typeType.srt2d ? 1 : 0)],
                        Id = ShaderParameterAnimationCurveTreeNodeInfo.texcoordComponentNames_[componentIndex + (param.type == shader_param_typeType.srt2d ? 1 : 0)],
                        ForeColor = colorPreset_texcoord[componentIndex + (param.type == shader_param_typeType.srt2d ? 1 : 0)],
                        Category = CurveTreeNodeCategory.ShaderParamAnimTarget,
                        AnimationCurve = animationCurve,
                        IsModified = animTarget == null ? false : animTarget.IsModified,
                    };
        }

        /// <summary>
        /// カーブを選択する
        /// </summary>
        public void Select(IEnumerable<string> materials, string uniform)
        {
            // 表示対象の設定
            var panel = Owner as AnimationObjectPropertyPanel;
            if (panel == null)
            {
                return;
            }

            panel.SetAllShowNode(true);
            panel.SetOnlyTargetShowNode(false);
            CurveEditorPanel.InvisibleAllNode();

            // 選択の変更
            var curves = CurveEditorPanel.AllCurves.Where(x => x.Name == uniform && materials.Contains(x.ParentName)).ToList();
            curves.ForEach(curve => curve.KeyFrames.ForEach(key => key.Selected = true));
            CurveEditorPanel.VisibledCurves = curves;
            CurveEditorPanel.SelectedCurves = curves;

            TreeNode firstnode = null;
            var parents = new List<TreeNode>();
            // ツリービューの設定
            foreach (var node in CurveEditorPanel.CurveTreeView.AllTreeNodes)
            {
                if (!curves.Contains(node.Tag))
                {
                    continue;
                }

                this.CurveEditorPanel.CheckNodeIndividually(node, true);

                // ツリーで選択状態にするノード
                if (firstnode == null)
                {
                    firstnode = node;
                }

                // 親ノードはすべて開く
                var parent = node.Parent;
                while (parent != null)
                {
                    if (!parents.Contains(parent))
                    {
                        parents.Add(parent);
                    }
                    parent = parent.Parent;
                }
            }

            foreach (var parent in parents)
            {
                parent.Expand();
            }

            CurveEditorPanel.CurveTreeView.SelectedNode = firstnode;
            CurveEditorPanel.UpdateTreeViewVisibled();
        }

        /// <summary>
        /// カーブを選択しダイアログを開いた後にフィットする
        /// </summary>
        public void FitAfterSelect()
        {
            // スケールの設定
            CurveEditorPanel.FitAllViewIn(CurveEditorPanel.SelectedCurves, true);
        }

        #region コピー＆ペースト
        private class CopyData
        {
            public int frame_count { get; set; }
            public bool loop { get; set; }
            public float quantize_tolerance_tex_scale { get; set; }
            public float quantize_tolerance_tex_rotate { get; set; }
            public float quantize_tolerance_tex_translate { get; set; }
            public ShaderParameterAnimation.ShaderParamAnimList ShaderParamMatAnims { get; set; }
            public GuiObjectID SourceObjectID { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(ShaderParameterAnimation target)
        {
            return
                new CopyData()
                {
                    frame_count = target.Data.shader_param_anim_info.frame_count,
                    loop = target.Data.shader_param_anim_info.loop,
                    quantize_tolerance_tex_scale = target.Data.shader_param_anim_info.quantize_tolerance_tex_scale,
                    quantize_tolerance_tex_rotate = target.Data.shader_param_anim_info.quantize_tolerance_tex_rotate,
                    quantize_tolerance_tex_translate = target.Data.shader_param_anim_info.quantize_tolerance_tex_translate,
                    ShaderParamMatAnims = target.VisibleShaderParamMatAnim(),
                    SourceObjectID = target.ObjectID
                };
        }

        /// <summary>
        /// ペーストが有効か。
        /// </summary>
        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return CanPaste(copiedObjectInfo, copiedObject, ActiveTarget, Targets);
        }

        public static bool CanPaste(object copiedObjectInfo, object copiedObject, ShaderParameterAnimation activeTarget, GuiObjectGroup targets)
        {
            var copyData = (CopyData)copiedObject;
            var currentObjectID = activeTarget.ObjectID;

            if (currentObjectID != copyData.SourceObjectID)
            {
                return false;
            }

            foreach (ShaderParameterAnimation target in targets.GetObjects(GuiObjectID.ShaderParameterAnimation))
            {
                if (target.IsEditable == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject, ActiveTarget.ObjectID));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject, GuiObjectID objectID)
        {
            EditCommandSet commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
            using (var block = new Viewer.ViewerDrawSuppressBlock(AnimationCurveEditCommand.AnimationMessageFilter))
            {
                var copyData = (CopyData)pasteObject;
                commandSet.Add(ShaderParameterAnimationGeneralPage.CreateEditCommand_frame_count(targets, copyData.frame_count).Execute());
                commandSet.Add(ShaderParameterAnimationGeneralPage.CreateEditCommand_loop(targets, copyData.loop).Execute());

                commandSet.Add(
                    ShaderAnimationSettingPopup.CreateEditCommand_quantize(
                        targets,
                        targets.Active.ObjectID,
                    //
                        copyData.quantize_tolerance_tex_scale,
                        copyData.quantize_tolerance_tex_rotate,
                        copyData.quantize_tolerance_tex_translate,
                        false
                    ).Execute());
                commandSet.Add(
                    (new GeneralGroupReferenceEditCommand<ShaderParameterAnimation.ShaderParamAnimList>(
                        targets,
                        objectID,
                        targets.GetObjects(objectID).OfType<ShaderParameterAnimation>().Select(x => x.MergeInvisibleShaderParamAnimation(copyData.ShaderParamMatAnims)).ToArray(),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            Debug.Assert(target is ShaderParameterAnimation);
                            var shaderParamAnim = target as ShaderParameterAnimation;
                            swap = shaderParamAnim.ShaderParamAnims;
                            shaderParamAnim.ShaderParamAnims = (ShaderParameterAnimation.ShaderParamAnimList)data;

                            // 量子化後に送信するのでここでは転送しない！
                        }
                    )).Execute()
                );

                ShaderParameterAnimation active = (ShaderParameterAnimation)targets.Active;
                commandSet.Add(ShaderAnimationSettingPopup.CreateEditCommand_quantize(
                    targets,
                    objectID,
                    active.Data.shader_param_anim_info.quantize_tolerance_tex_scale,
                    active.Data.shader_param_anim_info.quantize_tolerance_tex_rotate,
                    active.Data.shader_param_anim_info.quantize_tolerance_tex_translate, false).Execute());

                var command = active.CreateUpdateBindCommand();
                if (command != null)
                {
                    commandSet.Add(command.Execute());
                }

                // 再転送を設定
                EventHandler reload = (s, e) =>
                {
                    foreach (var target in targets.GetObjects(objectID).OfType<ShaderParameterAnimation>())
                    {
                        Viewer.LoadOrReloadAnimation.Send(target);
                    }
                };

                commandSet.OnPostEdit += reload;
                reload(null, null);
            }

            commandSet.Reverse();
            return commandSet;
        }

        #endregion
    }
}

