﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class ShaderDefinitionShadingModelSubPage : ShaderDefinitionPropertyPage
    {
        public ShaderDefinitionShadingModelSubPage(int index) :
            base(PropertyPageID.ShaderDefinitionShadingModelSub)
        {
            index_ = index;

            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_shader_definition_property_window_shading_model_page;
            }
        }

        private readonly int index_;

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new ShaderDefinitionShadingModelSubPage((int)arg);
        }

        protected override void InitializeFormInternal()
        {
            tabSource.SelectedIndexChanged += delegate
            {
                ((ShaderDefinitionPropertyPanel)Owner).SelectedTabPageIndex = tabSource.SelectedIndex;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelSubPageTabIndex = tabSource.SelectedIndex;
            };
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            tabSource.SelectedIndex = ((ShaderDefinitionPropertyPanel)Owner).SelectedTabPageIndex;
            UpdateShadingModel();
        }

        // 無駄にスクロールされないように変更があったときのみ更新するために以前の Text を置いておく
        // TextBox.Text は set した値そのままが get できない場合がある
        private string lastVertexShaderText = null;
        private string lastFragmentShaderText = null;
        private string lastComputeShaderText = null;
        private string lastGeometryShaderText = null;

        private void UpdateShadingModel()
        {
            var targetShadingModel = ActiveTarget.Data.shading_model_array.shading_model[index_];

            // リビジョン
            var savedShadingModel = ActiveTarget.GetSavedShadingModel(targetShadingModel.name);
            lblRevision.IsModified = savedShadingModel != null && ActiveTarget.IsValueChanged(targetShadingModel, savedShadingModel, x => x.revision);
            ltbRevision.Text = targetShadingModel.revision.ToString();
            lblMaterialShader.IsModified = savedShadingModel != null && ActiveTarget.IsValueChanged(targetShadingModel, savedShadingModel, x => x.material_shader);
            ltbMaterialShader.Text = UIText.FlagYesNo(targetShadingModel.material_shader);

            var streams = ActiveTarget.BinaryStreams;
            if (targetShadingModel.macro_array != null)
            {
                UpdateList(lvwMacro, targetShadingModel.macro_array.macro);
            }
            else
            {
                UpdateList(lvwMacro, Enumerable.Empty<macroType>());
            }

            // 頂点シェーダー
            var vshader = targetShadingModel.vertex_stage;
            ltbVertexSouceName.Text = "";
            var vertexShaderText = "";
            if (vshader != null)
            {
                tabVertexShader.Enabled = true;
                if (0 <= vshader.src_index && vshader.src_index < ActiveTarget.Data.shader_src_array.shader_src.Length)
                {
                    var shader_src = ActiveTarget.Data.shader_src_array.shader_src[vshader.src_index];
                    ltbVertexSouceName.Text = shader_src.path;
                    if (0 <= shader_src.stream_index && shader_src.stream_index < streams.Count)
                    {
                        vertexShaderText = streams[shader_src.stream_index].StringData.Replace("\n", "\r\n");
                    }
                }
            }
            else
            {
                tabVertexShader.Enabled = false;
            }

            if (lastVertexShaderText != vertexShaderText)
            {
                tbxVertexShader.Text = vertexShaderText;
                lastVertexShaderText = vertexShaderText;
            }

            // フラグメントシェーダー
            var fshader = targetShadingModel.fragment_stage;
            ltbFragmentSourceName.Text = "";
            var fragmentShaderText = "";
            if (fshader != null)
            {
                tabFragmentShader.Enabled = true;
                if (0 <= fshader.src_index && fshader.src_index < ActiveTarget.Data.shader_src_array.shader_src.Length)
                {
                    var shader_src = ActiveTarget.Data.shader_src_array.shader_src[fshader.src_index];
                    ltbFragmentSourceName.Text = shader_src.path;
                    if (0 <= shader_src.stream_index && shader_src.stream_index < streams.Count)
                    {
                        fragmentShaderText = streams[shader_src.stream_index].StringData.Replace("\n", "\r\n");
                    }
                }
            }
            else
            {
                tabFragmentShader.Enabled = false;
            }

            if (lastFragmentShaderText != fragmentShaderText)
            {
                tbxFragmentShader.Text = fragmentShaderText;
                lastFragmentShaderText = fragmentShaderText;
            }

            // ジオメトリシェーダー
            var gshader = targetShadingModel.geometry_stage;
            ltbGeometrySourceName.Text = "";
            var geometryShaderText = "";
            if (gshader != null)
            {
                tabGeometryShader.Enabled = true;
                if (0 <= gshader.src_index && gshader.src_index < ActiveTarget.Data.shader_src_array.shader_src.Length)
                {
                    var shader_src = ActiveTarget.Data.shader_src_array.shader_src[gshader.src_index];
                    ltbGeometrySourceName.Text = shader_src.path;
                    if (0 <= shader_src.stream_index && shader_src.stream_index < streams.Count)
                    {
                        geometryShaderText = streams[shader_src.stream_index].StringData.Replace("\n", "\r\n");
                    }
                }
            }
            else
            {
                tabGeometryShader.Enabled = false;
            }

            if (lastGeometryShaderText != geometryShaderText)
            {
                tbxGeometryShader.Text = geometryShaderText;
                lastGeometryShaderText = geometryShaderText;
            }

            // コンピュートシェーダー
            var cshader = targetShadingModel.compute_stage;
            ltbComputeSourceName.Text = "";
            var computeShaderText = "";
            if (cshader != null)
            {
                tabComputeShader.Enabled = true;
                if (0 <= cshader.src_index && cshader.src_index < ActiveTarget.Data.shader_src_array.shader_src.Length)
                {
                    var shader_src = ActiveTarget.Data.shader_src_array.shader_src[cshader.src_index];
                    ltbComputeSourceName.Text = shader_src.path;
                    if (0 <= shader_src.stream_index && shader_src.stream_index < streams.Count)
                    {
                        computeShaderText = streams[shader_src.stream_index].StringData.Replace("\n", "\r\n");
                    }
                }
            }
            else
            {
                tabComputeShader.Enabled = false;
            }

            if (lastComputeShaderText != computeShaderText)
            {
                tbxComputeShader.Text = computeShaderText;
                lastComputeShaderText = computeShaderText;
            }

            UpdateUniformList(			lvwUniform,			targetShadingModel.UniformBlocks());
            UpdateBufferList(           lvwBuffer,          targetShadingModel.BufferBlocks());
            UpdateSamplerList(			lvwSampler,			targetShadingModel.Samplers());
            UpdateAttribList(			lvwAttrib,			targetShadingModel.Attributes());
            UpdateOptionList(			lvwOption,			targetShadingModel.Options());
            UpdateRenderInfoSlotList(	lvwRenderInfoSlot,	targetShadingModel.RenderInfoSlots());
            UpdateInterleaveList(		lvwInterleave,		targetShadingModel.Interleave());
            UpdateGroupList(lvwGroup, targetShadingModel.GroupsExceptDefaultAttribGroup());
            UpdatePageList(lvwPage, targetShadingModel.Pages());
            UpdateTextBlockList(lvwTextBlock, targetShadingModel.textblock_array);

            SetupContextMenu(lvwUniform,        ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.UniformColumns);
            SetupContextMenu(lvwBuffer,         ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.BufferColumns);
            SetupContextMenu(lvwSampler,        ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.SamplerColumns);
            SetupContextMenu(lvwAttrib,         ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.AttribColumns);
            SetupContextMenu(lvwOption,         ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.OptionColumns);
            SetupContextMenu(lvwRenderInfoSlot, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.RenderInfoSlotColumns);
            SetupContextMenu(lvwInterleave,     ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.InterleaveColumns);
            SetupContextMenu(lvwGroup,          ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.GroupColumns);
            SetupContextMenu(lvwPage,           ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.PageColumns);
            SetupContextMenu(lvwMacro,          ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.MacroColumns);
            SetupContextMenu(lvwTextBlock,      ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.TextBlockColumns);

            tmsResetColumnOrder.Click += (s, e) =>
            {
                var tag = tmsResetColumnOrder.Tag as Tuple<UIListView, List<string>>;
                if (tag == null) return;

                var lvw = tag.Item1;
                tag.Item2.Clear();
                var headers = lvw.Columns.OfType<System.Windows.Forms.ColumnHeader>().ToArray();
                using (new UpdateBlock(lvw))
                {
                    foreach (var column in headers)
                    {
                        column.DisplayIndex = column.Index;
                    }
                }
            };
        }

        void SetupContextMenu(UIListView lvwList, List<string> config)
        {
            lvwList.ColumnContextMenuPopup += (s, e) =>
            {
                tmsResetColumnOrder.Tag = new Tuple<UIListView, List<string>>(lvwList, config);
                cmsResetColumnOrder.Show(this, PointToClient(lvwList.PointToScreen(e.Location)));
            };

        }

        private void UpdateList(UIListView lvwList, IEnumerable<macroType> macros)
        {
            using(var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(macros.Count());
                //lvwList.Clear();

                int index = 0;
                foreach (var macro in macros)
                {
                    var item = lvwList.Items[index];

                    Debug.Assert(clhIndexV.Index >= 0);
                    item.SubItems[clhIndexV.Index].Text = index.ToString();
                    item.SubItems[clhNameV.Index].Text = macro.name;
                    item.SubItems[clhValueV.Index].Text = macro.value;

                    ++ index;
                }
            }
        }

        private void UpdateUniformList(UIListView lvwList, IEnumerable<block_varType> blocks)
        {
            using(var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(blocks.Where(x => x.uniform_var_array != null).Select(x => x.uniform_var_array.uniform_var.Length).DefaultIfEmpty()
                    .Aggregate((x, y)=>x + y));

                int index = 0;
                foreach (var block in blocks)
                {
                    if (block.uniform_var_array == null)
                    {
                        continue;
                    }
                    foreach (var numericParam in block.uniform_var_array.uniform_var)
                    {
                        var item = lvwList.Items[index];

                        item.SubItems[clhUniformIndexV.Index].Text = index.ToString();
                        item.SubItems[clhUniformNameV.Index].Text = numericParam.id;
                        item.SubItems[clhUniformHintV.Index].Text = numericParam.hint;
                        item.SubItems[clhUniformTypeV.Index].Text = numericParam.type.ToString();
                        item.SubItems[clhUniformDefaultV.Index].Text = numericParam.@default != null ? IfUtility.MakeArrayString(numericParam.@default).TrimEnd("\t".ToArray()).Replace("\t", ", ") : string.Empty;
                        item.SubItems[clhConverter.Index].Text = numericParam.converter;
                        item.SubItems[clhDepend.Index].Text = numericParam.depend;

                        item.SubItems[clhUniformVertexSymbolV.Index].Text   = (numericParam.vertex_symbol   != null) ? numericParam.vertex_symbol.name   : string.Empty;
                        item.SubItems[clhUniformGeometrySymbolV.Index].Text = (numericParam.geometry_symbol != null) ? numericParam.geometry_symbol.name : string.Empty;
                        item.SubItems[clhUniformFragmentSymbolV.Index].Text = (numericParam.fragment_symbol != null) ? numericParam.fragment_symbol.name : string.Empty;
                        item.SubItems[clhUniformComputeSymbolV.Index].Text  = (numericParam.compute_symbol  != null) ? numericParam.compute_symbol.name  : string.Empty;

                        item.SubItems[clhUniformNostripV.Index].Text = UIText.FlagYesNo(numericParam.nostrip);

                        item.SubItems[clhUniformUiLabelV.Index].Text = numericParam.Label();
                        item.SubItems[clhUniformUiCommentV.Index].Text = numericParam.Comment();
                        item.SubItems[clhUniformUiGroupV.Index].Text = numericParam.Group();
                        item.SubItems[clhUniformUiOrderV.Index].Text = numericParam.Order().ToString();
                        item.SubItems[clhUniformUiItemV.Index].Text = UIText.EnumValue(numericParam.Item());

                        item.SubItems[clhUniformUiMinV.Index].Text = numericParam.ui_min != null ? numericParam.ui_min.value.ToString() : string.Empty;
                        item.SubItems[clhUniformUiMaxV.Index].Text = numericParam.ui_max != null ? numericParam.ui_max.value.ToString() : string.Empty;
                        item.SubItems[clhUniformUiDefaultMin.Index].Text = numericParam.ui_default_min != null ? numericParam.ui_default_min.value.ToString() : string.Empty;
                        item.SubItems[clhUniformUiDefaultMax.Index].Text = numericParam.ui_default_max != null ? numericParam.ui_default_max.value.ToString() : string.Empty;

                        item.SubItems[clhUniformUiVisibleV.Index].Text = UIText.EnumValue(numericParam.Visible());
                        item.SubItems[clhUniformBlock.Index].Text = block.id;
                        item.SubItems[clhUniformBlockType.Index].Text = UIText.EnumValue(block.type);
                        ++index;
                    }
                }
            }
        }

        private void UpdateBufferList(UIListView lvwList, IEnumerable<shader_storage_block_varType> buffers)
        {
            using (var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(buffers.Where(x => x.buffer_var_array != null).Select(x => x.buffer_var_array.buffer_var.Length).DefaultIfEmpty()
                    .Aggregate((x, y) => x + y));

                int index = 0;
                foreach (var buffer in buffers)
                {
                    if (buffer.buffer_var_array == null)
                    {
                        continue;
                    }
                    foreach (var numericParam in buffer.buffer_var_array.buffer_var)
                    {
                        var item = lvwList.Items[index];

                        item.SubItems[clhBufferIndexV.Index].Text = index.ToString();
                        item.SubItems[clhBufferNameV.Index].Text = numericParam.id;

                        item.SubItems[clhBufferVertexSymbolV.Index].Text = (numericParam.vertex_symbol != null) ? numericParam.vertex_symbol.name : string.Empty;
                        item.SubItems[clhBufferGeometrySymbolV.Index].Text = (numericParam.geometry_symbol != null) ? numericParam.geometry_symbol.name : string.Empty;
                        item.SubItems[clhBufferFragmentSymbolV.Index].Text = (numericParam.fragment_symbol != null) ? numericParam.fragment_symbol.name : string.Empty;
                        item.SubItems[clhBufferComputeSymbolV.Index].Text = (numericParam.compute_symbol != null) ? numericParam.compute_symbol.name : string.Empty;

                        item.SubItems[clhBufferNostripV.Index].Text = UIText.FlagYesNo(numericParam.nostrip);
                        item.SubItems[clhBufferBlock.Index].Text = buffer.id;
                        item.SubItems[clhBufferBlockType.Index].Text = UIText.EnumValue(buffer.type);

                        ++index;
                    }
                }
            }
        }

        private void UpdateSamplerList(UIListView lvwList, IEnumerable<sampler_varType> samplerParams)
        {
            Debug.Assert(samplerParams != null);

            using(var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(samplerParams.Count());

                int index = 0;
                foreach (var samplerParam in samplerParams)
                {
                    var item = lvwList.Items[index];

                    item.SubItems[clhSamplerIndexV.Index].Text	= index.ToString();
                    item.SubItems[clhSamplerNameV.Index].Text	= samplerParam.id;
                    item.SubItems[clhSamplerHintV.Index].Text	= samplerParam.hint;
                    item.SubItems[clhSamplerTypeV.Index].Text	= samplerParam.type.ToString();

                    item.SubItems[clhSamplerVertexSymbolV.Index].Text	= (samplerParam.vertex_symbol   != null) ? samplerParam.vertex_symbol.name   : string.Empty;
                    item.SubItems[clhSamplerGeometrySymbolV.Index].Text	= (samplerParam.geometry_symbol != null) ? samplerParam.geometry_symbol.name : string.Empty;
                    item.SubItems[clhSamplerFragmentSymbolV.Index].Text	= (samplerParam.fragment_symbol != null) ? samplerParam.fragment_symbol.name : string.Empty;
                    item.SubItems[clhSamplerComputeSymbolV.Index].Text  = (samplerParam.compute_symbol  != null) ? samplerParam.compute_symbol.name : string.Empty;

                    item.SubItems[clhSamplerAltV.Index].Text = samplerParam.alt;
                    item.SubItems[clhSamplerNostripV.Index].Text	= UIText.FlagYesNo(samplerParam.nostrip);

                    item.SubItems[clhSamplerUiLabelV.Index].Text	= samplerParam.Label();
                    item.SubItems[clhSamplerUiCommentV.Index].Text	= samplerParam.Comment();
                    item.SubItems[clhSamplerUiGroupV.Index].Text	= samplerParam.Group();
                    item.SubItems[clhSamplerUiOrderV.Index].Text	= samplerParam.Order().ToString();
                    item.SubItems[clhSamplerUiVisibleV.Index].Text = UIText.EnumValue(samplerParam.Visible());//UIText.FlagYesNo(samplerParam.Visible());

                    ++ index;
                }
            }
        }

        private void UpdateAttribList(UIListView lvwList, IEnumerable<attrib_varType> attribParams)
        {
            Debug.Assert(attribParams != null);

            using(var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(attribParams.Count());

                int index = 0;
                foreach (var attribParam in attribParams)
                {
                    var item = lvwList.Items[index];

                    item.SubItems[clhAttribIndexV.Index].Text	= index.ToString();
                    item.SubItems[clhAttribNameV.Index].Text	= attribParam.id;
                    item.SubItems[clhAttribHintV.Index].Text	= attribParam.hint;
                    item.SubItems[clhAttribTypeV.Index].Text	= attribParam.type.ToString();
                    item.SubItems[clhAttribVertexSymbol.Index].Text = (attribParam.vertex_symbol != null) ? attribParam.vertex_symbol.name : string.Empty;
                    item.SubItems[clhAttribUiEditable.Index].Text = UIText.FlagYesNo(attribParam.Editable());
                    item.SubItems[clhAttribUiComment.Index].Text = attribParam.Comment();
                    item.SubItems[clhAttribUiLabel.Index].Text = attribParam.Label();
                    item.SubItems[clhAttribUiGroup.Index].Text = attribParam.Group();
                    item.SubItems[clhAttribUiOrder.Index].Text = attribParam.Order().ToString();

                    ++ index;
                }
            }
        }

        private void UpdateOptionList(UIListView lvwList, IEnumerable<option_varType> optionParams)
        {
            Debug.Assert(optionParams != null);

            using(var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(optionParams.Count());

                int index = 0;
                foreach (var optionParam in optionParams)
                {
                    var item = lvwList.Items[index];

                    item.SubItems[clhOptionIndexV.Index].Text	= index.ToString();
                    item.SubItems[clhOptionNameV.Index].Text	= optionParam.id;
                    item.SubItems[clhOptionTypeV.Index].Text = optionParam.type.ToString();
                    item.SubItems[clhOptionChoiceV.Index].Text	= optionParam.choice;
                    item.SubItems[clhOptionDefaultV.Index].Text = optionParam.@default != null ? optionParam.@default.Trim().TrimEnd("\t".ToArray()).Replace("\t", ", ") : string.Empty;
                    item.SubItems[clhOptionBranch.Index].Text = UIText.FlagYesNo(optionParam.branch);

                    item.SubItems[clhOptionVertexSymbolV.Index].Text	= (optionParam.vertex_symbol   != null) ? optionParam.vertex_symbol.name   : string.Empty;
                    item.SubItems[clhOptionGeometrySymbolV.Index].Text	= (optionParam.geometry_symbol != null) ? optionParam.geometry_symbol.name : string.Empty;
                    item.SubItems[clhOptionFragmentSymbolV.Index].Text	= (optionParam.fragment_symbol != null) ? optionParam.fragment_symbol.name : string.Empty;
                    item.SubItems[clhOptionComputeSymbolV.Index].Text   = (optionParam.compute_symbol  != null) ? optionParam.compute_symbol.name  : string.Empty;

                    item.SubItems[clhOptionUiLabelV.Index].Text		= optionParam.Label();
                    item.SubItems[clhOptionUiCommentV.Index].Text	= optionParam.Comment();
                    item.SubItems[clhOptionUiGroupV.Index].Text		= optionParam.Group();
                    item.SubItems[clhOptionUiOrderV.Index].Text     = optionParam.Order().ToString();
                    item.SubItems[clhOptionUiVisibleV.Index].Text   = UIText.EnumValue(optionParam.Visible());

                    ++ index;
                }
            }
        }

        private void UpdateInterleaveList(UIListView lvwList, IEnumerable<interleaveType> interleaveParams)
        {
            Debug.Assert(interleaveParams != null);

            using(var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(interleaveParams.Count());

                int index = 0;
                foreach (var interleaveParam in interleaveParams)
                {
                    var item = lvwList.Items[index];

                    item.SubItems[clhInterleaveIndexV.Index].Text		= index.ToString();
                    item.SubItems[clhInterleaveIdSetV.Index].Text		= interleaveParam.id_set;
                    item.SubItems[clhInterleaveBinarizeV.Index].Text	= UIText.FlagYesNo(interleaveParam.binarize);

                    ++ index;
                }
            }
        }

        private void UpdateRenderInfoSlotList(UIListView lvwList, IEnumerable<render_info_slotType> renderInfoSlotParams)
        {
            Debug.Assert(renderInfoSlotParams != null);

            using(var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(renderInfoSlotParams.Count());

                int index = 0;
                foreach (var renderInfoSlotParam in renderInfoSlotParams)
                {
                    var item = lvwList.Items[index];

                    item.SubItems[clhRenderInfoSlotIndexV.Index].Text		= index.ToString();
                    item.SubItems[clhRenderInfoSlotNameV.Index].Text		= renderInfoSlotParam.name;
                    item.SubItems[clhRenderInfoSlotTypeV.Index].Text = renderInfoSlotParam.type.ToString();
                    item.SubItems[clhRenderInfoSlotChoiceV.Index].Text = renderInfoSlotParam.choice;
                    item.SubItems[clhRenderInfoSlotCountV.Index].Text		= renderInfoSlotParam.count.ToString();
                    item.SubItems[clhRenderInfoSlotOptionalV.Index].Text = UIText.FlagYesNo(renderInfoSlotParam.optional);
                    item.SubItems[clhRenderInfoSlotDefault.Index].Text = renderInfoSlotParam.Default();
                    item.SubItems[clhRenderInfoSlotUiLabelV.Index].Text		= renderInfoSlotParam.Label();
                    item.SubItems[clhRenderInfoSlotUiCommentV.Index].Text	= renderInfoSlotParam.Comment();
                    item.SubItems[clhRenderInfoSlotUiGroupV.Index].Text = renderInfoSlotParam.Group();
                    item.SubItems[clhRenderInfoSlotUiOrderV.Index].Text = renderInfoSlotParam.Order().ToString();
                    item.SubItems[clhRenderInfoSlotUiItemV.Index].Text = UIText.EnumValue(renderInfoSlotParam.Item());
                    item.SubItems[clhRenderInfoSlotUiVisibleV.Index].Text = UIText.EnumValue(renderInfoSlotParam.Visible());
                    ++ index;
                }
            }
        }

        private void UpdateGroupList(UIListView lvwList, IEnumerable<groupType> groups)
        {
            Debug.Assert(groups != null);

            using (var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(groups.Count());

                int index = 0;
                foreach (var group in groups)
                {
                    var item = lvwList.Items[index];

                    item.SubItems[clhGroupIndexV.Index].Text = index.ToString();
                    item.SubItems[clhGroupNameV.Index].Text = group.name;
                    item.SubItems[clhGroupConditionV.Index].Text = group.condition;
                    item.SubItems[clhGroupUiLableV.Index].Text = group.Label();
                    item.SubItems[clhGroupUiCommentV.Index].Text = group.Comment();
                    item.SubItems[clhGroupUiGroup.Index].Text = group.Group();
                    item.SubItems[clhGroupPage.Index].Text = group.page_name;
                    item.SubItems[clhGroupUiOrderV.Index].Text = group.Order().ToString();
                    ++index;
                }
            }
        }

        private void UpdatePageList(UIListView lvwList, IEnumerable<pageType> pages)
        {
            Debug.Assert(pages != null);

            using (var ub = new UpdateBlock(lvwList))
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(pages.Count());

                int index = 0;
                foreach (var page in pages)
                {
                    var item = lvwList.Items[index];

                    item.SubItems[clhPageIndex.Index].Text = index.ToString();
                    item.SubItems[clhPageName.Index].Text = page.name;
                    item.SubItems[clhPageUiLabel.Index].Text = page.Label();
                    item.SubItems[clhPageUiComment.Index].Text = page.Comment();
                    item.SubItems[clhPageUiOrder.Index].Text = page.Order().ToString();
                    ++index;
                }
            }
        }

        private void UpdateTextBlockList(UIListView lvwList, textblock_arrayType textblockArray)
        {
            if (textblockArray != null && textblockArray.textblock != null)
            {
                // リストアイテムを予め用意する
                lvwList.SetItemCount(textblockArray.textblock.Length);

                int index = 0;
                foreach (var textblock in textblockArray.textblock)
                {
                    var item = lvwList.Items[index];

                    item.SubItems[clhTextBlockIndexV.Index].Text = index.ToString();
                    item.SubItems[clhTextBlockNameV.Index].Text = textblock.id;
                    item.SubItems[clhTextBlockTextV.Index].Text = textblock.text;
                    item.SubItems[clhTextBlockUiGroupV.Index].Text = (textblock.ui_group != null) ? textblock.ui_group.group_name : string.Empty;
                    item.SubItems[clhTextBlockUiOrderV.Index].Text = (textblock.ui_order != null) ? textblock.ui_order.value.ToString() : string.Empty;
                    ++index;
                }
            }
        }

        private void UpdateOrder(UIListView lvw, List<string> columns)
        {
            if (columns.Any())
            {
                var order = new Dictionary<string, int>();
                foreach (var column in columns)
                {
                    order[column] = order.Count;
                }

                var headers = lvw.Columns.OfType<System.Windows.Forms.ColumnHeader>().ToArray();
                var ordered = headers.OrderBy(x =>
                {
                    int v;
                    if (!order.TryGetValue(x.Text, out v))
                    {
                        v = order.Count;
                    }
                    return new Tuple<int, int>(v, x.DisplayIndex);
                }).ToArray();

                using (new UpdateBlock(lvw))
                {
                    for (int i = 0; i < ordered.Length; i++)
                    {
                        ordered[i].DisplayIndex = i;
                    }
                }
            }
        }

        private void UpdateConfig(UIListView lvw, List<string> config, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            config.Clear();
            config.AddRange(lvw.Columns.OfType<System.Windows.Forms.ColumnHeader>().OrderBy(x => x.DisplayIndex == e.OldDisplayIndex ? e.NewDisplayIndex - 0.5 : x.DisplayIndex)
                .Select(x => x.Text));
        }

        private void lvwUniform_VisibleChanged(object sender, System.EventArgs e)
        {
            UpdateOrder(lvwUniform, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.UniformColumns);
        }

        private void lvwUniform_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwUniform, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.UniformColumns, e);
        }

        private void lvwSampler_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwSampler, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.SamplerColumns);
        }

        private void lvwSampler_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwSampler, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.SamplerColumns, e);
        }

        private void lvwAttrib_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwAttrib, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.AttribColumns);
        }

        private void lvwAttrib_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwAttrib, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.AttribColumns, e);
        }

        private void lvwOption_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwOption, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.OptionColumns, e);
        }

        private void lvwOption_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwOption, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.OptionColumns);
        }

        private void lvwRenderInfoSlot_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwRenderInfoSlot, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.RenderInfoSlotColumns, e);
        }

        private void lvwRenderInfoSlot_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwRenderInfoSlot, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.RenderInfoSlotColumns);
        }

        private void lvwInterleave_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwInterleave, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.InterleaveColumns);
        }

        private void lvwInterleave_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwInterleave, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.InterleaveColumns, e);
        }

        private void lvwGroup_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwGroup, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.GroupColumns);
        }

        private void lvwGroup_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwGroup, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.GroupColumns, e);
        }

        private void lvwPage_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwPage, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.PageColumns);
        }

        private void lvwPage_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwPage, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.PageColumns, e);
        }

        private void lvwMacro_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwMacro, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.MacroColumns);
        }

        private void lvwMacro_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwMacro, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.MacroColumns, e);
        }

        private void lvwBuffer_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwBuffer, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.BufferColumns);
        }

        private void lvwBuffer_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwBuffer, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.BufferColumns, e);
        }

        private void lvwTextBlock_VisibleChanged(object sender, EventArgs e)
        {
            UpdateOrder(lvwTextBlock, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.TextBlockColumns);
        }

        private void lvwTextBlock_ColumnReordered(object sender, System.Windows.Forms.ColumnReorderedEventArgs e)
        {
            UpdateConfig(lvwTextBlock, ConfigData.ApplicationConfig.Setting.PropertyEdit.ShadingModelPage.TextBlockColumns, e);
        }

        private int lvw_CompareItem(object sender, CompareListViewItemEventArgs e)
        {
            if (e.ColumnHeader.Index == 0)
            {
                int v1;
                int v2;
                int.TryParse(e.Item1.Text, out v1);
                int.TryParse(e.Item2.Text, out v2);
                return e.CompareInt(v1, v2);
            }

            return e.CompareDefault();
        }
    }
}
