﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Windows.Forms;
using App.Data;

namespace App.PropertyEdit
{
    public partial class ShaderDefinitionRootPage : ShaderDefinitionPropertyPage
    {
        public ShaderDefinitionRootPage() :
            base(PropertyPageID.ShaderDefinitionRoot)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_shader_definition_property_window;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new ShaderDefinitionRootPage();
        }

        protected override void InitializeFormInternal()
        {
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            UpdateDefinitionInfo();
            UpdateList();
            SetShaderSources();
        }

        private void UpdateDefinitionInfo()
        {
            lblCodePage.IsModified = ActiveTarget.IsValueChanged(x => x.shader_definition_info.code_page);
            ltbCodePage.Text = ActiveTarget.Data.shader_definition_info.code_page.ToString();
            lblConfigPath.IsModified = ActiveTarget.IsStringChanged(x => x.shader_definition_info.config_path);
            ltbConfigPath.Text = ActiveTarget.Data.shader_definition_info.config_path;
        }

        public static bool IsModified(ShaderDefinition activeTarget)
        {
            return activeTarget != null &&
                (activeTarget.IsValueChanged(x => x.shader_definition_info.code_page) ||
                activeTarget.IsStringChanged(x => x.shader_definition_info.config_path));
        }
        private void UpdateList()
        {
            lvwList.BeginUpdate();
            {
                if (ActiveTarget != null)
                {
                    // リストアイテムを予め用意する
                    lvwList.SetItemCount(ActiveTarget.Definitions.Count());

                    int index = 0;
                    foreach (Data.Definition program in ActiveTarget.Definitions)
                    {
                        ListViewItem item = lvwList.Items[index];

                        item.SubItems[clhIndex.Index].Text = index.ToString();
                        item.SubItems[clhName.Index].Text = program.Name;

                        item.SubItems[clhVShader.Index].Text = (program.Data.vertex_stage == null)
                            ? ""
                            : ActiveTarget.Data.shader_src_array.shader_src[program.Data.vertex_stage.src_index].path;
                        item.SubItems[clhFShader.Index].Text = (program.Data.fragment_stage == null)
                            ? ""
                            : ActiveTarget.Data.shader_src_array.shader_src[program.Data.fragment_stage.src_index].path;
                        item.SubItems[clhGShader.Index].Text = (program.Data.geometry_stage == null)
                            ? ""
                            : ActiveTarget.Data.shader_src_array.shader_src[program.Data.geometry_stage.src_index].path;
                        item.SubItems[clhCShader.Index].Text = (program.Data.compute_stage == null)
                            ? ""
                            : ActiveTarget.Data.shader_src_array.shader_src[program.Data.compute_stage.src_index].path;

                        ++index;
                    }
                }
                else
                {
                    lvwList.SetItemCount(0);
                }
            }
            lvwList.EndUpdate();
        }

        private void lvwList_SelectionChanged(object sender, EventArgs e)
        {
            UpdateForm(false, false);
        }

        // 無駄にスクロールされないように変更があったときのみ更新するために以前の Text を置いておく
        // TextBox.Text は以前 set した値のまま get されない場合がある
        private string lastVertexShaderText = null;
        private string lastFragmentShaderText = null;
        private string lastGeometryShaderText = null;
        private string lastComputeShaderText = null;

        private void SetShaderSources()
        {
            var vertexShaderText = "";
            var fragmentShaderText = "";
            var geometryShaderText = "";
            var computeShaderText = "";

            int row = lvwList.SelectedIndex;
            if (0<= row && row < ActiveTarget.Definitions.Count)
            {
                Data.Definition program = ActiveTarget.Definitions[row];
                var stream = ActiveTarget.BinaryStreams;
                if (program.Data.vertex_stage != null)
                {
                    int stream_index = ActiveTarget.Data.shader_src_array.shader_src[program.Data.vertex_stage.src_index].stream_index;
                    if (0 <= stream_index && stream_index < stream.Count)
                    {
                        vertexShaderText = stream[stream_index].StringData.Replace("\n", "\r\n");
                    }
                }
                if (program.Data.fragment_stage != null)
                {
                    int stream_index = ActiveTarget.Data.shader_src_array.shader_src[program.Data.fragment_stage.src_index].stream_index;
                    if (0 <= stream_index && stream_index < stream.Count)
                    {
                        fragmentShaderText = stream[stream_index].StringData.Replace("\n", "\r\n");
                    }
                }
                if (program.Data.geometry_stage != null)
                {
                    int stream_index = ActiveTarget.Data.shader_src_array.shader_src[program.Data.geometry_stage.src_index].stream_index;
                    if (0 <= stream_index && stream_index < stream.Count)
                    {
                        geometryShaderText = stream[stream_index].StringData.Replace("\n", "\r\n");
                    }
                }
                if (program.Data.compute_stage != null)
                {
                    int stream_index = ActiveTarget.Data.shader_src_array.shader_src[program.Data.compute_stage.src_index].stream_index;
                    if (0 <= stream_index && stream_index < stream.Count)
                    {
                        computeShaderText = stream[stream_index].StringData.Replace("\n", "\r\n");
                    }
                }
            }

            if (lastVertexShaderText != vertexShaderText)
            {
                tbxVertexShader.Text = vertexShaderText;
                lastVertexShaderText = vertexShaderText;
            }

            if (lastFragmentShaderText != fragmentShaderText)
            {
                tbxFragmentShader.Text = fragmentShaderText;
                lastFragmentShaderText = fragmentShaderText;
            }

            if (lastGeometryShaderText != geometryShaderText)
            {
                tbxGeometryShader.Text = geometryShaderText;
                lastGeometryShaderText = geometryShaderText;
            }

            if (lastComputeShaderText != computeShaderText)
            {
                tbxComputeShader.Text = computeShaderText;
                lastComputeShaderText = computeShaderText;
            }
        }
    }
}
