﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public class ShaderParameterAnimationCurveTreeNodeInfo2 : IAnimationCurve
    {
        public bool IsEditable { get; set; }
        public AnimationDocument.NonEditableKind NonEditableKind { get; set; }
        public object NonEditableKindDisplayAux { get; set; }
        public int ColorComponentIndex { get { return ComponentIndex; } }

        private readonly MaterialAnimation	target_;
        private readonly uniform_varType				param_;
        private readonly string						materialName_;
        private readonly int							componentIndex_;
        private readonly bool						isSingleComponent_;

        public uniform_varType UniformVar{ get{ return param_; } }

        public string GroupLabel { get; set; }
        public bool IsColorCurve
        {
            get
            {
                return
                    (UniformVar != null) &&
                    (UniformVar.ui_item != null) &&
                    (UniformVar.ui_item.value == nw.g3d.nw4f_3dif.ui_item_valueType.color);
            }
        }

        // componentIndex == -1で要素数１つ
        public ShaderParameterAnimationCurveTreeNodeInfo2(MaterialAnimation target, uniform_varType param, string materialName, int componentIndex)
        {
            target_			= target;
            param_			= param;
            materialName_	= materialName;

            if (componentIndex == -1)
            {
                isSingleComponent_	= true;
                componentIndex_		= 0;
            }
            else
            {
                isSingleComponent_	= false;
                componentIndex_		= componentIndex;
            }

            MakeCurvePrimitiveType();
        }

        public string GetMaterialName()
        {
            return materialName_;
        }

        public ShaderParameterAnimation.ParamAnim GetAnimParam(GuiObject editTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            return GetAnimParam(animation.PerMaterialAnims);
        }

        private ShaderParameterAnimation.ParamAnim GetAnimParam(MaterialAnimation.PerMaterialAnimList shaderParamAnims)
        {
            var shaderParamAnim = shaderParamAnims[materialName_];
            if (shaderParamAnim == null)
            {
                return null;
            }

            var paramAnim = shaderParamAnim.ParamAnims.FirstOrDefault(x => x.id == param_.id);

            return paramAnim;
        }

        public AnimTarget GetAnimTarget(GuiObject editTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            return GetAnimTarget(animation.PerMaterialAnims);
        }

        private AnimTarget GetAnimTarget(MaterialAnimation.PerMaterialAnimList shaderParamAnims)
        {
            var paramAnim = GetAnimParam(shaderParamAnims);
            if (paramAnim == null)
            {
                return null;
            }

            if (paramAnim.ParamAnimTargets.Count <= componentIndex_)
            {
                return null;
            }

            return paramAnim.ParamAnimTargets[componentIndex_];
        }

        public void UpdateIsModified(GuiObject editTarget)
        {
            var animation = (MaterialAnimation)editTarget;
            var animTarget = (ShaderParameterAnimation.ParamAnimTarget)GetAnimTarget(animation.PerMaterialAnims);
            var savedTarget = (ShaderParameterAnimation.ParamAnimTarget)GetAnimTarget(animation.savedPerMaterialAnims);
            if (savedTarget == null && animTarget.IsTexSrt && componentIndex_ == 0)
            {
                var key = animation.CreateInitialSrtModesKey(materialName_, param_.id);
                animation.initialSrtModes.TryGetValue(key, out savedTarget);
            }

            animation.UpdateIsModifiedAnimTarget(
                animTarget,
                savedTarget);
        }

        public AnimTarget CreateTemporaryTarget(GuiObject editTarget)
        {
            var source = (ShaderParameterAnimation.ParamAnimTarget)GetAnimTarget(editTarget);
            return new ShaderParameterAnimation.ParamAnimTarget()
            {
                component_index = source.component_index,
                IsTexSrt = source.IsTexSrt,
                CurveInterpolationType = CurvePrimitiveType == PrimitiveTypeKind.Float ? InterpolationType.Hermite: InterpolationType.Step,
            };
        }

        public AnimTarget CreateAnimTarget(GuiObject editTarget)
        {
            Debug.Assert(false); // 呼ばれないはず？

            var paramAnim = CreateTemporaryParamAnim();
            var shaderParamAnim = (MaterialAnimation)editTarget;
            shaderParamAnim.PerMaterialAnims.First(x => x.mat_name == materialName_).ParamAnims.Add(paramAnim);

            // TexSRTの場合、MaterialのModeの値をデフォルト値として取得する。
            if (paramAnim.IsTexSRT)
            {
                ShaderParameterAnimation.SetTexSRTModeValue(paramAnim, materialName_);
            }

            return paramAnim.ParamAnimTargets[componentIndex_];
        }

        public void SetAnimTarget(GuiObject editTarget, AnimTarget animTarget)
        {
            var shaderParamAnim = (MaterialAnimation)editTarget;
            var paramAnim = shaderParamAnim.PerMaterialAnims.First(x => x.mat_name == materialName_);
            var index = paramAnim.ParamAnims.FindIndex(x => x.id == param_.id);
            if (index == -1)
            {
                index = paramAnim.ParamAnims.Count;
                paramAnim.ParamAnims.Add(CreateTemporaryParamAnim());
            }
            paramAnim.ParamAnims[index].ParamAnimTargets[componentIndex_] = (ShaderParameterAnimation.ParamAnimTarget)animTarget;
        }

        public float GetDefaultValue(GuiObject targetOwner)
        {
            // バインドされたマテリアル
            var materials = from model in DocumentManager.Models
                            where model.AllAnimations.Contains(targetOwner)
                            from material in model.Materials
                            where material.Name == materialName_
                            select material;

            var values = from material in materials
                         from param in material.MaterialShaderAssign.ShaderParams
                         where param.id == param_.id && param.type == param_.type
                         select G3dDataParser.ParseFloatArray(param.Value)[componentIndex_];
            return values.Concat(Enumerable.Range(0, 1).Select(x => GetAnimTarget(targetOwner).GetBaseValue())).First();
        }

        public void IntermediateFileIndices(MaterialAnimation target, out int materialIndex, out int curveIndex, out int componentIndex)
        {
            componentIndex = componentIndex_;
            materialIndex = 0;
            curveIndex = 0;
            foreach (var perMaterialAnim in target.PerMaterialAnims)
            {
                if (perMaterialAnim.mat_name == materialName_)
                {
                    foreach (var paramAnim in perMaterialAnim.ParamAnims)
                    {
                        if (paramAnim.id == param_.id)
                        {
                            int index = 0;
                            foreach (var paramAnimTarget in paramAnim.ParamAnimTargets)
                            {
                                if (index == componentIndex_)
                                {
                                    return;
                                }
                                switch (paramAnimTarget.ExportType)
                                {
                                    case CurveExportType.Curve:
                                    case CurveExportType.Constant:
                                        curveIndex++;
                                        break;
                                }
                                index++;
                            }
                        }
                        else
                        {
                            foreach (var paramAnimTarget in paramAnim.ParamAnimTargets)
                            {
                                switch (paramAnimTarget.ExportType)
                                {
                                    case CurveExportType.Curve:
                                    case CurveExportType.Constant:
                                        curveIndex++;
                                        break;
                                }
                            }
                        }
                    }
                }
                else
                {
                    var curveCount = ShaderParameterAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count() +
                        TexturePatternAnimationCurveTreeNodeInfo2.CurvesInBinary(perMaterialAnim).Count() +
                        (MaterialVisibilityAnimationCurveTreeNodeInfo2.AnyCurveInBinary(perMaterialAnim) ? 1 : 0);
                    if (curveCount > 0)
                    {
                        curveIndex += curveCount;
                        materialIndex++;
                    }
                }
            }
        }

        public static IEnumerable<AnimTarget> CurvesInBinary(MaterialAnimation.PerMaterialAnim perMaterialAnim)
        {
            return perMaterialAnim.ParamAnims.SelectMany(x => x.ParamAnimTargets.Where(y => { var t = y.ExportType; return t == CurveExportType.Curve || t == CurveExportType.Constant; }));
        }

        public static int curveNumberInBinary(MaterialAnimation.PerMaterialAnim perMaterialAnim)
        {
            int count = 0;
            foreach (var paramAnim in perMaterialAnim.ParamAnims)
            {
                foreach (var paramAnimTarget in paramAnim.ParamAnimTargets)
                {
                    switch (paramAnimTarget.ExportType)
                    {
                        case CurveExportType.Curve:
                        case CurveExportType.Constant:
                            count++;
                            break;
                    }
                }
            }
            return count;
        }

        public List<KeyFrame> KeyFrames
        {
            get
            {
                var paramAnim = target_.GetParamAnimFromId(materialName_, param_.id);

                return
                    (paramAnim != null) ?
                        paramAnim.ParamAnimTargets[componentIndex_].KeyFrames :
                        new List<KeyFrame>();
            }
        }

        public InterpolationType CurveInterpolationType
        {
            get
            {
                var paramAnim = target_.GetParamAnimFromId(materialName_, param_.id);
                return
                    (paramAnim != null) ?
                        paramAnim.ParamAnimTargets[componentIndex_].CurveInterpolationType :
                        InterpolationType.Linear;
            }

            set
            {
                var paramAnim = target_.GetParamAnimFromId(materialName_, param_.id);
                if (paramAnim != null)
                {
                    paramAnim.ParamAnimTargets[componentIndex_].CurveInterpolationType = value;
                }
            }
        }

        public string ParentName{		get { return materialName_;		} }
        public string Name{				get { return param_.id;			} }
        public string UniformVarLabel { get; set; }
        public string FullPath
        {
            get;
            set;
        }

        public int ComponentIndex{		get { return componentIndex_;	} }
        public bool IsRotate
        {
            get
            {
                if (param_.type == shader_param_typeType.srt3d)
                {
                    return (componentIndex_ == (int)Srt3dComponentIndex.RotateU) ||
                           (componentIndex_ == (int)Srt3dComponentIndex.RotateV) ||
                           (componentIndex_ == (int)Srt3dComponentIndex.RotateW);
                }
                else
                if (IsTextureSrt(param_))
                {
                    return componentIndex_ + (param_.type== shader_param_typeType.srt2d ? 1: 0)== (int)TexcoordComponentIndex.Rotate;
                }
                else
                {
                    return false;
                }
            }
        }

        public float? MinClampValue
        {
            get;
            private set;
        }

        public float? MaxClampValue
        {
            get;
            private set;
        }

        public float? MinFitValue
        {
            get;
            private set;
        }
        public float? MaxFitValue
        {
            get;
            private set;
        }
        public PrimitiveTypeKind CurvePrimitiveType { get; private set; }

        public Color CurveColor
        {
            get
            {
                if (IsTextureSrt(UniformVar))
                {
                    if (UniformVar.type == shader_param_typeType.srt3d)
                    {
                        // srt それぞれが uvw を持つ
                        return CurveView.MakeRGBAColorFromIndex(componentIndex_ % 3);
                    }
                    else
                    {
                        // texsrt, texsrt_ex は 0番目にMode パラメーターを持つ
                        var texCompIndex =
                            (TexcoordComponentIndex)(componentIndex_ + (UniformVar.type == shader_param_typeType.srt2d ? 1 : 0));
                        switch (texCompIndex)
                        {
                            case TexcoordComponentIndex.ScaleU:
                            case TexcoordComponentIndex.TranslateU:
                                return CurveView.MakeRGBAColorFromIndex(0);
                            case TexcoordComponentIndex.ScaleV:
                            case TexcoordComponentIndex.TranslateV:
                                return CurveView.MakeRGBAColorFromIndex(1);
                            default:
                                return CurveView.MakeColorFromString(target_.Name + ParentName + Name);
                        }

                    }
                }
                return isSingleComponent_ ? CurveView.MakeColorFromString(target_.Name + ParentName + Name) : CurveView.MakeRGBAColorFromIndex(componentIndex_);
            }
        }

        // TexcoordComponentIndex と texcoordComponentNames_ を合わせる必要があります
        public enum TexcoordComponentIndex
        {
            Mode = 0,
            ScaleU = 1,
            ScaleV = 2,
            Rotate = 3,
            TranslateU = 4,
            TranslateV = 5,
        };

        // Srt3dComponentIndex と srt3dComponentNames_ を合わせる必要があります
        public enum Srt3dComponentIndex
        {
            ScaleU = 0,
            ScaleV = 1,
            ScaleW = 2,
            RotateU = 3,
            RotateV = 4,
            RotateW = 5,
            TranslateU = 6,
            TranslateV = 7,
            TranslateW = 8
        };

        public static readonly string[] colorComponentNames_	= new[]{"R", "G", "B", "A"};
        public static readonly string[] vectorComponentNames_	= new[]{"X", "Y", "Z", "W"};

        public static readonly string[] texcoordComponentNames_	= new[]{"Mode", "Scale U", "Scale V", "Rotate", "Translate U", "Translate V"};

        public static readonly string[] srt3dComponentNames_	= new[]{"Scale U",		"Scale V",		"Scale W",
                                                                        "Rotate U",		"Rotate V",		"Rotate W",
                                                                        "Translate U",	"Translate V",	"Translate W"};
        public static readonly string[] matrix2ComponentNames_	= new[]{"00", "01",
                                                                        "10", "11",
                                                                        "20", "21",
                                                                        "30", "31"};
        public static readonly string[] matrix3ComponentNames_	= new[]{"00", "01", "02",
                                                                        "10", "11", "12",
                                                                        "20", "21", "22",
                                                                        "30", "31", "32"};
        public static readonly string[] matrix4ComponentNames_	= new[]{"00", "01", "02", "03",
                                                                        "10", "11", "12", "13",
                                                                        "20", "21", "22", "23",
                                                                        "30", "31", "32", "33"};

        private string ComponentString
        {
            get
            {
                if ((UniformVar.ui_item != null) && (UniformVar.ui_item.value == ui_item_valueType.color))
                {
                    return colorComponentNames_[componentIndex_];
                }
                else
                {
                    if (param_.type == shader_param_typeType.srt3d)
                    {
                        return srt3dComponentNames_[componentIndex_];
                    }
                    else if (IsTextureSrt(param_))
                    {
                        return texcoordComponentNames_[componentIndex_ + (param_.type == shader_param_typeType.srt2d ? 1 : 0)];
                    }
                    else
                    {
                        switch(param_.type)
                        {
                            case shader_param_typeType.@bool:	case shader_param_typeType.@int:	case shader_param_typeType.@uint:	case shader_param_typeType.@float:
                            case shader_param_typeType.bool2:	case shader_param_typeType.int2:	case shader_param_typeType.uint2:	case shader_param_typeType.float2:
                            case shader_param_typeType.bool3:	case shader_param_typeType.int3:	case shader_param_typeType.uint3:	case shader_param_typeType.float3:
                            case shader_param_typeType.bool4:	case shader_param_typeType.int4:	case shader_param_typeType.uint4:	case shader_param_typeType.float4:
                                return vectorComponentNames_[componentIndex_];

                            case shader_param_typeType.float2x2:	case shader_param_typeType.float3x2:	case shader_param_typeType.float4x2:
                                return matrix2ComponentNames_[componentIndex_];

                            case shader_param_typeType.float2x3:	case shader_param_typeType.float3x3:	case shader_param_typeType.float4x3:
                                return matrix3ComponentNames_[componentIndex_];

                            case shader_param_typeType.float2x4:	case shader_param_typeType.float3x4:	case shader_param_typeType.float4x4:
                                return matrix4ComponentNames_[componentIndex_];
                        }
                    }
                }

                Debug.Assert(false);
                return string.Empty;
            }
        }

        public static bool	IsTextureSrt(uniform_varType param)
        {
            return
                (param.type == shader_param_typeType.srt2d) ||
                (param.type == shader_param_typeType.srt3d) ||
                //
                (param.type == shader_param_typeType.texsrt);
        }

        private float? GetParamMinMax(float[] src)
        {
            if (src == null)
            {
                return null;
            }
            else
            {
                if ((ComponentIndex < 0) ||
                    (ComponentIndex >= src.Length))
                {
                    return null;
                }
                else
                {
                    return src[ComponentIndex];
                }
            }
        }

        private void MakeCurvePrimitiveType()
        {
            var info = IfUtility.GetShaderParamTypeInfo(param_.type);

            switch(info.PrimitiveType[componentIndex_])
            {
                case IfUtility.ShaderParamPrimitiveType.@bool:
                {
                    CurvePrimitiveType = PrimitiveTypeKind.Bool;
                    MinFitValue = 0;
                    MaxFitValue = 1;
                    MinClampValue = 0;
                    MaxClampValue = 1;
                    break;
                }

                case IfUtility.ShaderParamPrimitiveType.@int:
                {
                    CurvePrimitiveType = PrimitiveTypeKind.Int;
                    MinFitValue = GetParamMinMax(param_.Min());
                    MaxFitValue = GetParamMinMax(param_.Max());
                    break;
                }

                case IfUtility.ShaderParamPrimitiveType.@uint:
                {
                    CurvePrimitiveType = PrimitiveTypeKind.Uint;
                    MinFitValue = GetParamMinMax(param_.Min());
                    MaxFitValue = GetParamMinMax(param_.Max());
                    if (!MinFitValue.HasValue || MinFitValue.Value < 0)
                    {
                        MinFitValue = 0;
                    }
                    MinClampValue = 0;

                    if (param_.type == shader_param_typeType.texsrt)
                    {
                        MinFitValue = 0;
                        MaxFitValue = 2;
                        MinClampValue = 0;
                        MaxClampValue = 2;
                    }
                    break;
                }

                case IfUtility.ShaderParamPrimitiveType.@float:
                {
                    CurvePrimitiveType = PrimitiveTypeKind.Float;
                    MinFitValue = GetParamMinMax(param_.Min());
                    MaxFitValue = GetParamMinMax(param_.Max());
                    if (param_.Item() == ui_item_valueType.color)
                    {
                        if (MinFitValue == null)
                        {
                            MinFitValue = 0;
                            MinClampValue = 0;
                        }
                        else
                        {
                            MinClampValue = MinFitValue;
                        }

                        if (MaxFitValue == null)
                        {
                            MaxFitValue = 1;
                            MaxClampValue = 1;
                        }
                        else
                        {
                            MaxClampValue = MaxFitValue;
                        }
                    }

                    break;
                }
            }

            MinClampValue = MinFitValue;
            MaxClampValue = MaxFitValue;
        }

        public ShaderParameterAnimation.ParamAnim CreateTemporaryParamAnim()
        {
            //var targetNumericParam = DocumentManager.EnumrateNumericParams()[curveName];

            var targetShaderParamTypeInfo = IfUtility.GetShaderParamTypeInfo(param_.type);

            var paramAnim = new ShaderParameterAnimation.ParamAnim();
            {
                paramAnim.id = Name;
                paramAnim.type = targetShaderParamTypeInfo.Type;
                paramAnim.ParamAnimTargets = Enumerable.Range(0, targetShaderParamTypeInfo.PrimitiveType.Length).Select(
                    i => new ShaderParameterAnimation.ParamAnimTarget()
                    {
                        component_index = i,
                        CurveInterpolationType = targetShaderParamTypeInfo.PrimitiveType[i] == IfUtility.ShaderParamPrimitiveType.@float ?
                            InterpolationType.Hermite :
                            InterpolationType.Step,
                        IsTexSrt = paramAnim.IsTexSRT,
                    }).ToList();
            }

            return paramAnim;
        }

        public bool IsSame(IAnimationCurve curve)
        {
            var paramCurve = curve as ShaderParameterAnimationCurveTreeNodeInfo2;
            return paramCurve != null &&
                target_ == paramCurve.target_ &&
                ParentName == paramCurve.ParentName &&
                Name == paramCurve.Name &&
                CurvePrimitiveType == paramCurve.CurvePrimitiveType &&
                ComponentIndex == paramCurve.ComponentIndex &&
                MinClampValue == paramCurve.MinClampValue &&
                MaxClampValue == paramCurve.MaxClampValue &&
                MinFitValue == paramCurve.MinFitValue &&
                MaxFitValue == paramCurve.MaxFitValue;
        }
    }
}
