﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class TexCoord : ShaderParamControl
    {
        public override string ParamName
        {
            get
            {
                return base.ParamName;
            }
            set
            {
                base.ParamName = value;
                chkLinkScale.Checked = !notCheckedNames.Contains(value);
            }
        }

        private IEnumerable<FloatEditPanel> panels
        {
            get
            {
                yield return edpScaleU;
                yield return edpScaleV;
                yield return edpRotate;
                yield return edpTranslateU;
                yield return edpTranslateV;
            }
        }
        public float[] Value
        {
            get{	return panels.Select(x => x.Value).ToArray();	}
            set{
                foreach(var p in panels.Zip(value, (x, y)=>new {x, y}))
                {
                    p.x.Value = p.y;
                }
            }
        }

        public float[] MinimumValue
        {
            get{	return panels.Select(x => x.Minimum).ToArray(); }
            set{
                foreach (var p in panels.Zip(value, (x, y) => new { x, y }))
                {
                    p.x.Minimum = p.y;
                }
            }
        }

        public float[] MaximumValue
        {
            get { return panels.Select(x => x.Maximum).ToArray(); }
            set
            {
                foreach (var p in panels.Zip(value, (x, y) => new { x, y }))
                {
                    p.x.Maximum = p.y;
                }
            }
        }

        private System.Drawing.Rectangle? initialBounds = null;
        private Dictionary<Control, System.Drawing.Rectangle> initialControlBounds = null;

        public shader_param_typeType type;
        public TexCoord(shader_param_typeType type)
        {
            InitializeComponent();

            chkLinkScale.Text = Strings.ShaderParamControl_LinkScale;

            this.type = type;
            uint bit = 1;
            foreach (var panel in panels)
            {
                panel.Tag = bit;
                bit <<= 1;
            }
        }

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            float[] paramArray = G3dDataParser.ParseFloatArray(value);
            if (paramArray == null || paramArray.Length != 5)
            {
                Enabled = false;
                return false;
            }

            Value = paramArray;
            return false;
        }

        private void edpValue_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            float[] param = Value.ToArray();
            uint bits = (uint)(((Control)sender).Tag);
            if (chkLinkScale.Checked)
            {
                if (bits == 1)
                {
                    using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                    {
                        edpScaleV.Value = param[1] = param[0];
                    }
                    bits = 3;
                }
                else if (bits == 2)
                {
                    using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                    {
                        edpScaleU.Value = param[0] = param[1];
                    }
                    bits = 3;
                }
            }

            ShaderParamValueChangedEventArgs args = new ShaderParamValueChangedEventArgs(type); // 必要なら考え直す
            {
                args.ParamName							= ParamName;
                args.ParamValue							= param;
                args.ElementBits						= bits;
                args.SequentialValueChangedEventArgs	= e;
            }

            InvokeValueChanged(this, args);
        }

        public static HashSet<string> notCheckedNames = new HashSet<string>();
        private void chkLinkScale_CheckedChanged(object sender, EventArgs e)
        {
            if (chkLinkScale.Checked)
            {
                notCheckedNames.Remove(ParamName);
            }
            else
            {
                notCheckedNames.Add(ParamName);
            }
        }

        public override IEnumerable<IHasShaderParameterAnimation> FilterAnimation(IEnumerable<IHasShaderParameterAnimation> animations)
        {
            return animations.Where(x => x.IsTextureSrtType());
        }

        public override ParamType ParamType
        {
            get { return ShaderParamControls.ParamType.uniform_var; }
        }

        public override void Align()
        {
            // FloatEditPanel の右横に「カーブエディタを開く」ボタンを配置。
            bool showButtons = false;
            do
            {
                var floatEditPanels = new FloatEditPanel[]
                {
                    edpScaleU,
                    edpScaleV,
                    edpRotate,
                    edpTranslateU,
                    edpTranslateV
                };

                var numComponents = ComponentCurveEditorOpeningButtons.Count;
                if ((numComponents <= 1) || (numComponents != floatEditPanels.Length) || !floatEditPanels.All(x => x != null))
                {
                    break;
                }

                if (initialControlBounds == null)
                {
                    // 配置調整前に、復元のための境界を保持しておく。
                    initialControlBounds = new Dictionary<Control, System.Drawing.Rectangle>();
                    foreach (var ctrl in Controls.OfType<Control>())
                    {
                        initialControlBounds.Add(ctrl, ctrl.Bounds);
                    }
                }

                // 配置調整。
                // Dictionary にアクセスするので、念のため try catch する。
                // キーはすべて揃っているので、不正な記述を行わない限り例外は発生しないはず。
                try
                {
                    int offset = 0;

                    Action<Control> align = ctrl =>
                    {
                        var initCtrlBounds = initialControlBounds[ctrl];
                        ctrl.Bounds = new System.Drawing.Rectangle(
                            initCtrlBounds.Left + offset,
                            initCtrlBounds.Top,
                            initCtrlBounds.Width,
                            initCtrlBounds.Height);
                    };
                    Action<Control, DropDownButton> alignedAdd = (ctrl, button) =>
                    {
                        align(ctrl);
                        button.Location = new System.Drawing.Point(ctrl.Right + 2, ctrl.Top + Math.Max(0, (ctrl.Height - button.Height) / 2));
                        offset += button.Left - ctrl.Right + button.Width;
                    };

                    // Scale 行。
                    offset = 0;
                    alignedAdd(edpScaleU, ComponentCurveEditorOpeningButtons[0]);
                    alignedAdd(edpScaleV, ComponentCurveEditorOpeningButtons[1]);

                    // Rotate 行。
                    offset = 0;
                    alignedAdd(edpRotate, ComponentCurveEditorOpeningButtons[2]);

                    // Translate 行。
                    offset = 0;
                    alignedAdd(edpTranslateU, ComponentCurveEditorOpeningButtons[3]);
                    alignedAdd(edpTranslateV, ComponentCurveEditorOpeningButtons[4]);
                }
                catch (System.Exception)
                {
                    break;
                }

                // コントロールサイズ拡大の妨げになるアンカーを一時的に外しておく。
                var oColorEditPanelAnchors = new System.Windows.Forms.AnchorStyles[Controls.Count];
                for (int i = 0; i < Controls.Count; i++)
                {
                    oColorEditPanelAnchors[i] = Controls[i].Anchor;
                    Controls[i].Anchor &= ~(System.Windows.Forms.AnchorStyles.Right | System.Windows.Forms.AnchorStyles.Bottom);
                }

                // 成分毎の「カーブエディタで開く」ボタンを追加する前の境界を取得しておく。
                if (!initialBounds.HasValue)
                {
                    initialBounds = Bounds;
                }

                // すべてのコントロールを内包する境界。
                // 座標はクラス外部で配置されるので、サイズだけを更新。
                var bounds = ComponentCurveEditorOpeningButtons.Select(x => x.Bounds).Concat(Enumerable.Repeat(initialBounds.Value, 1)).Aggregate(System.Drawing.Rectangle.Union);
                Size = bounds.Size;

                // アンカーを戻す。
                for (int i = 0; i < Controls.Count; i++)
                {
                    Controls[i].Anchor = oColorEditPanelAnchors[i];
                }

                showButtons = true;
            }
            while (false);

            if (showButtons)
            {
                // 成分毎の「カーブエディタで開く」ボタンを追加する。
                Controls.AddRange(ComponentCurveEditorOpeningButtons.Where(x => !Controls.Contains(x)).ToArray());
            }
            else
            {
                foreach (var button in ComponentCurveEditorOpeningButtons)
                {
                    Controls.Remove(button);
                }

                if (initialBounds.HasValue)
                {
                    // 座標はクラス外部で配置されるので、サイズだけを更新。
                    Size = initialBounds.Value.Size;
                }

                if (initialControlBounds != null)
                {
                    foreach (var items in initialControlBounds)
                    {
                        items.Key.Bounds = items.Value;
                    }
                }
            }
        }
    }
}
