﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditUintMatrix : ShaderParamControl
    {
        public uint[] Value
        {
            get
            {
                return uintEditPanels.Select(x => (uint)x.Value).ToArray();
            }
            set
            {
                for (int i= 0; i< Row*Column; i++)
                {
                    uintEditPanels[i].Value = value[i];
                }
            }
        }

        public void SetMinimumValue(uint minimum, uint inputMin)
        {
            for (int i = 0; i < Row * Column; i++)
            {
                uintEditPanels[i].MinBound = inputMin;
                uintEditPanels[i].Minimum = uintEditPanels[i].MinDefault = minimum;
            }
        }

        public void SetMaximumValue(uint maximum, uint inputMax)
        {
            for (int i = 0; i < Row * Column; i++)
            {
                uintEditPanels[i].MaxBound = inputMax;
                uintEditPanels[i].Maximum = uintEditPanels[i].MaxDefault = maximum;
            }
        }

        public void UpdateIncrement()
        {
            var Increment = 1u;// (uint)Math.Pow(10, Math.Max(0, Math.Round(Math.Log10(Math.Max(0.1, (double)uintEditPanels[0].Maximum - (double)uintEditPanels[0].Minimum))) - 2));
            var LargeIncrement = (uint)(Increment * 10);
            for (int i = 0; i < Row * Column; i++)
            {
                uintEditPanels[i].Increment = Increment;
                uintEditPanels[i].LargeIncrement = LargeIncrement;
            }
        }

        readonly Controls.UintEditPanel[] uintEditPanels;
        readonly UICheckBox linkValueCheck;
        int row = 0;
        int column = 0;

        public int Row
        {
            get
            {
                return row;
            }
            private set
            {
                row = value;
                NumComponentCurveEditorOpeningButtons = row * Column;
            }
        }
        public int Column
        {
            get
            {
                return column;
            }
            private set
            {
                column = value;
                NumComponentCurveEditorOpeningButtons = Row * column;
            }
        }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public shader_param_typeType type;

        public static HashSet<string> CheckedNames = new HashSet<string>();
        public override string ParamName
        {
            get
            {
                return base.ParamName;
            }
            set
            {
                base.ParamName = value;
                if (linkValueCheck != null)
                {
                    linkValueCheck.Checked = CheckedNames.Contains(value);
                }
            }
        }

        private void chkLinkValueCheck_CheckedChanged(object sender, EventArgs e)
        {
            if (linkValueCheck.Checked)
            {
                CheckedNames.Add(ParamName);
            }
            else
            {
                CheckedNames.Remove(ParamName);
            }
        }

        public EditUintMatrix(int row, int column, shader_param_typeType type)
        {
            InitializeComponent();
            Row = row;
            Column = column;
            this.type = type;
            uintEditPanels = new UintEditPanel[Row * Column];
            for (int i = 0; i < Row; i++)
            {
                for (int j = 0; j < Column; j++)
                {
                    // Row mager
                    int k = Column * i + j;
                    var control = new UintEditPanel();
                    control.Tag = 1u << k;
                    control.SliderRangeAdjustable = true;
                    control.Value = Convert.ToUInt32(i * 10 + j);
                    control.SequentialValueChanged += edpValue_SequentialValueChanged;
                    Controls.Add(control);
                    uintEditPanels[k] = control;
                }
                if (Row == 1 && Column != 1) // add link checkbox
                {
                    var chk = new UICheckBox();
                    chk.Text = Strings.ShaderParamControl_LinkScale;
                    chk.CheckedChanged += chkLinkValueCheck_CheckedChanged;
                    Controls.Add(chk);
                    linkValueCheck = chk;
                }
            }
        }

        public override void Align()
        {
            var numUintEditPanels = uintEditPanels.Length;
            bool showButtons = (numUintEditPanels > 1) && (numUintEditPanels == ComponentCurveEditorOpeningButtons.Count);
            var extraWidth = showButtons ? ((ComponentCurveEditorOpeningButtonDefaultSize.Width + 2) * 4) : 0;

            {
                int width = DefaultWidth + extraWidth - MarginX + marginBetweenColumn;
                for (int i = 0; i < Row; i++)
                {
                    for (int j = 0; j < Column; j++)
                    {
                        Control control = uintEditPanels[Column * i + j];
                        control.Size = new Size((width - extraWidth) / 4 - marginBetweenColumn, subControlHeight);
                        control.Location = new Point(MarginX + (width * j) / 4, MarginY + subControlHeight * i);
                    }
                    if (linkValueCheck != null && Row == 1 && Column != 1) // link value checkbox
                    {
                        linkValueCheck.Location = new Point(MarginX + (width * Column) / 4, MarginY);
                    }
                }
            }

            if (showButtons)
            {
                for (int i = 0; i < numUintEditPanels; i++)
                {
                    var uintEditPanel = uintEditPanels[i];
                    ComponentCurveEditorOpeningButtons[i].Location = new Point(uintEditPanel.Right + 2, uintEditPanel.Top + Math.Max(0, (uintEditPanel.Height - ComponentCurveEditorOpeningButtonDefaultSize.Height) / 2));
                }

                Controls.AddRange(ComponentCurveEditorOpeningButtons.Where(x => !Controls.Contains(x)).ToArray());
            }
            else
            {
                foreach (var button in ComponentCurveEditorOpeningButtons)
                {
                    Controls.Remove(button);
                }
            }
        }

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            uint[] paramArray = G3dDataParser.ParseUIntArray(value);
            if (paramArray == null || paramArray.Length != Row * Column)
            {
                Enabled = false;
                return false;
            }

            Value = paramArray;
            return false;
        }

        private void edpValue_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var args = new ShaderParamValueChangedEventArgs(type);
            uint elemBits = 0;

            // 連動がオンのときはUIを同じ値で更新しまとめて一つのイベントにする
            if (linkValueCheck != null && linkValueCheck.Checked)
            {
                var modifiedPanel = sender as UintEditPanel;
                Debug.Assert(modifiedPanel != null);

                using (var block = new UIControlEventSuppressBlock())
                {
                    foreach (var panel in uintEditPanels)
                    {
                        if (modifiedPanel != panel)
                        {
                            panel.Value = modifiedPanel.Value;
                        }
                        elemBits |= (uint)panel.Tag;
                    }
                }
            }
            else
            {
                elemBits = (uint)(((Control)sender).Tag);
            }
            args.ParamName = ParamName;
            args.ParamValue = Value;
            args.ElementBits = elemBits;
            args.SequentialValueChangedEventArgs = e;

            InvokeValueChanged(this, args);
        }

        public override ParamType ParamType
        {
            get { return ShaderParamControls.ParamType.uniform_var; }
        }
    }
}
