﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using System.Drawing;
using App.Controls;
using App.Data;
using App.res;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditTexSrtSliders : ShaderParamControl
    {
        public override string ParamName
        {
            get
            {
                return base.ParamName;
            }
            set
            {
                base.ParamName = value;
                chkLinkScale.Checked = !notCheckedNames.Contains(value);
            }
        }

        private FloatUpDown[] upds;

        private FloatTrackBar[] trbs;

        private System.Drawing.Rectangle? initialBounds = null;

        /// <summary>
        /// EditTexSrtSliders カーブの componentId は 0 ではなく 1 から開始される。
        /// </summary>
        private const int ComponentIdOffset = 1;

        public float[] Value
        {
            get{	return Enumerable.Repeat((float)cmbMode.SelectedItemData, 1).Concat(upds.Select(x => x.Value)).ToArray();	}
            set{
                cmbMode.SetSelectedItemData(value[0]);
                foreach(var p in upds.Zip(value.Skip(1), (x, y)=>new {x, y}))
                {
                    p.x.Value = p.y;
                }
                foreach (var p in trbs.Zip(value.Skip(1), (x, y) => new { x, y }))
                {
                    p.x.Value = p.y;
                }
            }
        }
#if false
        public float[] MinimumValue
        {
            get{	return upds.Select(x => x.Minimum).ToArray(); }
            set{
                foreach (var p in upds.Zip(value, (x, y) => new { x, y }))
                {
                    p.x.Minimum = p.y;
                }
                foreach (var p in trbs.Zip(value, (x, y) => new { x, y }))
                {
                    p.x.Minimum = p.y;
                }
            }
        }

        public float[] MaximumValue
        {
            get { return upds.Select(x => x.Maximum).ToArray(); }
            set
            {
                foreach (var p in upds.Zip(value, (x, y) => new { x, y }))
                {
                    p.x.Maximum = p.y;
                }
                foreach (var p in trbs.Zip(value, (x, y) => new { x, y }))
                {
                    p.x.Maximum = p.y;
                }
            }
        }
#endif
        void updateRangeScale(float min, float max)
        {
            double Increment = Math.Min(1, Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.001, (double)max - (double)min)) - 2)));
            double LargeIncrement = Increment * 10;
            trbScaleU.Minimum = trbScaleV.Minimum = updScaleU.Minimum = updScaleV.Minimum = min;
            trbScaleU.Maximum = trbScaleV.Maximum = updScaleU.Maximum = updScaleV.Maximum = max;
            trbScaleU.SmallChange = trbScaleV.SmallChange = updScaleU.Increment = updScaleV.Increment = Increment;
            trbScaleU.LargeChange = trbScaleV.LargeChange = updScaleU.LargeIncrement = updScaleV.LargeIncrement = LargeIncrement;
            var range = new FloatRangeProperty(min, max, 1, 1, float.MinValue, float.MaxValue, -1, 1);
            btnRangeDownScale.Enabled = range.Scale(false);
            range = new FloatRangeProperty(min, max, 1, 1, float.MinValue, float.MaxValue, -1, 1);
            btnRangeUpScale.Enabled = range.Scale(true);
            lblMinimumScale.Text = trbScaleU.Minimum.ToString("R");
            lblMaximumScale.Text = trbScaleU.Maximum.ToString("R");
            updateThumbColor();
        }

        void updateRangeRotate(float min, float max)
        {
            double Increment = Math.Min(1, Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.001, (double)max - (double)min)) - 2)));
            double LargeIncrement = Increment * 10;
            trbRotate.Minimum = updRotate.Minimum = min;
            trbRotate.Maximum = updRotate.Maximum = max;
            trbRotate.SmallChange = updRotate.Increment = Increment;
            trbRotate.LargeChange = updRotate.LargeIncrement = LargeIncrement;
            var range = new FloatRangeProperty(min, max, 1, 1, float.MinValue, float.MaxValue, -360, 360);
            btnRangeDownRotate.Enabled = range.Scale(false);
            range = new FloatRangeProperty(min, max, 1, 1, float.MinValue, float.MaxValue, -360, 360);
            btnRangeUpRotate.Enabled = range.Scale(true);
            lblMinimumRotate.Text = trbRotate.Minimum.ToString("R");
            lblMaximumRotate.Text = trbRotate.Maximum.ToString("R");
            updateThumbColor();
        }

        void updateRangeTranslate(float min, float max)
        {
            double Increment = Math.Min(1, Math.Pow(10, Math.Round(Math.Log10(Math.Max(0.001, (double)max - (double)min)) - 2)));
            double LargeIncrement = Increment * 10;
            trbTranslateU.Minimum = trbTranslateV.Minimum = updTranslateU.Minimum = updTranslateV.Minimum = min;
            trbTranslateU.Maximum = trbTranslateV.Maximum = updTranslateU.Maximum = updTranslateV.Maximum = max;
            trbTranslateU.SmallChange = trbTranslateV.SmallChange = updTranslateU.Increment = updTranslateV.Increment = Increment;
            trbTranslateU.LargeChange = trbTranslateV.LargeChange = updTranslateU.LargeIncrement = updTranslateV.LargeIncrement = LargeIncrement;
            var range = new FloatRangeProperty(min, max, 1, 1, float.MinValue, float.MaxValue, -1, 1);
            btnRangeDownTranslate.Enabled = range.Scale(false);
            range = new FloatRangeProperty(min, max, 1, 1, float.MinValue, float.MaxValue, -1, 1);
            btnRangeUpTranslate.Enabled = range.Scale(true);
            lblMinimumTranslate.Text = trbTranslateU.Minimum.ToString("R");
            lblMaximumTranslate.Text = trbTranslateU.Maximum.ToString("R");
            updateThumbColor();
        }

        void updateButtonEnable()
        {
            {
                var range = new FloatRangeProperty(trbTranslateU.Minimum, trbTranslateU.Maximum, 1, 1, float.MinValue, float.MaxValue, -1, 1);
                btnRangeDownScale.Enabled = range.Scale(false);
                range = new FloatRangeProperty(trbTranslateU.Minimum, trbTranslateU.Maximum, 1, 1, float.MinValue, float.MaxValue, -1, 1);
                btnRangeUpScale.Enabled = range.Scale(true);
            }
            {
                var range = new FloatRangeProperty(trbRotate.Minimum, trbRotate.Maximum, 1, 1, float.MinValue, float.MaxValue, -360, 360);
                btnRangeDownRotate.Enabled = range.Scale(false);
                range = new FloatRangeProperty(trbRotate.Minimum, trbRotate.Maximum, 1, 1, float.MinValue, float.MaxValue, -360, 360);
                btnRangeUpRotate.Enabled = range.Scale(true);
            }
            {
                var range = new FloatRangeProperty(trbScaleU.Minimum, trbScaleU.Maximum, 1, 1, float.MinValue, float.MaxValue, -1, 1);
                btnRangeDownTranslate.Enabled = range.Scale(false);
                range = new FloatRangeProperty(trbScaleU.Minimum, trbScaleU.Maximum, 1, 1, float.MinValue, float.MaxValue, -1, 1);
                btnRangeUpTranslate.Enabled = range.Scale(true);
            }
        }

        void updateThumbColor()
        {
            for (int i = 0; i < trbs.Length; i++)
            {
                trbs[i].DrawRedThumb = trbs[i].Value < trbs[i].Minimum || trbs[i].Maximum < trbs[i].Value;
            }
        }
        private static readonly Bitmap imgValueEditPanelRangeDown_ = App.Properties.Resources.Control_ValueEditPanelRangeDown;
        private static readonly Bitmap imgValueEditPanelRangeUp_ = App.Properties.Resources.Control_ValueEditPanelRangeUp;
        public shader_param_typeType type;
        public EditTexSrtSliders(shader_param_typeType type)
        {
            InitializeComponent();

            upds = new[]{
                    updScaleU,
                    updScaleV,
                    updRotate,
                    updTranslateU,
                    updTranslateV,
                };
            trbs = new[] {
                    trbScaleU,
                    trbScaleV,
                    trbRotate,
                    trbTranslateU,
                    trbTranslateV,
                };

            chkLinkScale.Text = Strings.ShaderParamControl_LinkScale;

            cmbMode.Tag = -1;
            for (int i = 0; i < 3; i++)
            {
                cmbMode.Items.Add(new UIListControlItem(UIText.TextureSrtMode((float)i), (float)i));
            }
            cmbMode.SetSelectedItemData((float)0);

            this.type = type;
            for (int i = 0; i < trbs.Length; i++)
            {
                trbs[i].Tag = upds[i].Tag = i;
                upds[i].EnableInputMinMax = true;
                upds[i].InputMin = float.MinValue;
                upds[i].InputMax = float.MaxValue;
            }

            btnRangeDownScale.Image = btnRangeDownRotate.Image = btnRangeDownTranslate.Image = imgValueEditPanelRangeDown_;
            btnRangeUpScale.Image = btnRangeUpRotate.Image = btnRangeUpTranslate.Image = imgValueEditPanelRangeUp_;

            updateRangeScale(-1, 1);
            updateRangeRotate(-360, 360);
            updateRangeTranslate(-1, 1);

            NumComponentCurveEditorOpeningButtons = ComponentIdOffset + trbs.Length;
        }

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            float[] paramArray = G3dDataParser.ParseFloatArray(value);
            if (paramArray == null || paramArray.Length != 6)
            {
                Enabled = false;
                return false;
            }

            for (int i = 0; i < 5; i++)
            {
                if (Value[i + 1] != paramArray[i + 1])
                {
                    FitRange(Value[i + 1], i);
                }
            }
            //FitRange(paramArray);

            Value = paramArray;
            updateThumbColor();
            return false;
        }

        private void FitRange(float value, int i)
        {
            var range = new FloatRangeProperty(upds[i].Minimum, upds[i].Maximum, 1, 1, float.MinValue, float.MaxValue, i == 2 ? -360 : -1, i == 2 ? 360 : 1);
            bool scaled = false;
            while (!range.Contains(value) && range.Scale(true))
            {
                scaled = true;
            }
            if (scaled && range.Contains(value))
            {
                switch (i)
                {
                    case 0:
                    case 1:
                        updateRangeScale(range.Minimum, range.Maximum);
                        break;
                    case 2:
                        updateRangeRotate(range.Minimum, range.Maximum);
                        break;
                    case 3:
                    case 4:
                        updateRangeTranslate(range.Minimum, range.Maximum);
                        break;
                }
            }
        }

        private void edpValue_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            float[] param = Value.ToArray();
            int i = (int)(((Control)sender).Tag);
            uint bits = 1u<<(i+1);
            if (chkLinkScale.Checked)
            {
                if (bits == 2)
                {
                    using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                    {
                        updScaleV.Value = param[2] = param[1];
                    }
                    bits = 6;
                }
                else if (bits == 4)
                {
                    using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                    {
                        updScaleU.Value = param[1] = param[2];
                    }
                    bits = 6;
                }
            }

            FitRange(param[i+1], i);

            ShaderParamValueChangedEventArgs args = new ShaderParamValueChangedEventArgs(type); // 必要なら考え直す
            {
                args.ParamName							= ParamName;
                args.ParamValue							= param;
                args.ElementBits						= bits;
                args.SequentialValueChangedEventArgs	= e;
            }

            InvokeValueChanged(this, args);
        }

        private void trbScale_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            float[] param = Value.ToArray();
            int i = (int)(((Control)sender).Tag);
            param[i+1] = ((FloatTrackBar)sender).Value;
            uint bits = 1u<<(i+1);
            if (chkLinkScale.Checked)
            {
                if (bits == 2)
                {
                    using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                    {
                        trbScaleV.Value = updScaleV.Value = param[2] = param[1];
                    }
                    bits = 6;
                }
                else if (bits == 4)
                {
                    using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                    {
                        trbScaleU.Value = updScaleU.Value = param[1] = param[2];
                    }
                    bits = 6;
                }
            }

            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                upds[i].Value = param[i+1];
                updateThumbColor();
            }

            ShaderParamValueChangedEventArgs args = new ShaderParamValueChangedEventArgs(type); // 必要なら考え直す
            {
                args.ParamName = ParamName;
                args.ParamValue = param;
                args.ElementBits = bits;
                args.SequentialValueChangedEventArgs = e;
            }

            InvokeValueChanged(this, args);
        }
        public static HashSet<string> notCheckedNames = new HashSet<string>();
        private void chkLinkScale_CheckedChanged(object sender, EventArgs e)
        {
            if (chkLinkScale.Checked)
            {
                notCheckedNames.Remove(ParamName);
            }
            else
            {
                notCheckedNames.Add(ParamName);
            }
        }

        public override IEnumerable<IHasShaderParameterAnimation> FilterAnimation(IEnumerable<IHasShaderParameterAnimation> animations)
        {
            return animations.Where(x => x.IsTextureSrtType());
        }

        public override void Align()
        {
            // スライダーバーの右横に「カーブエディタを開く」ボタンを配置。
            bool showButtons = false;
            do
            {
                var floatTrackBars = new FloatTrackBar[]
                {
                    trbScaleU,
                    trbScaleV,
                    trbRotate,
                    trbTranslateU,
                    trbTranslateV
                };

                var componentCurveEditorOpeningButtons = ComponentCurveEditorOpeningButtons.Skip(ComponentIdOffset).ToArray();
                var numComponents = componentCurveEditorOpeningButtons.Length;
                if ((numComponents <= 1) || (numComponents != floatTrackBars.Length) || !floatTrackBars.All(x => x != null))
                {
                    break;
                }

                for (int i = 0; i < numComponents; i++)
                {
                    var floatTrackBar = floatTrackBars[i];
                    componentCurveEditorOpeningButtons[i].Location = new System.Drawing.Point(floatTrackBar.Right, floatTrackBar.Top);
                }

                // コントロールサイズ拡大の妨げになるアンカーを一時的に外しておく。
                var oColorEditPanelAnchors = new System.Windows.Forms.AnchorStyles[Controls.Count];
                for (int i = 0; i < Controls.Count; i++)
                {
                    oColorEditPanelAnchors[i] = Controls[i].Anchor;
                    Controls[i].Anchor &= ~(System.Windows.Forms.AnchorStyles.Right | System.Windows.Forms.AnchorStyles.Bottom);
                }

                // 成分毎の「カーブエディタで開く」ボタンを追加する前の境界を取得しておく。
                if (!initialBounds.HasValue)
                {
                    initialBounds = Bounds;
                }

                // すべてのコントロールを内包する境界を設定。
                // EditTexSrtSliders はクラス外部で配置されるので、サイズだけを設定。
                var bounds = componentCurveEditorOpeningButtons.Select(x => x.Bounds).Concat(Enumerable.Repeat(initialBounds.Value, 1)).Aggregate(System.Drawing.Rectangle.Union);
                Size = bounds.Size;

                // アンカーを戻す。
                for (int i = 0; i < Controls.Count; i++)
                {
                    Controls[i].Anchor = oColorEditPanelAnchors[i];
                }

                showButtons = true;
            }
            while (false);

            if (showButtons)
            {
                // 成分毎の「カーブエディタで開く」ボタンを追加する。
                Controls.AddRange(ComponentCurveEditorOpeningButtons.Skip(ComponentIdOffset).Where(x => !Controls.Contains(x)).ToArray());
            }
            else
            {
                foreach (var button in ComponentCurveEditorOpeningButtons.Skip(ComponentIdOffset))
                {
                    Controls.Remove(button);
                }

                if (initialBounds.HasValue)
                {
                    // 座標はクラス外部で配置されるので、サイズだけを更新。
                    Size = initialBounds.Value.Size;
                }
            }
        }

        private void btnRangeScale_Click(object sender, EventArgs e)
        {
            var range = new FloatRangeProperty(updScaleU.Minimum, updScaleU.Maximum, 1, 1, float.MinValue, float.MaxValue, -1, 1);
            range.Scale(sender == btnRangeUpScale);
            updateRangeScale(range.Minimum, range.Maximum);
        }

        private void btnRangeRotate_Click(object sender, EventArgs e)
        {
            var range = new FloatRangeProperty(updRotate.Minimum, updRotate.Maximum, 1, 1, float.MinValue, float.MaxValue, -360, 360);
            range.Scale(sender == btnRangeUpRotate);
            updateRangeRotate(range.Minimum, range.Maximum);
        }

        private void btnRangeTranslate_Click(object sender, EventArgs e)
        {
            var range = new FloatRangeProperty(updTranslateU.Minimum, updTranslateU.Maximum, 1, 1, float.MinValue, float.MaxValue, -1, 1);
            range.Scale(sender == btnRangeUpTranslate);
            updateRangeTranslate(range.Minimum, range.Maximum);
        }

        private void cmbMode_SelectedIndexChanged(object sender, EventArgs e)
        {
            ShaderParamValueChangedEventArgs args = new ShaderParamValueChangedEventArgs(type); // 必要なら考え直す
            {
                args.ParamName = ParamName;
                args.ParamValue = Value.ToArray();
                args.ElementBits = 1;
                args.SequentialValueChangedEventArgs = null;
            }

            InvokeValueChanged(this, args);
        }

        public override void UpdateEnabled(bool value)
        {
            base.UpdateEnabled(value);
            if (value)
            {
                updateButtonEnable();
            }
        }

        public override ParamType ParamType
        {
            get { return ShaderParamControls.ParamType.uniform_var; }
        }
    }
}

