﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditRenderInfoFloat : EditRenderInfoBase
    {
        public float[] Value
        {
            get
            {
                return floatEditPanels.Take(ItemCount).Select(x => x.Value).ToArray();
            }
            set
            {
                ItemCount = 0;
                for (int i= 0; i< Math.Min(value.Length, Row*Column); i++)
                {
                    floatEditPanels[i].Value = value[i];
                    ItemCount++;
                }
            }
        }

        int ItemCount = 0;
        readonly Controls.FloatEditPanel[] floatEditPanels;

        public int Row { get; private set; }
        public int Column { get; private set; }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public render_info_slotType type;
        public const int minItemWidth = 100;
        public const int textMargin = 30;
        public EditRenderInfoFloat(int row, int column, render_info_slotType type)
        {
            InitializeComponent();
            Row = row;
            Column = column;
            this.type = type;
            floatEditPanels = new FloatEditPanel[Row * Column];
            int tabIndex = 0;
            for (int i = 0; i < Row; i++)
            {
                for (int j = 0; j < Column; j++)
                {
                    // Row mager
                    int k = Column * i + j;
                    var control = new FloatEditPanel()
                    {
                        Width = minItemWidth,
                    };
                    control.Tag = 1u << k;
                    control.SequentialValueChanged += cbxValue_ValueChanged;
                    control.TabIndex = tabIndex;
                    tabIndex++;
                    Controls.Add(control);
                    floatEditPanels[k] = control;
                }
            }
            if (type != null && type.optional)
            {
                btnIncrement.TabIndex = tabIndex;
                tabIndex++;
                btnDecrement.TabIndex = tabIndex;
            }
            else
            {
                btnIncrement.Visible = false;
                btnDecrement.Visible = false;
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);
        }
        public override void Align()
        {
            int width = DefaultWidth - MarginX + marginBetweenColumn;
            for (int i = 0; i < Row; i++)
            {
                for (int j = 0; j < Column; j++)
                {
                    Control control = floatEditPanels[Column * i + j];
                    control.Location = new Point(MarginX + (width * j) / 4, MarginY + subControlHeight * i);
                }
            }

            if (type != null && type.optional)
            {
                btnIncrement.Location = new Point(floatEditPanels[Row - 1].Right + 4, MarginY);
                btnDecrement.Location = new Point(btnIncrement.Right + 4, btnIncrement.Top);
            }
            else
            {
                // Align の都合上左上に寄せておく
                btnIncrement.Location = new Point(0, 0);
                btnDecrement.Location = new Point(0, 0);
            }
        }

        public override bool SetValues(Material material, List<string> values, bool candidateModified, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel, Material.ValueResolvedState valueResolvedState)
        {
            Value = values.Select(x => { float result; float.TryParse(x, out result); return result; }).ToArray();

            bool rangeSet = false;
            float min = 0;
            float max = 0;
            float[] defaults = new float[0];
            bool packFromHio = false;

            if (Viewer.Manager.Instance.IsConnected)
            {
                var candidate = material.RenderInfoPackFromHio.FloatItems.FirstOrDefault(x => x.name == ParamName);
                if (candidate != null)
                {
                    if (candidate.hasMinMax)
                    {
                        min = candidate.min;
                        max = candidate.max;
                        rangeSet = true;
                    }
                    defaults = candidate.defaults.ToArray();
                    packFromHio = true;
                }
            }

            if (type != null && !packFromHio)
            {
                if (IfShaderAssignUtility.TryParseFloatRange(type.choice, out min, out max))
                {
                    rangeSet = true;
                }

                defaults = G3dDataParser.Tokenize(type.Default()).Select(x => { float result; float.TryParse(x, out result); return result; }).ToArray();
            }

            lblParameter.BackColor = GetBgColor(candidateModified, packFromHio, rangeSet);

            // デフォルト値を設定
            for (int i = ItemCount; i < Math.Min(floatEditPanels.Length, defaults.Length); i++)
            {
                floatEditPanels[i].Value = defaults[i];
            }

            bool validDefault = defaults.All(x => min <= x && x <= max);

            for (int i = 0; i < floatEditPanels.Length; i++)
            {
                var control = floatEditPanels[i];
                if (!rangeSet)
                {
                    control.Enabled = false;
                }
                else
                {
                    control.Enabled = i < ItemCount && type != null && validDefault;
                    control.Minimum = min;
                    control.Maximum = max;
                }
            }

            btnIncrement.Enabled = ItemCount < floatEditPanels.Length && rangeSet && type != null && validDefault;
            btnDecrement.Enabled = 0 < ItemCount && rangeSet && type != null && validDefault;
            return false;
        }

        private void cbxValue_ValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                ParamValue = Value.Select(x => x.ToString()).ToList(),
                SequentialValueChangedEventArgs = e,
                type = render_info_typeType.@float,
            };
            InvokeValueChanged(this, args);
        }

        private void btnIncrement_Click(object sender, EventArgs e)
        {
            ItemCount++;
            Debug.Assert(ItemCount <= floatEditPanels.Length);
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                ParamValue = Value.Select(x => x.ToString()).ToList(),
                SequentialValueChangedEventArgs = null,
                type = render_info_typeType.@float,
            };
            InvokeValueChanged(this, args);
        }

        private void btnDecrement_Click(object sender, EventArgs e)
        {
            ItemCount--;
            Debug.Assert(ItemCount >= 0);
            var args = new RenderInfoValueChangedEventArgs()
            {
                ParamName = ParamName,
                ParamValue = Value.Select(x => x.ToString()).ToList(),
                SequentialValueChangedEventArgs = null,
                type = render_info_typeType.@float,
            };
            InvokeValueChanged(this, args);
        }
    }
}
