﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using App.Data;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditReadOnly : ShaderParamControl
    {
        public string Value
        {
            get;
            set;
        }

        public EditReadOnly(option_varType option, HorizontalAlignment align)
        {
            InitializeComponent();

            lowTextBox1.TextAlign = align;
            lowTextBox1.Width = minItemWidth;
            ParamName = option.id;
            lowTextBox1.Tag = (uint)(1 << 0);

            Aliases = IfShaderOptionUtility.Alias(option.choice);
            paramType = ParamType.option_var;
        }

        readonly Dictionary<string, string> Aliases = new Dictionary<string, string>();
        public EditReadOnly(string id, ParamType paramType)
        {
            InitializeComponent();
            lowTextBox1.Width = minItemWidth;
            ParamName = id;
            lowTextBox1.Tag = (uint)(1 << 0);
            this.paramType = paramType;
        }

        public const int minItemWidth = 100;
        public const int textMargin = 30;

        private ChoiceHelper choiceHelper = new ChoiceHelper();

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            choiceHelper.SetLabel(Aliases, customUI, table, visibleGroups, visiblePages, showId, showOriginalLabel);

            bool updated = false;
            Value = value;
            lowTextBox1.Text = choiceHelper.labels.ContainsKey(value) ? choiceHelper.labels[value] : value;
            using (var g = CreateGraphics())
            {
                int width = Math.Max(minItemWidth, (int)g.MeasureString(lowTextBox1.Text, lowTextBox1.Font).Width + textMargin);
                if (width != lowTextBox1.Width)
                {
                    lowTextBox1.Width = width;
                    updated = true;
                }
            }

            {
                LabelHelper helper = null;
                choiceHelper.LabelHelpers.TryGetValue(Value, out helper);
                lowTextBox1.ForeColor = helper != null && helper.color.HasValue ? helper.color.Value : SystemColors.ControlText;
                lowTextBox1.BackColor = helper != null && helper.color.HasValue ? helper.backGroundColor.Value : SystemColors.Control;
            }
            return UpdateLinkButtons(updated);
        }

        LinkButton linkButton;
        public bool UpdateLinkButtons(bool updated)
        {
            LabelHelper helper = null;
            choiceHelper.LabelHelpers.TryGetValue(Value, out helper);
            if (helper != null)
            {
                if (linkButton == null)
                {
                    if (helper.linkTarget != null)
                    {
                        linkButton = LabelHelper.CreateLinkButton();
                        linkButton.Click += (s, e) =>
                        {
                            var value = (string)Value;
                            OnLinkClicked(helper.ClickArgs());
                        };
                        Controls.Add(linkButton);
                        updated = true;

                        if (updated)
                        {
                            linkButton.Location = new Point(lowTextBox1.Right + 5, lowTextBox1.Top);
                        }

                        {
                            linkButton.Enabled =
                                helper.linkTarget != null &&
                                !helper.LinkError;
                        }
                    }
                    else if (linkButton != null)
                    {
                        Controls.Remove(linkButton);
                        linkButton.Dispose();
                        linkButton = null;
                        updated = true;
                    }
                }
            }
            else if (linkButton != null)
            {
                Controls.Remove(linkButton);
                linkButton.Dispose();
                linkButton = null;
                updated = true;
            }

            UpdateLinkButtonToolTip(false);

            return updated;
        }

        private void UpdateLinkButtonToolTip(bool force)
        {
            LabelHelper helper;
            choiceHelper.LabelHelpers.TryGetValue(Value, out helper);
            var toolTip = helper != null ? helper.GetLinkLabelString() : "";
            var current = parentHint.GetToolTip(linkButton);
            if (toolTip != current || force)
            {
                parentHint.SetToolTip(linkButton, toolTip);
            }
        }

        ParamType paramType;
        public override ParamType ParamType
        {
            get { return paramType; }
        }
    }
}
