﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using Microsoft.WindowsAPICodePack.Dialogs;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib.nw3de;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditCombiner : ShaderParamControl
    {
        class Target : IDisposable
        {
            private readonly EditCombiner owner;
            private System.IO.FileSystemWatcher ecmbDirWatcher = null;
            private System.IO.FileSystemWatcher ecmbFileWatcher = null;

            /// <summary>
            /// combiners ディレクトリパス
            /// </summary>
            private string location = null;
            public string Location
            {
                get { return location; }
            }

            /// <summary>
            /// 拡張子付きのファイル名を取得もしくは設定する。
            /// </summary>
            private string name = null;
            public string Name
            {
                get
                {
                    return name;
                }
                set
                {
                    if (!string.IsNullOrEmpty(value) && !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(System.IO.Path.GetFileNameWithoutExtension(value)))
                    {
                        name = value;

                        // ecmbFileWatcher での監視ファイルを更新する。
                        ReleaseFileSystemWatcher(ref ecmbFileWatcher);
                        CreateEcmbFileWatcher();
                    }
                    else
                    {
                        name = string.Empty;

                        // ecmbFileWatcher での監視を停止。
                        ReleaseFileSystemWatcher(ref ecmbFileWatcher);
                    }
                }
            }

            /// <summary>
            /// 拡張子を除いたファイル名を取得もしくは設定する。
            /// </summary>
            public string Value
            {
                get
                {
                    if (!string.IsNullOrEmpty(Name))
                    {
                        var ext = System.IO.Path.GetExtension(Name);
                        var index = !string.IsNullOrEmpty(ext) ? Name.LastIndexOf(ext) : -1;
                        return (index != -1) ? Name.Remove(index) : Name;
                    }
                    else
                    {
                        return CombinerShaderConverterManager.ReservedValue.Unset;
                    }
                }
                private set
                {
                    if (!string.IsNullOrEmpty(value) && !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(value))
                    {
                        Name = value + CombinerShaderConverterManager.CombinerShaderFileExtension;
                        if (!CorrectFileExtension() && string.IsNullOrEmpty(value))
                        {
                            Name = value;
                        }
                    }
                    else
                    {
                        Name = string.Empty;
                    }
                }
            }

            /// <summary>
            /// ecmb ファイルパス
            /// </summary>
            public string Path
            {
                get { return System.IO.Path.Combine(Location, Name); }
            }

            public event Action Notified;

            private bool CorrectFileExtension()
            {
                bool ret = false;
                try
                {
                    var info = new System.IO.FileInfo(Path);
                    if (info.Exists)
                    {
                        var ext = System.IO.Path.GetExtension(info.FullName);
                        System.IO.Path.ChangeExtension(name, ext);
                        ret = true;
                    }
                }
                catch
                { }
                return ret;
            }

            /// <summary>
            /// ecmbFileWatcher を作成する
            /// </summary>
            /// <returns>成功したかどうか</returns>
            private bool CreateEcmbFileWatcher()
            {
                System.IO.FileSystemWatcher watcher = null;
                try
                {
                    watcher = new System.IO.FileSystemWatcher(Location, Name);

                    watcher.NotifyFilter = System.IO.NotifyFilters.FileName;
                    watcher.IncludeSubdirectories = false;
                    watcher.SynchronizingObject = owner;
                    Action action = () =>
                    {
                        if (Notified != null)
                        {
                            Notified();
                        }
                    };
                    watcher.Changed += (ss, ee) => { action(); };
                    watcher.Created += (ss, ee) => { action(); };
                    watcher.Deleted += (ss, ee) => { action(); };
                    watcher.Renamed += (ss, ee) => { action(); };
                    watcher.EnableRaisingEvents = true;

                    ecmbFileWatcher = watcher;
                    return true;
                }
                catch
                {
                    ReleaseFileSystemWatcher(ref watcher);
                    return false;
                }
            }

            /// <summary>
            /// FileSystemWatcher を解放して null を代入する
            /// </summary>
            /// <param name="watcher">解放対象</param>
            private void ReleaseFileSystemWatcher(ref System.IO.FileSystemWatcher watcher)
            {
                if (watcher != null)
                {
                    watcher.EnableRaisingEvents = false;
                    watcher.Dispose();
                    watcher = null;
                }
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="material">マテリアル</param>
            /// <param name="value">拡張子を除いたファイル名</param>
            public Target(EditCombiner owner, Material material, string value)
            {
                this.owner = owner;
                var ownerDoc = material.OwnerDocument;
                location = System.IO.Path.Combine(ownerDoc.FileLocation, Material.CombinerShaderDirectoryName);
                Value = value;

                // fmdb ディレクトリ内の combiners ディレクトリを監視する ecmbDirWatcher を作成する。
                ecmbDirWatcher = new System.IO.FileSystemWatcher(ownerDoc.FileLocation, Material.CombinerShaderDirectoryName);
                ecmbDirWatcher.NotifyFilter = System.IO.NotifyFilters.DirectoryName | System.IO.NotifyFilters.LastWrite;
                ecmbDirWatcher.IncludeSubdirectories = false;
                ecmbDirWatcher.SynchronizingObject = owner;
                Action action = () =>
                {
                    if (Notified != null)
                    {
                        Notified();
                    }

                    // FileSystemWatcher は存在しないディレクトリ下のものは作成できない。
                    // つまり combiners ディレクトリが存在しない場合には ecmbFileWatcher を作成できない。
                    // そこで combiners ディレクトリの変更時に ecmbFileWatcher の作成を試みる。
                    if (ecmbFileWatcher == null)
                    {
                        // ecmbFileWatcher の作成を試みる。
                        CreateEcmbFileWatcher();
                    }
                };
                ecmbDirWatcher.Changed += (ss, ee) => { action(); };
                ecmbDirWatcher.Created += (ss, ee) => { action(); };
                ecmbDirWatcher.Deleted += (ss, ee) => { action(); };
                ecmbDirWatcher.Renamed += (ss, ee) => { action(); };
                ecmbDirWatcher.EnableRaisingEvents = true;

                // ecmbFileWatcher の作成を試みる。
                // combiners ディレクトリが存在しない場合は作成に失敗する。
                // 失敗時には ecmbDirWatcher に割りてた action イベントハンドラで作成される。
                CreateEcmbFileWatcher();
            }

            public void Dispose()
            {
                ReleaseFileSystemWatcher(ref ecmbFileWatcher);
                ReleaseFileSystemWatcher(ref ecmbDirWatcher);
            }
        };

        private readonly Color defaultTextColor;
        private Target target = null;
        private string prevText = null;

        public override ParamType ParamType
        {
            get { return ParamType.option_var; }
        }

        public EditCombiner(option_varType option)
        {
            InitializeComponent();

            defaultTextColor = tbxValue.ForeColor;

            ParamName = option.id;
            tbxValue.Tag = (uint)(1 << 0);

            this.Disposed += (ss, ee) =>
            {
                if (target != null)
                {
                    target.Dispose();
                    target = null;
                }
            };
        }

        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            if (!CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(value))
            {
                var combinerOption = material.CombinerOptionInfo?.CombinerOptions?.FirstOrDefault(x => x.Id == ParamName);
                if (combinerOption?.FileName != null)
                {
                    value = combinerOption.FileName;
                }
            }

            if (target != null)
            {
                target.Dispose();
                target = null;
            }

            if (!string.IsNullOrEmpty(material.OwnerDocument.FileLocation))
            {
                target = new Target(this, material, value);
                target.Notified += () =>
                {
                    UpdateTextColor();
                };

                prevText = target.Name;

                tbxValue.Text = target.Name;
                UpdateTextColor();

                Enabled = true;
            }
            else
            {
                // コンバイナーファイルディレクトリは、マテリアルを所有するドキュメントの場所に依存する。
                // ドキュメントの場所が決まっていなければファイルを読み込むことができないためコントロールを無効化する。
                Enabled = false;
            }
            return false;
        }

        private void UpdateTextColor()
        {
            if (!tbxValue.IsDisposed)
            {
                if ((target != null) && !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(target.Value))
                {
                    string path = target.Path;
                    tbxValue.ForeColor = System.IO.File.Exists(path) ? defaultTextColor : Color.Red;
                }
                else
                {
                    tbxValue.ForeColor = defaultTextColor;
                }
            }
        }

        private void tbxValue_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                tbxValue_Leave(sender, EventArgs.Empty);
            }
        }

        private void tbxValue_Leave(object sender, EventArgs e)
        {
            if ((target != null) && (target.Name != tbxValue.Text))
            {
                var oldName = target.Name;
                var oldValue = target.Value;

                target.Name = tbxValue.Text;

                var newValue = target.Value;
                if (newValue != oldValue)
                {
                    CombinerOptionValueChangedEventArgs args = new CombinerOptionValueChangedEventArgs()
                    {
                        ParamName = ParamName,
                        ParamValue = newValue
                    };
                    InvokeValueChanged(this, args);
                }
                else
                {
                    tbxValue.Text = oldName;
                }
            }
        }

        private void btnValue_Click(object sender, EventArgs e)
        {
            try
            {
                using (var dialog = new CommonOpenFileDialog()
                {
                    Title = res.Strings.IO_SpecifyTargetFile,
                    IsFolderPicker = false,
                    EnsureFileExists = true,
                })
                {
                    dialog.Filters.Add(new CommonFileDialogFilter("FCMB File", CombinerShaderConverterManager.CombinerShaderFileExtension));

                    if (!string.IsNullOrEmpty(target.Location))
                    {
                        // 期待通りに動作させるにはディレクトリセパレータを正しておく必要がある。
                        dialog.InitialDirectory = target.Location.Replace(System.IO.Path.AltDirectorySeparatorChar, System.IO.Path.DirectorySeparatorChar);
                    }

                    if (dialog.ShowDialog(Handle) == CommonFileDialogResult.Ok)
                    {
                        Func<string, string> normalize = (path) =>
                        {
                            path = path.Replace(System.IO.Path.AltDirectorySeparatorChar, System.IO.Path.DirectorySeparatorChar);

                            // . や .. を含んだパスを正規化する。
                            // ただし Win32.NativeMethods.PathCanonicalize() では連続するディレクトリセパレータの正規化は行えないので後段で手動削除する。
                            var pathBuilder = new StringBuilder(Math.Max(path.Length, Win32.Constants.MAX_PATH));
                            if (Win32.NativeMethods.PathCanonicalize(pathBuilder, path))
                            {
                                path = pathBuilder.ToString();
                            }

                            // UNC パスでない場合は、連続するディレクトリセパレータをひとつにまとめる。
                            var pathUri = new Uri(path);
                            if (!pathUri.IsUnc)
                            {
                                // 二連続以上のディレクトリセパレータをひとつにするために置換ではなくループで処理する必要がある。
                                var target = new string(Enumerable.Repeat(System.IO.Path.DirectorySeparatorChar, 2).ToArray());
                                for (var i = path.IndexOf(target); i != -1; i = path.IndexOf(target, i))
                                {
                                    path = path.Remove(i, 1);
                                }
                            }

                            return path.TrimEnd(System.IO.Path.DirectorySeparatorChar);
                        };
                        var selectedDir = System.IO.Path.GetDirectoryName(dialog.FileName);
                        var selectedName = System.IO.Path.GetFileName(dialog.FileName);
                        var normalizedSelectedDir = normalize(selectedDir);
                        var normalizedCombinerDir = normalize(target.Location);
                        if (string.Compare(normalizedSelectedDir, normalizedCombinerDir, true) == 0)
                        {
                            tbxValue.Text = selectedName;
                            tbxValue_Leave(sender, EventArgs.Empty);
                        }
                        else
                        {
                            // combiners ディレクトリ外 (不正な場所) のファイルが選択された。
                            UIMessageBox.Error(res.Strings.Combiner_InvalidFileLocation, target.Location);
                        }
                    }
                }
            }
            catch
            { }
        }

        private void tbxValue_TextChanged(object sender, EventArgs e)
        {
            // ファイル名として使えない文字が含まれていた場合は、変更前の状態に戻す。
            if (tbxValue.Text.IndexOfAny(System.IO.Path.GetInvalidFileNameChars()) != -1)
            {
                tbxValue.Text = prevText;
                return;
            }
            prevText = tbxValue.Text;
        }
    }
}
