﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using App.Utility;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit.ShaderParamControls
{
    public partial class EditBoolMatrix : ShaderParamControl
    {
        public bool[] Value
        {
            get
            {
                return boolEditPanels.Select(x => x.Checked).ToArray();
            }
            set
            {
                for (int i= 0; i< Row*Column; i++)
                {
                    boolEditPanels[i].Checked = value[i];
                }
            }
        }

        readonly Controls.UICheckBox[] boolEditPanels;

        public int Row { get; private set; }
        public int Column { get; private set; }
        public const int MarginX = 4;
        public const int MarginY = 20;
        public int subControlHeight = 24;
        public int marginBetweenColumn = 6;
        public shader_param_typeType type;
        private readonly ShaderTypeUtility.ParamPrimitiveTypeKind PrimitiveType;
        public bool option = false;
        private readonly float checkOff = 0;
        private readonly float checkOn = 1;
        private readonly float eps = 1.0e-6f;

        ParamType paramType;
        public override ParamType ParamType
        {
            get { return paramType; }
        }
        public EditBoolMatrix(int row, int column, shader_param_typeType type, float min=0, float max=1)
        {
            checkOff = min;
            checkOn = max;
            eps = Math.Min(Math.Abs(checkOff - checkOn) / 2, eps);
            InitializeComponent();
            Row = row;
            Column = column;
            this.type = type;
            PrimitiveType = ShaderTypeUtility.ParamPrimitiveTypeFromType(type);
            boolEditPanels = new UICheckBox[Row * Column];
            for (int i = 0; i < Row; i++)
            {
                for (int j = 0; j < Column; j++)
                {
                    // Row mager
                    int k = Column * i + j;
                    UICheckBox control = (row * column == 1) ?
                        (UICheckBox)new UIModifiedMarkAndImageCheckBox() :
                        (UICheckBox)new UICheckBox2();
                    control.Tag = 1u << k;
                    control.CheckedChanged += cbxValue_CheckedChanged;
                    Controls.Add(control);
                    boolEditPanels[k] = control;
                }
            }
            paramType = ParamType.uniform_var;
        }

        public EditBoolMatrix(option_varType option)
        {
            InitializeComponent();
            Row = 1;
            Column = 1;
            this.option = true;
            ParamName = option.id;
            boolEditPanels = new UICheckBox[1];
            var control = new UIModifiedMarkAndImageCheckBox();
            control.CheckedChanged += cbxValue_CheckedChanged;
            Controls.Add(control);
            boolEditPanels[0] = control;
            PrimitiveType = ShaderTypeUtility.ParamPrimitiveTypeKind.@bool;
            paramType = ParamType.option_var;
        }

        public override void Align()
        {
            if (boolEditPanels == null)
            {
                return;
            }

            if (boolEditPanels.Length == 1 && CanEnabled)
            {
                var control = boolEditPanels[0];
                control.AutoSize = true;
                lblParameter.Visible = false;
                if (lblParameter.ContextMenuStrip != null) // ラベルが消されるので、コンテクストメニューを付け替える
                {
                    control.ContextMenuStrip = lblParameter.ContextMenuStrip;
                }
                UpdateLabel();
            }
            else
            {
                int width = DefaultWidth - MarginX + marginBetweenColumn;
                for (int i = 0; i < Row; i++)
                {
                    for (int j = 0; j < Column; j++)
                    {
                        Control control = boolEditPanels[Column * i + j];
                        control.Size = new Size(width / 4 - marginBetweenColumn, subControlHeight);
                        control.Location = new Point(MarginX + (width * j) / 4, MarginY + subControlHeight * i);
                    }
                }
            }
        }

        public override void UpdateLabel()
        {
            if (boolEditPanels != null && !lblParameter.Visible && boolEditPanels.Length == 1 && CanEnabled)
            {
                var checkBox = boolEditPanels[0] as UIModifiedMarkAndImageCheckBox;
                Debug.Assert(checkBox != null);

                if (checkBox != null)
                {
                    var label = labelHelper.GetLabelString(false);
                    checkBox.Text = label;
                    checkBox.ForeColor = labelHelper.color ?? SystemColors.ControlText;
                    checkBox.BackColor = labelHelper.backGroundColor ?? SystemColors.Control;
                    checkBox.TailImages = lblParameter.TailImages;
                    checkBox.ResetSize();
                    UpdateDropDownButtonLocation();
                    UpdateLinkToolTip(false);
                }
            }
            else
            {
                base.UpdateLabel();
            }
        }
        public override bool SetValue(Material material, string value, CustomUI customUI, Definition.ShadingModelTable table, Predicate<string> visibleGroups, HashSet<string> visiblePages, bool showId, bool showOriginalLabel)
        {
            bool[] paramArray = null;
            switch(PrimitiveType)
            {
                case ShaderTypeUtility.ParamPrimitiveTypeKind.@bool:
                    paramArray = G3dDataParser.ParseInt2BoolArray(value);
                    break;
                case ShaderTypeUtility.ParamPrimitiveTypeKind.@float:
                    paramArray = G3dDataParser.ParseFloatArray(value).Select(x => Math.Abs(x - checkOff) >= eps).ToArray();
                    break;
                case ShaderTypeUtility.ParamPrimitiveTypeKind.@int:
                    paramArray = G3dDataParser.ParseIntArray(value).Select(x => Math.Abs(x - checkOff) >= eps).ToArray();
                    break;
                case ShaderTypeUtility.ParamPrimitiveTypeKind.@uint:
                    paramArray = G3dDataParser.ParseUIntArray(value).Select(x => Math.Abs(x - checkOff) >= eps).ToArray();
                    break;
            }

            if (paramArray == null || paramArray.Length != Row * Column)
            {
                Debug.Assert(false);
                Enabled = false;
                return false;
            }

            Value = paramArray;

            return false;
        }

        private void cbxValue_CheckedChanged(object sender, EventArgs e)
        {
            if (!option)
            {
                ShaderParamValueChangedEventArgs args = new ShaderParamValueChangedEventArgs(type);
                {
                    args.ParamName = ParamName;
                    switch (PrimitiveType)
                    {
                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@bool:
                            args.ParamValue = Value;
                            break;
                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@float:
                            args.ParamValue = Value.Select(x => x ? checkOn : checkOff).ToArray();
                            break;
                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@int:
                            args.ParamValue = Value.Select(x => (int)(x ? checkOn : checkOff)).ToArray();
                            break;
                        case ShaderTypeUtility.ParamPrimitiveTypeKind.@uint:
                            args.ParamValue = Value.Select(x => (uint)(x ? checkOn : checkOff)).ToArray();
                            break;
                    }

                    args.ElementBits = (uint)(((Control)sender).Tag);
                }
                InvokeValueChanged(this, args);
            }
            else
            {
                OptionValueChangedEventArgs args = new OptionValueChangedEventArgs();
                {
                    args.ParamName = ParamName;
                    args.ParamValue = Value[0] ? "1" : "0";
                }
                InvokeValueChanged(this, args);
            }
        }

        public override void SetHint(HintToolTip hint)
        {
            if (boolEditPanels != null && boolEditPanels.Length == 1)
            {
                parentHint = hint;

                if (dropDownButton1 != null)
                {
                    hint.SetToolTip(dropDownButton1, res.Strings.ShaderParamControl_ShowCurveEditor);
                }

                if (!string.IsNullOrWhiteSpace(ParamComment))
                {
                    hint.SetToolTip(boolEditPanels[0], ParamComment);
                }
            }
            else
            {
                base.SetHint(hint);
            }
        }

        private bool isModified = false;
        public override bool IsModified
        {
            set
            {
                if (boolEditPanels.Length == 1)
                {
                    var checkBox = boolEditPanels[0] as UIModifiedMarkAndImageCheckBox;
                    Debug.Assert(checkBox != null);

                    if (checkBox != null)
                    {
                        checkBox.IsModified = value;
                    }
                }
                else
                {
                    base.IsModified = value;
                }
                isModified = value;
            }
            get
            {
                return isModified;
            }
        }

        public override void UpdateDropDownButtonLocation()
        {
            if (boolEditPanels == null || boolEditPanels.Length != 1 || !CanEnabled)
            {
                base.UpdateDropDownButtonLocation();
                return;
            }

            if (dropDownButton1 != null)
            {
                dropDownButton1.Left = boolEditPanels[0].Right + 5;
            }

            if (labelHelper.linkTarget != null)
            {
                if (btnLink == null)
                {
                    btnLink = LabelHelper.CreateLinkButton();
                    btnLink.Click += (s, e) =>
                    {
                        OnLinkClicked(labelHelper.ClickArgs());
                    };
                    btnLink.Top = 0;
                    Controls.Add(btnLink);
                }

                if (dropDownButton1 != null)
                {
                    btnLink.Left = dropDownButton1.Right + 5;
                }
                else
                {
                    btnLink.Left = boolEditPanels[0].Right + 5;
                }

                btnLink.Enabled = !labelHelper.LinkError;
            }
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                var child = GetChildAtPoint(e.Location);
                if (child?.ContextMenuStrip != null && !child.Enabled)
                    child.ContextMenuStrip.Show(this, e.Location);
            }
            base.OnMouseUp(e);
        }
    }
}
