﻿using System;
using System.Windows.Forms;
using System.Collections.Generic;
using System.IO;
using System.Xml;
using System.Linq;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using Microsoft.VisualBasic.FileIO;

namespace App.PropertyEdit
{
    public partial class MaterialGeneralPage : MaterialPropertyPage
    {
        public MaterialGeneralPage() :
            base(PropertyPageID.MaterialGeneral)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_material_property_window_general_page;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new MaterialGeneralPage();
        }

        private enum ParseKeyColumn
        {
            Id,
            Branch,
            Type,
            KeyBlock,
            KeyPos,
            KeyWidth,
            KeyMask,
            Default,
            Choice,
        }

        private enum ShaderDebugDumpColumn
        {
            name,
            vs,
            fs,
        }

        private enum ShaderOptionColumn
        {
            id,
            choice,
            varType,
        }

        private enum ShaderDebugDumpParameter
        {
            AttributeMemUsage,
            Latency,
            Occupancy,
            NumDivergentBranches,
            ProgramSize,
            RequiresGlobalLoadUniformEmulation,
            SpillMem_NumLmemSpillBytes,
            SpillMem_NumLmemRefillBytes,
            SpillMem_NumSmemSpillBytes,
            SpillMem_NumSmemRefillBytes,
            SpillMem_Size,
            NonSpillLMem_LoadBytes,
            NonSpillLMem_StoreBytes,
            NonSpillLMem_Size,
            ThroughputLimiter_Issue,
            ThroughputLimiter_Fp,
            ThroughputLimiter_Half,
            ThroughputLimiter_Trancedental,
            ThroughputLimiter_Ipa,
            ThroughputLimiter_Shared,
            ThroughputLimiter_ControlFlow,
            ThroughputLimiter_TexLoadStore,
            ThroughputLimiter_Reg,
            ThroughputLimiter_warp,
            LoopData_PartiallyUnrolled,
            LoopData_NonUnrolled,
        }

        private List<Dictionary<string, string>> optionVariations_ = new List<Dictionary<string, string>>();
        private ShaderDebugDumpInformation vsDumpInformation_ = new ShaderDebugDumpInformation();
        private ShaderDebugDumpInformation fsDumpInformation_ = new ShaderDebugDumpInformation();

        private void GetShaderVariation(shader_assignType shaderAssign, UIComboBox variationComboBox)
        {
            // バリエーションが何も存在しない場合はオプション設定がない
            if (shaderAssign.shader_option_array == null)
            {
                variationComboBox.AddItem("0", null);
            }
            else
            {
                // スタティックオプションを抜き出す
                Dictionary<string, string> staticOption = new Dictionary<string, string>();
                foreach (shader_optionType option in shaderAssign.shader_option_array.Items)
                {
                    staticOption.Add(option.id, option.value);
                }

                // スタティック部分がすべて一致するものをバリエーションに加える
                int variation = 0;
                foreach (Dictionary<string, string> option in optionVariations_)
                {
                    int matchCount = 0;
                    foreach (KeyValuePair<string, string> pair in staticOption)
                    {
                        if (option[pair.Key] == pair.Value)
                        {
                            ++matchCount;
                        }
                    }

                    if (matchCount == staticOption.Count)
                    {
                        variationComboBox.AddItem(variation.ToString(), null);
                    }

                    ++variation;
                }
            }
        }

        private void SetupShaderVariation(List<uint[]> shaderKeys, Dictionary<string, Key> keys)
        {
            foreach (uint[] shaderKey in shaderKeys)
            {
                Dictionary<string, string> dictionary = new Dictionary<string, string>();
                foreach (KeyValuePair<string, Key> pair in keys)
                {
                    uint choice = shaderKey[pair.Value.KeyBlock] & pair.Value.KeyMask;
                    choice = choice >> (32 - pair.Value.KeyPos - pair.Value.KeyWidth);
                    dictionary.Add(pair.Key, pair.Value.Choices[choice]);
                }

                optionVariations_.Add(dictionary);
            }
        }

        private static bool EnableShaderDebugDumpInformation()
        {
            if (DocumentManager.OptimizeShader && Utility.TemporaryFileUtility.ShaderFolderPath != null)
            {
                // 実機に接続中のみ使用可能
                if (Viewer.Manager.Instance.IsConnected)
                {
                    if (App.AppContext.SelectedPlatformPreset.Name == "Nx")
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        protected override void InitializeFormInternal()
        {
            radDisplayFaceBoth.Tag  = render_state_display_faceType.both;
            radDisplayFaceFront.Tag = render_state_display_faceType.front;
            radDisplayFaceBack.Tag  = render_state_display_faceType.back;
            radDisplayFaceNone.Tag  = render_state_display_faceType.none;

            gbxDisplayFace.Visible = ApplicationConfig.DefaultValue.RenderStateInfoVisible;

            // 値が空欄の最適化パラメータを作成する
            foreach (ShaderDebugDumpParameter param in Enum.GetValues(typeof(ShaderDebugDumpParameter)))
            {
                glslCperfStatsData.Items.Add(new ListViewItem(new string[] { param.ToString(), "", "" }));
            }
        }

        class Key
        {
            public int KeyBlock;
            public int KeyPos;
            public int KeyWidth;
            public uint KeyMask;
            public string[] Choices;
        }

        public class SpillMem
        {
            public void Initialize(XmlNode xmlNode)
            {
                XmlElement xmlElement = (XmlElement)xmlNode.SelectSingleNode("numLmemSpillBytes");
                numLmemSpillBytes = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("numLmemRefillBytes");
                numLmemRefillBytes = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("numSmemSpillBytes");
                numSmemSpillBytes = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("numSmemRefillBytes");
                numSmemRefillBytes = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("size");
                size = int.Parse(xmlElement.InnerText);
            }

            public int numLmemSpillBytes;
            public int numLmemRefillBytes;
            public int numSmemSpillBytes;
            public int numSmemRefillBytes;
            public int size;
        }

        public class NonSpillLMem
        {
            public void Initialize(XmlNode xmlNode)
            {
                XmlElement xmlElement = (XmlElement)xmlNode.SelectSingleNode("loadBytes");
                loadBytes = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("storeBytes");
                storeBytes = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("size");
                size = int.Parse(xmlElement.InnerText);
            }

            public int loadBytes;
            public int storeBytes;
            public int size;
        }

        public class ThroughputLimiter
        {
            public void Initialize(XmlNode xmlNode)
            {
                XmlElement xmlElement = (XmlElement)xmlNode.SelectSingleNode("issue");
                issue = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("fp");
                fp = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("half");
                half = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("trancedental");
                trancedental = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("ipa");
                ipa = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("shared");
                shared = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("controlFlow");
                controlFlow = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("texLoadStore");
                texLoadStore = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("reg");
                reg = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("warp");
                warp = float.Parse(xmlElement.InnerText);
            }

            public float issue;
            public float fp;
            public float half;
            public float trancedental;
            public float ipa;
            public float shared;
            public float controlFlow;
            public float texLoadStore;
            public float reg;
            public float warp;
        }

        public class LoopData
        {
            public void Initialize(XmlNode xmlNode)
            {
                XmlElement xmlElement = (XmlElement)xmlNode.SelectSingleNode("partiallyUnrolled");
                partiallyUnrolled = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("nonUnrolled");
                nonUnrolled = int.Parse(xmlElement.InnerText);
            }

            public int partiallyUnrolled;
            public int nonUnrolled;
        }

        public class GLSLCperfStatsData
        {
            public GLSLCperfStatsData()
            {
                spillMem = new SpillMem();
                nonSpillLMem = new NonSpillLMem();
                throughputLimiter = new ThroughputLimiter();
                loopData = new LoopData();
            }

            public void Initialize(XmlNode xmlNode)
            {
                XmlNode childNode = xmlNode.SelectSingleNode("spillMem");
                spillMem.Initialize(childNode);
                childNode = xmlNode.SelectSingleNode("nonSpillLMem");
                nonSpillLMem.Initialize(childNode);
                childNode = xmlNode.SelectSingleNode("throughputLimiter");
                throughputLimiter.Initialize(childNode);
                childNode = xmlNode.SelectSingleNode("loopData");
                loopData.Initialize(childNode);

                XmlElement xmlElement = (XmlElement)xmlNode.SelectSingleNode("latency");
                latency = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("occupancy");
                occupancy = float.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("numDivergentBranches");
                numDivergentBranches = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("attributeMemUsage");
                attributeMemUsage = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("programSize");
                programSize = int.Parse(xmlElement.InnerText);
                xmlElement = (XmlElement)xmlNode.SelectSingleNode("requiresGlobalLoadUniformEmulation");
                requiresGlobalLoadUniformEmulation = bool.Parse(xmlElement.InnerText);
            }

            public int latency;
            public SpillMem spillMem;
            public NonSpillLMem nonSpillLMem;
            public float occupancy;
            public int numDivergentBranches;
            public int attributeMemUsage;
            public int programSize;
            public bool requiresGlobalLoadUniformEmulation;
            public ThroughputLimiter throughputLimiter;
            public LoopData loopData;
        }

        public class ShaderDebugDumpInformation
        {
            public ShaderDebugDumpInformation()
            {
                glslCperfStatsData = new GLSLCperfStatsData();
            }

            public void Initialize(XmlNode xmlNode)
            {
                XmlNode childNode = xmlNode.SelectSingleNode("GLSLCperfStatsData");
                glslCperfStatsData.Initialize(childNode);
            }

            public GLSLCperfStatsData glslCperfStatsData;
        }

        private void SetDumpShaderSourceParameter(int index, string text0, string text1)
        {
            glslCperfStatsData.Items[index].SubItems[(int)ShaderDebugDumpColumn.vs].Text = text0;
            glslCperfStatsData.Items[index].SubItems[(int)ShaderDebugDumpColumn.fs].Text = text1;
        }

        private void UpdateDumpShaderSource(string optimizedLogFolder, string shaderArchiveName, int shaderVariation)
        {
            // マテリアルに使用されるシェーダバリエーションのログ
            string filename = optimizedLogFolder;
            filename += "\\";
            filename += shaderArchiveName;
            filename += "_";
            filename += shaderVariation.ToString("D5");

            // 頂点シェーダのログを解析する
            string vsFileName = filename + "_vs.xml";
            if (System.IO.File.Exists(vsFileName))
            {
                XmlDocument vsDocument = new XmlDocument();
                vsDocument.Load(vsFileName);
                XmlNode vsNode = vsDocument.SelectSingleNode("shader_debug_dump_information");
                if (vsNode != null)
                {
                    vsDumpInformation_.Initialize(vsNode);
                }
            }

            // フラグメントシェーダのログを解析する
            string fsFileName = filename + "_fs.xml";
            if (System.IO.File.Exists(fsFileName))
            {
                XmlDocument fsDocument = new XmlDocument();
                fsDocument.Load(fsFileName);
                XmlNode fsNode = fsDocument.SelectSingleNode("shader_debug_dump_information");
                if (fsNode != null)
                {
                    fsDumpInformation_.Initialize(fsNode);
                }
            }

            // GLSLCperfStatsData
            int index = 0;
            GLSLCperfStatsData vsData = vsDumpInformation_.glslCperfStatsData;
            GLSLCperfStatsData fsData = fsDumpInformation_.glslCperfStatsData;
            SetDumpShaderSourceParameter(index++, vsData.attributeMemUsage.ToString(), fsData.attributeMemUsage.ToString());
            SetDumpShaderSourceParameter(index++, vsData.latency.ToString(), fsData.latency.ToString());
            SetDumpShaderSourceParameter(index++, vsData.occupancy.ToString(), fsData.occupancy.ToString());
            SetDumpShaderSourceParameter(index++, vsData.numDivergentBranches.ToString(), fsData.numDivergentBranches.ToString());
            SetDumpShaderSourceParameter(index++, vsData.programSize.ToString(), fsData.programSize.ToString());
            SetDumpShaderSourceParameter(index++, vsData.requiresGlobalLoadUniformEmulation.ToString(), fsData.requiresGlobalLoadUniformEmulation.ToString());

            // SpillMem
            SpillMem vsSpillMem = vsData.spillMem;
            SpillMem fsSpillMem = fsData.spillMem;
            SetDumpShaderSourceParameter(index++, vsSpillMem.numLmemSpillBytes.ToString(), fsSpillMem.numLmemSpillBytes.ToString());
            SetDumpShaderSourceParameter(index++, vsSpillMem.numLmemRefillBytes.ToString(), fsSpillMem.numLmemRefillBytes.ToString());
            SetDumpShaderSourceParameter(index++, vsSpillMem.numSmemSpillBytes.ToString(), fsSpillMem.numSmemSpillBytes.ToString());
            SetDumpShaderSourceParameter(index++, vsSpillMem.numSmemRefillBytes.ToString(), fsSpillMem.numSmemRefillBytes.ToString());
            SetDumpShaderSourceParameter(index++, vsSpillMem.size.ToString(), fsSpillMem.size.ToString());

            // NonSpillLMem
            NonSpillLMem vsNonSpillLMem = vsData.nonSpillLMem;
            NonSpillLMem fsNonSpillLMem = fsData.nonSpillLMem;
            SetDumpShaderSourceParameter(index++, vsNonSpillLMem.loadBytes.ToString(), fsNonSpillLMem.loadBytes.ToString());
            SetDumpShaderSourceParameter(index++, vsNonSpillLMem.storeBytes.ToString(), fsNonSpillLMem.storeBytes.ToString());
            SetDumpShaderSourceParameter(index++, vsNonSpillLMem.size.ToString(), fsNonSpillLMem.size.ToString());

            // ThroughputLimiter
            ThroughputLimiter vsThroughputLimiter = vsData.throughputLimiter;
            ThroughputLimiter fsThroughputLimiter = fsData.throughputLimiter;
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.issue.ToString(), fsThroughputLimiter.issue.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.fp.ToString(), fsThroughputLimiter.fp.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.half.ToString(), fsThroughputLimiter.half.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.trancedental.ToString(), fsThroughputLimiter.trancedental.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.ipa.ToString(), fsThroughputLimiter.ipa.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.shared.ToString(), fsThroughputLimiter.shared.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.controlFlow.ToString(), fsThroughputLimiter.controlFlow.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.texLoadStore.ToString(), fsThroughputLimiter.texLoadStore.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.reg.ToString(), fsThroughputLimiter.reg.ToString());
            SetDumpShaderSourceParameter(index++, vsThroughputLimiter.warp.ToString(), fsThroughputLimiter.warp.ToString());

            // LoopData
            LoopData vsLoopData = vsData.loopData;
            LoopData fsLoopData = fsData.loopData;
            SetDumpShaderSourceParameter(index++, vsLoopData.partiallyUnrolled.ToString(), vsLoopData.partiallyUnrolled.ToString());
            SetDumpShaderSourceParameter(index++, vsLoopData.nonUnrolled.ToString(), vsLoopData.nonUnrolled.ToString());
        }

        // バリエーション番号の選択変更
        private void cmbShaderVariations_SelectedIndexChanged(object sender, EventArgs e)
        {
            // プラットフォームに NX が選択されている場合に使用できる
            if (EnableShaderDebugDumpInformation())
            {
                UIComboBox senderComboBox = (UIComboBox)sender;
                string variation = senderComboBox.SelectedItem.ToString();
                ChangeShaderVariation(int.Parse(variation));
            }
        }

        private void ChangeShaderVariation(int variation)
        {
            // 最適化ログを更新
            UpdateDumpShaderSource(txtOptimizeShader.Text, ActiveTarget.Data.shader_assign.shading_model, variation);

            // オプション項目を更新
            int variationIndex = cmbShaderVariation.FindString(variation.ToString());
            if (variationIndex >= 0)
            {
                Dictionary<string, string> dictionary = optionVariations_[variationIndex];
                foreach (ListViewItem viewItem in lstShaderOption.Items)
                {
                    viewItem.SubItems[(int)ShaderOptionColumn.choice].Text = dictionary[viewItem.Text];
                }
            }
        }

        private void SetupShaderOptions()
        {
            // ダイナミックオプションだけ表示する
            var shaderDefinition = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == ActiveTarget.Data.shader_assign.shader_archive);
            if (shaderDefinition != null &&
                shaderDefinition.Data.shading_model_array != null &&
                shaderDefinition.Data.shading_model_array.shading_model != null &&
                shaderDefinition.Data.shading_model_array.shading_model.Length > 0)
            {
                var shadingModel = shaderDefinition.Data.shading_model_array.shading_model.FirstOrDefault(x => x.name == ActiveTarget.Data.shader_assign.shading_model);
                if (shadingModel.option_var_array != null)
                {
                    int index = 0;
                    foreach (var optionVar in shadingModel.option_var_array.option_var)
                    {
                        if (optionVar.type == option_var_typeType.dynamic)
                        {
                            lstShaderOption.Items.Add(new ListViewItem(new string[] { optionVar.id, "", optionVar.choice }));
                        }
                        ++index;
                    }
                }
            }
        }

        private void ClearShaderDebugDumpInformation()
        {
            txtOptimizeShader.Text = null; // ログフォルダのパス
            lstShaderOption.Items.Clear(); // ダイナミックオプションの項目
            optionVariations_.Clear(); // ダイナミックオプションのバリエーション
            cmbShaderVariation.Items.Clear(); // バリエーションのコンボボックス
            cmbShaderVariation.Enabled = false;
            ShaderDumpTargetName = null;

            for (int idx = 0; idx < glslCperfStatsData.Items.Count; ++idx)
            {
                SetDumpShaderSourceParameter(idx, string.Empty, string.Empty);
            }
        }

        private string GetOptimizedLogFolder()
        {
            // 情報取得の対象を先頭のモデルとしているが、特に意味はない。
            var model = ActiveTarget.Referrers.FirstOrDefault();
            int index = (model != null) ? model.Materials.IndexOf(ActiveTarget) : -1;
            if (index >= 0)
            {
                if (model.lastMaterialOptimizeData != null &&
                    model.lastMaterialOptimizeData[index] != null)
                {
                    string bfshaPath = model.lastMaterialOptimizeData[index].Item2;
                    if (bfshaPath != null)
                    {
                        return model.GetOptimizedLogFolder(bfshaPath);
                    }
                }
            }

            return null;
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            gbxDisplayFace.Visible = App.AppContext.SelectedPlatformPreset.UseNw;
            gbxDisplayFace.IsModified = ActiveTarget.IsValueChanged(x => x.render_state.display_face);
            radDisplayFaceBoth.SetCheckedByTag( ActiveTarget.Data.render_state.display_face);
            radDisplayFaceFront.SetCheckedByTag(ActiveTarget.Data.render_state.display_face);
            radDisplayFaceBack.SetCheckedByTag( ActiveTarget.Data.render_state.display_face);
            radDisplayFaceNone.SetCheckedByTag( ActiveTarget.Data.render_state.display_face);
            cbxVisibility.IsModified = ActiveTarget.IsValueChanged(x => x.material_info.visibility);
            cbxVisibility.Checked = ActiveTarget.Data.material_info.visibility;
            cbxCompress.IsModified = ActiveTarget.IsValueChanged(x => x.material_info.compress_enable);
            cbxCompress.Checked = ActiveTarget.Data.material_info.compress_enable;
            ltbMeshAdjacency.Text = UIText.FlagYesNo(ActiveTarget.Data.material_info.mesh_adjacency);
            lblMeshAdjacency.IsModified = ActiveTarget.MaterialShaderAssign.ShaderName != null && ActiveTarget.IsMeshAdjacencyModified();

            // シェーダ最適化情報を最新にする
            UpdateShaderDebugDumpInformation(ActiveTarget.MaterialShaderAssign.ShaderName);
        }

        private Dictionary<string, Key> ParseKey(string filename)
        {
            Dictionary<string, Key> result = new Dictionary<string, Key>();
            TextFieldParser parser = new TextFieldParser(filename, System.Text.Encoding.GetEncoding("Shift_JIS"));
            using (parser)
            {
                parser.TextFieldType = FieldType.Delimited;
                parser.SetDelimiters(",");
                parser.ReadFields(); // ヘッダ部分を読み飛ばす
                while (!parser.EndOfData)
                {
                    string[] fields = parser.ReadFields();
                    fields[(int)ParseKeyColumn.KeyMask] = fields[(int)ParseKeyColumn.KeyMask].Replace("_", "");

                    Key key = new Key();
                    key.KeyBlock = int.Parse(fields[(int)ParseKeyColumn.KeyBlock]);
                    key.KeyPos = int.Parse(fields[(int)ParseKeyColumn.KeyPos]);
                    key.KeyWidth = int.Parse(fields[(int)ParseKeyColumn.KeyWidth]);
                    key.KeyMask = uint.Parse(fields[(int)ParseKeyColumn.KeyMask], System.Globalization.NumberStyles.HexNumber);
                    key.Choices = fields[(int)ParseKeyColumn.Choice].Split(',');
                    result[fields[(int)ParseKeyColumn.Id]] = key;
                }
            }

            return result;
        }

        private uint[] ParseShaderKeyBlock(string field)
        {
            uint[] result = null;
            string[] keyBlocks = field.Split('_');
            if (keyBlocks.Length > 0)
            {
                int index = 0;
                result = new uint[keyBlocks.Length];
                foreach (var keyBlock in keyBlocks)
                {
                    result[index] = uint.Parse(keyBlock, System.Globalization.NumberStyles.HexNumber);
                    ++index;
                }
            }

            return result;
        }

        private List<uint[]> ParseShaderKey(string filename)
        {
            List<uint[]> result = new List<uint[]>();
            TextFieldParser parser = new TextFieldParser(filename, System.Text.Encoding.GetEncoding("Shift_JIS"));
            using (parser)
            {
                parser.TextFieldType = FieldType.Delimited;
                parser.SetDelimiters(",");
                parser.ReadFields(); // ヘッダ部分を読み飛ばす
                while (!parser.EndOfData)
                {
                    string[] fields = parser.ReadFields();
                    uint[] keyBlock = ParseShaderKeyBlock(fields[1]);
                    result.Add(keyBlock);
                }
            }
            return result;
        }

        private string ShaderDumpTargetName = null;

        private void UpdateShaderDebugDumpInformation(string shaderName)
        {
            // プラットフォームに実機が選択されている場合に使用できる
            if (EnableShaderDebugDumpInformation())
            {
                // シェーダ最適化情報へのパスを取得
                string optimizedLogFolder = GetOptimizedLogFolder();
                if (optimizedLogFolder != null && Directory.Exists(optimizedLogFolder))
                {
                    // 連続して更新が呼ばれることがあるので、ターゲットの変更が無ければ更新は不要
                    if (ActiveTarget.Name != ShaderDumpTargetName)
                    {
                        // シェーダ情報を一旦クリア
                        ClearShaderDebugDumpInformation();
                        txtOptimizeShader.Text = optimizedLogFolder;
                        ShaderDumpTargetName = ActiveTarget.Name;

                        // バリエーション情報ファイル
                        string shaderKeyFileName = optimizedLogFolder;
                        shaderKeyFileName += "\\";
                        shaderKeyFileName += shaderName;
                        shaderKeyFileName += "_shader_key.csv";
                        List<uint[]> shaderKeys = ParseShaderKey(shaderKeyFileName);

                        string keyFileName = optimizedLogFolder;
                        keyFileName += "\\";
                        keyFileName += shaderName;
                        keyFileName += "_key.csv";
                        Dictionary<string, Key> keys = ParseKey(keyFileName);

                        // シェーダオプション項目を作成する
                        SetupShaderOptions();
                        // シェーダバリエーションを列挙する
                        SetupShaderVariation(shaderKeys, keys);

                        // バリエーションを取得してシェーダ最適化情報を更新
                        GetShaderVariation(ActiveTarget.Data.shader_assign, cmbShaderVariation);

                        // 最初はトップを選択する
                        cmbShaderVariation.Enabled = cmbShaderVariation.Items.Count > 0;
                        if (cmbShaderVariation.Enabled)
                        {
                            string variation = cmbShaderVariation.Items[0].ToString();
                            cmbShaderVariation.SelectedIndex = 0;
                            ChangeShaderVariation(int.Parse(variation));
                        }
                    }
                }
            }
            else
            {
                ClearShaderDebugDumpInformation();
            }
        }

        public static bool IsModified(Material activeTarget)
        {
            return activeTarget != null &&
                (activeTarget.IsValueChanged(x => x.render_state.display_face) ||
                activeTarget.IsValueChanged(x => x.material_info.visibility) ||
                activeTarget.IsValueChanged(x => x.material_info.compress_enable) ||
                activeTarget.IsMeshAdjacencyModified());
        }

        private void cbxVisibility_CheckedChanged(object sender, EventArgs e)
        {
            bool visibility = (bool)((UICheckBox)sender).Checked;

            TheApp.CommandManager.Execute(CreateEditCommand_visibility(Targets, visibility));
        }

        private void cbxCompress_CheckedChanged(object sender, EventArgs e)
        {
            bool compress = (bool)((UICheckBox)sender).Checked;

            TheApp.CommandManager.Execute(CreateEditCommand_compress_enable(Targets, compress));
        }

        private void radDisplayFace_RadioChecked(object sender, EventArgs e)
        {
            render_state_display_faceType displayFace = (render_state_display_faceType)((Control)sender).Tag;

            TheApp.CommandManager.Execute(CreateEditCommand_render_state_display_face(Targets, displayFace));
        }

        #region コマンド

        static private GroupEditCommand CreateEditCommand_visibility(GuiObjectGroup targets, bool visibility)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.Material,
                    visibility,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.material_info.visibility;
                        material.Data.material_info.visibility = (bool)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterialVisibility.Send(editTargets, data, 0xFFFFFFFF, 0xFFFFFFFF);
                    }
                );
        }

        static private GroupEditCommand CreateEditCommand_compress_enable(GuiObjectGroup targets, bool compress_enable)
        {
            return
                 new GeneralGroupValueEditCommand<bool>(
                    targets,
                    GuiObjectID.Material,
                    compress_enable,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.material_info.compress_enable;
                        material.Data.material_info.compress_enable = (bool)data;
                    }
                );
        }

        static private GroupEditCommand CreateEditCommand_render_state_display_face(GuiObjectGroup targets, render_state_display_faceType render_state_display_face)
        {
            return
                 new GeneralGroupValueEditCommand<render_state_display_faceType>(
                    targets,
                    GuiObjectID.Material,
                    render_state_display_face,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var material = target as Material;

                        swap = material.Data.render_state.display_face;
                        material.Data.render_state.display_face = (render_state_display_faceType)data;
                    },
                    postEditDelegate : (editTargets, data) =>
                    {
                        Viewer.SetMaterial_render_state_display_face.Send(editTargets, data, 0xFFFFFFFF, 0xFFFFFFFF);
                    }
                );
        }
        #endregion

        #region コピー＆ペースト
        private class CopyData
        {
            public bool visibility { get; set; }
            public bool compress_enable { get; set; }
            public render_state_display_faceType render_state_display_face { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(Material target)
        {
            return
                new CopyData()
                {
                    visibility = target.Data.material_info.visibility,
                    compress_enable = target.Data.material_info.compress_enable,
                    render_state_display_face = target.Data.render_state.display_face,
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            var commandSet = new EditCommandSet();
            {
                var copyData = (CopyData)pasteObject;

                commandSet.Add(CreateEditCommand_visibility(targets, copyData.visibility));
                commandSet.Add(CreateEditCommand_compress_enable(targets, copyData.compress_enable));
                commandSet.Add(CreateEditCommand_render_state_display_face(targets, copyData.render_state_display_face));
            }
            return commandSet.Execute();
        }
        #endregion

        private class ListViewInputBox : TextBox
        {
            private MaterialGeneralPage ownerPage_;
            private ListViewItem.ListViewSubItem target_;
            private string varType_;

            public ListViewInputBox(MaterialGeneralPage ownerPage, UIListView parent,
                                        ListViewItem.ListViewSubItem target, string varType)
            {
                ownerPage_ = ownerPage;
                target_ = target;
                varType_ = varType;
                Parent = parent;
                Text = target_.Text;
                Top = target_.Bounds.Top;
                Left = target_.Bounds.Left;
                Width = target_.Bounds.Width;
                Height = target_.Bounds.Height;
                KeyDown += textbox_KeyDown;
                LostFocus += textbox_LostFocus;
            }

            // エンターキーで入力完了
            public void textbox_KeyDown(object sender, KeyEventArgs e)
            {
                if (e.KeyCode == Keys.Enter)
                {
                    e.Handled = true;
                    Parent.Focus();
                }
            }

            // フォーカスが離れたら更新
            public void textbox_LostFocus(Object sender, EventArgs e)
            {
                List<string> candidate = new List<string>();
                bool isCorrectionRange = false;

                // 入力できる値の候補一覧を作成
                if (varType_ == "bool")
                {
                    isCorrectionRange = true;
                    candidate.Add("0");
                    candidate.Add("1");
                }
                else if (varType_.StartsWith("[") && varType_.EndsWith("]"))
                {
                    isCorrectionRange = true;
                    varType_ = varType_.Replace(" ", "");
                    varType_ = varType_.Replace("[", "");
                    varType_ = varType_.Replace("]", "");
                    string[] split = varType_.Split(',');
                    if (split.Length == 2)
                    {
                        int min = int.Parse(split[0]);
                        int max = int.Parse(split[1]);
                        for (int i = min; i <= max; ++i)
                        {
                            candidate.Add(i.ToString());
                        }
                    }
                }
                else
                {
                    varType_ = varType_.Replace(" ", "");
                    foreach (string toke in varType_.Split(','))
                    {
                        candidate.Add(toke);
                    }
                }

                // 数値を範囲内に補正する
                if (isCorrectionRange)
                {
                    if (int.Parse(Text) < int.Parse(candidate.First()))
                    {
                        Text = candidate.First();
                    }
                    if (int.Parse(Text) > int.Parse(candidate.Last()))
                    {
                        Text = candidate.Last();
                    }
                }

                // 入力候補に適合するか
                if (candidate.IndexOf(Text) >= 0)
                {
                    bool isChange = target_.Text != Text;
                    target_.Text = Text;
                    if (isChange)
                    {
                        ownerPage_.lstShaderOption_TextChanged();
                    }
                }

                Hide();
            }
        }

        // バリエーションを探す
        private int GetCurrentVariationIndex()
        {
            // UI からダイナミックオプションを取得する
            Dictionary<string, string> dictionary = new Dictionary<string, string>(optionVariations_[0]);
            foreach (ListViewItem item in lstShaderOption.Items)
            {
                string id = item.SubItems[(int)ShaderOptionColumn.id].Text;
                string choice = item.SubItems[(int)ShaderOptionColumn.choice].Text;
                dictionary[id] = choice;
            }

            // 該当するバリエーションを探す
            int variation = 0;
            foreach (Dictionary<string, string> option in optionVariations_)
            {
                int matchCount = 0;
                foreach (KeyValuePair<string, string> pair in dictionary)
                {
                    if (option[pair.Key] == pair.Value)
                    {
                        ++matchCount;
                    }
                }

                // すべてオプション項目が一致するものがそれ
                if (matchCount == option.Count)
                {
                    return variation;
                }

                ++variation;
            }

            return -1;
        }

        // ダイナミックオプションの項目を編集する
        private void lstShaderOption_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            UIListView target = (UIListView)sender;
            ListViewItem.ListViewSubItem subItem = target.SelectedItems[0].SubItems[(int)ShaderOptionColumn.choice];
            string varType = target.SelectedItems[0].SubItems[(int)ShaderOptionColumn.varType].Text;
            ListViewInputBox viewInputBox = new ListViewInputBox(this, target, subItem, varType);
            viewInputBox.Show();
            viewInputBox.Focus();
        }

        // ダイナミックオプションの値が変更された
        public void lstShaderOption_TextChanged()
        {
            int variationIndex = GetCurrentVariationIndex();
            if (variationIndex >= 0 && variationIndex < cmbShaderVariation.Items.Count)
            {
                cmbShaderVariation.SelectedIndex = variationIndex;
            }
        }
    }

    /// <summary>
    /// 環境設定の変更通知
    /// </summary>
    public class DocumentPropertyChangedEnvRefArgs : DocumentContentsChangedArgs
    {
        public DocumentPropertyChangedEnvRefArgs(GuiObject target, EditCommand command) : base(target, command) { }
    }
}
