﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Data;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public class CameraAnimationCurveTreeNodeInfo : IAnimationCurve
    {
        public AnimationDocument.NonEditableKind NonEditableKind { get; set; }
        public object NonEditableKindDisplayAux { get; set; }
        public bool IsColorCurve { get { return false; } }
        public int ColorComponentIndex { get { return -1; } }

        public bool IsEditable
        {
            get
            {
                // カメラタイプを考慮した値を返す
                if (target_ == null || !isEditable)
                {
    	            return false;
                }

                var validType = true;
                switch (targetType_)
                {
                    case camera_anim_target_targetType.aim_x:
                    case camera_anim_target_targetType.aim_y:
                    case camera_anim_target_targetType.aim_z:
                    case camera_anim_target_targetType.twist:
                        validType = target_.Data.rotate_mode == camera_anim_rotate_modeType.aim;
                        break;
                    case camera_anim_target_targetType.rotate_x:
                    case camera_anim_target_targetType.rotate_y:
                    case camera_anim_target_targetType.rotate_z:
                        validType = target_.Data.rotate_mode == camera_anim_rotate_modeType.euler_zxy;
                        break;
                    case camera_anim_target_targetType.ortho_height:
                        validType = target_.Data.projection_mode == camera_anim_projection_modeType.ortho;
                        break;
                    case camera_anim_target_targetType.persp_fovy:
                        validType = target_.Data.projection_mode == camera_anim_projection_modeType.persp;
                        break;
                }
                return validType;
            }
            set
            {
                isEditable = value;
            }
        }

        public AnimTarget GetAnimTarget(GuiObject editTarget)
        {
            Debug.Assert(editTarget is CameraAnimation);
            var animation = (CameraAnimation)editTarget;
            return animation.GetTarget(targetType_);
        }

        public void UpdateIsModified(GuiObject editTarget)
        {
            var animation = (CameraAnimation)editTarget;
            var animTarget = GetAnimTarget(animation);
            var savedTarget = animation.savedCameraAnimTargets.FirstOrDefault(x => x.targetType == targetType_);
            animation.UpdateIsModifiedAnimTarget(
                (CameraAnimationTarget)animTarget,
                savedTarget);
        }

        public AnimTarget CreateAnimTarget(GuiObject editTarget)
        {
            Debug.Assert(false);
            return null;
        }

        public AnimTarget CreateTemporaryTarget(GuiObject editTarget)
        {
            var source = (CameraAnimationTarget)GetAnimTarget(editTarget);
            return new CameraAnimationTarget()
            {
                targetType = source.targetType,
            };
        }

        public void SetAnimTarget(GuiObject editTarget, AnimTarget animTarget)
        {
            var anims = ((CameraAnimation)editTarget).CameraAnimTargets;
            var index = anims.FindIndex(x => x.targetType == targetType_);
            Debug.Assert(index >= 0);
            anims[index] = (CameraAnimationTarget)animTarget;
        }

        public float GetDefaultValue(GuiObject editTarget)
        {
            return GetAnimTarget(editTarget).GetBaseValue();
        }

        public void IntermediateFileIndices(CameraAnimation target, out int cameraIndex, out int curveIndex, out int curveComponentIndex)
        {
            curveComponentIndex = (int)targetType_;
            cameraIndex = target.Owner.CameraAnims.IndexOf(target);
            curveIndex = 0;

            var isAim = target.Data.rotate_mode == camera_anim_rotate_modeType.aim;
            var isPerspective = target.Data.projection_mode == camera_anim_projection_modeType.persp;

            foreach (var animTarget in target.CameraAnimTargets)
            {
                if (animTarget.targetType == targetType_)
                {
                    return;
                }

                // カメラのタイプにより、カーブのインデクスを変える
                switch (animTarget.targetType)
                {
                    case camera_anim_target_targetType.aim_x:
                    case camera_anim_target_targetType.aim_y:
                    case camera_anim_target_targetType.aim_z:
                    case camera_anim_target_targetType.twist:
                        if (!isAim)
                        {
                            continue;
                        }
                        break;
                    case camera_anim_target_targetType.rotate_x:
                    case camera_anim_target_targetType.rotate_y:
                    case camera_anim_target_targetType.rotate_z:
                        if (isAim)
                        {
                            continue;
                        }
                        break;
                    case camera_anim_target_targetType.ortho_height:
                        if (isPerspective)
                        {
                            continue;
                        }
                        break;
                    case camera_anim_target_targetType.persp_fovy:
                        if (!isPerspective)
                        {
                            continue;
                        }
                        break;
                }

                curveIndex++;
            }

            curveIndex = -1;
        }

        private readonly CameraAnimation	target_;
        private readonly camera_anim_target_targetType targetType_;

        private bool isEditable;

        public CameraAnimationCurveTreeNodeInfo(CameraAnimation target, camera_anim_target_targetType targetType)
        {
            //Debug.Assert(this is IHasAnimTarget);
            target_			= target;
            targetType_	= targetType;
        }

        public List<KeyFrame> KeyFrames
        {
            get
            {
                return
                    (paramAnim != null) ?
                        paramAnim.KeyFrames :
                        new List<KeyFrame>();
            }
        }

        public CameraAnimationTarget paramAnim { get { return target_.GetTarget(targetType_); } }
        public InterpolationType CurveInterpolationType
        {
            get
            {
                return paramAnim != null ? paramAnim.CurveInterpolationType : InterpolationType.Linear;
            }

            set
            {
                ;   // 無視
            }
        }

        public string ParentName{		get { return string.Empty;		} }
        public string Name{				get { return string.Empty;		} }
        public string FullPath
        {
            get
            {
                switch(targetType_)
                {
                    case camera_anim_target_targetType.position_x:		return target_.Name + " / position / X";
                    case camera_anim_target_targetType.position_y:		return target_.Name + " / position / Y";
                    case camera_anim_target_targetType.position_z:		return target_.Name + " / position / Z";
                    case camera_anim_target_targetType.aim_x:			return target_.Name + " / aim / X";
                    case camera_anim_target_targetType.aim_y:			return target_.Name + " / aim / Y";
                    case camera_anim_target_targetType.aim_z:			return target_.Name + " / aim / Z";
                    case camera_anim_target_targetType.twist:			return target_.Name + " / twist";
                    case camera_anim_target_targetType.rotate_x:		return target_.Name + " / rotate / X";
                    case camera_anim_target_targetType.rotate_y:		return target_.Name + " / rotate / Y";
                    case camera_anim_target_targetType.rotate_z:		return target_.Name + " / rotate / Z";
                    case camera_anim_target_targetType.aspect:			return target_.Name + " / aspect";
                    case camera_anim_target_targetType.near:			return target_.Name + " / near";
                    case camera_anim_target_targetType.far:				return target_.Name + " / far";
                    case camera_anim_target_targetType.ortho_height:	return target_.Name + " / ortho_height";
                    case camera_anim_target_targetType.persp_fovy:		return target_.Name + " / persp_fovy";
                }

                Debug.Assert(false);
                return string.Empty;
            }
        }

        public int ComponentIndex{		get { return (int)targetType_;	} }
        public bool IsRotate
        {
            get {
                switch (targetType_)
                {
                    case camera_anim_target_targetType.rotate_x:
                    case camera_anim_target_targetType.rotate_y:
                    case camera_anim_target_targetType.rotate_z:
                    case camera_anim_target_targetType.twist:
                    case camera_anim_target_targetType.persp_fovy:
                        return true;
                }
                return false;
            }
        }
        public float? MinClampValue{	get { return MinFitValue;	} }
        public float? MaxClampValue{	get { return MaxFitValue;	} }
        public float? MinFitValue
        {
            get
            {
                switch (targetType_)
                {
                    case camera_anim_target_targetType.aspect:
                    case camera_anim_target_targetType.near:
                    case camera_anim_target_targetType.far:
                    case camera_anim_target_targetType.ortho_height:
                    case camera_anim_target_targetType.persp_fovy:
                        return 0;

                }
                return null;
            }
        }

        public float? MaxFitValue{
            get {
                switch (targetType_)
                {
                    case camera_anim_target_targetType.persp_fovy:
                        return 180;
                }
                return null;
            }
        }

        public PrimitiveTypeKind CurvePrimitiveType { get { return PrimitiveTypeKind.Float; } }
        public Color CurveColor
        {
            get
            {
                switch (targetType_)
                {
                    case camera_anim_target_targetType.position_x:
                    case camera_anim_target_targetType.aim_x:
                    case camera_anim_target_targetType.rotate_x:
                        return CurveView.MakeRGBAColorFromIndex(0);
                    case camera_anim_target_targetType.position_y:
                    case camera_anim_target_targetType.aim_y:
                    case camera_anim_target_targetType.rotate_y:
                        return CurveView.MakeRGBAColorFromIndex(1);

                    case camera_anim_target_targetType.position_z:
                    case camera_anim_target_targetType.aim_z:
                    case camera_anim_target_targetType.rotate_z:
                        return CurveView.MakeRGBAColorFromIndex(2);
                }
                return CurveView.MakeColorFromString(FullPath);
            }
        }

        public bool IsSame(IAnimationCurve curve)
        {
            var cameraCurve = (CameraAnimationCurveTreeNodeInfo)curve;
            return cameraCurve.target_ == target_ &&
                cameraCurve.targetType_ == targetType_;
        }
    }
}
