﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Command;
using App.Controls;
using App.Data;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.PropertyEdit
{
    public partial class CameraAnimationCurveEditPage : CameraAnimationPropertyPage, IAnimationEditPage
    {
        public CameraAnimationCurveEditPage() :
            base(PropertyPageID.CameraAnimationCurveEdit)
        {
            InitializeComponent();
        }

        public override Utility.HelpUtility.PageKey HelpKey
        {
            get
            {
                return Utility.HelpUtility.PageKey.p_curve_editor;
            }
        }

        public static ObjectPropertyPage CreateInstance(object arg)
        {
            return new CameraAnimationCurveEditPage();
        }

        protected override void InitializeFormInternal()
        {
            AnimationObjectPropertyPanel animationObjectPropertyPanel = Owner as AnimationObjectPropertyPanel;
            Debug.Assert(animationObjectPropertyPanel != null);

            CurveEditorPanel.Initialize(animationObjectPropertyPanel, ActiveTarget.ObjectID);

            CurveEditorPanel.UpdateTreeView       += (s, e) => UpdateTreeView();
            CurveEditorPanel.ChangeSelectedCurves += (s, e) => UpdateSelected();
            CurveEditorPanel.ChangeVisibledCurves += (s, e) => UpdateVisibled();
            CurveEditorPanel.GetFrameCount        = () => ActiveTarget.Data.frame_count;
        }

        public override Size DefaultPageSize
        {
            get
            {
                return new Size(
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width),
                    Math.Max(25, ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height));
            }
        }

        protected override void OnSizeChanged(EventArgs e)
        {
            if ((Owner != null) && (Owner.ActivePage == this) && !ObjectPropertyDialog.InternallyChangingSize.IsChanging &&
                Owner.Owner.WindowState == System.Windows.Forms.FormWindowState.Normal)
            {
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Width = Width;
                ConfigData.ApplicationConfig.Setting.PropertyEdit.CurveEditorPageSize.Height = Height;
            }

            base.OnSizeChanged(e);
        }

        protected override void UpdateFormInternal(UpdateFormInfo updateFormInfo)
        {
            CurveEditorPanel.TargetGroup = Targets;

            CurveEditorPanel.UpdateForm(updateFormInfo);
        }

        protected override void UpdateFormOnPageCreatedInternal()
        {
            CurveEditorPanel.UpdateFormOnPageCreatedInternal();
        }


        public override void BeforePageDeactivated()
        {
            CurveEditorPanel.BeforePageDeactivated();
        }

        public override void AfterPageActiveted(ObjectPropertyPage oldPage)
        {
            CurveEditorPanel.AfterPageActivated();
        }

        private void UpdateSelected()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private void UpdateVisibled()
        {
            CurveEditorPanel.InvalidateCurveView();
        }

        private Dictionary<string, bool> visibledNodes_ = new Dictionary<string, bool>();
        private Dictionary<string, bool>	isExpandedNodes_	= new Dictionary<string, bool>();

        private void UpdateTreeView()
        {
            var	isExpandedNodes	= new Dictionary<string, bool>();

            using (var sb = new UIControlEventSuppressBlock())
            {
                // ノードの状態を保存しておく
                if (CurveEditorPanel.CurveTreeView.Nodes.Count > 0)
                {
                    var root = (CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0];
                    foreach (var tuple in CurveTreeView.NodesAndFullPath(root, root.id))
                    {
                        // チェック状態
                        visibledNodes_[tuple.Item2] = tuple.Item1.Checked;

                        // 開閉状態
                        isExpandedNodes_[tuple.Item2] = tuple.Item1.Nodes.Count == 0 || tuple.Item1.IsExpanded;
                    }
                }

                bool isAllShowNode = (Owner as AnimationObjectPropertyPanel).IsAllShowNode;

                // つくり直す
                {
                    var rootNode = new CurveTreeInfo()
                    {
                        Text = ActiveTarget.Name,
                        Id = ActiveTarget.Name,
                        ShowEmptyNode = true,
                    };

                    var groups = (from target in ActiveTarget.CameraAnimTargets.OrderBy(x => x.targetType)
                                  group target by SceneAnimation.Group(target.targetType)).ToArray();
                    var isAim = ActiveTarget.Data.rotate_mode == camera_anim_rotate_modeType.aim;
                    var isPerspective = ActiveTarget.Data.projection_mode == camera_anim_projection_modeType.persp;

                    foreach (var group in groups)
                    {
                        List<CurveTreeInfo> nodes = new List<CurveTreeInfo>();
                        foreach (var cameraAnimationTarget in group)
                        {
                            var targetType = cameraAnimationTarget.targetType;
                            switch (targetType)
                            {
                                case camera_anim_target_targetType.aim_x:
                                case camera_anim_target_targetType.aim_y:
                                case camera_anim_target_targetType.aim_z:
                                case camera_anim_target_targetType.twist:
                                    if (!isAim)
                                    {
                                        continue;
                                    }
                                    break;
                                case camera_anim_target_targetType.rotate_x:
                                case camera_anim_target_targetType.rotate_y:
                                case camera_anim_target_targetType.rotate_z:
                                    if (isAim)
                                    {
                                        continue;
                                    }
                                    break;
                                case camera_anim_target_targetType.ortho_height:
                                    if (isPerspective)
                                    {
                                        continue;
                                    }
                                    break;
                                case camera_anim_target_targetType.persp_fovy:
                                    if (!isPerspective)
                                    {
                                        continue;
                                    }
                                    break;
                            }

                            var targetNode = new CurveTreeInfo()
                            {
                                Id = SceneAnimation.Subscript(group.Key, targetType.ToString()),
                                Text = SceneAnimation.Subscript(group.Key, targetType.ToString()),
                                AnimationCurve = new CameraAnimationCurveTreeNodeInfo(ActiveTarget, targetType),
                                ShowEmptyNode = isAllShowNode,
                                IsModified = cameraAnimationTarget.IsModified,
                            };

                            nodes.Add(targetNode);
                        }

                        if (nodes.Count > 0)
                        {
                            if (group.Count() == 1)
                            {
                                rootNode.Nodes.AddRange(nodes);
                            }
                            else
                            {
                                var groupNode = new CurveTreeInfo()
                                {
                                    Text = group.Key,
                                    Id = group.Key,
                                    Nodes = nodes,
                                    ShowEmptyNode = isAllShowNode,
                                };
                                rootNode.Nodes.Add(groupNode);
                            }
                        }
                    }

                    rootNode.TrimInvisibleNodes();

                    // 文字列によるフィルター
                    CurveTreeInfo.FilterCurveTreeInfoRoot(rootNode, (Owner as AnimationObjectPropertyPanel).SearchTexts);

                    {
                        var nodes = rootNode.ConvertToTreeNode();
                        if (nodes.IsSameStructure(CurveEditorPanel.CurveTreeView.Nodes) == false)
                        {
                            if ((CurveEditorPanel.UpdateFormInfo == null) || (CurveEditorPanel.UpdateFormInfo.TargetOrPageChanged == false))
                            {
                                // つくり直す


                                foreach (var tuple in CurveTreeView.NodesAndFullPath((CurveTreeNode)nodes, nodes.id))
                                {
                                    // チェック状態を設定する
                                    bool isChecked = false;
                                    if (visibledNodes_.TryGetValue(tuple.Item2, out isChecked))
                                    {
                                        if (isChecked != tuple.Item1.Checked)
                                        {
                                            tuple.Item1.Checked = isChecked;
                                        }
                                    }
                                    else
                                    {
                                        tuple.Item1.Checked = true;
                                    }

                                    bool isExpanded = false;
                                    if (isExpandedNodes_.TryGetValue(tuple.Item2, out isExpanded))
                                    {
                                        if (isExpanded)
                                        {
                                            if (!tuple.Item1.IsExpanded)
                                            {
                                                tuple.Item1.Expand();
                                            }
                                        }
                                        else
                                        {
                                            if (tuple.Item1.IsExpanded)
                                            {
                                                tuple.Item1.Collapse();
                                            }
                                        }
                                    }
                                    else
                                    {
                                        tuple.Item1.Expand();
                                    }
                                }

                                CurveEditorPanel.CurveTreeView.Nodes.Clear();
                                CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);
                            }
                            else
                            {
                                nodes.ExpandAll();

                                // つくり直す
                                CurveEditorPanel.CurveTreeView.Nodes.Clear();
                                CurveEditorPanel.CurveTreeView.Nodes.Add(nodes);

                                // 新規に開いたときは全表示にする
                                CurveEditorPanel.VisibleAllNode();
                            }
                        }
                        else
                        {
                            ((CurveTreeNode)CurveEditorPanel.CurveTreeView.Nodes[0]).CopyInfo(nodes);
                        }
                    }
                }
            }
        }

        #region コピー＆ペースト
        private class CopyData
        {
            public int frame_count { get; set; }
            public bool loop { get; set; }
            public camera_anim_rotate_modeType rotate { get; set; }
            public camera_anim_projection_modeType projection { get; set; }
            public List<CameraAnimationTarget> CameraAnims { get; set; }
        }

        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy()
        {
            return true;
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public override object Copy(ref object copyObjectInfo)
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static object Copy(CameraAnimation target)
        {
            return
                new CopyData()
                {
                    frame_count = target.Data.frame_count,
                    loop = target.Data.loop,
                    rotate = target.Data.rotate_mode,
                    projection = target.Data.projection_mode,
                    CameraAnims = ObjectUtility.Clone(target.CameraAnimTargets)
                };
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(object pasteObject)
        {
            TheApp.CommandManager.Add(Paste(Targets, pasteObject));
        }

        public override bool CanPaste(object copiedObjectInfo, object copiedObject)
        {
            return CanPaste(ActiveTarget, copiedObjectInfo, copiedObject);
        }

        public static bool CanPaste(CameraAnimation ActiveTarget, object copiedObjectInfo, object copiedObject)
        {
            return true;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static ICommand Paste(GuiObjectGroup targets, object pasteObject)
        {
            EditCommandSet commandSet = new EditCommandSet();
            commandSet.SetViewerDrawSuppressBlockDelegate(AnimationCurveEditCommand.AnimationMessageFilter);
            using (var block = new Viewer.ViewerDrawSuppressBlock(AnimationCurveEditCommand.AnimationMessageFilter))
            {
                var copyData = (CopyData)pasteObject;
                commandSet.Add(CameraAnimationGeneralPage.CreateEditCommand_frame_count(targets, copyData.frame_count).Execute());
                commandSet.Add(CameraAnimationGeneralPage.CreateEditCommand_loop(targets, copyData.loop).Execute());
                commandSet.Add(
                    new GeneralGroupReferenceEditCommand<List<CameraAnimationTarget>>(
                        targets,
                        GuiObjectID.CameraAnimation,
                        ObjectUtility.MultipleClone(copyData.CameraAnims, targets.Objects.Count),
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            Debug.Assert(target is CameraAnimation);
                            var animation = (CameraAnimation)target;

                            swap = animation.CameraAnimTargets;
                            animation.CameraAnimTargets = (List<CameraAnimationTarget>)data;
                            animation.UpdateIsModifiedAnimTargetAll();

                            // TODO: Viewer 通信
                        }
                    ).Execute()
                );

                // 他のアニメーションの量子化も更新される!
                var active = (CameraAnimation)targets.Active;

                var command =
                    SceneAnimationGeneralPage.CreateEditCommand_quantize(
                        new GuiObjectGroup(active.Owner),
                        GuiObjectID.SceneAnimation,
                        active.Owner.Data.scene_anim_info.quantize_tolerance_rotate,
                        active.Owner.Data.scene_anim_info.quantize_tolerance_translate,
                        false);

                if (command != null)
                {
                    commandSet.Add(command.Execute());
                }

                // 再転送を設定
                EventHandler reload = (s, e) =>
                {
                    foreach (var target in targets.Objects.OfType<CameraAnimation>().Select(x => x.Owner).Distinct())
                    {
                        Viewer.LoadOrReloadAnimation.Send(target);
                    }
                };

                commandSet.OnPostEdit += reload;
                reload(null, null);
            }

            commandSet.Reverse();
            return commandSet;
        }
        #endregion
    }
}
