﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using System.Linq;
using App.Data;
using App.ObjectView.Schematic;

namespace App.ObjectView.List
{
    /// <summary>
    /// 標準動作オブジェクトリストビュークラス。
    /// シーンコンポーネント以外のオブジェクト用です。
    /// </summary>
    public class StandardObjectListView : ObjectListView
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected StandardObjectListView()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected StandardObjectListView(ViewID viewID, AppConfig.ObjectViewMode viewMode)
            : base(viewID, viewMode, true)
        {
            // イベント登録
            App.AppContext.PropertyChanged += Document_PropertyChanged;
            App.AppContext.DocumentAddedOrRemoved += Document_AddedOrRemoved;
            App.AppContext.SelectedTargetChanged += GuiObjectManager_SelectedChanged;

            // 初期項目作成
            RefreshList();
        }

        /// <summary>
        /// リスト更新。
        /// </summary>
        protected override void RefreshList()
        {
            // 項目更新
            BeginUpdate();
            {
                // 既存項目破棄
                if (Items.Count > 0)
                {
                    Items.Clear();
                }

                // 項目作成
                CreateItem();
            }
            EndUpdate();

            // 選択状態更新
            SynchronizeSelectedState();
        }

        /// <summary>
        /// 項目作成。
        /// </summary>
        protected virtual void CreateItem()
        {
            throw new NotSupportedException();
        }

        /// <summary>
        /// ドキュメント構成変更時に項目更新の必要があるか。
        /// </summary>
        protected virtual bool NeedRefreshItem(DocumentComponentChangedEventArgs e)
        {
            return true;
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnVisibleChanged(EventArgs e)
        {
            // 表示されたら選択状態を更新する
            if (Visible)
            {
                SynchronizeSelectedState();
            }
            base.OnVisibleChanged(e);
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            switch (keyData & Keys.KeyCode)
            {
                case Keys.F:
                    var schematicView = TheApp.MainFrame.ObjectViewClient.GetViewControl(AppConfig.ObjectViewMode.SchematicView);
                    if (schematicView != null)
                    {
                        ((ObjectSchematicViewPanel)schematicView).View.CenteringSelected();
                        return true;
                    }
                    break;
            }
            return base.ProcessCmdKey(ref msg, keyData);
        }
        #endregion

        #region イベントハンドラ

        protected virtual void Document_AddedOrRemoved(object sender, IEnumerable<Document> opened, IEnumerable<Document> closed, Dictionary<GuiObject, GuiObject> swaped, IEnumerable<Document> reloaded)
        {
            RefreshList();
        }

        //---------------------------------------------------------------------------
        // ドキュメント内容
        protected virtual void Document_PropertyChanged(object sender, IEnumerable<DocumentPropertyChangedArgs> e)
        {
            if (e.Any(x => x is App.FileView.DocumentPropertyChangedShowInObjViewArgs))
            {
                RefreshList();
            }
            Invalidate();
        }

        //---------------------------------------------------------------------------
        // 選択オブジェクト
        private void GuiObjectManager_SelectedChanged(object sender, EventArgs e)
        {
            if (!IsSelectionChanging)
            {
                SynchronizeSelectedState();
            }
        }
        #endregion

        /// <summary>
        /// 列の幅の合計の上限
        /// </summary>
        public const int ColumnWidhtSumUpperBound = 32768;

        public int ColumnWidthSum { get; set; }
        private List<int> Widths = new List<int>();
        protected override void OnColumnWidthChanged(ColumnWidthChangedEventArgs e)
        {
            if (Columns.Count < Widths.Count)
            {
                for (int i = Columns.Count; i < Widths.Count; i++)
                {
                    ColumnWidthSum -= Widths[i];
                }
                Widths.RemoveRange(Columns.Count, Widths.Count - Columns.Count);
            }
            else if (Columns.Count > Widths.Count)
            {
                int count = Widths.Count;
                Widths.AddRange(Enumerable.Range(Widths.Count, Columns.Count - Widths.Count).Select(x => Columns[x].Width));
                for (int i = count; i < Widths.Count; i++)
                {
                    ColumnWidthSum += Widths[i];
                }
            }

            Debug.Assert(e.ColumnIndex < Columns.Count);

            ColumnWidthSum += Columns[e.ColumnIndex].Width - Widths[e.ColumnIndex];
            Widths[e.ColumnIndex] = Columns[e.ColumnIndex].Width;

            base.OnColumnWidthChanged(e);
        }

        #region 検索
        private string searchText = string.Empty;
        private string[] splittedSearchTexts = new string[0];
        public string SearchText
        {
            get
            {
                return searchText;
            }
            set
            {
                if (searchText != value)
                {
                    searchText = value;
                    splittedSearchTexts = searchText.ToLower().Split(new char[0] { }, StringSplitOptions.RemoveEmptyEntries);
                    RefreshList();
                }
            }
        }

        protected bool IsMatchedToSearchText(string text)
        {
            var lower = text.ToLower();
            return splittedSearchTexts.All(x => lower.Contains(x));
        }
        #endregion
    }

    /// <summary>
    /// 画像オブジェクトリストビュークラス。
    /// テクスチャ、パレット用です。
    /// </summary>
    public class ImageObjectListView : StandardObjectListView
    {
        // ３２サイズイメージリスト
        private static readonly ImageList _iml32Size = new ImageList();
        // ６４サイズイメージリスト
        private static readonly ImageList _iml64Size = new ImageList();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        protected ImageObjectListView(ViewID viewID, AppConfig.ObjectViewMode viewMode)
            : base(viewID, viewMode)
        {
            // 行の高さを変更
            SmallImageList = _iml32Size;
            LargeImageList = _iml64Size;
        }

        /// <summary>
        /// タイプコンストラクタ。
        /// </summary>
        static ImageObjectListView()
        {
            // ３２サイズイメージリスト
            _iml32Size.ImageSize = new Size(32, 32);
            _iml32Size.ColorDepth = ColorDepth.Depth8Bit;
            _iml32Size.TransparentColor = Color.Magenta;

            // ６４サイズイメージリスト
            _iml64Size.ImageSize = new Size(64, 64);
            _iml64Size.ColorDepth = ColorDepth.Depth8Bit;
            _iml64Size.TransparentColor = Color.Magenta;
        }
    }
}
