﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;

namespace App.ObjectView.List
{
    /// <summary>
    /// モデルリストビュークラス。
    /// </summary>
    [ToolboxItem(false)]
    public sealed class ModelListView : StandardObjectListView
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ModelListView()
            : base(ViewID.Model, AppConfig.ObjectViewMode.ModelList)
        {
        }

        protected override void Document_AddedOrRemoved(object sender, System.Collections.Generic.IEnumerable<Document> opened, System.Collections.Generic.IEnumerable<Document> closed, System.Collections.Generic.Dictionary<GuiObject, GuiObject> swaped, System.Collections.Generic.IEnumerable<Document> reloaded)
        {
            IEnumerable<Document> documents = opened.Concat(closed).Concat(reloaded);
            if (documents.Any(x => x.ObjectID == GuiObjectID.Model))
            {
                // 列の更新
                SetupColumn();
                NotifyColumnSettingChanged();
            }
            else
            {
                base.Document_AddedOrRemoved(sender, opened, closed, swaped, reloaded);
            }
        }

        /// <summary>
        /// 列項目初期化。
        /// </summary>
        public static void InitializeColumn()
        {
            SetupColumn();
        }

        public static void SetupColumn()
        {
            BeginRegisterColumn(ViewID.Model, "ModelListView");
            {
                CurrentPropertyPageID = PropertyEdit.PropertyPageID.ModelRoot;

                // ColumnIDで宣言している順番で登録します。
                // モデル名
                RegisterColumn(ColumnID.Name, "Name", res.Strings.ListView_Model_Column_Name, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Model).Name);
                // コメント
                RegisterCommentColorColumn(ColumnID.CommentColor, "CommentColor", res.Strings.ListView_Object_Column_CommentColor, HorizontalAlignment.Left, 16, true);
                RegisterColumn(ColumnID.CommentText, "CommentText", res.Strings.ListView_Object_Column_CommentText, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((Model)data).Comment ?? string.Empty, ColumnID.CommentText.ToString());
                RegisterColumn(ColumnID.CommentLabel, "CommentLabel", res.Strings.ListView_Object_Column_CommentLabel, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((Model)data).Label ?? string.Empty, ColumnID.CommentLabel.ToString());

                var showingModels = DocumentManager.Models.Where(x => x.IsShowInObjView).ToArray();
                var lodCount = 1;
                if (showingModels.Any())
                {
                    lodCount = Math.Max(1, showingModels.Max(x => x.LodInfoArray.Length));
                }

                for (int ii = 0; ii < lodCount; ii++)
                {
                    int i = ii;
                    Predicate<object> isValid = data => i < ((Model)data).LodInfoArray.Length;

                    // 処理頂点数
                    RegisterColumn(
                        ColumnID.ProcessVertexCount,
                        "ProcessVertexCount",
                        string.Format(res.Strings.ListView_Model_Column_ProcessVertexCount, i),
                        HorizontalAlignment.Right,
                        ColumnDefaultWidthM,
                        true,
                        data => ((Model)data).LodInfoArray[i].HasProcessVertexCount ? ((Model)data).LodInfoArray[i].ProcessVertexCount.ToString() : res.Strings.ProcessVertexCount_UnsupportedPlatform,
                        isValid:isValid);
                    // 頂点数
                    RegisterColumn(
                        ColumnID.VertexCount,
                        "VertexCount",
                        string.Format(res.Strings.ListView_Model_Column_VertexCount, i),
                        HorizontalAlignment.Right,
                        ColumnDefaultWidthM,
                        true,
                        data => ((Model)data).LodInfoArray[i].VertexCount,
                        isValid:isValid);
                    // ポリゴン数
                    RegisterColumn(
                        ColumnID.PolygonCount,
                        "PolygonCount",
                        string.Format(res.Strings.ListView_Model_Column_PolygonCount, i),
                        HorizontalAlignment.Right,
                        ColumnDefaultWidthM,
                        true,
                        data => ((Model)data).LodInfoArray[i].PolygonCount,
                        isValid:isValid);
                }

                // シェイプ数
                RegisterColumn(ColumnID.ShapeArray, "ShapeArray", res.Strings.ListView_Model_Column_ShapeArray, HorizontalAlignment.Right, ColumnDefaultWidthS, true,
                    data => (data as Model).Shapes.Count);
                // マテリアル数
                RegisterColumn(ColumnID.MaterialArray, "MaterialArray", res.Strings.ListView_Model_Column_MaterialArray, HorizontalAlignment.Right, ColumnDefaultWidthS, true,
                    data => (data as Model).Materials.Count);
                // スキニング
                RegisterColumn(ColumnID.Skinning, "Skinning", res.Strings.ListView_Model_Column_Skinning, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((data as Model).Data.skeleton != null));
                // ボーン行列のスケール計算方法
                RegisterColumn(ColumnID.ScaleMode, "ScaleMode", res.Strings.ListView_Model_Column_ScaleMode, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((data as Model).Data.skeleton != null) ? UIText.EnumValue((skeleton_info_scale_modeType)(data as Model).Data.skeleton.skeleton_info.scale_mode) : string.Empty);
                // ボーン行列の回転計算方法
                RegisterColumn(ColumnID.RotateMode, "RotateMode", res.Strings.ListView_Model_Column_RotateMode, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((data as Model).Data.skeleton != null) ? UIText.EnumValue((skeleton_info_rotate_modeType)(data as Model).Data.skeleton.skeleton_info.rotate_mode) : string.Empty);
                // ボーン数
                RegisterColumn(ColumnID.BoneCount, "BoneCount", res.Strings.ListView_Model_Column_BoneCount, HorizontalAlignment.Right, ColumnDefaultWidthS, true,
                    data => ((data as Model).Data.skeleton != null) ? (data as Model).Data.skeleton.bone_array.bone.Length : 0);
                // DCC プリセット
                RegisterColumn(ColumnID.DccPreset, "DccPreset", res.Strings.ListView_Model_Column_DccPreset, HorizontalAlignment.Right, ColumnDefaultWidthS, true,
                    data => (data as Model).Data.model_info.dcc_preset);
                // DCC スケール
                RegisterColumn(ColumnID.DccMagnify, "DccMagnify", res.Strings.ListView_Model_Column_DccMagnify, HorizontalAlignment.Right, ColumnDefaultWidthS, true,
                    data => (data as Model).Data.model_info.dcc_magnify);
                // DCC 開始フレーム
                RegisterColumn(ColumnID.DccStartFrame, "DccStartFrame", res.Strings.ListView_Model_Column_DccStartFrame, HorizontalAlignment.Right, ColumnDefaultWidthS, true,
                    data => (data as Model).Data.model_info.dcc_start_frame);
            }
            EndRegisterColumn();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void CreateItem()
        {
            foreach (var model in DocumentManager.OrderedModels.Where(x => x.IsShowInObjView))
            {
                if (IsMatchedToSearchText(model.Name))
                {
                    RegisterItem(model, GuiObjectID.Model, ViewID.Model);
                }
            }
        }

        #region ColumnID
        /// <summary>
        /// 列項目ＩＤ。
        /// </summary>
        public enum ColumnID
        {
            /// <summary>名前。</summary>
            Name,
            /// <summary>処理頂点数。</summary>
            ProcessVertexCount,
            /// <summary>頂点数。</summary>
            VertexCount,
            /// <summary>ポリゴン数。</summary>
            PolygonCount,
            /// <summary>シェイプ数。</summary>
            ShapeArray,
            /// <summary>マテリアル数。</summary>
            MaterialArray,
            /// <summary>スキニング。</summary>
            Skinning,
            /// <summary>ボーン行列のスケール計算方法。</summary>
            ScaleMode,
            /// <summary>ボーン行列の回転計算方法。</summary>
            RotateMode,
            /// <summary>数。</summary>
            BoneCount,
            /// <summary>コメント。</summary>
            Comment,
            /// <summary>DCC プリセット。</summary>
            DccPreset,
            /// <summary>DCC スケール。</summary>
            DccMagnify,
            /// <summary>DCC 開始フレーム。</summary>
            DccStartFrame,
            /// <summary>コメントカラー。</summary>
            CommentColor,
            /// <summary>コメントテキスト。</summary>
            CommentText,
            /// <summary>コメントラベル。</summary>
            CommentLabel,
        }
        #endregion
    }
}
