﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using App.Data;
using App.PropertyEdit;
using App.Utility;
using App.res;
using App.ConfigData;
using App.Controls;
using App.Properties;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.ObjectView.List
{
    using IdLabel = KeyValuePair<string, string>;

    public enum ColumnTypes
    {
        render_info,
        shader_option,
        sampler_assign,
        shader_param,
        attrib_assign,
    }

    /// <summary>
    /// モデルリストビュークラス。
    /// </summary>
    [ToolboxItem(false)]
    public sealed class MaterialListView : StandardObjectListView
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialListView()
            : base(ViewID.Material, AppConfig.ObjectViewMode.MaterialList)
        {
            SetupColumn();
            Model.IsAttachedChanged += (x, y) =>
                {
                    if (IsHandleCreated)
                    {
                        BeginInvoke((Action)(()=>{Invalidate();}));
                    }
                };
            App.AppContext.SpecChanged += () =>
                {
                    SetupColumn();
                    NotifyColumnSettingChanged();
                };
        }

        protected override void Document_AddedOrRemoved(object sender, IEnumerable<Document> opened, IEnumerable<Document> closed, Dictionary<GuiObject, GuiObject> swaped, IEnumerable<Document> reloaded)
        {
            foreach (var material in opened.OfType<Model>().SelectMany(x => x.Materials))
            {
                material.EnableCombinerWatchers();
            }
            foreach (var material in closed.OfType<Model>().SelectMany(x => x.Materials))
            {
                material.DisableCombinerWatchers();
            }

            IEnumerable<Document> documents = opened.Concat(closed).Concat(reloaded);
            if (documents.Any(x => x.ObjectID == GuiObjectID.Model || x.ObjectID == GuiObjectID.ShaderDefinition))
            {
                // 列の更新
                SetupColumn();
                NotifyColumnSettingChanged();
            }
            else
            {
                base.Document_AddedOrRemoved(sender, opened, closed, swaped, reloaded);
            }
        }

        protected override void Document_PropertyChanged(object sender, IEnumerable<DocumentPropertyChangedArgs> e)
        {
            if (e.Any(x => x is App.PropertyEdit.DocumentPropertyChangedShaderArgs ||
                x is App.PropertyEdit.DocumentPropertyChangedSamplerCountArgs ||
                x is App.PropertyEdit.DocumentPropertyChangedEnvRefArgs ||
                x is App.FileView.DocumentPropertyChangedShowInObjViewArgs))
            {
                SetupColumn();
                NotifyColumnSettingChanged();
            }
            else
            {
                base.Document_PropertyChanged(sender, e);
            }
        }

        /// <summary>
        /// 列項目初期化。
        /// </summary>
        public static void InitializeColumn()
        {
            SetupColumn();
        }

        private static Image optimizedEdittedShaderSent = Resources.ListView_OptimizedEdittedShaderSent;
        private static Image optimizedShaderSent = Resources.ListView_OptimizedShaderSent;
        private static Image edittedShaderSent = Resources.ListView_EdittedShaderSent;
        private static Image shaderNotSent = Resources.ListView_ShaderNotSent;
        private static Image modelNotSent = Resources.ListView_Empty;
        //private static Image attachedEdittedShaderSent = Resources.ListView_AttachedEdittedShaderSent;
        private static Image attachedShaderSent = Resources.ListView_AttachedShaderSent;
        private static Image shaderConvertError = Resources.ListView_Error12x12;

        // MainFrame初期化中に呼ばれることが有るので、mainFrameを渡す
        public static void SetupColumn()
        {
            BeginRegisterColumn(ViewID.Material, "MaterialListView");
            {
                // ルートページ
                CurrentPropertyPageID = PropertyPageID.MaterialRoot;

                // シェーダー最適化状態
                {
                    var item = RegisterMaterialColumn(
                        Strings.ListView_Material_Column_ShaderState,
                        material =>
                        {
                            if (material.ShaderSent)
                            {
                                if (material.OptimizeShaderSent)
                                {

                                    if (material.ModifiedShaderSent)
                                    {
                                        return Strings.MaterialPreview_OptimizedEdittedShaderSent;
                                    }
                                    else
                                    {
                                        return Strings.MaterialPreview_OptimizedShaderSent;
                                    }
                                }
                                else
                                {
                                    return Strings.MaterialPreview_EdittedShaderSent;
                                }
                            }
                            else
                            {
                                // アタッチシェーダー
                                var shaderDefinition = material.MaterialShaderAssign.ShaderDefinition;
                                if (shaderDefinition != null)
                                {
                                    if (shaderDefinition.FailedToBinarize)
                                    {
                                        return Strings.MaterialPreview_ShaderConvertError;
                                    }

                                    var shadingModelName = material.MaterialShaderAssign.ShaderName;
                                    if (shaderDefinition.IsAttached &&
                                        shaderDefinition.updatedAttachedShadingModel.Contains(shadingModelName))
                                    {
                                        return Strings.MaterialPreview_AttachedShaderSent;
                                    }
                                }

                                return Strings.MaterialPreview_ShaderNotSent;
                            }
                        },
                        null,
                        null,
                        HorizontalAlignment.Left,
                        16);
                    item.GetSpecificImage = (args, m) =>
                    {
                        var material = (Material)m;
                        if (material.Referrers.All(x => !x.IsAttached))
                        {
                            return modelNotSent;
                        }
                        else if (material.ShaderSent)
                        {
                            if (material.OptimizeShaderSent)
                            {

                                if (material.ModifiedShaderSent)
                                {
                                    return optimizedEdittedShaderSent;
                                }
                                else
                                {
                                    return optimizedShaderSent;
                                }
                            }
                            else
                            {
                                return edittedShaderSent;
                            }
                        }
                        else
                        {
                            // アタッチシェーダー
                            var shaderDefinition = material.MaterialShaderAssign.ShaderDefinition;
                            if (shaderDefinition != null)
                            {
                                if (shaderDefinition.FailedToBinarize)
                                {
                                    return shaderConvertError;
                                }
                                var shadingModelName = material.MaterialShaderAssign.ShaderName;
                                if (shaderDefinition.IsAttached && shaderDefinition.updatedAttachedShadingModel.Contains(shadingModelName))
                                {
                                    return attachedShaderSent;
                                }
                            }

                            return shaderNotSent;
                        }
                    };

                }
                // マテリアル
                {
                    var item = RegisterMaterialColumn(
                        Strings.ListView_Material_Column_Name,
                        x => x.Name,
                        null,
                        null,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthL);

                    item.GetSpecificForeColor = (args, m) =>
                    {
                        var material = m as Material;
                        return (material != null && (!material.TextureValidity() || !material.ParentMaterialsValidity() || !material.CombinerValidity())) ? Color.Red : (Color?) null;
                    };
                }

                RegisterMaterialColumn(
                    Strings.ListView_Material_Model,
                    x => string.Join(", ", x.Referrers.Select(y => y.Name)),
                    null,
                    null,
                    HorizontalAlignment.Left,
                    ColumnDefaultWidthL);

                // コメント
                RegisterCommentColorColumn(ColumnID.CommentColor, "CommentColor", res.Strings.ListView_Object_Column_CommentColor, HorizontalAlignment.Left, 16, true);
                RegisterColumn(ColumnID.CommentText, "CommentText", res.Strings.ListView_Object_Column_CommentText, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((Material)data).Comment ?? string.Empty, ColumnID.CommentText.ToString());
                RegisterColumn(ColumnID.CommentLabel, "CommentLabel", res.Strings.ListView_Object_Column_CommentLabel, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((Material)data).Label ?? string.Empty, ColumnID.CommentLabel.ToString());

                // シェーダーページ
                CurrentPropertyPageID = PropertyPageID.MaterialShader;
                RegisterMaterialColumn(
                    Strings.ListView_Material_Column_ShaderArchive,
                    x => x.MaterialShaderAssign.ShaderDefinitionFileName,
                    null,
                    null,
                    HorizontalAlignment.Left,
                    ColumnDefaultWidthL);
                RegisterMaterialColumn(
                    Strings.ListView_Material_Column_Program,
                    x => x.MaterialShaderAssign.ShaderName,
                    null,
                    null,
                    HorizontalAlignment.Left,
                    ColumnDefaultWidthL);
                RegisterMaterialColumn(
                    Strings.ListView_Material_Column_ShaderRevision,
                    x => x.MaterialShaderAssign.Revision,
                    null,
                    x => !string.IsNullOrEmpty(x.MaterialShaderAssign.ShaderName),
                    HorizontalAlignment.Right,
                    ColumnDefaultWidthL);

                // 全般ページ
                CurrentPropertyPageID = PropertyPageID.MaterialGeneral;
                RegisterMaterialColumn(
                    Strings.ListView_Material_Column_Visibility,
                    x => x.Data.material_info.visibility,
                    UIText.FlagYesNo,
                    null,
                    HorizontalAlignment.Left,
                    ColumnDefaultWidthS);
                RegisterMaterialColumn(
                    Strings.ListView_Material_Column_Compressable,
                    x => x.Data.material_info.compress_enable,
                    UIText.FlagYesNo,
                    null,
                    HorizontalAlignment.Left,
                    ColumnDefaultWidthL);
                RegisterMaterialColumn(
                    Strings.ListView_Material_Column_MeshAdjacency,
                    x => x.Data.material_info.mesh_adjacency,
                    UIText.FlagYesNo,
                    x => !string.IsNullOrEmpty(x.MaterialShaderAssign.ShaderName),
                    HorizontalAlignment.Left,
                    ColumnDefaultWidthS);

                if (App.AppContext.SelectedPlatformPreset.UseNw)
                {
                    RegisterMaterialColumn(
                        Strings.ListView_Material_Column_DisplayFace,
                        x => x.Data.render_state.display_face,
                        UIText.EnumValue,
                        null,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthS);
                }

                // プレビュー
                CurrentPropertyPageID = PropertyPageID.MaterialPreview;
                RegisterMaterialColumn(
                    res.Strings.ListView_Material_Column_OptimizeShader,
                    x => x.OptimizeShader,
                    UIText.FlagYesNo,
                    null,
                    HorizontalAlignment.Left,
                    ColumnDefaultWidthM);
                RegisterMaterialColumn(
                    res.Strings.ListView_Material_Column_OptimizeShaderOnReload,
                    x => x.OptimizeShaderOnReload,
                    UIText.FlagYesNo,
                    null,
                    HorizontalAlignment.Left,
                    ColumnDefaultWidthM);

                // レンダーステート
                if (ApplicationConfig.DefaultValue.RenderStateInfoVisible && App.AppContext.SelectedPlatformPreset.UseNw)
                {
                    CurrentPropertyPageID = PropertyPageID.MaterialRenderState;
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_RenderStateMode,
                        x => x.Data.render_state.mode,
                        UIText.EnumValue,
                        null,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    //Predicate<Material> isCustomRenderState = x => x.Data.render_state.mode == render_state_modeType.custom;
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_DepthTest,
                        x => x.Data.render_state.depth_test.enable,
                        UIText.FlagEnableDisable,
                        null,
                        //isCustomRenderState,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthS);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_DepthTestFunc,
                        x => x.Data.render_state.depth_test.func,
                        UIText.EnumValue,
                        null,
                        //isCustomRenderState,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_DepthTestWrite,
                        x => x.Data.render_state.depth_test.write,
                        UIText.FlagYesNo,
                        null,
                        //isCustomRenderState,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_AlphaTest,
                        x => x.Data.render_state.alpha_test.enable,
                        UIText.FlagEnableDisable,
                        null,
                        //isCustomRenderState,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_AlphaTestFunc,
                        x => x.Data.render_state.alpha_test.func,
                        UIText.EnumValue,
                        null,
                        //isCustomRenderState,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_AlphaTestValue,
                        x => x.Data.render_state.alpha_test.value,
                        null,
                        null,
                        //isCustomRenderState,
                        HorizontalAlignment.Right,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_Blend,
                        x => x.Data.render_state.blend_mode,
                        UIText.EnumValue,
                        null,
                        //isCustomRenderState,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    // Predicate<Material> isValidColorBlend =
                    //	 x => isCustomRenderState(x) &&
                    //		 x.Data.render_state.blend_mode == render_state_blend_modeType.color;
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_BlendRgbOp,
                        x => x.Data.render_state.color_blend.rgb_op,
                        UIText.EnumValue,
                        null,
                        //isValidColorBlend,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_BlendRgbSrcFunc,
                        x => x.Data.render_state.color_blend.rgb_src_func,
                        UIText.EnumValue,
                        null,
                        //isValidColorBlend,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_BlendRgbDstFunc,
                        x => x.Data.render_state.color_blend.rgb_dst_func,
                        UIText.EnumValue,
                        null,
                        //isValidColorBlend,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_BlendAlphaOp,
                        x => x.Data.render_state.color_blend.alpha_op,
                        UIText.EnumValue,
                        null,
                        //isValidColorBlend,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_BlendAlphaSrcFunc,
                        x => x.Data.render_state.color_blend.alpha_src_func,
                        UIText.EnumValue,
                        null,
                        //isValidColorBlend,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_BlendAlphaDstFunc,
                        x => x.Data.render_state.color_blend.alpha_dst_func,
                        UIText.EnumValue,
                        null,
                        //isValidColorBlend,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                    RegisterColorColumn(
                        res.Strings.ListView_Material_Column_BlendColor,
                        x => x.Data.render_state.color_blend.const_color,
                        null,
                        //isValidColorBlend,
                        ColumnDefaultWidthColor4);
                    //Predicate<Material> isValidLogicBlend =
                    //	x => isCustomRenderState(x) &&
                    //		x.Data.render_state.blend_mode == render_state_blend_modeType.logic;
                    RegisterMaterialColumn(
                        res.Strings.ListView_Material_Column_LogicalBlend,
                        x => x.Data.render_state.logical_blend.op,
                        UIText.EnumValue,
                        null,
                        //isValidLogicBlend,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM);
                }
                //表示されるマテリアル
                var showingMaterials = DocumentManager.Materials
                    .Where(x => (x.OwnerDocument == null) || (x.OwnerDocument is SeparateMaterial) || ((x.OwnerDocument is Model) && ((Model)x.OwnerDocument).IsShowInObjView))
                    .ToList();

                // サンプラー
                CurrentPropertyPageID = PropertyPageID.MaterialSampler;
                var samplerCount = 0;

                if (showingMaterials.Any())
                {
                    samplerCount = showingMaterials.Max(x => x.ResolvedSamplers.Count());
                }

                for (int i = 0; i < samplerCount; i++)
                {
                    // 匿名デリゲーのためにスコープを制限
                    int ii = i;
                    Predicate<Material> isValidSampler = x => x.sampler_array != null &&
                        x.ResolvedSamplers.Count() > ii;
                    Func<Material, samplerType> sampler = x => x.ResolvedSamplers.ToArray()[ii];
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerName, i),
                        x => sampler(x).name,
                        null,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerHint, i),
                        x => sampler(x).hint,
                        null,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerTexture, i),
                        x => sampler(x).tex_name,
                        null,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerWrapU, i),
                        x => sampler(x).wrap.u,
                        UIText.EnumValue,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerWrapV, i),
                        x => sampler(x).wrap.v,
                        UIText.EnumValue,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerWrapW, i),
                        x => sampler(x).wrap.w,
                        UIText.EnumValue,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerFilterMag, i),
                        x => sampler(x).filter.mag,
                        UIText.EnumValue,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerFilterMin, i),
                        x => sampler(x).filter.min,
                        UIText.EnumValue,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerFilterMip, i),
                        x => sampler(x).filter.mip,
                        UIText.EnumValue,
                        isValidSampler,
                        HorizontalAlignment.Left,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerFilterMaxAniso, i),
                        x => sampler(x).filter.max_aniso,
                        UIText.EnumValue,
                        isValidSampler,
                        HorizontalAlignment.Right,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerLodMin, i),
                        x => sampler(x).lod.min,
                        null,
                        isValidSampler,
                        HorizontalAlignment.Right,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerLodMax, i),
                        x => sampler(x).lod.max,
                        null,
                        isValidSampler,
                        HorizontalAlignment.Right,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                    RegisterMaterialColumn(
                        string.Format(Strings.ListView_Material_SamplerLodBias, i),
                        x => sampler(x).lod.bias,
                        null,
                        isValidSampler,
                        HorizontalAlignment.Right,
                        ColumnDefaultWidthM,
                        fitWidthToName: true);
                }

                // シェーダーパラメーター
                CurrentPropertyPageID = PropertyPageID.MaterialShader;
                optionParameterSelectors.Clear();
                numericParameterSelectors.Clear();
                samplerParameterSelectors.Clear();
                attributeParameterSelectors.Clear();
                renderInfoParameterSelectors.Clear();
                var labelCounter = new Dictionary<string, int>();
                var registerActions = new List<Action>();
                foreach (var material in showingMaterials)
                {
                    var program = material.MaterialShaderAssign.ShadingModel;
                    //if (program != null)
                    {
                        // オプション
                        foreach (var param0 in material.MaterialShaderAssign.ShaderOptions)
                        {
                            var param = param0;
                            var option = program != null ? program.Options().FirstOrDefault(x => x.id == param.id) : null;

                            if (option != null && !option.OptionVisible())
                            {
                                continue;
                            }

                            IdLabel idLabel = new IdLabel(param.id, option != null ? option.Label() : null);
                            HashSet<Material> parameterSelector;
                            if (!optionParameterSelectors.ContainsKey(idLabel))
                            {
                                parameterSelector = new HashSet<Material>();
                                optionParameterSelectors.Add(idLabel, parameterSelector);
                                var label = idLabel.Value ?? idLabel.Key;
                                if (labelCounter.ContainsKey(label))
                                {
                                    labelCounter[label]++;
                                }
                                else
                                {
                                    labelCounter[label] = 1;
                                }

                                registerActions.Add(() => RegisterShaderOptionColumn(idLabel, parameterSelector, option, param, program, labelCounter[label]));
                            }
                            else
                            {
                                parameterSelector = optionParameterSelectors[idLabel];
                            }
                            parameterSelector.Add(material);
                        }

                        // シェーダーパラメーター
                        foreach (var param0 in material.MaterialShaderAssign.ShaderParams)
                        {
                            var param = param0;
                            var uniform = program != null ? program.MaterialUniforms().FirstOrDefault(x => x.id == param.id) : null;

                            if (uniform != null && !uniform.IsParamVisible())
                            {
                                continue;
                            }

                            var idLabel = new IdLabel(param.id, uniform != null ? uniform.Label() : null);
                            HashSet<Material> parameterSelector;
                            if (!numericParameterSelectors.ContainsKey(idLabel))
                            {
                                parameterSelector = new HashSet<Material>();
                                numericParameterSelectors.Add(idLabel, parameterSelector);

                                var label = idLabel.Value ?? idLabel.Key;
                                if (labelCounter.ContainsKey(label))
                                {
                                    labelCounter[label]++;
                                }
                                else
                                {
                                    labelCounter[label] = 1;
                                }

                                registerActions.Add(() => RegisterShaderParamColumn(idLabel, parameterSelector, param, uniform, program, labelCounter[label]));
                            }
                            else
                            {
                                parameterSelector = numericParameterSelectors[idLabel];
                            }
                            parameterSelector.Add(material);
                        }

                        // サンプラ割り当て
                        foreach (var param0 in material.MaterialShaderAssign.SamplerAssigns)
                        {
                            var param = param0;
                            var sampler = program != null ? program.Samplers().FirstOrDefault(x => x.id == param.id) : null;

                            if (sampler != null && !sampler.SamplerVisible())
                            {
                                continue;
                            }

                            var idLabel = new IdLabel(param.id, sampler != null ? sampler.Label() : null);
                            HashSet<Material> parameterSelector;
                            if (!samplerParameterSelectors.ContainsKey(idLabel))
                            {
                                parameterSelector = new HashSet<Material>();
                                samplerParameterSelectors.Add(idLabel, parameterSelector);
                                var label = idLabel.Value ?? idLabel.Key;
                                if (labelCounter.ContainsKey(label))
                                {
                                    labelCounter[label]++;
                                }
                                else
                                {
                                    labelCounter[label] = 1;
                                }
                                registerActions.Add(() => RegisterShaderSamplerColumn(idLabel, parameterSelector, sampler, program, labelCounter[label]));
                            }
                            else
                            {
                                parameterSelector = samplerParameterSelectors[idLabel];
                            }
                            parameterSelector.Add(material);
                        }

                        // アトリビュート
                        foreach (var param0 in material.MaterialShaderAssign.AttribAssigns)
                        {
                            var param = param0;
                            var attrib_var = program != null ? program.Attributes().FirstOrDefault(x => x.id == param.id) : null;
                            IdLabel idLabel = new IdLabel(param.id, attrib_var != null ? attrib_var.Label() : null);
                            HashSet<Material> parameterSelector;
                            if (!attributeParameterSelectors.ContainsKey(idLabel))
                            {
                                parameterSelector = new HashSet<Material>();
                                attributeParameterSelectors.Add(idLabel, parameterSelector);
                                var label = idLabel.Value ?? idLabel.Key;
                                if (labelCounter.ContainsKey(label))
                                {
                                    labelCounter[label]++;
                                }
                                else
                                {
                                    labelCounter[label] = 1;
                                }
                                registerActions.Add(() => RegisterAttributeAssignColumn(idLabel, parameterSelector, attrib_var, program, labelCounter[label]));
                            }
                            else
                            {
                                parameterSelector = attributeParameterSelectors[idLabel];
                            }
                            parameterSelector.Add(material);
                        }

                        // レンダーインフォ
                        foreach (var param0 in material.MaterialShaderAssign.RenderInfos)
                        {
                            var param = param0;
                            var slot = program != null ? program.RenderInfoSlots().FirstOrDefault(x => x.name == param.name) : null;

                            if (slot != null && !slot.RenderInfoVisible())
                            {
                                continue;
                            }

                            var idLabel = new IdLabel(param.name, slot != null ? slot.Label() : null);
                            HashSet<Material> parameterSelector;
                            if (!renderInfoParameterSelectors.ContainsKey(idLabel))
                            {
                                parameterSelector = new HashSet<Material>();
                                renderInfoParameterSelectors.Add(idLabel, parameterSelector);
                                var label = idLabel.Value ?? idLabel.Key;
                                if (labelCounter.ContainsKey(label))
                                {
                                    labelCounter[label]++;
                                }
                                else
                                {
                                    labelCounter[label] = 1;
                                }
                                registerActions.Add(() => RegisterRenderInfoColumn(idLabel, parameterSelector, slot, program, labelCounter[label]));
                            }
                            else
                            {
                                parameterSelector = renderInfoParameterSelectors[idLabel];
                            }
                            parameterSelector.Add(material);
                        }
                    }
                }

                foreach (var action in registerActions)
                {
                    action();
                }
            }
            EndRegisterColumn();
        }

        public string _filterString = PropertyPageID.MaterialShader.ToString();
        public string FilterCategoryString
        {
            get
            {
                return _filterString;
            }
            set
            {
                if (!string.IsNullOrEmpty(value))
                {
                    _filterString = value;
                }
            }
        }

        protected override void AddColumns(AppConfig.ListView config)
        {
            var root = PropertyPageID.MaterialRoot.ToString();
            var Null = PropertyPageID.Null.ToString();
            foreach (var column in config.Columns.Where(x => x.Show))
            {
                if (FilterCategoryString == Null ||
                    column.Category == FilterCategoryString ||
                    column.Category == root)
                {
                    var item = column.GetColumnItem();
                    if (item != null && item.Visible)
                    {
                        Columns.Add(item);
                    }
                }
            }
        }

        protected override void OnColumnReordered(ColumnReorderedEventArgs e)
        {
            // 順序未変更時は何もしない
            if (e.OldDisplayIndex == e.NewDisplayIndex) { return; }

            // 列情報を更新
            var config = ConfigData.ApplicationConfig.Setting.ObjectView.GetListView(ViewID.Material);
            // TODO:整理
            var rootStr = PropertyPageID.MaterialRoot.ToString();
            var nullStr = PropertyPageID.Null.ToString();
            var visibleColumns = config.Columns
                .Where(x => x.Show && x.GetColumnItem() != null && x.GetColumnItem().Visible)
                .Where(x => FilterCategoryString == nullStr || x.Category == FilterCategoryString || x.Category == rootStr)
                .ToArray();

            var oldPosItem = visibleColumns[e.OldDisplayIndex];
            var newPosItem = visibleColumns[e.NewDisplayIndex];
            var oldPosIndex = config.Columns.IndexOf(oldPosItem);
            var newPosIndex = config.Columns.IndexOf(newPosItem);
            DebugConsole.WriteLine("old: " + oldPosIndex + " new: " + newPosIndex);
            config.Columns.RemoveAt(oldPosIndex);
            config.Columns.Insert(newPosIndex, oldPosItem);
            DebugConsole.WriteLine("column reordered:{0}", string.Concat(config.Columns.Select(x => x.Name + ": ")));
        }



        private static readonly Dictionary<IdLabel, HashSet<Material>> optionParameterSelectors = new Dictionary<IdLabel, HashSet<Material>>();
        private static readonly Dictionary<IdLabel, HashSet<Material>> numericParameterSelectors = new Dictionary<IdLabel, HashSet<Material>>();
        private static readonly Dictionary<IdLabel, HashSet<Material>> samplerParameterSelectors = new Dictionary<IdLabel, HashSet<Material>>();
        private static readonly Dictionary<IdLabel, HashSet<Material>> attributeParameterSelectors = new Dictionary<IdLabel, HashSet<Material>>();
        private static readonly Dictionary<IdLabel, HashSet<Material>> renderInfoParameterSelectors = new Dictionary<IdLabel, HashSet<Material>>();

        private static HorizontalAlignment GetHorizontalAlignment(option_varType option_var, shader_optionType option)
        {
            if (option_var != null)
            {
                if (option_var.choice == "bool" || (option_var.choice.StartsWith("[") && option_var.choice.EndsWith("]")))
                {
                    return HorizontalAlignment.Right;
                }
            }
            else
            {
                int dummy;
                if (int.TryParse(option.value, out dummy))
                {
                    return HorizontalAlignment.Right;
                }
            }

            return HorizontalAlignment.Left;
        }

        private static HorizontalAlignment GetHorizontalAlignment(shader_param_typeType type)
        {
            switch (type)
            {
                case shader_param_typeType.@float:
                case shader_param_typeType.@int:
                case shader_param_typeType.@uint:
                    return HorizontalAlignment.Right;
            }
            return HorizontalAlignment.Left;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void CreateItem()
        {
            foreach (var material in DocumentManager.OrderedModels.Where(x => x.IsShowInObjView).SelectMany(x => x.Materials).Concat(DocumentManager.OrderedSeparateMaterials.SelectMany(x => x.Materials)))
            {
                if (IsMatchedToSearchText(material.Name))
                {
                    var tooltip = material.TextureValidity() ? "" : Strings.MaterialListView_CreateItem_TextureMissing;
                    if (!material.ParentMaterialsValidity())
                    {
                        tooltip += tooltip.Any() ? "\n" : "";
                        tooltip += Strings.MaterialListView_CreateItem_BaseMaterialInvalid;
                    }
                    RegisterItem(material, GuiObjectID.Material, List.ViewID.Material, tooltip);
                }
            }
        }

        /// <summary>
        /// マテリアル用のRegisterColumn
        /// </summary>
        private static ColumnItem RegisterMaterialColumn<T>(
            string name,
            Func<Material, T> memberSelect,
            Func<T, string> specificText,
            Predicate<Material> isValid,
            HorizontalAlignment TextAlign,
            int width,
            string id = null,
            string configName = null,
            bool fitWidthToName = false
            )
        {
            // 名前まで縮める
            if (fitWidthToName)
            {
                width = Math.Min(GraphicsUtility.MeasureString(name) + 10, width);
            }

            var item = new ColumnItem()
            {
                Text = name,
                TextAlign = TextAlign,
                Width = width,
                GetParamObject = x => memberSelect((Material)x),
            };

            if (specificText != null)
            {
                item.CustomDraw = (v, e, x) => e.SpecificText = specificText((T)x);
            }
            if (isValid != null)
            {
                item.IsValid = x => isValid((Material)x);
            }
            RegisterColumn(item, id, configName);
            return item;
        }

        /*
        private static ColumnItem RegisterMatrialNameColumn()
        {
            //string name =
        }*/

        /// <summary>
        /// マテリアル用のRegisterColumn
        /// </summary>
        private static void RegisterColorColumn(
            string name,
            Func<Material, float[]> memberSelect,
            Predicate<Material> isValid,
            int width)
        {
            var item = new ColumnItem()
            {
                Text = name,
                TextAlign = HorizontalAlignment.Left,
                Width = width,
                GetParamObject = x => memberSelect((Material)x),
            };
            {
                item.CustomDraw = (v, e, x) => v.DrawItem_Color(e, (float[])x);
            }
            if (isValid != null)
            {
                item.IsValid = x => isValid((Material)x);
            }
            RegisterColumn(item);
        }

        public static string valueString(shader_param_typeType type, string value)
        {
            int column = 1;
            int row = 1;
            switch (type)
            {
                case shader_param_typeType.@bool:
                case shader_param_typeType.bool2:
                case shader_param_typeType.bool3:
                case shader_param_typeType.bool4:
                    {
                        bool[] paramArray = G3dDataParser.ParseInt2BoolArray(value);
                        StringBuilder builder = new StringBuilder();
                        for (int i = 0; i < paramArray.Length; i++)
                        {
                            if (i > 0)
                            {
                                builder.Append(", ");
                            }
                            builder.Append(UIText.FlagOnOff(paramArray[i]));
                        }
                        return builder.ToString();
                    }
                case shader_param_typeType.@int:
                case shader_param_typeType.int2:
                case shader_param_typeType.int3:
                case shader_param_typeType.int4:
                    {
                        int[] paramArray = G3dDataParser.ParseIntArray(value);
                        StringBuilder builder = new StringBuilder();
                        for (int i = 0; i < paramArray.Length; i++)
                        {
                            if (i > 0)
                            {
                                builder.Append(", ");
                            }
                            builder.Append(paramArray[i].ToString());
                        }
                        return builder.ToString();
                    }
                case shader_param_typeType.@uint:
                case shader_param_typeType.uint2:
                case shader_param_typeType.uint3:
                case shader_param_typeType.uint4:
                    {
                        uint[] paramArray = G3dDataParser.ParseUIntArray(value);
                        StringBuilder builder = new StringBuilder();
                        for (int i = 0; i < paramArray.Length; i++)
                        {
                            if (i > 0)
                            {
                                builder.Append(", ");
                            }
                            builder.Append(paramArray[i].ToString());
                        }
                        return builder.ToString();
                    }

                case shader_param_typeType.float2x2: row = 2; column = 2; goto case shader_param_typeType.@float;
                case shader_param_typeType.float2x3: row = 2; column = 3; goto case shader_param_typeType.@float;
                case shader_param_typeType.float2x4: row = 2; column = 4; goto case shader_param_typeType.@float;
                case shader_param_typeType.float3x2: row = 3; column = 2; goto case shader_param_typeType.@float;
                case shader_param_typeType.float3x3: row = 3; column = 3; goto case shader_param_typeType.@float;
                case shader_param_typeType.float3x4: row = 3; column = 4; goto case shader_param_typeType.@float;
                case shader_param_typeType.float4x2: row = 4; column = 2; goto case shader_param_typeType.@float;
                case shader_param_typeType.float4x3: row = 4; column = 3; goto case shader_param_typeType.@float;
                case shader_param_typeType.float4x4: row = 4; column = 4; goto case shader_param_typeType.@float;
                case shader_param_typeType.@float:
                case shader_param_typeType.float2:
                case shader_param_typeType.float3:
                case shader_param_typeType.float4:
                    {
                        float[] paramArray = G3dDataParser.ParseFloatArray(value);
                        StringBuilder builder = new StringBuilder();
                        for (int i = 0; i < paramArray.Length; i++)
                        {
                            if (i > 0)
                            {
                                builder.Append(", ");
                            }
                            if (row > 1)
                            {
                                if (i % column == 0)
                                {
                                    builder.Append("[");
                                }
                            }
                            builder.Append(paramArray[i].ToString("F3"));
                            if (row > 1)
                            {
                                if ((i + 1) % column == 0)
                                {
                                    builder.Append("]");
                                }
                            }
                        }
                        return builder.ToString();
                    }
                case shader_param_typeType.srt3d:
                    {
                        float[] paramArray = G3dDataParser.ParseFloatArray(value);
                        {
                            Debug.Assert(paramArray.Length == 9);
                            return String.Format("S:{0:F3}, {1:F3}, {2:F3}  R:{3:F3}, {4:F3}, {5:F3}, T:{6:F}, {7:F3}, {8:F4}",
                                paramArray.Cast<object>().ToArray());
                        }
                    }
                case shader_param_typeType.srt2d:
                    {
                        float[] paramArray = G3dDataParser.ParseFloatArray(value);
                        {
                            Debug.Assert(paramArray.Length == 5);
                            return String.Format("S:{0:F3}, {1:F3}  R:{2:F3}  T:{3:F3}, {4:F3}",
                                paramArray.Cast<object>().ToArray());
                        }
                    }
                case shader_param_typeType.texsrt:
                    {
                        float[] paramArray = G3dDataParser.ParseFloatArray(value);
                        {
                            Debug.Assert(paramArray.Length == 6);
                            return String.Format("Mode:{0}, S:{1:F3}, {2:F3}  R:{3:F3}  T:{4:F3}, {5:F3}",
                                Enumerable.Repeat((object)UIText.TextureSrtMode(paramArray[0]), 1).Concat(paramArray.Skip(1).Cast<object>()).ToArray());
                        }
                    }
            }
            return string.Empty;
        }

        private static void RegisterShaderOptionColumn(
            IdLabel idLabel,
            HashSet<Material> parameterSelector,
            option_varType option_var,
            shader_optionType option,
            shading_modelType shadingModel,
            int count)
        {
            var item = new ColumnItem()
            {
                Text = idLabel.Value ?? idLabel.Key,
                GetParamObject = x => x,
                TextAlign = GetHorizontalAlignment(option_var, option),
                IsValid = x => parameterSelector.Contains((Material)x),
                Type = ColumnTypes.shader_option.ToString(),
            };

            if (count > 1 && shadingModel != null && option_var != null && option_var.ui_group != null)
            {
                var group = shadingModel.Groups().FirstOrDefault(x => x.name == option_var.Group());
                if (group != null)
                {
                    item.Text += string.Format("({0})", group.Label() ?? group.name);
                }
            }

            item.Width = GraphicsUtility.MeasureString(item.Text)+10;
            item.CustomDraw = (v, e, x) =>
                {
                    var p = ((Material)x).MaterialShaderAssign.ShaderOptions.FirstOrDefault(y => y.id == idLabel.Key);
                    if (p == null)
                    {
                        return;
                    }

                    string text = p.value;
                    if (!string.IsNullOrEmpty(text))
                    {
                        e.SpecificText = text;
                    }
                    else
                    {
                        v.DrawItem_Invalid(e);
                    }
                };
            item.CustomComparer = (o1, o2) =>
                {
                    var m1 = (Material)o1;
                    var m2 = (Material)o2;
                    var p1 = m1.MaterialShaderAssign.ShaderOptions.FirstOrDefault(x => x.id == idLabel.Key);
                    var p2 = m2.MaterialShaderAssign.ShaderOptions.FirstOrDefault(x => x.id == idLabel.Key);
                    if (p1 == p2)
                    {
                        return 0;
                    }
                    if (p1 == null)
                    {
                        return -1;
                    }
                    if (p2 == null)
                    {
                        return 1;
                    }
                    int v1;
                    bool isInt1 = int.TryParse(p1.value, out v1);
                    int v2;
                    bool isInt2 = int.TryParse(p2.value, out v2);
                    if (isInt1 != isInt2)
                    {
                        return isInt1.CompareTo(isInt2);
                    }
                    if (v1 != v2)
                    {
                        return v1.CompareTo(v2);
                    }
                    return p1.value.CompareTo(p2.value);
                };
            RegisterColumn(item, idLabel.Key, idLabel.Value, type:item.Type);
        }

        private static void RegisterShaderParamColumn(
            IdLabel idLabel,
            HashSet<Material> parameterSelector,
            shader_paramType param,
            uniform_varType uniform,
            shading_modelType shadingModel,
            int count)
        {
            var item = new ColumnItem()
            {
                Text = idLabel.Value ?? idLabel.Key,
                GetParamObject = x => x,
                TextAlign = GetHorizontalAlignment(param.type),
                IsValid = x => parameterSelector.Contains((Material)x),
                Type = ColumnTypes.shader_param.ToString(),
            };
            if (count > 1 && shadingModel != null && uniform != null && uniform.ui_group != null)
            {
                var group = shadingModel.Groups().FirstOrDefault(x => x.name == uniform.Group());
                if (group != null)
                {
                    item.Text += string.Format("({0})", group.Label() ?? group.name);
                }
            }

            int width = GraphicsUtility.MeasureString(item.Text) + 10;
            if (param != null)
            {
                int dim = ShaderTypeUtility.Dimension(param.type);
                var primitiveType = ShaderTypeUtility.ParamPrimitiveTypeFromType(param.type);
                int itemWidth = 35;
                switch (primitiveType)
                {
                    case ShaderTypeUtility.ParamPrimitiveTypeKind.@bool:
                        itemWidth = 25;
                        break;
                    case ShaderTypeUtility.ParamPrimitiveTypeKind.@int:
                    case ShaderTypeUtility.ParamPrimitiveTypeKind.@uint:
                        itemWidth = 30;
                        break;
                }
                width = Math.Max(width, itemWidth * Math.Min(dim, 4));
            }
            else
            {
                width = Math.Max(ColumnDefaultWidthM, width);
            }
            item.Width = width;

            item.CustomDraw = (v, e, x) =>
                {
                    var p = ((Material)x).MaterialShaderAssign.ShaderParams.FirstOrDefault(y => y.id == idLabel.Key);
                    if (p == null)
                    {
                        return;
                    }
                    string auxiliary = string.IsNullOrEmpty(p.depend) ? null : string.Format(" (depend: {0})", p.depend);
                    switch (p.type)
                    {
                        case shader_param_typeType.float3:
                        case shader_param_typeType.float4:
                            {
                                float[] paramArray = G3dDataParser.ParseFloatArray(p.Value);
                                var definition = ((Material)x).MaterialShaderAssign.ShadingModel;
                                if (definition != null && definition.block_var_array != null && definition.block_var_array.block_var != null)
                                {
                                    var shaderParam = definition.MaterialUniforms().FirstOrDefault(y => y.id == idLabel.Key && y.type == p.type);
                                    if (shaderParam != null && shaderParam.Item() == ui_item_valueType.color)
                                    {
                                        v.DrawItem_Color(e, paramArray, auxiliary);
                                        return;
                                    }
                                }
                            }
                            break;
                    }

                    string text = valueString(p.type, p.Value);
                    if (!string.IsNullOrEmpty(text))
                    {
                        if (auxiliary != null)
                        {
                            text += auxiliary;
                        }

                        e.SpecificText = text;
                    }
                    else
                    {
                        v.DrawItem_Invalid(e);
                    }
                };
            item.CustomComparer = (o1, o2) =>
            {
                var m1 = (Material)o1;
                var m2 = (Material)o2;
                var p1 = m1.MaterialShaderAssign.ShaderParams.FirstOrDefault(x => x.id == idLabel.Key);
                var p2 = m2.MaterialShaderAssign.ShaderParams.FirstOrDefault(x => x.id == idLabel.Key);
                if (p1 == p2)
                {
                    return 0;
                }
                if (p1 == null)
                {
                    return -1;
                }
                if (p2 == null)
                {
                    return 1;
                }
                if (p1.type != p2.type)
                {
                    return p1.type.CompareTo(p2.type);
                }
                float[] a1 = G3dDataParser.ParseFloatArray(p1.Value);
                float[] a2 = G3dDataParser.ParseFloatArray(p2.Value);
                for (int i = 0; i < Math.Min(a1.Length, a2.Length); i++)
                {
                    if (a1[i] != a2[i])
                    {
                        return a1[i].CompareTo(a2[i]);
                    }
                }
                return 0;
            };
            RegisterColumn(item, idLabel.Key, idLabel.Value, type:item.Type);
        }

        private static void RegisterShaderSamplerColumn(
            IdLabel idLabel,
            HashSet<Material> parameterSelector,
            sampler_varType sampler_var,
            shading_modelType shadingModel,
            int count)
        {
            var item = new ColumnItem()
            {
                Text = idLabel.Value ?? idLabel.Key,
                GetParamObject = x => x,
                TextAlign = HorizontalAlignment.Left,
                IsValid = x => parameterSelector.Contains((Material)x),
                Type = ColumnTypes.sampler_assign.ToString(),
            };

            if (count > 1 && shadingModel != null && sampler_var != null && sampler_var.ui_group != null)
            {
                var group = shadingModel.Groups().FirstOrDefault(x => x.name == sampler_var.Group());
                if (group != null)
                {
                    item.Text += string.Format("({0})", group.Label() ?? group.name);
                }
            }

            item.Width = GraphicsUtility.MeasureString(item.Text) + 10;
            item.CustomDraw = (v, e, x) =>
            {
                var p = ((Material)x).MaterialShaderAssign.SamplerAssigns.FirstOrDefault(y => y.id == idLabel.Key);
                if (p == null)
                {
                    return;
                }
                e.SpecificText = p.sampler_name;
            };

            item.GetSpecificForeColor = (args, m) =>
            {
                var material = m as Material;

                foreach(var samplerAssign in material.MaterialShaderAssign.SamplerAssigns.Where(x => (string.IsNullOrEmpty(x.sampler_name) == false) && (x.id == idLabel.Key)))
                {
                    var sampler = material.ResolvedSamplers.FirstOrDefault(x => x.name == samplerAssign.sampler_name);
                    var texture = sampler != null ? DocumentManager.Textures.FirstOrDefault(x => x.Name == sampler.tex_name) : null;

                    if (texture == null)
                    {
                        return Color.Red;
                    }
                }

                return null;
            };

            item.CustomComparer = (o1, o2) =>
            {
                var m1 = (Material)o1;
                var m2 = (Material)o2;
                var p1 = m1.MaterialShaderAssign.SamplerAssigns.FirstOrDefault(x => x.id == idLabel.Key);
                var p2 = m2.MaterialShaderAssign.SamplerAssigns.FirstOrDefault(x => x.id == idLabel.Key);
                if (p1 == p2)
                {
                    return 0;
                }
                if (p1 == null)
                {
                    return -1;
                }
                if (p2 == null)
                {
                    return 1;
                }
                string str1 = p1.sampler_name;
                string str2 = p2.sampler_name;
                return str1.CompareTo(str2);
            };
            RegisterColumn(item, idLabel.Key, idLabel.Value, type:item.Type);
        }

        private static void RegisterAttributeAssignColumn(
            IdLabel idLabel,
            HashSet<Material> parameterSelector,
            attrib_varType attrib,
            shading_modelType shadingModel,
            int count)
        {
            var item = new ColumnItem()
            {
                Text = idLabel.Value ?? idLabel.Key,
                GetParamObject = x => x,
                TextAlign = HorizontalAlignment.Left,
                IsValid = x => parameterSelector.Contains((Material)x),
                Type = ColumnTypes.attrib_assign.ToString(),
            };

            if (count > 1 && shadingModel != null && attrib != null && attrib.ui_group != null)
            {
                var group = shadingModel.Groups().FirstOrDefault(x => x.name == attrib.Group());
                if (group != null)
                {
                    item.Text += string.Format("({0})", group.Label() ?? group.name);
                }
            }
            else if (count > 1)
            {
                item.Text += string.Format("({0})", ShaderTypeUtility.AttribAssignGroupLabel());
            }

            item.Width = GraphicsUtility.MeasureString(item.Text) + 10;
            item.CustomDraw = (v, e, x) =>
            {
                var p = ((Material)x).MaterialShaderAssign.AttribAssigns.FirstOrDefault(y => y.id == idLabel.Key);
                if (p == null)
                {
                    return;
                }
                e.SpecificText = p.attrib_name;

            };
            item.CustomComparer = (o1, o2) =>
            {
                var m1 = (Material)o1;
                var m2 = (Material)o2;
                var p1 = m1.MaterialShaderAssign.AttribAssigns.FirstOrDefault(x => x.id == idLabel.Key);
                var p2 = m2.MaterialShaderAssign.AttribAssigns.FirstOrDefault(x => x.id == idLabel.Key);
                if (p1 == p2)
                {
                    return 0;
                }
                if (p1 == null)
                {
                    return -1;
                }
                if (p2 == null)
                {
                    return 1;
                }
                string str1 = p1.attrib_name;
                string str2 = p2.attrib_name;
                return str1.CompareTo(str2);
            };
            RegisterColumn(item, idLabel.Key, idLabel.Value, type:item.Type);
        }

        private static void RegisterRenderInfoColumn(
            IdLabel idLabel,
            HashSet<Material> parameterSelector,
            render_info_slotType slot,
            shading_modelType shadingModel,
            int count)
        {
            var item = new ColumnItem()
            {
                Text = idLabel.Value ?? idLabel.Key,
                GetParamObject = x => x,
                Width = ColumnDefaultWidthL,
                TextAlign = HorizontalAlignment.Left,
                IsValid = x => parameterSelector.Contains((Material)x),
                Type = ColumnTypes.render_info.ToString(),
            };

            if (count > 1 && shadingModel != null && slot != null && slot.ui_group != null)
            {
                var group = shadingModel.Groups().FirstOrDefault(x => x.name == slot.Group());
                if (group != null)
                {
                    item.Text += string.Format("({0})", group.Label() ?? group.name);
                }
            }

            int width = GraphicsUtility.MeasureString(item.Text) + 10;
            if (slot != null)
            {
                int itemWidth = 40;
                if (slot.type == render_info_slot_typeType.@int)
                {
                    itemWidth = 30;
                }
                width = Math.Max(Math.Min(slot.count, 4) * itemWidth, width);
                if (slot.type == render_info_slot_typeType.@string)
                {
                    width = Math.Max(ColumnDefaultWidthS, width);
                }
            }
            else
            {
                width = Math.Max(ColumnDefaultWidthS, width);
            }

            item.Width = width;
            item.CustomDraw = (v, e, x) =>
            {
                var p = ((Material)x).MaterialShaderAssign.RenderInfos.FirstOrDefault(y => y.name == idLabel.Key);
                if (p == null)
                {
                    return;
                }

                e.SpecificText = ConcatRenderInfo(p.values);
            };
            item.CustomComparer = (o1, o2) =>
            {
                var m1 = (Material)o1;
                var m2 = (Material)o2;
                var p1 = m1.MaterialShaderAssign.RenderInfos.FirstOrDefault(x => x.name == idLabel.Key);
                var p2 = m2.MaterialShaderAssign.RenderInfos.FirstOrDefault(x => x.name == idLabel.Key);
                if (p1 == p2)
                {
                    return 0;
                }
                if (p1 == null)
                {
                    return -1;
                }
                if (p2 == null)
                {
                    return 1;
                }
                string str1 = ConcatRenderInfo(p1.values);
                string str2 = ConcatRenderInfo(p2.values);
                return str1.CompareTo(str2);
            };
            RegisterColumn(item, idLabel.Key, idLabel.Value, type:item.Type);
        }

        private static string ConcatRenderInfo(List<string> values)
        {
            var builder = new StringBuilder();
            foreach (var value in values.Where(x => !string.IsNullOrEmpty(x)))
            {
                if (builder.Length > 0)
                {
                    builder.Append(", ");
                }
                builder.Append(value);
            }
            return builder.ToString();
        }
        #region ColumnID
        /// <summary>
        /// 列項目ＩＤ。
        /// </summary>
        public enum ColumnID
        {
            /// <summary>コメント。</summary>
            Comment = 0,
            /// <summary>コメントカラー。</summary>
            CommentColor,
            /// <summary>コメントテキスト。</summary>
            CommentText,
            /// <summary>コメントラベル。</summary>
            CommentLabel,
        }
        #endregion
    }
}
