﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;
using App.Controls;
using App.Data;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;

namespace App.ObjectView.List
{
    /// <summary>
    /// モデルリストビュークラス。
    /// </summary>
    [ToolboxItem(false)]
    public sealed class BoneListView : StandardObjectListView
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public BoneListView()
            : base(ViewID.Bone, AppConfig.ObjectViewMode.BoneList)
        {
        }

        /// <summary>
        /// 列項目初期化。
        /// </summary>
        public static void InitializeColumn()
        {
            BeginRegisterColumn(ViewID.Bone, "BoneListView");
            {
                CurrentPropertyPageID = PropertyEdit.PropertyPageID.BoneRoot;
                // ColumnIDで宣言している順番で登録します。
                // ボーンインデックス
                RegisterColumn(ColumnID.BoneIndex, "BoneIndex", res.Strings.ListView_Bone_Column_BoneIndex, HorizontalAlignment.Right, ColumnDefaultWidthNumber, true,
                    data => (data as Bone).Data.index);
                // ボーン名
                RegisterColumn(ColumnID.Name, "Name", res.Strings.ListView_Bone_Column_Name, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => (data as Bone).Data.name);
                // 親ボーン
                RegisterColumn(ColumnID.ParentName, "ParentName", res.Strings.ListView_Bone_Column_ParentName, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => (data as Bone).Data.parent_name);
                // オーナーモデル
                RegisterColumn(ColumnID.Owner, "Owner", res.Strings.ListView_Bone_Column_Owner, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => (data as Bone).Owner.Name);
                // コメント
                RegisterCommentColorColumn(ColumnID.CommentColor, "CommentColor", res.Strings.ListView_Object_Column_CommentColor, HorizontalAlignment.Left, 16, true);
                RegisterColumn(ColumnID.CommentText, "CommentText", res.Strings.ListView_Object_Column_CommentText, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((Bone)data).Comment ?? string.Empty, ColumnID.CommentText.ToString());
                RegisterColumn(ColumnID.CommentLabel, "CommentLabel", res.Strings.ListView_Object_Column_CommentLabel, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => ((Bone)data).Label ?? string.Empty, ColumnID.CommentLabel.ToString());
                // 描画に利用
                RegisterColumn(ColumnID.RigidBody, "RigidBody", res.Strings.ListView_Bone_Column_RigidBody, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => (data as Bone).Data.rigid_body);
                // 行列のリストにおけるインデックス
                RegisterColumn(ColumnID.MatrixIndex0, "MatrixIndex0", res.Strings.ListView_Bone_Column_MatrixIndex0, HorizontalAlignment.Right, ColumnDefaultWidthNumber, true,
                    data => (data as Bone).Data.matrix_index[0]);
                // 行列のリストにおけるインデックス
                RegisterColumn(ColumnID.MatrixIndex1, "MatrixIndex1", res.Strings.ListView_Bone_Column_MatrixIndex1, HorizontalAlignment.Right, ColumnDefaultWidthNumber, true,
                    data => (data as Bone).Data.matrix_index[1]);
                // スケール値
                RegisterColumn(ColumnID.Scale, "Scale", res.Strings.ListView_Bone_Column_Scale, HorizontalAlignment.Right, ColumnDefaultWidthColor3, true,
                    data => (data as Bone).Data.scale);
                // 回転値
                RegisterColumn(ColumnID.Rotate, "Rotate", res.Strings.ListView_Bone_Column_Rotate, HorizontalAlignment.Right, ColumnDefaultWidthColor4 + ColumnDefaultWidthS, true,
                    data => (data as Bone).Data.rotate);
                // 移動値
                RegisterColumn(ColumnID.Translate, "Translate", res.Strings.ListView_Bone_Column_Translate, HorizontalAlignment.Right, ColumnDefaultWidthColor3, true,
                    data => (data as Bone).Data.translate);
                // Maya のセグメントスケール補正
                RegisterColumn(ColumnID.ScaleCompensate, "ScaleCompensate", res.Strings.ListView_Bone_Column_ScaleCompensate, HorizontalAlignment.Right, ColumnDefaultWidthS, true,
                    data => (data as Bone).Data.scale_compensate);
                // ビルボード
                RegisterColumn(ColumnID.Billboard, "Billboard", res.Strings.ListView_Bone_Column_Billboard, HorizontalAlignment.Left, ColumnDefaultWidthL, true,
                    data => ( UIText.EnumValue( (bone_billboardType)(data as Bone).Data.billboard) ) );
                // ボーンに関連付けられた renderset の可視状態
                RegisterColumn(ColumnID.Visibility, "Visibility", res.Strings.ListView_Bone_Column_Visibility, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => (data as Bone).Data.visibility);
                // 圧縮許可
                RegisterColumn(ColumnID.CompressEnable, "CompressEnable", res.Strings.ListView_Bone_Column_CompressEnable, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => (data as Bone).Data.compress_enable);
                // ボーンサイド
                RegisterColumn(ColumnID.BoneSide, "BoneSide", res.Strings.ListView_Bone_Column_BoneSide, HorizontalAlignment.Left, ColumnDefaultWidthS, true,
                    data => (data as Bone).Data.side);
            }
            EndRegisterColumn();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void CreateItem()
        {
            foreach (var bone in DocumentManager.OrderedModels.Where(x => x.IsShowInObjView).SelectMany(x => x.Bones))
            {
                if (IsMatchedToSearchText(bone.Name))
                {
                    RegisterItem(bone, GuiObjectID.Bone, ViewID.Bone);
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnCustomDrawItem(CustomDrawListViewItemEventArgs e)
        {
            Bone o = (Bone)e.Item.Tag;
            ColumnItem item = (ColumnItem)e.ColumnHeader;

            // 特別な処理が必要な項目だけ、Case文を書いて処理する。
            // 文字列、数値、bool等、通常の場合は、defaultで処理します。
            switch ((ColumnID)item.Tag)
            {
                case ColumnID.Visibility:
                    DefaultCustomDrawItem(e, item.GetParamObject(o), BoolValueTextType.VisibleNonvisible);
                    break;
                case ColumnID.CompressEnable:
                    DefaultCustomDrawItem(e, item.GetParamObject(o), BoolValueTextType.EnableDisable);
                    break;
                default:
                    DefaultCustomDrawItem(e, item.GetParamObject(o));
                    break;
            }

            base.OnCustomDrawItem(e);
        }

        #region ColumnID
        /// <summary>
        /// 列項目ＩＤ。
        /// </summary>
        public enum ColumnID
        {
            /// <summary>インデックス。</summary>
            BoneIndex,
            /// <summary>名前。</summary>
            Name,
            /// <summary>親ボーン。</summary>
            ParentName,
            /// <summary>オーナーモデル。</summary>
            Owner,
            /// <summary>描画に利用。</summary>
            RigidBody,
            /// <summary>スムーススキニング用のインデックス。</summary>
            MatrixIndex0,
            /// <summary>リジッドスキニング用のインデックス。</summary>
            MatrixIndex1,
            /// <summary>スケール値。</summary>
            Scale,
            /// <summary>回転値。</summary>
            Rotate,
            /// <summary>移動値。</summary>
            Translate,
            /// <summary>Maya のセグメントスケール補正。</summary>
            ScaleCompensate,
            /// <summary>ビルボード。</summary>
            Billboard,
            /// <summary>ボーンに関連付けられた renderset の可視状態。</summary>
            Visibility,
            /// <summary>圧縮許可。</summary>
            CompressEnable,
            /// <summary>ボーンサイド。</summary>
            BoneSide,
            /// <summary>コメント。</summary>
            Comment,
            /// <summary>コメントカラー。</summary>
            CommentColor,
            /// <summary>コメントテキスト。</summary>
            CommentText,
            /// <summary>コメントラベル。</summary>
            CommentLabel,
        }
        #endregion
    }
}
