﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using App.Controls;

namespace App
{
    #region MenuCommandDesc
    /// <summary>
    /// メニューコマンド記述子クラス。
    /// </summary>
    public sealed class MenuCommandDesc<TCommandID> where TCommandID : struct
    {
        // コマンドＩＤ
        private readonly TCommandID _commandID;
        // コマンドハンドラ
        private readonly MenuCommandHandler _commandHandler;
        // コマンドデータ
        // メニューテキストＩＤ
        // ツールチップテキストＩＤ
        // メッセージＩＤ
        // イメージ

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MenuCommandDesc(TCommandID commandID, MenuCommandHandler commandHandler)
        {
            Image = null;
            Message = null;
            ToolTipText = null;
            MenuText = null;
            CommandData = null;
            _commandID      = commandID;
            _commandHandler = commandHandler;
        }

        /// <summary>
        /// コマンドＩＤ。
        /// </summary>
        public TCommandID CommandID
        {
            get { return _commandID; }
        }

        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        public MenuCommandHandler CommandHandler
        {
            get { return _commandHandler; }
        }

        /// <summary>
        /// コマンドデータ。
        /// </summary>
        public object CommandData { get; set; }

        /// <summary>
        /// メニューテキストＩＤ。
        /// </summary>
        public string MenuText { get; set; }

        /// <summary>
        /// ツールチップテキストＩＤ。
        /// </summary>
        public string ToolTipText { get; set; }

        /// <summary>
        /// メッセージＩＤ。
        /// </summary>
        public string Message { get; set; }

        /// <summary>
        /// イメージ。
        /// </summary>
        public Image Image { get; set; }
    }
    #endregion

    #region MenuCommandUI
    /// <summary>
    /// メニューコマンドＵＩクラス。
    /// </summary>
    public sealed class MenuCommandUI
    {
        // 対象項目
        private readonly ToolStripItem _targetItem;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MenuCommandUI(ToolStripItem targetItem)
        {
            _targetItem = targetItem;
        }

        /// <summary>
        /// 対象項目。
        /// </summary>
        public ToolStripItem TargetItem
        {
            get { return _targetItem; }
        }

        /// <summary>
        /// 有効状態。
        /// </summary>
        public bool Enabled
        {
            get {	return _targetItem.Enabled;		}
            set {	_targetItem.Enabled = value;	}
        }

        /// <summary>
        /// チェック状態。
        /// </summary>
        public bool Checked
        {
            get
            {
                // メニュー項目
                ToolStripMenuItem menuItem = _targetItem as ToolStripMenuItem;
                if (menuItem != null)
                {
                    return menuItem.Checked;
                }
                // ボタン
                ToolStripButton button = _targetItem as ToolStripButton;
                if (button != null)
                {
                    return button.Checked;
                }
                // その他
                return false;
            }
            set
            {
                // メニュー項目
                ToolStripMenuItem menuItem = _targetItem as ToolStripMenuItem;
                if (menuItem != null)
                {
                    menuItem.Checked = value;
                    return;
                }
                // ボタン
                ToolStripButton button = _targetItem as ToolStripButton;
                if (button != null)
                {
                    button.Checked = value;
                    return;
                }
                // その他は何もしない
            }
        }

        public bool IsObjectViewContextMenu
        {
            get
            {
                ToolStripMenuItem menuItem = _targetItem as ToolStripMenuItem;
                if (menuItem != null)
                {
                    var contextMenuStrip = menuItem.GetCurrentParent() as UIContextMenuStrip;
                    if (contextMenuStrip != null)
                    {
                        return contextMenuStrip.IsObjectViewContextMenu;
                    }
                }

                return false;
            }
        }
    }
    #endregion

    #region MenuCommandArgs
    /// <summary>
    /// メニューコマンド引数クラス。
    /// </summary>
    public sealed class MenuCommandArgs
    {
        // 更新要求フラグ
        private readonly bool _requireUpdate;
        // コマンドＵＩ
        private readonly MenuCommandUI _commandUI;
        // コマンドデータ 暫定もしかしたら必要か
        private readonly object _commandData;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MenuCommandArgs(bool requireUpdate, MenuCommandUI commandUI, object commandData = null)
        {
            _requireUpdate = requireUpdate;
            _commandUI = commandUI;
            _commandData = commandData;
        }

        /// <summary>
        /// 更新要求フラグ。
        /// </summary>
        public bool RequireUpdate
        {
            get { return _requireUpdate; }
        }

        /// <summary>
        /// 実行要求フラグ。
        /// </summary>
        public bool RequireExecute
        {
            get { return !_requireUpdate; }
        }

        /// <summary>
        /// コマンドＵＩ。
        /// </summary>
        public MenuCommandUI CommandUI
        {
            get { return _commandUI; }
        }

        /// <summary>
        /// コマンドデータ。
        /// </summary>
        public object CommandData
        {
            get { return _commandData; }
        }
    }
    #endregion

    #region MenuCommandHost

    /// <summary>
    /// メニューコマンドホストクラス。
    /// </summary>
    public sealed class MenuCommandHost : IDisposable
    {
        // コマンドイメージリスト
        private readonly ImageList commandImageList_ = new ImageList()
        {
            ImageSize = new Size(16, 16),
            ColorDepth = ColorDepth.Depth24Bit,
            TransparentColor = Color.Magenta
        };

        // メッセージハンドラ
        private MenuCommandMessageHandler messageHandler_ = null;
        // アイドル時更新項目リスト
        private readonly List<ToolStripItem> idleUpdateItems_ = new List<ToolStripItem>();
        // メッセージ更新中フラグ
        private bool isMessageUpdating_ = false;
        // ドロップダウンレベル
        private int dropDownLevel_ = 0;
        // コンテキストメニュー
        private readonly UIContextMenuStrip contextMenuStrip_ = new UIContextMenuStrip();

        // 関連動作
        private readonly Dictionary<ToolStripItem, UIToolStripMenuItem> bindTable_ = new Dictionary<ToolStripItem, UIToolStripMenuItem>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MenuCommandHost()
        {
            contextMenuStrip_.Opening += Event_ContextMenu_Opening;
            contextMenuStrip_.Closed += Event_ContextMenu_Closed;
        }

        public void Dispose()
        {
            commandImageList_.Dispose();
            contextMenuStrip_.Dispose();
        }

        /// <summary>
        /// メッセージハンドラ。
        /// </summary>
        public MenuCommandMessageHandler MessageHandler
        {
            get { return messageHandler_; }
            set { messageHandler_ = value; }
        }

        /// <summary>
        /// コマンド発行。
        /// </summary>
        public void InvokeCommand(ToolStripItem item)
        {
            // 実行ハンドラ呼び出し
            CallCommandHandler(item);
        }

        /// <summary>
        /// アイドル時更新処理。
        /// </summary>
        public void OnIdleUpdate()
        {
            // ドロップダウン中じゃなければ
            if (dropDownLevel_ == 0)
            {
                // 更新ハンドラ呼び出し
                foreach (ToolStripItem item in idleUpdateItems_)
                {
                    CallUpdateHandler(item);
                }

                // コマンドメッセージクリア
                if (messageHandler_ != null && !isMessageUpdating_)
                {
                    messageHandler_(string.Empty);
                }
            }
        }

        /// <summary>
        /// 汎用コンテキストメニュー作成。
        /// </summary>
        public UIContextMenuStrip CreateGeneralContextMenu()
        {
            contextMenuStrip_.Items.Clear();
            contextMenuStrip_.IsObjectViewContextMenu = false;
            return contextMenuStrip_;
        }

        /// <summary>
        /// 汎用セパレータ作成。
        /// </summary>
        public UIToolStripSeparator CreateGeneralSeparator()
        {
            UIToolStripSeparator separator = new UIToolStripSeparator();
            separator.MouseEnter += Event_ToolStripItem_MouseEnter;
            return separator;
        }

        //---------------------------------------------------------------------
        // セットアップ
        //---------------------------------------------------------------------
        /// <summary>
        /// セットアップ処理。
        /// </summary>
        public void Setup(ToolStrip toolStrip)
        {
            // マウスイベント
            toolStrip.MouseEnter += Event_ToolStrip_MouseEnter;
            toolStrip.MouseLeave += Event_ToolStrip_MouseLeave;

            // 項目をセットアップ
            foreach (ToolStripItem item in toolStrip.Items)
            {
                SetupItem(item);
            }

            // ツールバーなら
            if (toolStrip is UIToolStrip)
            {
                // 最上位項目をアイドル時更新対象にする
                foreach (ToolStripItem item in toolStrip.Items)
                {
                    if (!idleUpdateItems_.Contains(item))
                    {
                        idleUpdateItems_.Add(item);
                    }
                }

                // 無効項目の選択タイミング取得用
                ((UIToolStrip)toolStrip).BeforeMouseMove += Event_UIToolStrip_BeforeMouseMove;
            }
        }

        /// <summary>
        /// セットアップ処理。
        /// </summary>
        public void SetupItem(ToolStripItem item)
        {
            // 選択時イベント（全項目に設定）
            item.MouseEnter += Event_ToolStripItem_MouseEnter;

            // ドロップダウン項目
            ToolStripDropDownItem dropDownItem = item as ToolStripDropDownItem;
            if (dropDownItem != null)
            {
                // 子項目なし
                if (dropDownItem.DropDownItems.Count == 0)
                {
                    // クリック時イベント
                    dropDownItem.Click += Event_ToolStripItem_Click;

                    // ペイントイベント（無効状態の選択タイミング取得用）
                    dropDownItem.Paint += Event_DropDownItem_Paint;

                    // ショートカット処理前イベント
                    UIToolStripMenuItem menuItem = dropDownItem as UIToolStripMenuItem;
                    if (menuItem != null && menuItem.ShortcutKeys != Keys.None)
                    {
                        menuItem.BeforeShortcutKeyProcessing += Event_ToolStripMenuItem_BeforeShortcutKeyProcessing;
                    }
                }
                // 子項目あり
                else
                {
                    // ドロップダウン時イベント
                    dropDownItem.DropDownOpening += Event_ToolStripDropDownItem_DropDownOpening;
                    dropDownItem.DropDownClosed  += Event_ToolStripDropDownItem_DropDownClosed;

                    // 子項目の設定
                    foreach (ToolStripItem subItem in dropDownItem.DropDownItems)
                    {
                        SetupItem(subItem);
                    }
                }
                return;
            }

            // ボタン項目
            ToolStripButton button = item as ToolStripButton;
            if (button != null)
            {
                // クリック時イベント
                button.Click += Event_ToolStripItem_Click;
                return;
            }
        }

        //---------------------------------------------------------------------
        // コマンド記述子
        //---------------------------------------------------------------------
        public void BindCommandItem(UIToolStripMenuItem menuItem, params ToolStripItem[] commandItems)
        {
            foreach (var item in commandItems)
            {
                bindTable_[item] = menuItem;
            }
        }

        //---------------------------------------------------------------------
        // ハンドラ呼び出し
        //---------------------------------------------------------------------
        /// <summary>
        /// 更新ハンドラ呼び出し。
        /// </summary>
        private void CallUpdateHandler(ToolStripItem item)
        {
            UIToolStripMenuItem menuItem = item as UIToolStripMenuItem;
            if (menuItem == null && bindTable_.ContainsKey(item))
            {
                menuItem = bindTable_[item];
            }

            if (menuItem != null)
            {
                MenuCommandArgs args = new MenuCommandArgs(true, new MenuCommandUI(item), item.Tag);
                menuItem.InvokeMenuCommand(args);
            }
        }

        /// <summary>
        /// コマンドハンドラ呼び出し。
        /// </summary>
        private void CallCommandHandler(ToolStripItem item)
        {
            UIToolStripMenuItem menuItem = item as UIToolStripMenuItem;
            if (menuItem == null && bindTable_.ContainsKey(item))
            {
                menuItem = bindTable_[item];
            }

            if (menuItem != null)
            {
                MenuCommandArgs args = new MenuCommandArgs(false, new MenuCommandUI(item), item.Tag);
                menuItem.InvokeMenuCommand(args);
            }
        }

        /// <summary>
        /// メッセージハンドラ呼び出し。
        /// </summary>
        private void CallMessageHandler(ToolStripItem item)
        {
            if (messageHandler_ != null)
            {
                UIToolStripMenuItem menuItem = item as UIToolStripMenuItem;
                if (menuItem == null && bindTable_.ContainsKey(item))
                {
                    menuItem = bindTable_[item];
                }

                if (menuItem != null && menuItem.CaptionText != null)
                {
                    messageHandler_(menuItem.CaptionText);
                }
                else
                {
                    messageHandler_(string.Empty);
                }
            }
        }

        #region イベントハンドラ
        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStrip_MouseEnter(object sender, EventArgs e)
        {
            // メッセージ更新状態開始
            isMessageUpdating_ = true;
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStrip_MouseLeave(object sender, EventArgs e)
        {
            // メッセージ更新状態終了
            isMessageUpdating_ = false;
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_UIToolStrip_BeforeMouseMove(object sender, MouseEventArgs e)
        {
            UIToolStrip toolStrip = (UIToolStrip)sender;

            // マウス上の無効状態項目を取得
            ToolStripItem item = toolStrip.GetItemAt(e.Location);
            if (item != null && !item.Enabled)
            {
                // メッセージハンドラ呼び出し
                CallMessageHandler(item);
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripItem_MouseEnter(object sender, EventArgs e)
        {
            // メッセージハンドラ呼び出し
            CallMessageHandler((ToolStripItem)sender);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripItem_Click(object sender, EventArgs e)
        {
            if (!TheApp.MainFrame.CanFocus)
            {
                // モーダルダイアログ表示中やドラッグ中は無視する
                return;
            }

            // 長時間処理コマンドになる前の描画更新
            // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1932
            // 描画に時間のかかる場合にここでDoEventするとこの先の処理が止まるので無効に。
            // 重い処理の時に描画に遅れが出る可能性有り？
            // Application.DoEvents();

            // 実行ハンドラ呼び出し
            CallCommandHandler((ToolStripItem)sender);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_DropDownItem_Paint(object sender, PaintEventArgs e)
        {
            ToolStripDropDownItem item = (ToolStripDropDownItem)sender;

            // 無効状態の場合のみ
            if (!item.Enabled)
            {
                if (item.Owner != null)
                {
                    // マウスが領域内にあれば
                    Rectangle rect = item.Owner.RectangleToScreen(item.Bounds);
                    if (rect.Contains(Control.MousePosition))
                    {
                        // メッセージハンドラ呼び出し
                        CallMessageHandler(item);
                    }
                }
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripDropDownItem_DropDownOpening(object sender, EventArgs e)
        {
            // ドロップダウンレベルを上げる
            dropDownLevel_++;

            // 更新ハンドラ呼び出し
            ToolStripDropDownItem item = (ToolStripDropDownItem)sender;
            foreach (ToolStripItem subItem in item.DropDownItems)
            {
                CallUpdateHandler(subItem);
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripDropDownItem_DropDownClosed(object sender, EventArgs e)
        {
            // ドロップダウンレベルを下げる
            dropDownLevel_--;
            Debug.Assert(dropDownLevel_ >= 0);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        void Event_ContextMenu_Opening(object sender, CancelEventArgs e)
        {
            // ドロップダウンレベルを上げる
            dropDownLevel_++;

            // 更新ハンドラ呼び出し
            ToolStripDropDown dropDown = (ToolStripDropDown)sender;
            foreach (ToolStripItem subItem in dropDown.Items)
            {
                CallUpdateHandler(subItem);
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        void Event_ContextMenu_Closed(object sender, ToolStripDropDownClosedEventArgs e)
        {
            // ドロップダウンレベルを下げる
            dropDownLevel_--;
            Debug.Assert(dropDownLevel_ >= 0);
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ToolStripMenuItem_BeforeShortcutKeyProcessing(object sender, EventArgs e)
        {
            // 強制的に編集中のものを確定させる
            App.PropertyEdit.ObjectPropertyDialog.ForceFixEditting();

            // 更新ハンドラ呼び出し
            CallUpdateHandler((ToolStripItem)sender);
        }
        #endregion
    }
    #endregion

    /// <summary>
    /// メニューコマンドハンドラデリゲート。
    /// </summary>
    public delegate void MenuCommandHandler(MenuCommandArgs args);

    /// <summary>
    /// メニューコマンドメッセージハンドラデリゲート。
    /// </summary>
    public delegate void MenuCommandMessageHandler(string message);
}
