﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Threading;
using App.Controls;
using App.Data;
using App.Utility;
using App.res;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using System.Text;

namespace App
{
    // バイナリコンバーターマネージャー
    static public class BinaryConverterManager
    {
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate int InitDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ShutdownDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ClearDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate UInt32 GetCvtrVersionDelegate();

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool SetOptionsDelegate(
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool AddFileDelegate(
            IntPtr pData,
            int dataSize,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] paths,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] options
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate UInt64 CalcSizeDelegate(
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate UInt64 GetAlignmentSizeDelegate(
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool ConvertDelegate(
            IntPtr ppData,
            int pDataSize
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool BindDelegate(
            IntPtr ppData,
            int pDataSize
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate bool EndianSwapDelegate(
            IntPtr ppData,
            int pDataSize
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate IntPtr OpenLogFileDelegate(
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] outArgv,
            [MarshalAs(UnmanagedType.LPArray, ArraySubType = UnmanagedType.LPWStr)] string[] errArgv
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void CloseLogFileDelegate(
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void ForceNwArchiveDelegate(
        );

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        public delegate void ForceNnArchiveDelegate(
        );

        private static InitDelegate				Init;
        private static ShutdownDelegate			Shutdown;
        private static ClearDelegate			Clear;
        private static GetCvtrVersionDelegate	GetCvtrVersion;
        private static SetOptionsDelegate		SetOptions;
        private static AddFileDelegate			AddFile;
        private static CalcSizeDelegate			CalcSize;
        private static GetAlignmentSizeDelegate GetAlignmentSize;
        private static ConvertDelegate			Convert;
        private static BindDelegate				Bind;
        private static EndianSwapDelegate		EndianSwap;
        private static OpenLogFileDelegate		OpenLogFile;
        private static CloseLogFileDelegate		CloseLogFile;
        private static ForceNwArchiveDelegate ForceNwArchive;
        private static ForceNnArchiveDelegate ForceNnArchive;

        private static string DllFilepath
        {
            get
            {
                var path = string.Format(
                        @"{0}\..\G3dTool\{1}\NW4F_g3dbincvtr.dll",
                        Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                        Environment.Is64BitProcess ?
                            "win64" :
                            "win32"
                    );

                // 新構成対応
                if (!File.Exists(path))
                {
                    path = string.Format(
                        @"{0}\..\3dTools\{1}3dBinaryConverter.dll",
                        Environment.GetEnvironmentVariable("NW4F_3DEDITOR_ROOT"),
                        Environment.Is64BitProcess ?
                            "":
                            @"x86\"
                    );
                }

                return path;
            }
        }

        private static IntPtr dllHandle_ = IntPtr.Zero;
        public static void Initialize()
        {
            // dll が読み込めないようなので、一時的にカレントディレクトリを移す
            string currentDir = System.Environment.CurrentDirectory;
            try
            {
                System.IO.Directory.SetCurrentDirectory(Path.GetDirectoryName(DllFilepath));
                dllHandle_ = App.Win32.NativeMethods.LoadLibrary(DllFilepath);

                SetExportFunction();

                Init();
            }
            catch(Exception e)
            {
                UIMessageBox.Error(e.ToString());
            }
            finally
            {
                System.Environment.CurrentDirectory = currentDir;
            }
        }

        public static void Destroy()
        {
            if (dllHandle_ != IntPtr.Zero)
            {
                Shutdown();

                App.Win32.NativeMethods.FreeLibrary(dllHandle_);
                dllHandle_ = IntPtr.Zero;
            }
        }

        private static void SetExportFunction()
        {
            Init = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterInitialize"), typeof(InitDelegate)) as InitDelegate;
            Shutdown = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterShutdown"), typeof(ShutdownDelegate)) as ShutdownDelegate;
            Clear = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterClear"), typeof(ClearDelegate)) as ClearDelegate;
            GetCvtrVersion = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterGetConverterVersion"), typeof(GetCvtrVersionDelegate)) as GetCvtrVersionDelegate;
            SetOptions = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterSetOptions"), typeof(SetOptionsDelegate)) as SetOptionsDelegate;
            AddFile = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterAddFile"), typeof(AddFileDelegate)) as AddFileDelegate;
            CalcSize = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterCalculateSize"), typeof(CalcSizeDelegate)) as CalcSizeDelegate;
            GetAlignmentSize = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterGetAlignmentSize"), typeof(GetAlignmentSizeDelegate)) as GetAlignmentSizeDelegate;
            Convert = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterConvert"), typeof(ConvertDelegate)) as ConvertDelegate;
            Bind = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterBind"), typeof(BindDelegate)) as BindDelegate;
            EndianSwap = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterSwapEndian"), typeof(EndianSwapDelegate)) as EndianSwapDelegate;
            OpenLogFile = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterOpenLogFile"), typeof(OpenLogFileDelegate)) as OpenLogFileDelegate;
            CloseLogFile = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterCloseLogFile"), typeof(CloseLogFileDelegate)) as CloseLogFileDelegate;
            ForceNwArchive = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterForceNwArchive"), typeof(ForceNwArchiveDelegate)) as ForceNwArchiveDelegate;
            ForceNnArchive = Marshal.GetDelegateForFunctionPointer(App.Win32.NativeMethods.GetProcAddress(dllHandle_, "nng3dToolBinaryConverterForceNnArchive"), typeof(ForceNnArchiveDelegate)) as ForceNnArchiveDelegate;
        }

        public enum errCodeEnum
        {
            noErr = 0,
            notInitialize = 1,
            SetOptions = 2,
            AddFile = 3,
            CalcSize = 4,
            MallocOutBuf = 5,
            Convert = 6,
            Bind = 7,
            EndianSwap = 8,
            SaveFile = 9,
            UnknownErr = 10
        };

        private static string GetErrorMsg(int errCode)
        {
            string msg = string.Empty;
            switch ((errCodeEnum)errCode)
            {
                case errCodeEnum.notInitialize:
                    msg = Strings.BinalizeError_not_Initialize;
                    break;
                case errCodeEnum.SetOptions:
                    msg = Strings.BinalizeError_SetOptions;
                    break;
                case errCodeEnum.AddFile:
                    msg = Strings.BinalizeError_AddFile;
                    break;
                case errCodeEnum.CalcSize:
                    msg = Strings.BinalizeError_CalcSize;
                    break;
                case errCodeEnum.MallocOutBuf:
                    msg = Strings.BinalizeError_MallocOutBuf;
                    break;
                case errCodeEnum.Convert:
                    msg = Strings.BinalizeError_Convert;
                    break;
                case errCodeEnum.Bind:
                    msg = Strings.BinalizeError_Bind;
                    break;
                case errCodeEnum.EndianSwap:
                    msg = Strings.BinalizeError_EndianSwap;
                    break;
                case errCodeEnum.SaveFile:
                    msg = Strings.BinalizeError_SaveFile;
                    break;
                case errCodeEnum.UnknownErr:
                    msg = Strings.BinalizeError_UnknownErr;
                    break;
            }
            return msg;
        }

        private static void ProcessErrorCode(string filePath, int errCode)
        {
            string msg = GetErrorMsg(errCode);
            // バイナリ化失敗エラーメッセージ(ウインドウに表示に変えるかもしれません。)
            DebugConsole.WriteLine("BinarConverterDll Error : {0} : {1}", filePath, msg);
        }

        // 書き込みデリゲート
        private delegate void ShowBinalizeErrorMsgDelegate(List<string> errMsgs);
        // Invoke用書き込みメソッド
        private static void _ShowBinalizeErrorMsg(List<string> errMsgs) { ShowBinalizeErrorMsg(errMsgs); }
        public static void ShowBinalizeErrorMsg(List<string> errMsgs)
        {
            if (!errMsgs.Any())
            {
                // 文字列が入っていない場合は、出力しない。
                return;
            }

            // 別スレッドからの呼び出し用
            if (System.Threading.Thread.CurrentThread != TheApp.MainThread)
            {
                TheApp.MainFrame.BeginInvoke(new ShowBinalizeErrorMsgDelegate(_ShowBinalizeErrorMsg), new object[] { errMsgs });
                return;
            }

            // ダイアログ表示
            using (var dialog = new OkListBoxDialog())
            {
                dialog.Text = res.Strings.BinalizeError_DlgCaption;
                dialog.lblDescription.Text = res.Strings.BinalizeError_DlgDescription;
                foreach (var file in errMsgs)
                {
                    dialog.AddLine(file);
                }
                dialog.ShowDialog();
            }
        }

        private static string CreateTemporaryFileNameForBinalize(IntermediateFileDocument intermediateFile, string inputFilePath)
        {
            // 新規作成時等、ファイル名が設定されていない場合は、
            // 適当なファイル名を付けます。
            string tempFilePath = TemporaryFileUtility.MakeTemporaryFileName(".bfres");
            string[] inputExt = ObjectIDUtility.IdToExt(intermediateFile.ObjectID).ToArray();
            ////////////////////////////////////////////////////////////////////////////
            //// TODO:現状バイナリコンバーターがUserDataを含む場合、アスキーにしか対応していないので、
            //// アスキーでの出力に変更します。
            ////////////////////////////////////////////////////////////////////////////
            //// 中間バイナリファイルの拡張子にする。
            //return Path.GetDirectoryName(tempFilePath) + "\\" + intermediateFile.Name + "." + inputExt[0];

            // 中間バイナリファイルの拡張子にする。
            return Path.GetDirectoryName(tempFilePath) + "\\" + intermediateFile.Name + "." + inputExt[1];
        }

        private static string ChangeExtAtoB(string inputFilePath)
        {
            GuiObjectID? objId = ObjectIDUtility.ExtToId(Path.GetExtension(inputFilePath));
            if (objId == null)
            {
                // こんなこと、あり得ないはず。
            }
            GuiObjectID objID = (GuiObjectID)objId;
            string[] inputExt = ObjectIDUtility.IdToExt(objID).ToArray();
            return Path.ChangeExtension(inputFilePath, inputExt[1]);
        }

        private static int ConvertToBinary(byte[] fileImage, string inputFilePath, string outputFilePath, TeamConfig.PlatformPreset platformPreset)
        {
            using (var watch = new DebugStopWatch("ConvertToBinary " + outputFilePath))
            {
                try
                {
                    DebugConsole.WriteLine("ConvertToBinary ");
                    if (Clear == null)
                    {
                        Initialize();
                    }
                    if (!Clear())
                    {
                        DebugConsole.WriteLine("ConvertToBinary Clear() failed");
                        return (int)errCodeEnum.UnknownErr;
                    }

                    if (platformPreset.UseNw)
                    {
                        ForceNwArchive();
                    }
                    else
                    {
                        ForceNnArchive();
                    }

                    if (!SetOptions(new string[] { "--editor", "", null }))
                    {
                        DebugConsole.WriteLine("ConvertToBinary SetOptions() failed");
                        return (int)errCodeEnum.SetOptions;
                    }

                    unsafe
                    {
                        fixed (byte* pData = fileImage)
                        {
                            if (!AddFile(
                                (IntPtr)pData,
                                Marshal.SizeOf(fileImage[0]) * fileImage.Length,
                                new string[] { inputFilePath, Path.GetFileNameWithoutExtension(inputFilePath), Path.GetExtension(inputFilePath) },
                                new string[] { null }
                                ))
                            {
                                DebugConsole.WriteLine("ConvertToBinary AddFile() failed");
                                return (int)errCodeEnum.AddFile;
                            }

                            int size = (int)CalcSize();
                            if (size == 0)
                            {
                                DebugConsole.WriteLine("ConvertToBinary CalcSize() failed");
                                return (int)errCodeEnum.CalcSize;
                            }

                            byte[] output = new byte[size / Marshal.SizeOf(typeof(byte))];
                            fixed (byte* pOutput = output)
                            {
                                if (!Convert((IntPtr)pOutput, size))
                                {
                                    DebugConsole.WriteLine("ConvertToBinary Convert() failed");
                                    return (int)errCodeEnum.Convert;
                                }

                                if (!Bind((IntPtr)pOutput, size))
                                {
                                    DebugConsole.WriteLine("ConvertToBinary Bind() failed");
                                    return (int)errCodeEnum.Bind;
                                }

                                if (!EndianSwap((IntPtr)pOutput, size))
                                {
                                    DebugConsole.WriteLine("ConvertToBinary EndianSwap() failed");
                                    return (int)errCodeEnum.EndianSwap;
                                }

                                // バイナリ形式でファイルに書き出し。
                                using (BinaryWriter w = new BinaryWriter(File.OpenWrite(outputFilePath)))
                                {
                                    w.Write(output, 0, size);
                                }
                            }
                        }
                    }
                }
                catch (Exception e)
                {
                    DebugConsole.WriteLine("ConvertToBinary exception: " + e.ToString());
                    return (int)errCodeEnum.UnknownErr;
                }
                finally
                {
                    // リソースを解放
                    if (Clear != null)
                    {
                        if (!Clear())
                        {
                            DebugConsole.WriteLine("ConvertToBinary Clear() failed");
                        }
                    }
                }
            }
            return (int)errCodeEnum.noErr;
        }



        /// <summary>
        /// 複数ファイル用バイナリコンバート前処理
        /// </summary>
        public static int PreBinarize(TeamConfig.PlatformPreset platformPreset)
        {
            using (var watch = new DebugStopWatch("PreBinarize "))
            {
                try
                {
                    if (Clear == null)
                    {
                        Initialize();
                    }
                    if (!Clear())
                    {
                        DebugConsole.WriteLine("ConvertToBinaryForViewer Clear() failed");
                        return (int)errCodeEnum.UnknownErr;
                    }

                    if (platformPreset.UseNw)
                    {
                        ForceNwArchive();
                    }
                    else
                    {
                        ForceNnArchive();
                    }


                    if (!SetOptions(new string[] { "--editor", "", null }))
                    {
                        DebugConsole.WriteLine("ConvertToBinaryForViewer SetOptions() failed");
                        return (int)errCodeEnum.SetOptions;
                    }

                }
                catch (Exception e)
                {
                    DebugConsole.WriteLine("ConvertToBinaryForViewer exception: " + e.ToString());
                    return (int)errCodeEnum.UnknownErr;
                }
            }
            return (int)errCodeEnum.noErr;
        }

        /// <summary>
        /// 複数ファイル、バイナリコンバート用ファイル追加
        /// </summary>
        public static int FileAdd(byte[] fileImage, string inputFilePath, IEnumerable<string> options)
        {
            using (var watch = new DebugStopWatch("FileAdd "))
            {
                try
                {
                    unsafe
                    {
                        fixed (byte* pData = fileImage)
                        {
                            var ext = Path.GetExtension(inputFilePath);
                            // 拡張子がtgaのままでは送れないので、ftxbにする
                            if (ext.ToLower() == ".tga")
                            {
                                ext = ".ftxb";
                            }
                            if (!AddFile(
                                (IntPtr)pData,
                                Marshal.SizeOf(fileImage[0]) * fileImage.Length,
                                new string[] { inputFilePath, Path.GetFileNameWithoutExtension(inputFilePath), ext },
                                options.Concat(Enumerable.Repeat<string>(null, 1)).ToArray()
                                ))
                            {
                                DebugConsole.WriteLine("FileAdd AddFile() failed");
                                return (int)errCodeEnum.AddFile;
                            }
                        }
                    }
                }
                catch (Exception e)
                {
                    DebugConsole.WriteLine("FileAdd exception: " + e.ToString());
                    return (int)errCodeEnum.AddFile;
                }
            }

            return (int)errCodeEnum.noErr;
        }

        /// <summary>
        /// 複数ファイル、バイナリコンバート
        /// </summary>
        public static int Binarize(string outputFilePath, out uint alignmentSize)
        {
            using (var watch = new DebugStopWatch("Binarize "))
            {
                alignmentSize = 0;
                try
                {
                    unsafe
                    {
                        int size = (int)CalcSize();
                        if (size == 0)
                        {
                            DebugConsole.WriteLine("Binarize CalcArchiveSize() failed");
                            return (int)errCodeEnum.CalcSize;
                        }

                        // アライメントサイズを取得
                        alignmentSize = (uint)GetAlignmentSize();

                        byte[] output = new byte[size / Marshal.SizeOf(typeof(byte))];
                        fixed (byte* pOutput = output)
                        {
                            if (!Convert((IntPtr)pOutput, size))
                            {
                                DebugConsole.WriteLine("Binarize Convert() failed");
                                return (int)errCodeEnum.Convert;
                            }

                            if (!Bind((IntPtr)pOutput, size))
                            {
                                DebugConsole.WriteLine("Binarize Bind() failed");
                                return (int)errCodeEnum.Bind;
                            }

                            if (!EndianSwap((IntPtr)pOutput, size))
                            {
                                DebugConsole.WriteLine("Binarize EndianSwap() failed");
                                return (int)errCodeEnum.EndianSwap;
                            }

                            // バイナリ形式でファイルに書き出し。
                            using (BinaryWriter w = new BinaryWriter(File.OpenWrite(outputFilePath)))
                            {
                                w.Write(output, 0, size);
                            }
                        }
                    }
                }
                catch (Exception e)
                {
                    DebugConsole.WriteLine("Binarize exception: " + e.ToString());
                    return (int)errCodeEnum.UnknownErr;
                }
            }
            return (int)errCodeEnum.noErr;
        }

        /// <summary>
        /// バイナリコンバート(１ファイル用)
        /// </summary>
        public static string ConvertToBinary(
            IntermediateFileDocument intermediateFile,
            string filePath,
            string outPath,
            string outExt,
            List<string> errMsgs,
            bool forViewer,
            TeamConfig.PlatformPreset platformPreset)
        {
            // バイナリコンバーターDll側にログファイル名を設定する。
            string stdOutLogFilePath = GetLogFilePath();
            string stdErrLogFilePath = GetLogFilePath();
            SetBinaryConverterLogFileStream(stdOutLogFilePath, stdErrLogFilePath);

            string inputFilePath = filePath;
            string outFilePath = outPath;

            if (string.IsNullOrEmpty(inputFilePath))
            {
                inputFilePath = CreateTemporaryFileNameForBinalize(intermediateFile, inputFilePath);
            }

            using (var watch = new DebugStopWatch("Binalize File " + inputFilePath))
            {
                nw4f_3difType nw4f_3dif = intermediateFile.Create_nw4f_3difType();
                //nw4f_3dif.file_info = intermediateFile.file_info;
                nw4f_3dif.file_info = null;

                // テキスト中間ファイルはバイナリ中間ファイルにする。
                List<G3dStream> streams = intermediateFile.BinaryStreams;
                // バイナリ中間ファイルの場合、nw4f_3difからストリームを削除する。
                if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                {
                    nw4f_3dif.RootElement.stream_array = null;
                }

                // 中間テキストファイルの場合、中間バイナリファイルの拡張子に変更します。
                if (G3dPath.IsTextPath(inputFilePath))
                {
                    inputFilePath = ChangeExtAtoB(inputFilePath);
                }

                // バイナリデータに変換する。
                byte[] fileImage;
                lock (Viewer.Connecter.AbortLock)
                {
                    fileImage = IfBinaryFormatter.FormatStreamFast(nw4f_3dif, streams);
                }

                if (string.IsNullOrEmpty(outFilePath))
                {
                    // 出力ファイルが指定されていない場合は、テンポラリの名前を使用する。
                    outFilePath = TemporaryFileUtility.MakeTemporaryFileName(outExt);
                }

                try
                {
                    // コンバートします。
                    int errCode = ConvertToBinary(fileImage, inputFilePath, outFilePath, platformPreset);
                    if (errCode != 0)
                    {
                        outFilePath = string.Empty;
                        ProcessErrorCode(filePath, errCode);
                        if (errMsgs != null)
                        {
                            string msg = intermediateFile.FileName + ": " + GetErrorMsg(errCode);
                            errMsgs.Add(msg);
                        }
                    }
                }
                catch (ThreadAbortException)
                {
                    // Abortに対しては特に何も行わない
                }
                catch (Exception e)
                {
                    e.GetType();
                    outFilePath = string.Empty;
                    ProcessErrorCode(string.Empty, (int)errCodeEnum.UnknownErr);
                    if (errMsgs != null)
                    {
                        errMsgs.Add(intermediateFile.FileName + ": " + GetErrorMsg((int)errCodeEnum.UnknownErr));
                    }
                }
            }

            // バイナリコンバーターDll側から出力されたログを読み込む
            ResetBinaryConverterLogFileStream();
            SetErrMsgFromLogfile(errMsgs, stdErrLogFilePath, stdOutLogFilePath);

            return outFilePath;
        }

        /// <summary>
        /// バイナリコンバート(複数ファイル用)
        /// </summary>
        public static string ConvertToBinary(List<IntermediateFileDocument> intermediateFiles, string outPath, List<string> errMsgs, bool forViewer, TeamConfig.PlatformPreset platformPreset)
        {
            if (!intermediateFiles.Any())
            {
                // 一つもなかったら、return
                return string.Empty;
            }

            // バイナリコンバーターDll側にログファイル名を設定する。
            string stdOutLogFilePath = GetLogFilePath();
            string stdErrLogFilePath = GetLogFilePath();
            SetBinaryConverterLogFileStream(stdOutLogFilePath, stdErrLogFilePath);

            string outFilePath = string.Empty;
            using (var watch = new DebugStopWatch("Binalize File " + outPath))
            {
                // バイナリコンバーターの初期処理を呼び出す。
                int errCode = PreBinarize(platformPreset);
                if (errCode != 0)
                {
                    // 失敗したら、return
                    ProcessErrorCode(string.Empty, errCode);
                    return string.Empty;
                }

                // 個々のファイルをバイナリコンバーターへ追加していく
                foreach (IntermediateFileDocument doc in intermediateFiles)
                {
                    string inputFilePath = doc.FilePath;

                    // インプットファイルが指定されていない場合、適当な名前をでっちあげる。
                    if (string.IsNullOrEmpty(inputFilePath))
                    {
                        inputFilePath = CreateTemporaryFileNameForBinalize(doc, inputFilePath);
                    }

                    // コンバート事前処理
                    nw4f_3difType nw4f_3dif = doc.Create_nw4f_3difType();
                    //nw4f_3dif.file_info = doc.file_info;
                    nw4f_3dif.file_info = null;

                    // テキスト中間ファイルはバイナリ中間ファイルにする。
                    List<G3dStream> streams = doc.BinaryStreams;
                    // バイナリ中間ファイルの場合、nw4f_3difからストリームを削除する。
                    if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                    {
                        nw4f_3dif.RootElement.stream_array = null;
                    }
                    // 中間テキストファイルの場合、中間バイナリファイルの拡張子に変更します。
                    if (G3dPath.IsTextPath(inputFilePath))
                    {
                        inputFilePath = ChangeExtAtoB(inputFilePath);
                    }
                    byte[] fileImage;
                    lock (Viewer.Connecter.AbortLock)
                    {
                        fileImage = IfBinaryFormatter.FormatStreamFast(nw4f_3dif, streams);
                    }

                    try
                    {
                        // バイナリコンバーターへファイルを追加します。
                        errCode = FileAdd(fileImage, inputFilePath, Enumerable.Empty<string>());
                        if (errCode != 0)
                        {
                            string msg = doc.FileName + ": " + GetErrorMsg(errCode);
                            errMsgs.Add(msg);
                        }
                    }
                    catch (ThreadAbortException)
                    {
                        // Abortに対しては特に何も行わない
                    }
                    catch (Exception e)
                    {
                        e.GetType();
                        errMsgs.Add(doc.FileName + ": " + GetErrorMsg((int)errCodeEnum.UnknownErr));
                    }

                }

                // 最後に出力を行う。
                try
                {
                    // コンバート。
                    uint alignmentSize;
                    errCode = Binarize(outPath, out alignmentSize);
                    if (errCode != 0)
                    {
                        errMsgs.Add(GetErrorMsg(errCode));
                        ProcessErrorCode(string.Empty, errCode);
                    }
                }
                catch (ThreadAbortException)
                {
                    // Abortに対しては特に何も行わない
                }
                catch (Exception e)
                {
                    e.GetType();
                    errMsgs.Add(GetErrorMsg((int)errCodeEnum.UnknownErr));
                    ProcessErrorCode(string.Empty, (int)errCodeEnum.UnknownErr);
                }

            }

            // バイナリコンバーターDll側から出力されたログを読み込む
            ResetBinaryConverterLogFileStream();
            SetErrMsgFromLogfile(errMsgs, stdErrLogFilePath, stdOutLogFilePath);

            // リソースを解放
            if (Clear != null)
            {
                Clear();
            }

            outFilePath = outPath;
            return outFilePath;
        }

        // 単体の bntx を生成
        public static bool ConvertTextureBinary(nw4f_3difType data, List<G3dStream> streams, string name, TeamConfig.PlatformPreset platform, string outputPath, out uint alignment, out string error)
        {
            alignment = 0;
            using (var tempDirectory = TemporaryFileUtility.MakeDisposableDirectoryName())
            {
                try
                {
                    var builder = new StringBuilder();
                    var tempPath = Path.Combine(tempDirectory.Path, name + ".ftxb");
                    if (G3dStreamUtility.HasStreamArray(data))
                    {
                        data.RootElement.stream_array = null;
                    }
                    // 中断すると危険なのでロック
                    lock (Viewer.Connecter.AbortLock)
                    {
                        IfWriteUtility.Write(data, streams, tempPath);
                    }

                    builder.Append(string.Format("\"{0}\" ", tempPath));

                    builder.Append(string.Format("-o \"{0}\" ", outputPath));
                    builder.Append(Environment.ExpandEnvironmentVariables(platform.ActiveDeviceOption.TextureConverterAdditionalArgs ?? string.Empty) + " ");
                    var errorBuilder = new StringBuilder();
                    var ret = ProcessUtility.ProcessStart(TexcvtrManager.TextureConverterExeFilepath, builder.ToString(), null, (s, e) => { errorBuilder.Append(e.Data); }, e => errorBuilder.AppendLine(e.Message));

                    error = errorBuilder.ToString();

                    if (ret == 0)
                    {
                        using (var fs = new FileStream(outputPath, FileMode.Open))
                        {
                            fs.Position = 14;
                            var bytes = new byte[1];
                            fs.Read(bytes, 0, 1);
                            alignment = 1u << bytes[0];
                        }
                    }

                    return ret == 0;

                }
                catch (Exception e)
                {
                    error = e.ToString();
                    return false;
                }
            }
        }

        /// <summary>
        /// バイナリコンバート(複数ファイル用)
        /// </summary>
        public static string ConvertToBinary(List<Tuple<IntermediateFileDocument, nw4f_3difType, List<G3dStream>, string>> dataList, string outPath, List<string> errMsgs, bool forViewer, TeamConfig.PlatformPreset platform, out uint alignmentSize)
        {
            alignmentSize = 0;
            if (!dataList.Any())
            {
                // 一つもなかったら、return
                return string.Empty;
            }

            // バイナリコンバーターDll側にログファイル名を設定する。
            string stdOutLogFilePath = GetLogFilePath();
            string stdErrLogFilePath = GetLogFilePath();
            SetBinaryConverterLogFileStream(stdOutLogFilePath, stdErrLogFilePath);


            string outFilePath = string.Empty;
            var tempBinaryFiles = new List<string>();
#if false
            byte[] textureBinary = null;
            string textureBinaryPath = null;
#endif
            try
            {
#if false
                // テクスチャコンバート
                if (!platform.UseNw)
                {
                    var textures = dataList.Where(x => x.Item1 is Texture).ToList();
                    if (textures.Any())
                    {
                        string error;
                        textureBinary = ConvertTextureBianry(textures, platform, out error, out textureBinaryPath);
                        if (textureBinary == null)
                        {
                            errMsgs.Add(dataList[0].Item1.FileName + ": " + Strings.TextureConverter_error + error);
                            return string.Empty;
                        }
                    }
                }
#endif

                using (var watch = new DebugStopWatch("Binalize File " + outPath))
                {
                    // バイナリコンバーターの初期処理を呼び出す。
                    int errCode = PreBinarize(platform);
                    if (errCode != 0)
                    {
                        // 失敗したら、return
                        ProcessErrorCode(string.Empty, errCode);
                        errMsgs.Add(dataList[0].Item1.FileName + ": " + GetErrorMsg(errCode));
                        return string.Empty;
                    }


                    // 個々のファイルをバイナリコンバーターへ追加していく
                    foreach (var data in dataList)
                    {
                        if (!platform.UseNw && data.Item1 is Texture)
                        {
                            continue;
                        }

                        string inputFilePath = data.Item1.FilePath;

                        // インプットファイルが指定されていない場合、適当な名前をでっちあげる。
                        if (string.IsNullOrEmpty(inputFilePath))
                        {
                            inputFilePath = CreateTemporaryFileNameForBinalize(data.Item1, inputFilePath);
                        }

                        data.Item2.file_info = null;

                        // テキスト中間ファイルはバイナリ中間ファイルにする。
                        // 中間テキストファイルの場合、中間バイナリファイルの拡張子に変更します。
                        if (G3dPath.IsTextPath(inputFilePath))
                        {
                            inputFilePath = ChangeExtAtoB(inputFilePath);
                        }

                        byte[] fileImage;
                        lock (Viewer.Connecter.AbortLock)
                        {
                            fileImage = IfBinaryFormatter.FormatStreamFast(data.Item2, data.Item3);
                        }

                        try
                        {
                            // バイナリコンバーターへファイルを追加します。
                            errCode = FileAdd(fileImage, inputFilePath, Enumerable.Empty<string>());
                            if (errCode != 0)
                            {
                                string msg = data.Item1.FileName + ": " + GetErrorMsg(errCode);
                                errMsgs.Add(msg);
                            }
                        }
                        catch (ThreadAbortException)
                        {
                            // Abortに対しては特に何も行わない
                            //return;
                        }
                        catch (Exception e)
                        {
                            e.GetType();
                            errMsgs.Add(data.Item1.FileName + ": " + GetErrorMsg((int)errCodeEnum.UnknownErr));
                        }
                    }

#if false
                    if (textureBinary != null)
                    {
                        try
                        {
                            const int alignPos = 14;
                            int alignment = 1;
                            // アライメントのチェック
                            if (textureBinary.Length > alignPos && textureBinary[alignPos] >= 0 && textureBinary[alignPos] < 31)
                            {
                                alignment = 1 << textureBinary[alignPos];

                                // バイナリコンバーターへファイルを追加します。
                                errCode = FileAdd(textureBinary, textureBinaryPath, new [] { "--external-file", null, "--external-file-alignment", alignment.ToString(), });
                                if (errCode != 0)
                                {
                                    string msg = textureBinaryPath + ": " + GetErrorMsg(errCode);
                                    errMsgs.Add(msg);
                                }
                            }
                            else
                            {
                                errMsgs.Add(textureBinaryPath + ": " + Strings.BinarizeError_InvalidAlignment);
                            }
                        }
                        catch (ThreadAbortException)
                        {
                            // Abortに対しては特に何も行わない
                            //return;
                        }
                        catch (Exception e)
                        {
                            e.GetType();
                            errMsgs.Add(textureBinaryPath + ": " + GetErrorMsg((int)errCodeEnum.UnknownErr));
                        }
                    }
#endif

                    // 最後に出力を行う。
                    try
                    {
                        // コンバート。
                        errCode = Binarize(outPath, out alignmentSize);
                        if (errCode != 0)
                        {
                            errMsgs.Add(GetErrorMsg(errCode));
                            ProcessErrorCode(string.Empty, errCode);
                        }
                    }
                    catch (ThreadAbortException)
                    {
                        // Abortに対しては特に何も行わない
                    }
                    catch (Exception e)
                    {
                        e.GetType();
                        errMsgs.Add(GetErrorMsg((int)errCodeEnum.UnknownErr));
                        ProcessErrorCode(string.Empty, (int)errCodeEnum.UnknownErr);
                    }
                }
            }
            finally
            {
            }

            // バイナリコンバーターDll側から出力されたログを読み込む
            ResetBinaryConverterLogFileStream();
            SetErrMsgFromLogfile(errMsgs, stdErrLogFilePath, stdOutLogFilePath);

            outFilePath = outPath;
            return outFilePath;
        }


        private static string GetLogFilePath()
        {
            return TemporaryFileUtility.MakeTemporaryFileName(".txt");
        }

        private static void SetBinaryConverterLogFileStream(string stdOutLogFilePath, string stdErrLogFilePath)
        {
            try
            {
                const string mode = "w+";
                OpenLogFile(
                    new string[] { stdOutLogFilePath, mode },
                    new string[] { stdErrLogFilePath, mode }
                    );
            }
            catch (Exception e)
            {
                DebugConsole.WriteLine("SetBinaryConverterLogFileStream Exception {0}", e.Message);
            }
        }

        private static void ResetBinaryConverterLogFileStream()
        {
            try
            {
                // バイナリコンバーターのLogFileStreamをもとに戻す。
                CloseLogFile();
            }
            catch (Exception e)
            {
                DebugConsole.WriteLine("ResetBinaryConverterLogFileStream Exception {0}", e.Message);
            }

        }

        private static string GetLogFileString(string path)
        {
            string logFileStr = string.Empty;

            try
            {
                using (StreamReader reader = new StreamReader(path, System.Text.Encoding.Default))
                {
                    logFileStr = reader.ReadToEnd();
                }

                // 一時ファイルも削除しちゃう。
                System.IO.File.Delete(path);
            }
            catch (Exception e)
            {
                DebugConsole.WriteLine("GetLogFileString Exception {0}", e.Message);
            }

            return logFileStr;
        }

        private static void SetErrMsgFromLogfile(List<string> errMsgs, string stdErrLogFilePath, string stdOutLogFilePath)
        {
            string ErrorlogStr = GetLogFileString(stdErrLogFilePath);
            string OutlogStr = GetLogFileString(stdOutLogFilePath);
            if (!string.IsNullOrEmpty(ErrorlogStr))
            {
                const string fromConverter = "From BinaryConverter -------------------------------";
                errMsgs.Add(string.Empty);
                errMsgs.Add(fromConverter);
                errMsgs.Add(OutlogStr);
                errMsgs.Add(ErrorlogStr);
            }
        }

    }
}
