﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.PropertyEdit;
using App.PropertyEdit.ShaderParamControls;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using nw.g3d.iflib.nw3de;
using Viewer;

namespace App.Data
{
    using System.IO;
    using System.Runtime.Serialization;

    using NintendoWare.G3d.Edit;

    using EditCommand = App.Command.EditCommand;

    public sealed partial class Material : GuiObject, IHasUserData
    {
        public materialType Data{ get; private set; }

        public IMaterialOwner Owner { get { return OwnerDocument as IMaterialOwner; } }

        public override string Name
        {
            get
            {
                // マテリアル中間ファイルの場合はファイル名がマテリアル名となる。
                // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=280255310
                return (ownerDocument is SeparateMaterial) ? ownerDocument.Name : base.Name;
            }
            set
            {
                // マテリアル中間ファイルの場合はファイル名がマテリアル名となる。
                // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=280255310
                if (!(ownerDocument is SeparateMaterial))
                {
                    base.Name = value;
                }
            }
        }

        /// <summary>
        /// マテリアルが所属するドキュメントを取得する
        /// </summary>
        public override Document OwnerDocument
        {
            get { return ownerDocument; }
        }
        private IntermediateFileDocument ownerDocument;

        /// <summary>
        /// マテリアルを直接参照しているモデルを取得する (マテリアル参照による間接参照は含まない)
        /// </summary>
        public ReadOnlyList<Model> Referrers
        {
            get
            {
                if (OwnerDocument is Model)
                {
                    return new ReadOnlyList<Model>(new Model[] { (Model)OwnerDocument });
                }
                else if (OwnerDocument is SeparateMaterial)
                {
                    List<Model> models;
                    var materialDoc = (SeparateMaterial)OwnerDocument;
                    if (!materialDoc.Referrers.TryGetValue(this, out models))
                    {
                        models = new List<Model>();
                    }
                    return new ReadOnlyList<Model>(models);
                }
                else
                {
                    return new ReadOnlyList<Model>(new Model[] { });
                }
            }
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        // マテリアル名として正しい文字列かどうか？
        public static bool IsValidateNameString(string name)
        {
            var match = nameRegex.Match(name);

            return
                match.Success && match.Groups[0].Value == name;

        }
        static readonly Regex nameRegex = new Regex(@"[0-9A-Za-z\-\._]+", RegexOptions.Compiled);

        public CustomUI CustomUI
        {
            get
            {
                return ToolData.CustomUI;
            }
            set
            {
                ToolData.CustomUI = value;
            }
        }

        static Material()
        {
            AppContext.MaterialReferenceBehaviorSettingChanged += OnMaterialReferenceBehaviorSettingChanged;
        }

        public Material(materialType material, Model owner, int index)
            : this(material, (IntermediateFileDocument)owner, index)
        { }

        public Material(materialType material, SeparateMaterial owner, int index)
            : this(material, (IntermediateFileDocument)owner, index)
        { }

        private Material(materialType material, IntermediateFileDocument owner, int index) : base(GuiObjectID.Material, index)
        {
            Data = material;
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            ownerDocument = owner;

            // マテリアル中間ファイルの場合はファイル名がマテリアル名となる。
            // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=280255310
            if (!(ownerDocument is SeparateMaterial))
            {
                Name = Data.name;
            }

            MaterialShaderAssign = new MaterialShaderAssign(Data.shader_assign, false);

            sampler_array = ObjectUtility.Clone(Data.sampler_array) ?? new sampler_arrayType() { sampler = new samplerType[] { } };

            // 初期値はコンフィグから取得
            OptimizeShaderOnReload = ApplicationConfig.DefaultValue.OptimizeShaderAfterMaterialChanged;

            // HIOからのRenderInfoPack
            RenderInfoPackFromHio = new RenderInfoPackFromHio();

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, ownerDocument.BinaryStreams);
        }

        public void UpdateData()
        {
            UpdateData(ownerDocument.BinaryStreams, false);
        }

        public materialType UpdateData(List<G3dStream> stream, bool materialTemplate)
        {
            var data = materialTemplate ? ObjectUtility.Clone(Data) : Data;
            data.tool_data = GetToolData();//ToolData.SaveDefault();
            data.comment = GetComment();

            data.shader_assign = MaterialShaderAssign.To_shader_assign(this, materialTemplate);
            data.sampler_array = sampler_array.sampler.Any() ? ObjectUtility.Clone(sampler_array) : null;

            // ユーザーデータのシリアライズ
            if (UserDataArray == null ||
                UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                data.user_data_array = null;
                return data;
            }

            data.user_data_array = new user_data_arrayType();
            this.MakeSerializeData(data.user_data_array, UserDataArray, stream);
            return data;
        }

        public tool_dataType GetToolData()
        {
            var elements = ToolData.GetElements();
            elements.RemoveAll(x => x.Name == typeof(ToolData.nw3de_CustomUI).Name);
            {
                var nw3de_CustomUI = CreateNw3de_CustomUI();

                if (nw3de_CustomUI.AnyElements())
                {
                    elements.Add(IfToolData.ConvertToXmlElement(nw3de_CustomUI, 3));
                }
            }

            // コンバイナーオプション情報を持っていれば保存。
            elements.RemoveAll(x => x.Name == typeof(nw3de_CombinerOptionInfo).Name);
            do
            {
                var combinerOptionInfo = ObjectUtility.Clone(ToolData.CombinerOptionInfo);
                if ((combinerOptionInfo?.CombinerOptions == null) || !combinerOptionInfo.CombinerOptions.Any())
                {
                    // オプション情報を持っていない。
                    break;
                }

                // シェーダーアサインに含まれなくなったものや未設定値のコンバイナーオプション情報を取り除く。
                var shaderOptions = MaterialShaderAssign.ShaderOptions.ToDictionary(x => x.id, x => x);
                combinerOptionInfo.CombinerOptions.RemoveAll(x =>
                {
                    shader_optionType opt;
                    return
                        !shaderOptions.TryGetValue(x.Id, out opt) ||
                        CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(opt.value);
                });
                if (!combinerOptionInfo.CombinerOptions.Any())
                {
                    // オプション情報が空になった。
                    break;
                }

                // シェーディングモデルに含まれなくなったコンバイナーオプション情報を取り除く。
                if (MaterialShaderAssign.ShadingModel != null)
                {
                    var combinerOptions = new HashSet<string>(GetCombiners().Where(x => x.Item1 == this).Select(x => x.Item2.id));
                    combinerOptionInfo.CombinerOptions.RemoveAll(x => !combinerOptions.Contains(x.Id));
                    if (!combinerOptionInfo.CombinerOptions.Any())
                    {
                        // オプション情報が空になった。
                        break;
                    }
                }

                // 残ったコンバイナーオプション情報を保存。
                elements.Add(IfToolData.ConvertToXmlElement(combinerOptionInfo, 1));
            }
            while (false);

            elements.RemoveAll(x => x.Name == typeof(nw3de_MaterialReference).Name);
            // マテリアル参照情報を持つ設定のみ保存
            var materialReference = ObjectUtility.Clone(ToolData.MaterialReference);
            IfApplyBaseMaterialUtility.UpdateMaterialReference(materialReference, MaterialShaderAssign?.ShadingModel);

            materialReference?.ClearEmptyItems();
            if (materialReference?.IsEmpty() == false)
            {
                elements.Add(IfToolData.ConvertToXmlElement(materialReference, 1));
            }

            // マテリアル中間ファイルに追加参照パスが設定されていれば保存。
            if (OwnerDocument is SeparateMaterial)
            {
                var separateMaterial = (SeparateMaterial)OwnerDocument;
                elements.RemoveAll(x => x.Name == typeof(nw3de_SearchPath).Name);
                if ((separateMaterial.SearchPaths != null) && separateMaterial.SearchPaths.Any())
                {
                    var item = new nw3de_SearchPath()
                    {
                        SearchPaths = separateMaterial.SearchPaths,
                    };
                    elements.Add(IfToolData.ConvertToXmlElement(item, 1));
                }
            }

            return
                elements.Any()
                    ? new tool_dataType()
                    {
                        Any = elements.ToArray()
                    }
                    : null;
        }

        public ToolData.nw3de_CustomUI CreateNw3de_CustomUI()
        {

            return new ToolData.nw3de_CustomUI()
            {
                option_vars = ToCustomItems(MaterialShaderAssign.ShaderOptions.Select(x => x.id), CustomUI.option_vars),
                attrib_vars = ToCustomItems(MaterialShaderAssign.AttribAssigns.Select(x => x.id), CustomUI.attrib_vars),
                sampler_vars = ToCustomItems(MaterialShaderAssign.SamplerAssigns.Select(x => x.id), CustomUI.sampler_vars),
                uniform_vars = ToCustomItems(MaterialShaderAssign.ShaderParams.Select(x => x.id), CustomUI.uniform_vars),
                render_info_slots = ToCustomItems(MaterialShaderAssign.RenderInfos.Select(x => x.name), CustomUI.render_info_slots),
                groups = CreateGroupCustomItems(),
                colorControls = CreateColorControls(),
            };
        }

        public List<CustomItem> CreateGroupCustomItems()
        {
            var groupSource = MaterialShaderAssign.ShaderDefinition != null || String.IsNullOrEmpty(MaterialShaderAssign.ShaderName) ?
                MaterialShaderAssign.GetGroups(false).Select(x => x.groupName) :
                CustomUI.groups.Keys.Select(x => x);
            return ToCustomItems(groupSource, CustomUI.groups);
        }

        public List<CustomItem> ToCustomItems(IEnumerable<string> keys, Dictionary<string, string> dict)
        {
            return (from key in keys
                    where dict.ContainsKey(key)
                    orderby key
                    select new CustomItem { id = key, label = dict[key] }).ToList();
        }

        public List<string> CreateColorControls()
        {
            return MaterialShaderAssign.ShaderParams
                .Where(x => x.type == shader_param_typeType.float3 || x.type == shader_param_typeType.float4)
                .Select(x => x.id)
                .Where(x => CustomUI.colorControls.Contains(x))
                .ToList();
        }

        // コンバイナーシェーダーディレクトリウォッチャー
        private List<FileSystemWatcher> combinerShaderDirectoryWatchers = new List<FileSystemWatcher>();

        public void EnableCombinerWatchers()
        {
            DisableCombinerWatchers();

            var models = new HashSet<Model>();
            foreach (var material in GetCombiners().Select(x => x.Item1).Distinct())
            {
                models.UnionWith(material.Referrers);
            }
            foreach (var model in models)
            {
                var watcher = new FileSystemWatcher(model.FileLocation, Material.CombinerShaderDirectoryName);
                watcher.NotifyFilter = System.IO.NotifyFilters.DirectoryName | NotifyFilters.LastWrite;
                watcher.IncludeSubdirectories = false;
                watcher.SynchronizingObject = TheApp.MainFrame;
                watcher.Changed += (ss, ee) => { App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(model, null)); };
                watcher.Created += (ss, ee) => { App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(model, null)); };
                watcher.Deleted += (ss, ee) => { App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(model, null)); };
                watcher.Renamed += (ss, ee) => { App.AppContext.NotifyPropertyChanged(null, new DocumentContentsChangedArgs(model, null)); };
                watcher.EnableRaisingEvents = true;
                combinerShaderDirectoryWatchers.Add(watcher);
            }
        }

        public void DisableCombinerWatchers()
        {
            // ディレクトリウォッチャーを破棄。
            foreach (var watcher in combinerShaderDirectoryWatchers)
            {
                watcher.Dispose();
            }
            combinerShaderDirectoryWatchers.Clear();
        }

        public List<Tuple<Material, shader_optionType>> GetCombiners()
        {
            var shadingModel = MaterialShaderAssign.ShadingModel;
            var combinerIds = new HashSet<string>((shadingModel != null) ? shadingModel.Options().Where(x => x.ui_item?.value == ui_item_valueType.combiner).Select(x => x.id) : Enumerable.Empty<string>());
            var list = new List<Tuple<Material, shader_optionType>>();
            foreach (var combiner in MaterialShaderAssign.ShaderOptions.Where(x => combinerIds.Contains(x.id)))
            {
                bool found;
                Material material;
                ValueResolvedState state;
                var value = GetResolvedValue(combiner, out found, out material, out state);
                list.Add(Tuple.Create(material, value));
            }
            return list;
        }

        public const string CombinerShaderDirectoryName = "combiners";

        public List<string> GetCombinerShaderFilePaths()
        {
            List<string> list = new List<string>();
            var combinersPerMaterial = new Dictionary<Material, List<shader_optionType>>();
            foreach (var combiner in GetCombiners().Where(x => !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(x.Item2.value)))
            {
                List<shader_optionType> combiners;
                if (!combinersPerMaterial.TryGetValue(combiner.Item1, out combiners))
                {
                    combiners = new List<shader_optionType>();
                    combinersPerMaterial.Add(combiner.Item1, combiners);
                }
                combiners.Add(combiner.Item2);
            }
            foreach (var combinersInMaterial in combinersPerMaterial)
            {
                var material = combinersInMaterial.Key;
                var combinerOptions = material.CombinerOptionInfo?.CombinerOptions?.ToDictionary(x => x.Id, x => x);
                if (combinerOptions == null)
                {
                    combinerOptions = new Dictionary<string, CombinerOption>();
                }

                foreach (var combiner in combinersInMaterial.Value)
                {
                    CombinerOption combinerOption;
                    if (!combinerOptions.TryGetValue(combiner.id, out combinerOption) || string.IsNullOrEmpty(combinerOption.FileName))
                    {
                        continue;
                    }
                    list.Add(material.CreateCombinerShaderFilePath(combinerOption.FileName));
                }
            }
            return list.Distinct().ToList();
        }

        private string CreateCombinerShaderFilePath(string name)
        {
            return
                string.IsNullOrEmpty(name) ? string.Empty :
                CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(name) ? CombinerShaderConverterManager.ReservedValue.Unset :
                Path.Combine(Path.Combine(OwnerDocument.FileLocation, CombinerShaderDirectoryName), name + CombinerShaderConverterManager.CombinerShaderFileExtension);
        }

        /// <summary>
        /// CombinerOptionInfo.CombinerOptions から option.id を探し、そのファイル名からファイルパスを生成する。
        /// </summary>
        public string CreateCombinerShaderFilePath(shader_optionType option)
        {
            return
                CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(option?.value) ? string.Empty :
                CreateCombinerShaderFilePath(CombinerOptionInfo?.CombinerOptions?.FirstOrDefault(x => string.Equals(x.Id, option.id, StringComparison.Ordinal))?.FileName);
        }

        #region プレビュー
        public bool OptimizeShader = true;

        public bool OptimizeShaderOnReload;

        // シェーダー送信情報
        public bool ShaderSent;

        // 編集済みシェーダー編集情報
        public bool ModifiedShaderSent;

        // 最適化シェーダー送信状態
        public bool OptimizeShaderSent;

        // 送信済みの最適化シェーダー
        private string BfshaPath;

        public bool AfterModelReloaded(Model.ShaderOptimizeData.MaterialData data)
        {
            bool shaderSent = data != null && data.bfshaPath != null;
            bool modifiedShaderSent = data != null && data.modifiedCount > 0;
            bool optimizeShaderSent = data != null && (data.optimize || data.reusedData != null);
            bool bfshaPathSent = data != null && data.bfshaPath != null;
            bool changed = shaderSent != ShaderSent ||
                modifiedShaderSent != ModifiedShaderSent ||
                optimizeShaderSent != OptimizeShaderSent ||
                bfshaPathSent && BfshaPath != data.bfshaPath;
            ShaderSent = shaderSent;
            ModifiedShaderSent = modifiedShaderSent;
            OptimizeShaderSent = optimizeShaderSent;
            BfshaPath = data != null ? data.bfshaPath : null;
            return changed;
        }
        #endregion

        // 現在のシェーダー割り当て
        public MaterialShaderAssign	MaterialShaderAssign
        {
            get;
            set;
        }

        // シェーダー割り当て
        public MaterialShaderAssignPool MaterialShaderAssignPool = new MaterialShaderAssignPool();

        // 直前のプログラム
        public Dictionary<string, string> LastProgram = new Dictionary<string, string>();

        // この sampler_array.sampler は null にならない
        public sampler_arrayType sampler_array { get; set; }

        public samplerType[] Samplers
        {
            get { return sampler_array.sampler; }
            set { sampler_array.sampler = value; }
        }

        public samplerType[] ResolvedSamplers
        {
            get
            {
                UpdateSamplers();
                return Samplers.Select(x => GetResolvedSampler(x.name)).ToArray();
            }
        }
        public string[] ResolvedSamplerTextureNames
        {
            get
            {
                UpdateSamplers();
                if (MaterialReference?.ParentMaterials?.Any() == true || !ParentMaterialsValidity())
                {
                    return Samplers.Select(x => x.tex_name).ToArray();
                }
                return Samplers.Select(x => GetResolvedSamplerTextureName(x.name)).ToArray();
            }
        }


        // HIO に送信したマテリアルの状況
        public class LastRenderInfoHio
        {
        public Dictionary<string, string[]> StringRenderInfoHio = new Dictionary<string, string[]>();
        public Dictionary<string, int[]> IntRenderInfoHio = new Dictionary<string, int[]>();
        public Dictionary<string, float[]> FloatRenderInfoHio = new Dictionary<string, float[]>();
        public HashSet<string> DefaultEmptyRenderInfoHio = new HashSet<string>();
        }

        public LastRenderInfoHio LastRenderInfo = new LastRenderInfoHio();

        // HIOからのRenderInfo スロットの情報等
        public RenderInfoPackFromHio	RenderInfoPackFromHio{	get; private set; }

        // HIO にRenderInfo を問い合わせたか
        public bool RenderInfoQueried
        {
            get
            {
                lock (this)
                {
                    return renderInfoQueried;
                }
            }
            set
            {
                lock (this)
                {
                    renderInfoQueried = value;
                }
            }
        }
        private bool renderInfoQueried = false;

        public void SetRenderInfoPackFromHio(RenderInfoPackFromHio pack)
        {
            var old = RenderInfoPackFromHio;
            RenderInfoPackFromHio = pack;

            // デフォルトの設定
            // あえてコマンド化していない
            foreach (var info in MaterialShaderAssign.RenderInfos)
            {
                switch (info.type)
                {
                    case render_info_typeType.@int:
                        {
                            var slot = MaterialShaderAssign.ShadingModel.RenderInfoSlots().FirstOrDefault(x => x.name == info.name);
                            if (info.IsDefaultEmpty)
                            {
                                var item = pack.IntItems.FirstOrDefault(x => x.name == info.name);
                                if (item != null && slot != null)
                                {
                                    // Default を上書きする場合は要素数を減らさない
                                    info.internalValues = item.defaults.Select(x => x.ToString())
                                        .Concat(info.internalValues.Skip(item.defaults.Count)).Take(slot.count).ToList();
                                }
                            }

                            if (slot != null && !slot.optional && info.internalValues.Count < slot.count)
                            {
                                var item = pack.IntItems.FirstOrDefault(x => x.name == info.name);
                                if (item != null)
                                {
                                    for (int i = info.internalValues.Count; i < slot.count; i++)
                                    {
                                        if (item.defaults.Count <= i)
                                        {
                                            if (item.hasMinMax)
                                            {
                                                info.internalValues.Add(item.min.ToString());
                                            }
                                            else
                                            {
                                                break;
                                            }
                                        }
                                        else
                                        {
                                            info.internalValues.Add(item.defaults[i].ToString());
                                        }
                                    }
                                }
                            }
                            break;
                        }
                    case render_info_typeType.@float:
                        {
                            var slot = MaterialShaderAssign.ShadingModel.RenderInfoSlots().FirstOrDefault(x => x.name == info.name);
                            if (info.IsDefaultEmpty)
                            {
                                var item = pack.FloatItems.FirstOrDefault(x => x.name == info.name);
                                if (item != null && slot != null)
                                {
                                    // Default を上書きする場合は要素数を減らさない
                                    info.internalValues = item.defaults.Select(x => x.ToString()).Take(slot.count)
                                        .Concat(info.internalValues.Skip(item.defaults.Count)).Take(slot.count).ToList();
                                }
                            }

                            if (slot != null && !slot.optional && info.internalValues.Count < slot.count)
                            {
                                var item = pack.FloatItems.FirstOrDefault(x => x.name == info.name);
                                if (item != null)
                                {
                                    for (int i = info.internalValues.Count; i < slot.count; i++)
                                    {
                                        if (item.defaults.Count <= i)
                                        {
                                            if (item.hasMinMax)
                                            {
                                                info.internalValues.Add(item.min.ToString());
                                            }
                                            else
                                            {
                                                break;
                                            }
                                        }
                                        else
                                        {
                                            info.internalValues.Add(item.defaults[i].ToString());
                                        }
                                    }
                                }
                            }
                            break;
                        }
                    case render_info_typeType.@string:
                        {
                            var slot = MaterialShaderAssign.ShadingModel.RenderInfoSlots().FirstOrDefault(x => x.name == info.name);
                            if (info.IsDefaultEmpty)
                            {
                                var item = pack.StringItems.FirstOrDefault(x => x.name == info.name);
                                if (item != null && slot != null)
                                {
                                    info.internalValues = item.defaults.Take(slot.count).ToList();
                                }
                            }

                            if (slot != null && !slot.optional && info.internalValues.Count < slot.count)
                            {
                                var item = pack.StringItems.FirstOrDefault(x => x.name == info.name);
                                if (item != null)
                                {
                                    for (int i = info.internalValues.Count; i < slot.count; i++)
                                    {
                                        if (item.defaults.Count <= i)
                                        {
                                            if (item.Choices.Any())
                                            {
                                                info.internalValues.Add(item.Choices.First());
                                            }
                                            else
                                            {
                                                break;
                                            }
                                        }
                                        else
                                        {
                                            info.internalValues.Add(item.defaults[i]);
                                        }
                                    }
                                }
                            }
                            break;
                        }
                }
            }

            // 更新フラグの設定
            foreach (var info in MaterialShaderAssign.RenderInfos)
            {
                {
                    switch (info.type)
                    {
                        case render_info_typeType.@int:
                            {
                                var oldCandidates = old.IsNull ? null : old.IntItems.FirstOrDefault(x => x.name == info.name);
                                var newCandidates = pack.IntItems.FirstOrDefault(x => x.name == info.name);

                                if (newCandidates == null)
                                {
                                    break;
                                }

                                if (!newCandidates.hasMinMax)
                                {
                                    if (oldCandidates != null && oldCandidates.hasMinMax)
                                    {
                                        newCandidates.hasMinMax = true;
                                        newCandidates.min = oldCandidates.min;
                                        newCandidates.max = oldCandidates.max;
                                        break;
                                    }
                                }

                                if (info.candidateModified)
                                {
                                    break;
                                }

                                if (oldCandidates == null)
                                {
                                    info.candidateModified = true;
                                    break;
                                }

                                info.candidateModified = oldCandidates.min != newCandidates.min || oldCandidates.max != newCandidates.max;
                            }
                            break;
                        case render_info_typeType.@float:
                            {
                                var oldCandidates = old.IsNull ? null: old.FloatItems.FirstOrDefault(x => x.name == info.name);
                                var newCandidates = pack.FloatItems.FirstOrDefault(x => x.name == info.name);

                                if (newCandidates == null)
                                {
                                    break;
                                }

                                if (!newCandidates.hasMinMax)
                                {
                                    if (oldCandidates != null && oldCandidates.hasMinMax)
                                    {
                                        newCandidates.hasMinMax = true;
                                        newCandidates.min = oldCandidates.min;
                                        newCandidates.max = oldCandidates.max;
                                        break;
                                    }
                                }

                                if (info.candidateModified)
                                {
                                    break;
                                }

                                if (oldCandidates == null)
                                {
                                    info.candidateModified = true;
                                    break;
                                }

                                info.candidateModified = oldCandidates.min != newCandidates.min || oldCandidates.max != newCandidates.max;
                            }
                            break;
                        case render_info_typeType.@string:
                            {
                                if (info.candidateModified)
                                {
                                    break;
                                }

                                var oldCandidates = old.IsNull ? null: old.StringItems.FirstOrDefault(x => x.name == info.name);
                                var newCandidates = pack.StringItems.FirstOrDefault(x => x.name == info.name);

                                if (newCandidates == null)
                                {
                                    break;
                                }

                                if (oldCandidates == null)
                                {
                                    info.candidateModified = true;
                                    break;
                                }

                                info.candidateModified = oldCandidates.values.Count != newCandidates.values.Count ||
                                    !oldCandidates.values.Zip(newCandidates.values,
                                        (x, y) => x.Alias == y.Alias && x.Choice == y.Choice).All(x => x);
                            }
                            break;
                    }
                }
            }

            // 古いものと合わせる
            if (old != null)
            {
                RenderInfoPackFromHio.StringItems = pack.StringItems.Concat(old.StringItems.Where(x => !pack.StringItems.Any(y => y.name == x.name))).ToArray();
                RenderInfoPackFromHio.IntItems = pack.IntItems.Concat(old.IntItems.Where(x => !pack.IntItems.Any(y => y.name == x.name))).ToArray();
                RenderInfoPackFromHio.FloatItems = pack.FloatItems.Concat(old.FloatItems.Where(x => !pack.FloatItems.Any(y => y.name == x.name))).ToArray();
            }

            SetRenderInfoAll.Send(this);
        }

        public original_materialType originalMaterial
        {
            get
            {
                var model = OwnerDocument as Model;
                if (model != null &&
                    model.Data != null &&
                    model.Data.original_material_array != null &&
                    model.Data.original_material_array.original_material != null)
                {
                    return model.Data.original_material_array.original_material.FirstOrDefault(x => x.mat_name == Name);
                }

                return null;
            }
        }

        public bool TextureValidity()
        {
            return !(ResolvedSamplers.Where(sampler => !Owner.ReferenceTexturePaths.ContainsKey(sampler.tex_name))).Any();
        }

        public bool CombinerValidity()
        {
            bool ret = true;
            var combiners = GetCombiners().Where(x => x.Item1 == this && !CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(x.Item2.value)).Select(x => x.Item2).ToArray();
            if (combiners.Any())
            {
                var combinerOptions = CombinerOptionInfo?.CombinerOptions?.ToDictionary(x => x.Id, x => x);
                if (combinerOptions == null)
                {
                    ret = false;
                }
                else
                {
                    foreach (var combiner in combiners)
                    {
                        CombinerOption combinerOption;
                        if (!combinerOptions.TryGetValue(combiner.id, out combinerOption))
                        {
                            ret = false;
                            break;
                        }
                        var combinerShaderFilePath = CreateCombinerShaderFilePath(combinerOption.FileName);
                        ret &= System.IO.File.Exists(combinerShaderFilePath);
                    }
                }
            }
            return ret;

        }

        public List<Material> ParentMaterials => ParentMaterialsValidity() ? GetParentMaterials() : new List<Material>();
        public List<Material> ChildMaterials => ParentMaterialsValidity() ? GetChildMaterials() : new List<Material>();

        /// <summary>
        /// 親マテリアルのリストを返す。
        /// 整合性チェックをしていないので、通常は ParentMaterials を使用する。
        /// </summary>
        /// <returns>
        /// 親マテリアルのリスト。
        /// 開かれていない親マテリアルはnullとして返す。
        /// </returns>
        internal List<Material> GetParentMaterials()
        {
            var parentMaterials = new List<Material>();
            if (MaterialReference?.ParentMaterials?.Any() != true)
            {
                return parentMaterials;
            }

            foreach (var parentMaterial in MaterialReference.ParentMaterials)
            {
                if (string.IsNullOrEmpty(OwnerDocument.FilePath)) // モデルのパスが未確定の場合
                {
                    parentMaterials.Add(null);
                }
                else if (string.IsNullOrEmpty(parentMaterial.Path))
                {
                    parentMaterials.Add(Owner.Materials.FirstOrDefault(x => x.Name == parentMaterial.MaterialName));
                }
                else
                {
                    // まず子モデルからの相対で探し、見つからなければ親マテリアルパスから探す
                    var parentDocFullPath = IfApplyBaseMaterialUtility.GetParentModelFullPath(parentMaterial.Path, OwnerDocument.FilePath);
                    if (string.Equals(System.IO.Path.GetExtension(parentMaterial.Path), G3dPath.MaterialBinaryExtension, StringComparison.OrdinalIgnoreCase) ||
                        string.Equals(System.IO.Path.GetExtension(parentMaterial.Path), G3dPath.MaterialTextExtension, StringComparison.OrdinalIgnoreCase))
                    {
                        var materialDoc =
                            DocumentManager.SeparateMaterials.FirstOrDefault(
                                x => string.Equals(x.FilePath, parentDocFullPath, StringComparison.OrdinalIgnoreCase));
                        if (materialDoc == null)
                        {
                            parentDocFullPath = SearchParentModelFromParentMaterialPaths(parentMaterial.Path);
                            materialDoc = DocumentManager.SeparateMaterials.FirstOrDefault(
                                        x => string.Equals(x.FilePath, parentDocFullPath, StringComparison.OrdinalIgnoreCase));
                        }
                        parentMaterials.Add(materialDoc?.Materials.FirstOrDefault(x => x.Name == parentMaterial.MaterialName));
                    }
                    else
                    {
                        var model =
                            DocumentManager.Models.FirstOrDefault(
                                x => string.Equals(x.FilePath, parentDocFullPath, StringComparison.OrdinalIgnoreCase));
                        if (model == null)
                        {
                            parentDocFullPath = SearchParentModelFromParentMaterialPaths(parentMaterial.Path);
                            model = DocumentManager.Models.FirstOrDefault(
                                        x => string.Equals(x.FilePath, parentDocFullPath, StringComparison.OrdinalIgnoreCase));
                        }
                        parentMaterials.Add(model?.Materials.FirstOrDefault(x => x.Name == parentMaterial.MaterialName));
                    }
                }
            }
            return parentMaterials;
        }

        /// <summary>
        /// 先祖マテリアルのリストを返す。
        /// 親マテリアル検索順に全ての先祖マテリアルを返す
        /// </summary>
        /// <returns>
        /// 先祖マテリアルのリスト。
        /// 開かれていない先祖マテリアルはnullとして返す。
        /// </returns>
        public List<Material> GetAllAncestorMaterials()
        {
            var allAncestorMaterials = new List<Material>();
            Action<Material, List<Material>> action = null;
            action = (item, list) =>
            {
                list.Add(item);
                foreach (var subitem in item.ParentMaterials.AsEnumerable().Reverse())
                {
                    action(subitem, list);
                }
            };

            foreach (var material in ParentMaterials.AsEnumerable().Reverse())
            {
                action(material, allAncestorMaterials);
            }
            return allAncestorMaterials.ToList();
        }

        private List<Material> GetChildMaterials()
        {
            return DocumentManager.Materials.Where(x => x.ParentMaterials.Contains(this)).ToList();
        }

        public List<Material> GetAllDescendantMaterials()
        {
            var allDescendantMaterials = new List<Material>();
            Action<Material, List<Material>> action = null;
            action = (item, list) =>
            {
                list.Add(item);
                foreach (var subitem in item.ChildMaterials)
                {
                    action(subitem, list);
                }
            };

            foreach (var material in ChildMaterials)
            {
                action(material, allDescendantMaterials);
            }
            return allDescendantMaterials.ToList();
        }


        public List<Material> GetAllDescendantAttribAssignMaterials(string paramName)
        {
            var attribAssign = MaterialShaderAssign.AttribAssigns.FirstOrDefault(x => x.id == paramName);
            return (attribAssign != null) ? GetAllDescendantMaterials(attribAssign) : new List<Material>();
        }

        public List<Material> GetAllDescendantShaderOptionMaterials(string paramName)
        {
            var shaderOption = MaterialShaderAssign.ShaderOptions.FirstOrDefault(x => x.id == paramName);
            return (shaderOption != null) ? GetAllDescendantMaterials(shaderOption) : new List<Material>();
        }

        public List<Material> GetAllDescendantSamplerAssignMaterials(string paramName)
        {
            var samplerAssign = MaterialShaderAssign.SamplerAssigns.FirstOrDefault(x => x.id == paramName);
            return (samplerAssign != null) ? GetAllDescendantMaterials(samplerAssign) : new List<Material>();
        }

        public List<Material> GetAllDescendantShaderParamMaterials(string paramName)
        {
            var shaderParam = MaterialShaderAssign.ShaderParams.FirstOrDefault(x => x.id == paramName);
            return (shaderParam != null) ? GetAllDescendantMaterials(shaderParam) : new List<Material>();
        }
        public List<Material> GetAllDescendantRenderInfoMaterials(string paramName)
        {
            var renderInfo = MaterialShaderAssign.RenderInfos.FirstOrDefault(x => x.name == paramName);
            return (renderInfo != null) ? GetAllDescendantMaterials(renderInfo) : new List<Material>();
        }

        public List<Material> GetAllDescendantMaterials<T>(T shaderValue) where T : class
        {
            return GetAllDescendantMaterials().Where(x => x.SearchParentValueMaterial(shaderValue) == this).Distinct().ToList();
        }

        public IEnumerable<Material> GetAllDescendantSamplerMaterials(string samplerName)
        {
            if (Samplers.All(x => x.name != samplerName))
            {
                return new List<Material>();
            }

            return GetAllDescendantMaterials().Where(x => x.Samplers.Any(y => y.name == samplerName)).Distinct();
        }

        public List<Material> GetAllDescendantSamplerParamMaterials(string samplerName, SamplerParamID id)
        {
            if (Samplers.All(x => x.name != samplerName))
            {
                return new List<Material>();
            }

            return GetAllDescendantSamplerMaterials(samplerName).Where(x => x.SearchParentSamplerParamMaterial(samplerName, id) == this).Distinct().ToList();
        }


        internal IEnumerable<MaterialReferenceBehaviorItem> GetAllReferenceBehaviorsInGroup(string groupName, bool removeNull = true)
        {
            var shadingModel = MaterialShaderAssign?.ShadingModel;
            if (shadingModel == null)
            {
                return null;
            }

            var materialReferenceBehaviorItems =
                shadingModel.Attributes()
                    .Where(x => x.Group() == groupName)
                    .Select(x => GetReferenceBehaviorAttribAssign(x.id))
                .Concat(shadingModel.Options()
                    .Where(x => x.Group() == groupName)
                    .Select(x => GetReferenceBehaviorShaderOption(x.id)))
                .Concat(shadingModel.Samplers()
                    .Where(x => x.Group() == groupName)
                    .Select(x => GetReferenceBehaviorSamplerAssign(x.id)))
                .Concat(shadingModel.RenderInfoSlots()
                    .Where(x => x.Group() == groupName)
                    .Select(x => GetReferenceBehaviorRenderInfo(x.name)))
                .Concat(shadingModel.MaterialUniforms()
                    .Where(x => x.Group() == groupName)
                    .Select(x => GetReferenceBehaviorShaderParam(x.id)));
            return removeNull ? materialReferenceBehaviorItems.Where(x => x != null) : materialReferenceBehaviorItems;
        }


        public bool ParentMaterialsValidity()
        {
            var parentMaterials = GetParentMaterials();
            return !parentMaterials.Contains(null) &&
                   CheckParentMaterialsReference() &&
                   parentMaterials.All(parentMaterial => parentMaterial.ParentMaterialsValidity());
        }

        public bool CheckParentMaterialsReference(Material material, List<Material> list)
        {
            if (material == null || list.Contains(material))
            {
                return false;
            }
            list.Add(material);
            return material.GetParentMaterials().All(x => CheckParentMaterialsReference(x, list));
        }

        public bool CheckParentMaterialsReference()
        {
            return CheckParentMaterialsReference(new List<Material>());
        }

        public bool CheckParentMaterialsReference(List<Material> list)
        {
            return CheckParentMaterialsReference(this, list);
        }
        public nw3de_MaterialReference MaterialReference
        {
            get { return ToolData.MaterialReference; }
            set { ToolData.MaterialReference = value; }
            }

        public nw3de_CombinerOptionInfo CombinerOptionInfo
        {
            get { return ToolData.CombinerOptionInfo; }
            set { ToolData.CombinerOptionInfo = value; }
        }

        // マテリアル参照を解決した条件を表す
        public enum ValueResolvedState
        {
            Override,
            Refer,
            Default,
            ForceRefer,
            ForceOverride,
            None,
        }

        public static MaterialReferenceBehaviorItem GetDefaultReferenceBehaviorItem(string paramName)
        {
            return new MaterialReferenceBehaviorItem()
                {
                    Id = paramName,
                    Value = ShaderItemValueState.Refer,
                    ChildRestriction = ShaderItemRestrictionState.None,
                };
        }

        public MaterialReferenceBehaviorItem GetReferenceBehaviorAttribAssign(string paramName)
        {
            return MaterialReference?.AttribAssignBehaviors?.FirstOrDefault(x => x.Id == paramName);
        }
        public MaterialReferenceBehaviorItem GetReferenceBehaviorShaderOption(string paramName)
        {
            return MaterialReference?.ShaderOptionBehaviors?.FirstOrDefault(x => x.Id == paramName);
        }
        public MaterialReferenceBehaviorItem GetReferenceBehaviorSamplerAssign(string paramName)
        {
            return MaterialReference?.SamplerAssignBehaviors?.FirstOrDefault(x => x.Id == paramName);
        }
        public MaterialReferenceBehaviorItem GetReferenceBehaviorShaderParam(string paramName)
        {
            return MaterialReference?.ShaderParamBehaviors?.FirstOrDefault(x => x.Id == paramName);
        }
        public MaterialReferenceBehaviorItem GetReferenceBehaviorRenderInfo(string paramName)
        {
            return MaterialReference?.RenderInfoBehaviors?.FirstOrDefault(x => x.Id == paramName);
        }
        public MaterialReferenceSamplerBehaviors GetReferenceBehaviorSamplers()
        {
            return MaterialReference?.SamplerBehaviors;
        }

        public SamplerBehaviorItem GetReferenceBehaviorSampler(string samplerName)
        {
            return GetReferenceBehaviorSamplers()?.SamplerBehaviorItems?.FirstOrDefault(x => x.Id == samplerName);
        }
        public MaterialReferenceBehaviorItem GetReferenceBehaviorSamplerParam(string samplerName, SamplerParamID id)
        {
            return GetReferenceBehaviorSampler(samplerName)?.SamplerParamBehaviors?.FirstOrDefault(x => x.Id == Enum.GetName(typeof(SamplerParamID), id));
        }
        // このマテリアルのサンプラパラメータの制限を返す。サンプラが存在しない場合はnullを返す
        public ShaderItemRestrictionState? GetSamplerParamChildRestrictionState(string samplerName, SamplerParamID id)
        {
            if (Samplers.All(x => x.name != samplerName))
            {
                return null;
            }
            return GetReferenceBehaviorSamplerParam(samplerName, id)?.ChildRestriction ?? ShaderItemRestrictionState.None;
        }

        public ShaderItemRestrictionState? GetResolvedSamplerParamChildRestrictionState(string samplerName, SamplerParamID id, bool ignoreSelf = true)
        {
            ShaderItemRestrictionState? restriction = null;
            if (!ignoreSelf)
            {
                restriction = GetSamplerParamChildRestrictionState(samplerName, id);
            }

            // 祖先を ChildRestriction == ForceRefer, ForceOverride が見つかるまで探す
            // 全てでChildRestriction == null が返された場合は 親サンプラが存在しない場合なので区別のためにnullを返す
            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                var parentRestriction = parentMaterial.GetResolvedSamplerParamChildRestrictionState(samplerName, id, false);
                switch (parentRestriction)
                {
                    case ShaderItemRestrictionState.None:
                        // Noneが帰ってきた場合は親サンプラが存在しているので返値に反映する
                        if (restriction == null)
                        {
                            restriction = parentRestriction;
                        }
                        break;
                    case ShaderItemRestrictionState.ForceRefer:
                    case ShaderItemRestrictionState.ForceOverride:
                        return parentRestriction;
                    default:
                        break;
                }
            }

            return restriction;
        }

        public bool IsSamplerParamValueEditable(string samplerName, SamplerParamID id)
        {
            if (!ParentMaterials.Any())
            {
                return true;
            }

            var shaderItemValueState = GetReferenceBehaviorSamplerParam(samplerName, id)?.Value ?? ShaderItemValueState.Refer;
            return shaderItemValueState == ShaderItemValueState.Override;
        }

        public bool IsResolvedSamplerParamValueEditable(string samplerName, SamplerParamID id)
        {
            var restrictionState = GetResolvedSamplerParamChildRestrictionState(samplerName, id);
            switch (restrictionState)
            {
                case ShaderItemRestrictionState.ForceOverride:
                    return true;
                case ShaderItemRestrictionState.ForceRefer:
                    return false;
                case ShaderItemRestrictionState.None:
                {
                    var shaderItemValueState = GetReferenceBehaviorSamplerParam(samplerName, id)?.Value ?? ShaderItemValueState.Refer;
                    return shaderItemValueState == ShaderItemValueState.Override;
                }
                default: // null の場合は親サンプラーが見つからない場合なので、Default以外は編集可能にする
                {
                    var shaderItemValueState = GetReferenceBehaviorSamplerParam(samplerName, id)?.Value ?? ShaderItemValueState.Refer;
                    return shaderItemValueState != ShaderItemValueState.Default;
                }
            }
        }

        public bool IsResolvedReferenceBehavioSamplerParamEditable(string samplerName, SamplerParamID id)
        {
            var restrictionState = GetResolvedSamplerParamChildRestrictionState(samplerName, id);
            switch (restrictionState)
            {
                case ShaderItemRestrictionState.ForceRefer:
                case ShaderItemRestrictionState.ForceOverride:
                    return false;
                case ShaderItemRestrictionState.None:
                default:
                    return true;
            }
        }

        private List<samplerType> GetParentRequiredForChildSamplers()
        {
            var samplers = new List<samplerType>();
            if (!ParentMaterialsValidity())
            {
                return samplers;
            }

            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                samplers.AddRange(parentMaterial.GetParentRequiredForChildSamplers().Where(x => samplers.All(y => x.name != y.name)));
                var parentSamplers = parentMaterial.Samplers;
                foreach (var parentSampler in parentSamplers.Where(x => samplers.All(y => x.name != y.name)))
                {
                    var samplerBehavior = parentMaterial.GetReferenceBehaviorSampler(parentSampler.name);
                    if (samplerBehavior?.IsRequiredForChild == true)
                    {
                        if (samplers.All(x => x.name != parentSampler.name))
                        {
                            samplers.Add(parentSampler);
                        }
                    }
                }
            }

            return samplers;
        }

        public List<samplerType> GetParentRestrictedSamplers(bool checkValidity = true)
        {
            if (checkValidity && !ParentMaterialsValidity())
            {
                return null;
            }

            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                var restrictedSamplers = parentMaterial.GetParentRestrictedSamplers(false);
                if (restrictedSamplers?.Any() == true)
                {
                    return restrictedSamplers;
                }
                if (parentMaterial.GetChildSamplerStructureRestriction() == ChildSamplerStructureRestrictionState.DisallowAddOwnSampler)
                {
                    return ObjectUtility.Clone(parentMaterial.Samplers).ToList();
                }
            }

            return null;
        }

        public List<string> GetParentRestrictedSamplerNames(bool checkValidity = true)
        {
            if (checkValidity && !ParentMaterialsValidity())
            {
                return null;
            }

            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                var restrictedSamplerNames = parentMaterial.GetParentRestrictedSamplerNames(false);
                if (restrictedSamplerNames?.Any() == true)
                {
                    return restrictedSamplerNames;
                }
                if (parentMaterial.GetChildSamplerStructureRestriction() == ChildSamplerStructureRestrictionState.DisallowAddOwnSampler)
                {
                    return parentMaterial.Samplers.Select(x => x.name).ToList();
                }
            }

            return null;
        }

        // サンプラーのディープコピー
        private static List<samplerType> CloneSamplers(IEnumerable<samplerType> samplers)
        {
            var count = samplers.Count();
            var newSamplers = new List<samplerType>(count);
            newSamplers.AddRange(samplers.Select(sampler => new samplerType
            {
                name = sampler.name,
                hint = sampler.hint,
                tex_name = sampler.tex_name,
                index_hint = sampler.index_hint,
                sampler_index = sampler.sampler_index,
                wrap = new wrapType {u = sampler.wrap.u, v = sampler.wrap.v, w = sampler.wrap.w},
                filter = new filterType {mag = sampler.filter.mag, min = sampler.filter.min, mip = sampler.filter.mip, max_aniso = sampler.filter.max_aniso,},
                lod = new lodType {bias = sampler.lod.bias, max = sampler.lod.max, min = sampler.lod.min}
            }));
            return newSamplers;
        }

        // マテリアル参照のサンプラーの更新
        // 親マテリアルから継承するサンプラが存在しない場合には追加、追加が禁止されていたら、削除する
        public bool UpdateSamplers(EditCommandSet commandSet = null)
        {
            if (!ParentMaterialsValidity())
            {
                return false;
            }

            // 親マテリアルから継承するサンプラが存在しない場合に追加する
            var addSamplers = GetParentRequiredForChildSamplers().Where(x => Samplers.All(y => x.name != y.name)).ToList();
            var disallowAddSampler = DisallowAddSampler(false);
            if (!addSamplers.Any() && !disallowAddSampler)
            {
                return false;
            }

            var newSamplers = CloneSamplers(Samplers);
            newSamplers.AddRange(CloneSamplers(addSamplers));

            // 追加が禁止されていたら、親マテリアルにない不要なサンプラは削除する
            var samplerRemoved = false;
            if (disallowAddSampler)
            {
                var restrictedSamplers = GetParentRestrictedSamplers(false);
                var count = newSamplers.Count;
                newSamplers = newSamplers.Where(x => restrictedSamplers.Any(y => x.name == y.name)).ToList();
                samplerRemoved = newSamplers.Count != count;
            }

            if (!addSamplers.Any() && !samplerRemoved)
            {
                return false;
            }

            // サンプラーが全て一致する場合は何もしない
            if (Samplers.Length == newSamplers.Count)
            {
                var isSame = true;
                for (var i = 0; i < Samplers.Length; i++)
                {
                    var sampler = Samplers[i];
                    var newSampler = newSamplers[i];
                    if (sampler.name != newSampler.name)
                    {
                        isSame = false;
                        break;
                    }
                }
                if (isSame)
                {
                    return false;
                }
            }

            var command = CreateEditCommand_SamplerArray(new GuiObjectGroup(this), new List<samplerType[]> { newSamplers.ToArray() }, false);
            if (commandSet != null)
            {
                commandSet.Add(command);
            }
            else
            {
                TheApp.CommandManager.Execute(command);
            }
            return true;
        }

        public bool DisallowAddSampler(bool checkValidity = true)
        {
            var structureRestriction = GetResolvedChildSamplerStructureRestriction();
            return structureRestriction == ChildSamplerStructureRestrictionState.DisallowAddOwnSampler;
        }

        public bool IsRequiredChildSampler(string samplerName, bool checkValidity = true)
        {
            if (checkValidity && !ParentMaterialsValidity())
            {
                return false;
            }

            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                if (parentMaterial.GetReferenceBehaviorSampler(samplerName)?.IsRequiredForChild == true || parentMaterial.IsRequiredChildSampler(samplerName, false))
                {
                    return true;
                }
            }

            return false;
        }

        public bool IsReferenceSampler(string samplerName, bool checkValidity = true)
        {
            if (checkValidity && !ParentMaterialsValidity())
            {
                return false;
            }

            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                if (parentMaterial.Samplers.Any(x => x.name == samplerName) || parentMaterial.IsReferenceSampler(samplerName, false))
                {
                    return true;
                }
            }

            return false;
        }

        public IEnumerable<string> GetParentSamplerNames(bool checkValidity = true)
        {
            //var parentSamplers = new List<string>();
            if (checkValidity && !ParentMaterialsValidity())
            {
                return new string[0];
            }
            return Enumerable.Reverse(ParentMaterials)
                .SelectMany(material => material.Samplers.Select(x => x.name).Union(material.GetParentSamplerNames(false)));
        }

        public void SetReferenceBehaviorAttribAssign(MaterialReferenceBehaviorItem referenceBehaviorItem)
        {
            var materialReference = ObjectUtility.Clone(MaterialReference) ?? new nw3de_MaterialReference();
            if (materialReference.AttribAssignBehaviors == null)
            {
                materialReference.AttribAssignBehaviors = new List<MaterialReferenceBehaviorItem>();
            }
            SetMaterialReferenceBehaviorItem(referenceBehaviorItem, materialReference, materialReference.AttribAssignBehaviors);
        }

        public void SetReferenceBehaviorShaderOption(MaterialReferenceBehaviorItem referenceBehaviorItem)
        {
            var materialReference = ObjectUtility.Clone(MaterialReference) ?? new nw3de_MaterialReference();
            if (materialReference.ShaderOptionBehaviors == null)
            {
                materialReference.ShaderOptionBehaviors = new List<MaterialReferenceBehaviorItem>();
            }
            SetMaterialReferenceBehaviorItem(referenceBehaviorItem, materialReference, materialReference.ShaderOptionBehaviors);
        }
        public void SetReferenceBehaviorSamplerAssign(MaterialReferenceBehaviorItem referenceBehaviorItem)
        {
            var materialReference = ObjectUtility.Clone(MaterialReference) ?? new nw3de_MaterialReference();
            if (materialReference.SamplerAssignBehaviors == null)
            {
                materialReference.SamplerAssignBehaviors = new List<MaterialReferenceBehaviorItem>();
            }
            SetMaterialReferenceBehaviorItem(referenceBehaviorItem, materialReference, materialReference.SamplerAssignBehaviors);
        }
        public void SetReferenceBehaviorShaderParam(MaterialReferenceBehaviorItem referenceBehaviorItem)
        {
            var materialReference = ObjectUtility.Clone(MaterialReference) ?? new nw3de_MaterialReference();
            if (materialReference.ShaderParamBehaviors == null)
            {
                materialReference.ShaderParamBehaviors = new List<MaterialReferenceBehaviorItem>();
            }
            SetMaterialReferenceBehaviorItem(referenceBehaviorItem, materialReference, materialReference.ShaderParamBehaviors);
        }
        public void SetReferenceBehaviorRenderInfo(MaterialReferenceBehaviorItem referenceBehaviorItem)
        {
            var materialReference = ObjectUtility.Clone(MaterialReference) ?? new nw3de_MaterialReference();
            if (materialReference.RenderInfoBehaviors == null)
            {
                materialReference.RenderInfoBehaviors = new List<MaterialReferenceBehaviorItem>();
            }
            SetMaterialReferenceBehaviorItem(referenceBehaviorItem, materialReference, materialReference.RenderInfoBehaviors);
        }

        public EditCommand CreateSetReferenceBehaviorSamplerCommand(MaterialReferenceBehaviorItem referenceBehaviorItem, string samplerName)
        {
            var materialReference = ObjectUtility.Clone(MaterialReference) ??
                                    new nw3de_MaterialReference();
            if (materialReference.SamplerBehaviors == null)
            {
                materialReference.SamplerBehaviors = new MaterialReferenceSamplerBehaviors();
            }
            if (materialReference.SamplerBehaviors.SamplerBehaviorItems == null)
            {
                materialReference.SamplerBehaviors.SamplerBehaviorItems = new List<SamplerBehaviorItem>();
            }
            var samplerBehaviorItem = materialReference.SamplerBehaviors.SamplerBehaviorItems.FirstOrDefault(x => x.Id == samplerName);
            if (samplerBehaviorItem == null)
            {
                samplerBehaviorItem = new SamplerBehaviorItem() { Id = samplerName };
                materialReference.SamplerBehaviors.SamplerBehaviorItems.Add(samplerBehaviorItem);
            }

            if (samplerBehaviorItem.SamplerParamBehaviors == null)
            {
                samplerBehaviorItem.SamplerParamBehaviors = new List<MaterialReferenceBehaviorItem>();
            }
            return CreateSetMaterialReferenceBehaviorItemCommand(referenceBehaviorItem, materialReference, samplerBehaviorItem.SamplerParamBehaviors);
        }

        public EditCommand CreateSetMaterialReferenceBehaviorItemCommand(MaterialReferenceBehaviorItem shaderItem,
            nw3de_MaterialReference materialReferenceBehavior, List<MaterialReferenceBehaviorItem> referenceBehaviorItems)
        {
            if (!AddOrUpdateMaterialReferenceBehaviorItem(shaderItem, referenceBehaviorItems))
            {
                return new EditCommandSet();
            }
            return CreateEditCommand_MaterialReference(new GuiObjectGroup(this), materialReferenceBehavior);

        }

        public void SetMaterialReferenceBehaviorItem(MaterialReferenceBehaviorItem shaderItem, nw3de_MaterialReference materialReference, List<MaterialReferenceBehaviorItem> referenceBehaviorItems)
        {
            TheApp.CommandManager.Add(CreateSetMaterialReferenceBehaviorItemCommand(shaderItem, materialReference, referenceBehaviorItems).Execute());
        }

        public void SetSamplerBehaviorItem(SamplerBehaviorItem behaviorItem)
        {
            var command = CreateSetSamplerBehaviorItemCommand(behaviorItem);
            TheApp.CommandManager.Add(command.Execute());
        }

        internal App.Command.ICommand CreateSetSamplerBehaviorItemCommand(SamplerBehaviorItem behaviorItem)
        {
            var samplerName = behaviorItem.Id;
            var materialReference = ObjectUtility.Clone(MaterialReference) ??
                                    new nw3de_MaterialReference();
            if (materialReference.SamplerBehaviors == null)
            {
                materialReference.SamplerBehaviors = new MaterialReferenceSamplerBehaviors();
            }
            if (materialReference.SamplerBehaviors.SamplerBehaviorItems == null)
            {
                materialReference.SamplerBehaviors.SamplerBehaviorItems = new List<SamplerBehaviorItem>();
            }

            var samplerBehaviorItems = materialReference.SamplerBehaviors.SamplerBehaviorItems;
            var index = samplerBehaviorItems.FindIndex(x => x.Id == samplerName);

            if (index >= 0)
            {
                samplerBehaviorItems[index] = ObjectUtility.Clone(behaviorItem);
            }
            else
            {
                samplerBehaviorItems.Add(ObjectUtility.Clone(behaviorItem));
            }

            return CreateEditCommand_MaterialReference(new GuiObjectGroup(this), materialReference);
        }


        public void SetSamplerBehaviors(MaterialReferenceSamplerBehaviors samplerBehaviors)
        {
            var materialReference = ObjectUtility.Clone(MaterialReference) ??
                                    new nw3de_MaterialReference();
            materialReference.SamplerBehaviors = samplerBehaviors;
            TheApp.CommandManager.Add(
                CreateEditCommand_MaterialReference(new GuiObjectGroup(this), materialReference).Execute());
        }

        // リストにBehaviorItemを追加または更新する。
        // 変更がない場合はFalse
        internal static bool AddOrUpdateMaterialReferenceBehaviorItem(
            MaterialReferenceBehaviorItem shaderItem,
            List<MaterialReferenceBehaviorItem> referenceBehaviorItems)
        {
            var isDefault = shaderItem.Value == ShaderItemValueState.Refer
                && shaderItem.ChildRestriction == ShaderItemRestrictionState.None;
            var index = referenceBehaviorItems.FindIndex(x => x.Id == shaderItem.Id);
            if (index >= 0)
            {
                if (isDefault)
                {
                    referenceBehaviorItems.RemoveAt(index);
                }
                else
                {
                    var orgItem = referenceBehaviorItems[index];
                    if (orgItem.Value == shaderItem.Value
                        && orgItem.ChildRestriction == shaderItem.ChildRestriction)
                    {
                        return false;
                    }
                    referenceBehaviorItems[index] = shaderItem;
                }
            }
            else
            {
                if (isDefault)
                {
                    return false;
                }
                else
                {
                    referenceBehaviorItems.Add(shaderItem);
                }
            }
            referenceBehaviorItems.Sort((l, r) => String.CompareOrdinal(l.Id, r.Id));
            return true;
        }

        public ChildSamplerStructureRestrictionState GetChildSamplerStructureRestriction()
        {
            return GetReferenceBehaviorSamplers()?.ChildSamplerStructureRestriction ?? ChildSamplerStructureRestrictionState.None;
        }

        public ChildSamplerStructureRestrictionState GetResolvedChildSamplerStructureRestriction(bool ignoreSelf = true)
        {
            var restriction = ChildSamplerStructureRestrictionState.None;
            if (!ignoreSelf)
            {
                restriction = GetChildSamplerStructureRestriction();
            }

            // 祖先を SamplerStructureRestriction != None が見つかるまで探す
            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                var parentRestriction = parentMaterial.GetResolvedChildSamplerStructureRestriction(false);
                if (parentRestriction != ChildSamplerStructureRestrictionState.None)
                {
                    return parentRestriction;
                }
            }

            return restriction;
        }

        public const string SamplerParamID_tex_name = "tex_name";
        public const string SamplerParamID_wrap_u = "wrap_u";
        public const string SamplerParamID_wrap_v = "wrap_v";
        public const string SamplerParamID_wrap_w = "wrap_w";
        public const string SamplerParamID_filter_mag = "filter_mag";
        public const string SamplerParamID_filter_min = "filter_min";
        public const string SamplerParamID_filter_mip = "filter_mip";
        public const string SamplerParamID_filter_max_aniso = "filter_max_aniso";
        public const string SamplerParamID_lod_min = "lod_min";
        public const string SamplerParamID_lod_max = "lod_max";
        public const string SamplerParamID_lod_bias = "lod_bias";

        public enum SamplerParamID
        {
            tex_name,
            wrap_u,
            wrap_v,
            wrap_w,
            filter_mag,
            filter_min,
            filter_mip,
            filter_max_aniso,
            lod_min,
            lod_max,
            lod_bias,
        }


        public object GetSamplerParamValue(string samplerName, SamplerParamID id)
        {
            var sampler = Samplers?.FirstOrDefault(x => x.name == samplerName);
            if (sampler == null)
            {
                return null;
            }
            switch (id)
            {
                case SamplerParamID.tex_name:
                    return sampler.tex_name;
                case SamplerParamID.wrap_u:
                    return sampler.wrap.u;
                case SamplerParamID.wrap_v:
                    return sampler.wrap.v;
                case SamplerParamID.wrap_w:
                    return sampler.wrap.w;
                case SamplerParamID.filter_mag:
                    return sampler.filter.mag;
                case SamplerParamID.filter_min:
                    return sampler.filter.min;
                case SamplerParamID.filter_mip:
                    return sampler.filter.mip;
                case SamplerParamID.filter_max_aniso:
                    return sampler.filter.max_aniso;
                case SamplerParamID.lod_min:
                    return sampler.lod.min;
                case SamplerParamID.lod_max:
                    return sampler.lod.max;
                case SamplerParamID.lod_bias:
                    return sampler.lod.bias;
            }
            return null;
        }
        public Material SearchParentSamplerParamMaterial(string samplerName, SamplerParamID id)
        {
            bool found;
            Material foundMaterial;
            ValueResolvedState valueResolvedState;
            var value = GetResolvedSamplerParamValue(samplerName, id, out found, out foundMaterial, out valueResolvedState);
            return foundMaterial;
        }


        public object GetResolvedSamplerParamValue(string samplerName, SamplerParamID id)
        {
            bool found;
            Material foundMaterial;
            ValueResolvedState valueResolvedState;
            return GetResolvedSamplerParamValue(samplerName, id, out found, out foundMaterial, out valueResolvedState);
        }

        public object GetResolvedSamplerParamValue(string samplerName, SamplerParamID id, out ValueResolvedState valueResolvedState)
        {
            bool found;
            Material foundMaterial;
            return GetResolvedSamplerParamValue(samplerName, id, out found, out foundMaterial, out valueResolvedState);
        }

        public object GetResolvedSamplerParamValue(string samplerName, SamplerParamID id, out bool found, out Material foundMaterial, out ValueResolvedState valueResolvedState)
        {
            foundMaterial = this;

            var samplerParamBehavior = GetReferenceBehaviorSamplerParam(samplerName, id);
            var childRestrictionState = GetResolvedSamplerParamChildRestrictionState(samplerName, id);

            ShaderItemValueState valueState;
            var hasParentSampler = childRestrictionState.HasValue;
            switch (childRestrictionState)
            {
                case ShaderItemRestrictionState.ForceRefer:
                    // 親マテリアルまたはその祖先のどれかがForceReferの場合はForceReferになっている親マテリアルを探し
                    // その値を採用する（その親マテリアル自身が持つ値ではなく、参照を解決した値を採用する）
                    Debug.Assert(ParentMaterials.Any(), "ForceRefer: ParentMaterial is not exist.");

                    // 後に指定されたベースマテリアルから検索
                    foreach (var parentMaterial in Enumerable.Reverse(ParentMaterials))
                    {
                        // この親マテリアルまたはその祖先がForceReferの場合
                        var resolvedRestriction = parentMaterial.GetResolvedSamplerParamChildRestrictionState(samplerName, id, ignoreSelf: false);
                        if (resolvedRestriction == ShaderItemRestrictionState.ForceRefer)
                        {
                            // この親マテリアル自身がForceReferの場合、その値を採用する
                            var restriction = parentMaterial.GetSamplerParamChildRestrictionState(samplerName, id);
                            if (restriction == ShaderItemRestrictionState.ForceRefer)
                            {
                                var parentShaderParam = parentMaterial.GetResolvedSamplerParamValue(samplerName, id, out found, out foundMaterial, out valueResolvedState);
                                found = true;
                                valueResolvedState = ValueResolvedState.ForceRefer;
                                return parentShaderParam;
                            }
                        }
                    }
                    valueState = ShaderItemValueState.Refer;
                    break;
                case ShaderItemRestrictionState.ForceOverride:
                    found = true;
                    valueResolvedState = ValueResolvedState.ForceOverride;
                    return GetSamplerParamValue(samplerName, id);
                case ShaderItemRestrictionState.None:
                default:
                    valueState = samplerParamBehavior?.Value ?? ShaderItemValueState.Refer;
                    break;
            }

            var value = GetSamplerParamValue(samplerName, id);

            switch (valueState)
            {
                case ShaderItemValueState.Override:
                    found = true;
                    valueResolvedState = ValueResolvedState.Override;
                    return value;
                case ShaderItemValueState.Default:
                    found = true;
                    valueResolvedState = ValueResolvedState.Default;
                    return null;
                case ShaderItemValueState.Refer:
                default:
                    object parentShaderParam = null;
                    if (hasParentSampler && ParentMaterials.Any())
                    {
                        // 後に指定されたベースマテリアルから検索
                        foreach (var parentMaterial in Enumerable.Reverse(ParentMaterials))
                        {
                            parentShaderParam = parentMaterial.GetResolvedSamplerParamValue(samplerName, id, out found, out foundMaterial, out valueResolvedState);
                            if (found)
                            {
                                valueResolvedState = ValueResolvedState.Refer;
                                return parentShaderParam;
                            }
                        }

                        // 値が確定しない場合(全てRefer) found は false で 最後の値を返す
                        found = false;
                        valueResolvedState = ValueResolvedState.Refer;
                        return parentShaderParam;
                    }
                    else
                    {
                        // 親マテリアルがない場合は found は false で 自分の値を返す
                        found = false;
                        valueResolvedState = ValueResolvedState.Override;
                        return value;
                    }
            }
        }

        // 全てのパラメータを解決したサンプラー情報を返す
        public samplerType GetResolvedSampler(string samplerName)
        {
            var sampler = Samplers?.FirstOrDefault(x => x.name == samplerName);
            Debug.Assert(sampler != null, "sampler != null");

            if (!ParentMaterials.Any() || !IsReferenceSampler(samplerName) || !ParentMaterialsValidity())
            {
                return sampler;
            }

            // デフォルト値を入れておく
            var newSampler = new samplerType
            {
                name = samplerName,
                hint = sampler?.hint ?? String.Empty,
                tex_name = String.Empty,
                wrap = new wrapType()
                {
                    u = ApplicationConfig.DefaultValue.wrap.u,
                    v = ApplicationConfig.DefaultValue.wrap.v,
                    w = ApplicationConfig.DefaultValue.wrap.w,
                },
                filter = new filterType()
                {
                    mag = ApplicationConfig.DefaultValue.filter.mag,
                    min = ApplicationConfig.DefaultValue.filter.min,
                    mip = ApplicationConfig.DefaultValue.filter.mip,
                    max_aniso = ApplicationConfig.DefaultValue.filter.max_aniso,
                },
                lod = new lodType()
                {
                    min = ApplicationConfig.DefaultValue.lod.min,
                    max = ApplicationConfig.DefaultValue.lod.max,
                    bias = ApplicationConfig.DefaultValue.lod.bias,
                },
            };

            newSampler.tex_name = GetResolvedSamplerParamValue(samplerName, SamplerParamID.tex_name) as string ?? newSampler.tex_name;
            newSampler.wrap.u = GetResolvedSamplerParamValue(samplerName, SamplerParamID.wrap_u) as wrap_uvwType? ?? newSampler.wrap.u;
            newSampler.wrap.v = GetResolvedSamplerParamValue(samplerName, SamplerParamID.wrap_v) as wrap_uvwType? ?? newSampler.wrap.v;
            newSampler.wrap.w = GetResolvedSamplerParamValue(samplerName, SamplerParamID.wrap_w) as wrap_uvwType? ?? newSampler.wrap.w;
            newSampler.filter.mag = GetResolvedSamplerParamValue(samplerName, SamplerParamID.filter_mag) as filter_mag_minType? ?? newSampler.filter.mag;
            newSampler.filter.min = GetResolvedSamplerParamValue(samplerName, SamplerParamID.filter_min) as filter_mag_minType? ?? newSampler.filter.min;
            newSampler.filter.mip = GetResolvedSamplerParamValue(samplerName, SamplerParamID.filter_mip) as filter_mipType? ?? newSampler.filter.mip;
            newSampler.filter.max_aniso = GetResolvedSamplerParamValue(samplerName, SamplerParamID.filter_max_aniso) as filter_max_anisoType? ?? newSampler.filter.max_aniso;
            newSampler.lod.min = GetResolvedSamplerParamValue(samplerName, SamplerParamID.lod_min) as float? ?? newSampler.lod.min;
            newSampler.lod.max = GetResolvedSamplerParamValue(samplerName, SamplerParamID.lod_max) as float? ?? newSampler.lod.max;
            newSampler.lod.bias = GetResolvedSamplerParamValue(samplerName, SamplerParamID.lod_bias) as float? ?? newSampler.lod.bias;

            return newSampler;
        }

        // 参照を解決したテクスチャ名を返す
        public string GetResolvedSamplerTextureName(string samplerName)
        {
            var sampler = Samplers?.FirstOrDefault(x => x.name == samplerName);
            Debug.Assert(sampler != null, "sampler != null");

            if (!IsReferenceSampler(samplerName, false))
            {
                return sampler.tex_name ?? string.Empty;
            }

            return GetResolvedSamplerParamValue(samplerName, SamplerParamID.tex_name) as string ?? string.Empty;
        }

        public bool IsReferenceBehaviorEditable(string paramName, Type type)
        {
            if (!ParentMaterials.Any())
            {
                return true;
            }
            return GetResolvedChildRestrictionState(paramName, type) == ShaderItemRestrictionState.None;
        }

        private ShaderItemRestrictionState GetChildRestrictionState(string paramName, Type type)
        {
            if (type == typeof(attrib_assignType))
                return GetReferenceBehaviorAttribAssign(paramName)?.ChildRestriction ?? ShaderItemRestrictionState.None;
            if (type == typeof(shader_optionType))
                return GetReferenceBehaviorShaderOption(paramName)?.ChildRestriction ?? ShaderItemRestrictionState.None;
            if (type == typeof(sampler_assignType))
                return GetReferenceBehaviorSamplerAssign(paramName)?.ChildRestriction ?? ShaderItemRestrictionState.None;
            if (type == typeof(shader_paramType))
                return GetReferenceBehaviorShaderParam(paramName)?.ChildRestriction ?? ShaderItemRestrictionState.None;
            if (type == typeof(render_infoType))
                return GetReferenceBehaviorRenderInfo(paramName)?.ChildRestriction ?? ShaderItemRestrictionState.None;

            Debug.Assert(false);
            return ShaderItemRestrictionState.None;
        }

        public ShaderItemRestrictionState GetResolvedChildRestrictionState(string paramName, Type type, bool ignoreSelf = true)
        {
            var restriction = ShaderItemRestrictionState.None;
            if (!ignoreSelf)
            {
                restriction = GetChildRestrictionState(paramName, type);
            }

            // 祖先を ChildRestriction != None が見つかるまで探す
            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                var parentRestriction = parentMaterial.GetResolvedChildRestrictionState(paramName, type, false);
                if (parentRestriction != ShaderItemRestrictionState.None)
                {
                    return parentRestriction;
                }
            }

            return restriction;
        }

        public bool IsValueEditable(string paramName, Type type)
        {
            if (!ParentMaterials.Any())
            {
                return true;
            }
            var shaderItemValueState = GetValueState(paramName, type);
            return shaderItemValueState == ShaderItemValueState.Override;
        }

        public bool IsResolvedValueEditable(string paramName, Type t)
        {
            var restrictionState = GetResolvedChildRestrictionState(paramName, t);
            switch (restrictionState)
            {
                case ShaderItemRestrictionState.ForceOverride:
                    return true;
                case ShaderItemRestrictionState.ForceRefer:
                    return false;
                case ShaderItemRestrictionState.None:
                default:
                    return this.IsValueEditable(paramName, t);
            }
        }

        private ShaderItemValueState GetValueState(string paramName, Type type)
        {
            if (type == typeof(attrib_assignType))
                return GetReferenceBehaviorAttribAssign(paramName)?.Value ?? ShaderItemValueState.Refer;
            if (type == typeof(shader_optionType))
                return GetReferenceBehaviorShaderOption(paramName)?.Value ?? ShaderItemValueState.Refer;
            if (type == typeof(sampler_assignType))
                return GetReferenceBehaviorSamplerAssign(paramName)?.Value ?? ShaderItemValueState.Refer;
            if (type == typeof(shader_paramType))
                return GetReferenceBehaviorShaderParam(paramName)?.Value ?? ShaderItemValueState.Refer;
            if (type == typeof(render_infoType))
                return GetReferenceBehaviorRenderInfo(paramName)?.Value ?? ShaderItemValueState.Refer;

            Debug.Assert(false);
            return ShaderItemValueState.Refer;
        }


        public ShaderItemValueState? GetResolvedValueState(string paramName, Type type)
        {
            var valueEditableState = GetValueState(paramName, type);
            if (valueEditableState != ShaderItemValueState.Refer)
            {
                return valueEditableState;
            }

            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                var parentValueEditableState = parentMaterial.GetValueState(paramName, type);
                if (parentValueEditableState != ShaderItemValueState.Refer)
                {
                    return parentValueEditableState;
                }
            }

            return null;//ShaderItemValueState.Refer;
        }


        private ShaderItemValueState GetValueState<T>(T shaderValue) where T : class
        {
            if (shaderValue is attrib_assignType)
                return GetReferenceBehaviorAttribAssign((shaderValue as attrib_assignType).id)?.Value ?? ShaderItemValueState.Refer;
            if (shaderValue is shader_optionType)
                return GetReferenceBehaviorShaderOption((shaderValue as shader_optionType).id)?.Value ?? ShaderItemValueState.Refer;
            if (shaderValue is sampler_assignType)
                return GetReferenceBehaviorSamplerAssign((shaderValue as sampler_assignType).id)?.Value ?? ShaderItemValueState.Refer;
            if (shaderValue is shader_paramType)
                return GetReferenceBehaviorShaderParam((shaderValue as shader_paramType).id)?.Value ?? ShaderItemValueState.Refer;
            if (shaderValue is render_infoType)
                return GetReferenceBehaviorRenderInfo((shaderValue as render_infoType).name)?.Value ?? ShaderItemValueState.Refer;

            Debug.Assert(false);
            return ShaderItemValueState.Refer;
        }

        public ShaderItemValueState GetResolvedValueState<T>(T shaderValue) where T : class
        {
            var valueState = GetValueState(shaderValue);
            if (valueState != ShaderItemValueState.Refer)
            {
                return valueState;
            }

            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                var parentValueState = parentMaterial.GetValueState(shaderValue);
                if (parentValueState != ShaderItemValueState.Refer)
                {
                    return parentValueState;
                }
            }

            return ShaderItemValueState.Refer;
        }
        public Material GetResolvedValueParentMaterial<T>(T shaderValue) where T : class
        {
            var valueState = GetValueState(shaderValue);
            if (valueState != ShaderItemValueState.Refer)
            {
                return this;
            }

            foreach (var parentMaterial in ParentMaterials.AsEnumerable().Reverse())
            {
                var parentValueState = parentMaterial.GetValueState(shaderValue);
                if (parentValueState != ShaderItemValueState.Refer)
                {
                    return parentMaterial;
                }
            }

            return null;
        }

        public T GetResolvedValue<T>(T shaderValue) where T : class
        {
            bool found;
            Material foundMaterial;
            ValueResolvedState valueResolvedState;
            return GetResolvedValue(shaderValue, out found, out foundMaterial, out valueResolvedState);
        }
        public T GetResolvedValue<T>(T shaderValue, out ValueResolvedState valueResolvedState) where T : class
        {
            bool found;
            Material foundMaterial;
            return GetResolvedValue(shaderValue, out found, out foundMaterial, out valueResolvedState);
        }
        public T GetResolvedValue<T>(T shaderValue, out Material foundMaterial, out ValueResolvedState valueResolvedState) where T : class
        {
            bool found;
            return GetResolvedValue(shaderValue, out found, out foundMaterial, out valueResolvedState);
        }
        public T GetResolvedValue<T>(T shaderValue, out bool found) where T : class
        {
            Material foundMaterial;
            ValueResolvedState valueResolvedState;
            return GetResolvedValue(shaderValue, out found, out foundMaterial, out valueResolvedState);
        }
        public Material SearchParentValueMaterial<T>(T shaderValue) where T : class
        {
            if (shaderValue == null)
            {
                return null;
            }
            bool found;
            Material foundMaterial;
            ValueResolvedState valueResolvedState;
            var value = GetResolvedValue(shaderValue, out found, out foundMaterial, out valueResolvedState);
            return foundMaterial;
        }


        public T GetResolvedValue<T>(T shaderValue, out bool found, out Material foundMaterial, out ValueResolvedState valueResolvedState) where T : class
        {
            MaterialReferenceBehaviorItem referenceBehaviorItem;
            var material = this.Data;
            foundMaterial = this;
            string valueId;
            T value = null;
            Type type;
            var childRestrictionState = ShaderItemRestrictionState.None;

            if (shaderValue is attrib_assignType)
            {
                type = typeof(attrib_assignType);
                valueId = (shaderValue as attrib_assignType).id;
                value =
                    (material?.shader_assign?.attrib_assign_array?.attrib_assign?.FirstOrDefault(x => x.id == valueId)
                     ?? new attrib_assignType() { id = valueId }) as T;
                referenceBehaviorItem = MaterialReference?.AttribAssignBehaviors?.FirstOrDefault(x => x.Id == valueId);
                childRestrictionState = GetResolvedChildRestrictionState(valueId, type);
            }
            else if (shaderValue is render_infoType)
            {
                type = typeof(render_infoType);
                valueId = (shaderValue as render_infoType).name;
                value =
                    (material?.shader_assign?.render_info_array?.render_info?.FirstOrDefault(x => x.name == valueId)
                     ?? new render_infoType() { name = valueId }) as T;
                referenceBehaviorItem = MaterialReference?.RenderInfoBehaviors?.FirstOrDefault(x => x.Id == valueId);
                childRestrictionState = GetResolvedChildRestrictionState(valueId, type);
            }
            else if (shaderValue is RenderInfo)
            {
                type = typeof(render_infoType);
                valueId = (shaderValue as RenderInfo).name;
                value =
                    (MaterialShaderAssign.RenderInfos.FirstOrDefault(x => x.name == valueId)
                     ?? new RenderInfo() { name = valueId }) as T;
                referenceBehaviorItem = MaterialReference?.RenderInfoBehaviors?.FirstOrDefault(x => x.Id == valueId);
                childRestrictionState = GetResolvedChildRestrictionState(valueId, type);
            }
            else if (shaderValue is shader_optionType)
            {
                type = typeof(shader_optionType);
                valueId = (shaderValue as shader_optionType).id;
                value =
                    (material?.shader_assign?.shader_option_array?.shader_option?.FirstOrDefault(x => x.id == valueId)
                     ?? new shader_optionType() { id = valueId }) as T;
                referenceBehaviorItem = MaterialReference?.ShaderOptionBehaviors?.FirstOrDefault(x => x.Id == valueId);
                childRestrictionState = GetResolvedChildRestrictionState(valueId, type);
            }
            else if (shaderValue is sampler_assignType)
            {
                type = typeof(sampler_assignType);
                valueId = (shaderValue as sampler_assignType).id;
                value =
                    (material?.shader_assign?.sampler_assign_array?.sampler_assign?.FirstOrDefault(x => x.id == valueId)
                     ?? new sampler_assignType() { id = valueId }) as T;
                referenceBehaviorItem = MaterialReference?.SamplerAssignBehaviors?.FirstOrDefault(x => x.Id == valueId);
                childRestrictionState = GetResolvedChildRestrictionState(valueId, type);
            }
            else if (shaderValue is shader_paramType)
            {
                type = typeof(shader_paramType);
                valueId = (shaderValue as shader_paramType).id;
                value =
                    (material?.shader_assign?.shader_param_array?.shader_param?.FirstOrDefault(x => x.id == valueId)
                     ?? new shader_paramType() { id = valueId }) as T;
                referenceBehaviorItem = MaterialReference?.ShaderParamBehaviors?.FirstOrDefault(x => x.Id == valueId);
                childRestrictionState = GetResolvedChildRestrictionState(valueId, typeof(shader_paramType));
            }
            else
            {
                Debug.Assert(false);
                found = false;
                valueResolvedState = ValueResolvedState.Override;
                return null;
            }

            ShaderItemValueState valueState;
            switch (childRestrictionState)
            {
                case ShaderItemRestrictionState.ForceRefer:
                    // 親マテリアルまたはその祖先のどれかがForceReferの場合はForceReferになっている親マテリアルを探し
                    // その値を採用する（その親マテリアル自身が持つ値ではなく、参照を解決した値を採用する）
                    Debug.Assert(ParentMaterials.Any(), "ForceRefer: ParentMaterial is not exist.");

                    // 後に指定されたベースマテリアルから検索
                    foreach (var parentMaterial in Enumerable.Reverse(ParentMaterials))
                    {
                        // この親マテリアルまたはその祖先がForceReferの場合
                        var resolvedRestriction = parentMaterial.GetResolvedChildRestrictionState(valueId, type, ignoreSelf: false);
                        if (resolvedRestriction == ShaderItemRestrictionState.ForceRefer)
                        {
                            // この親マテリアル自身がForceReferの場合、その値を採用する
                            var restriction = parentMaterial.GetChildRestrictionState(valueId, type);
                            if (restriction == ShaderItemRestrictionState.ForceRefer)
                            {
                                var parentShaderParam = parentMaterial.GetResolvedValue(shaderValue, out found, out foundMaterial, out valueResolvedState);
                                found = true;
                                valueResolvedState = ValueResolvedState.ForceRefer;
                                return parentShaderParam;
                            }
                        }
                    }

                    valueState = ShaderItemValueState.Refer;
                    break;
                case ShaderItemRestrictionState.ForceOverride:
                    found = true;
                    valueResolvedState = ValueResolvedState.ForceOverride;
                    return value;
                case ShaderItemRestrictionState.None:
                default:
                    valueState = referenceBehaviorItem?.Value ?? ShaderItemValueState.Refer;
                    break;
            }

            switch (valueState)
            {
                case ShaderItemValueState.Override:
                    found = true;
                    valueResolvedState = ValueResolvedState.Override;
                    return value;
                case ShaderItemValueState.Default:
                    found = true;
                    valueResolvedState = ValueResolvedState.Default;
                    return null;
                case ShaderItemValueState.Refer:
                default:
                    T parentShaderParam = null;
                    if (ParentMaterials.Any())
                    {
                        // 後に指定されたベースマテリアルから検索
                        foreach (var parentMaterial in Enumerable.Reverse(ParentMaterials))
                        {
                            parentShaderParam = parentMaterial.GetResolvedValue(shaderValue, out found, out foundMaterial, out valueResolvedState);
                            if (found)
                            {
                                valueResolvedState = ValueResolvedState.Refer;
                                return parentShaderParam;
                            }
                        }

                        // 値が確定しない場合(全てRefer) found は false で 最後の値を返す
                        found = false;
                        valueResolvedState = ValueResolvedState.Refer;
                        return parentShaderParam;
                    }
                    else
                    {
                        // 親マテリアルがない場合は found は false で 自分の値を返す
                        found = false;
                        valueResolvedState = ValueResolvedState.Override;
                        return value;
                    }
            }
        }

        // マテリアル参照解決済みの値を取得する
        public string GetResolvedOptionValue(shader_optionType option)
        {
            ValueResolvedState valueResolvedState;
            Material foundMaterial;
            return GetResolvedOptionValue(option, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedOptionValue(shader_optionType option, out ValueResolvedState valueResolvedState)
        {
            Material foundMaterial;
            return GetResolvedOptionValue(option, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedOptionValue(shader_optionType option, out Material foundMaterial, out ValueResolvedState valueResolvedState)
        {
            foundMaterial = this;
            if (!ParentMaterials.Any() ||!ParentMaterialsValidity())
            {
                var behaviorItem = GetReferenceBehaviorShaderOption(option.id);
                valueResolvedState = behaviorItem?.Value == ShaderItemValueState.Default ? ValueResolvedState.Default : ValueResolvedState.Override;
                return option.value;
            }
            var parentOption = GetResolvedValue(option, out foundMaterial, out valueResolvedState);
            if (parentOption != null)
            {
                return parentOption.value ?? String.Empty;
            }
            else // ShaderItemValueState.Default
            {
                var defaultShaderOption = MaterialShaderAssign?.ShadingModel?.option_var_array?.option_var?.FirstOrDefault(x => x.id == option.id);
                return defaultShaderOption?.@default ?? option.value;
            }
        }

        public string GetResolvedSamplerAssignValue(sampler_assignType samplerAssign)
        {
            ValueResolvedState valueResolvedState;
            Material foundMaterial;
            return GetResolvedSamplerAssignValue(samplerAssign, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedSamplerAssignValue(sampler_assignType samplerAssign, out ValueResolvedState valueResolvedState)
        {
            Material foundMaterial;
            return GetResolvedSamplerAssignValue(samplerAssign, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedSamplerAssignValue(sampler_assignType samplerAssign, out Material foundMaterial, out ValueResolvedState valueResolvedState)
        {
            foundMaterial = this;
            if (!ParentMaterials.Any() || !ParentMaterialsValidity())
            {
                var behaviorItem = GetReferenceBehaviorSamplerAssign(samplerAssign.id);
                valueResolvedState = behaviorItem?.Value == ShaderItemValueState.Default ? ValueResolvedState.Default : ValueResolvedState.Override;
                return samplerAssign.sampler_name;
            }
            var parentSamplerAssign = GetResolvedValue(samplerAssign, out foundMaterial, out valueResolvedState);
            if (parentSamplerAssign != null)
            {
                return parentSamplerAssign.sampler_name ?? String.Empty;
            }
            else // ShaderItemValueState.Default
            {
                var defaultSampler = MaterialShaderAssign?.ShadingModel?.sampler_var_array?.sampler_var?.FirstOrDefault(x => x.id == samplerAssign.id);

                if (defaultSampler != null)
                {
                    return IfSamplerAssignUtility.GetSamplerNameFromHint(Data.sampler_array?.sampler, defaultSampler) ??
                           String.Empty;
                }
            }
            return  samplerAssign.sampler_name;
        }

        public string GetResolvedShaderParamValue(shader_paramType shaderParam)
        {
            ValueResolvedState valueResolvedState;
            Material foundMaterial;
            return GetResolvedShaderParamValue(shaderParam, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedShaderParamValue(shader_paramType shaderParam, out ValueResolvedState valueResolvedState)
        {
            Material foundMaterial;
            return GetResolvedShaderParamValue(shaderParam, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedShaderParamValue(shader_paramType shaderParam, out Material foundMaterial, out ValueResolvedState valueResolvedState)
        {
            foundMaterial = this;
            if (!ParentMaterials.Any() || !ParentMaterialsValidity())
            {
                var behaviorItem = GetReferenceBehaviorShaderParam(shaderParam.id);
                valueResolvedState = behaviorItem?.Value == ShaderItemValueState.Default ? ValueResolvedState.Default : ValueResolvedState.Override;
                return shaderParam.Value;
            }
            var parentShaderParam = GetResolvedValue(shaderParam, out foundMaterial, out valueResolvedState);
            if (parentShaderParam != null)
            {
                return parentShaderParam.Value ?? String.Empty;
            }
            else // ShaderItemValueState.Default
            {
                return MaterialShaderAssign?.ShadingModel?.MaterialUniforms().FirstOrDefault(x => x.id == shaderParam.id)?.Default() ?? shaderParam.Value;
            }
        }

        public string GetResolvedAttribAssignValue(attrib_assignType attribAssign)
        {
            ValueResolvedState valueResolvedState;
            Material foundMaterial;
            return GetResolvedAttribAssignValue(attribAssign, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedAttribAssignValue(attrib_assignType attribAssign, out ValueResolvedState valueResolvedState)
        {
            Material foundMaterial;
            return GetResolvedAttribAssignValue(attribAssign, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedAttribAssignValue(attrib_assignType attribAssign, out Material foundMaterial, out ValueResolvedState valueResolvedState)
        {
            foundMaterial = this;
            if (!ParentMaterials.Any() || !ParentMaterialsValidity())
            {
                var behaviorItem = GetReferenceBehaviorAttribAssign(attribAssign.id);
                valueResolvedState = behaviorItem?.Value == ShaderItemValueState.Default ? ValueResolvedState.Default : ValueResolvedState.Override;
                return attribAssign.attrib_name;
            }
            var parentAttribAssign = GetResolvedValue(attribAssign, out foundMaterial, out valueResolvedState);
            if (parentAttribAssign != null)
            {
                return parentAttribAssign.attrib_name ?? String.Empty;
            }
            else // ShaderItemValueState.Default
            {
                var defaultAttrib = MaterialShaderAssign?.ShadingModel?.attrib_var_array?.attrib_var?.FirstOrDefault(x => x.id == attribAssign.id);
                if (defaultAttrib != null)
                {
                    return Referrers.Any() ? IfAttributeAssignUtility.GetAttribNameFromHint(Referrers.First().Data, Name, defaultAttrib) ?? String.Empty : string.Empty;
                }
            }
            return attribAssign.attrib_name;
        }

        public List<string> GetResolvedRenderInfoValues(RenderInfo renderInfo)
        {
            ValueResolvedState valueResolvedState;
            Material foundMaterial;
            return GetResolvedRenderInfoValues(renderInfo, out foundMaterial, out valueResolvedState);
        }
        public List<string> GetResolvedRenderInfoValues(RenderInfo renderInfo, out ValueResolvedState valueResolvedState)
        {
            Material foundMaterial;
            return GetResolvedRenderInfoValues(renderInfo, out foundMaterial, out valueResolvedState);
        }
        public List<string> GetResolvedRenderInfoValues(RenderInfo renderInfo, out Material foundMaterial, out ValueResolvedState valueResolvedState)
        {
            foundMaterial = this;
            if (!ParentMaterials.Any() || !ParentMaterialsValidity())
            {
                var behaviorItem = GetReferenceBehaviorRenderInfo(renderInfo.name);
                valueResolvedState = behaviorItem?.Value == ShaderItemValueState.Default ? ValueResolvedState.Default : ValueResolvedState.Override;
                return renderInfo.values;
            }
            var parentRenderInfo = GetResolvedValue(renderInfo, out foundMaterial, out valueResolvedState);
            if (parentRenderInfo != null)
            {
                return parentRenderInfo.values ?? new List<string>();
            }
            else // ShaderItemValueState.Default
            {
                var defaultRenderInfo = MaterialShaderAssign?.ShadingModel?.render_info_slot_array?.render_info_slot?.FirstOrDefault(x => x.name == renderInfo.name);
                if (defaultRenderInfo != null)
                {
                    return IfRenderInfoUtility.CreateRenderInfoValue(defaultRenderInfo, null);
                }
            }
            return renderInfo.values;
        }

        public string GetResolvedRenderInfoValue(render_infoType render_info)
        {
            ValueResolvedState valueResolvedState;
            Material foundMaterial;
            return GetResolvedRenderInfoValue(render_info, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedRenderInfoValue(render_infoType render_info, out ValueResolvedState valueResolvedState)
        {
            Material foundMaterial;
            return GetResolvedRenderInfoValue(render_info, out foundMaterial, out valueResolvedState);
        }
        public string GetResolvedRenderInfoValue(render_infoType render_info, out Material foundMaterial, out ValueResolvedState valueResolvedState)
        {
            foundMaterial = this;
            if (!ParentMaterials.Any() || !ParentMaterialsValidity())
            {
                var behaviorItem = GetReferenceBehaviorRenderInfo(render_info.name);
                valueResolvedState = behaviorItem?.Value == ShaderItemValueState.Default ? ValueResolvedState.Default : ValueResolvedState.Override;
                return render_info.Value;
            }
            var parentRenderInfo = GetResolvedValue(render_info, out foundMaterial, out valueResolvedState);
            if (parentRenderInfo != null)
            {
                return parentRenderInfo.Value ?? String.Empty;
            }
            else // ShaderItemValueState.Default
            {
                render_info_slotType defaultRenderInfo = MaterialShaderAssign?.ShadingModel?.render_info_slot_array?.render_info_slot?.FirstOrDefault(x => x.name == render_info.name);
                if (defaultRenderInfo != null)
                {
                    return defaultRenderInfo.choice;
                }
            }
            return render_info.Value;
        }

        #region コマンド
        public static EditCommand CreateEditCommand_MaterialReference(GuiObjectGroup targets, nw3de_MaterialReference materialReference)
        {
            var materialReferences = ObjectUtility.MultipleClone(materialReference, targets.Objects.Count).ToArray();
            return CreateEditCommand_MaterialReference(targets, materialReferences);
        }

        public static EditCommand CreateEditCommand_MaterialReference(GuiObjectGroup targets, IEnumerable<nw3de_MaterialReference> materialReferences)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.NotNull(targets);
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.NotNull(materialReferences);

            var materials = targets.Objects.Where(x => x.ObjectID == GuiObjectID.Material).OfType<Material>().ToArray();
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.AreEqual(materials.Length, materialReferences.Count());

            var commandSet = new EditCommandSet();
            commandSet.Add(new GeneralGroupReferenceEditCommand<nw3de_MaterialReference>(
                new GuiObjectGroup(materials),
                null,
                materialReferences,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material) target;
                    swap = material.MaterialReference;
                    material.MaterialReference = (nw3de_MaterialReference) data;
                },
                postEditDelegate: (a, o) =>
                {
                    foreach (var model in materials.SelectMany(x => x.Referrers).Distinct())
                    {
                        LoadOrReloadModel.Send(model);
                    }
                },
                createEventArgsDelegate: (target, command) => new DocumentShaderPropertyChangedArgs(target, command)));
            var descendantMaterials = materials.SelectMany(x => x.GetAllDescendantMaterials()).Distinct().ToList();
            descendantMaterials = materials.Concat(descendantMaterials).Distinct().ToList();
            commandSet.Add(new LazyCommand(() => CreateEditCommand_UpdateSamplers(new GuiObjectGroup(descendantMaterials))));
            commandSet.OnPostEdit += (sender, args) =>
            {
                // コンバイナー参照が切り替わる可能性があるのでウォッチャーを更新する。
                foreach (var material in materials)
                {
                    material.DisableCombinerWatchers();
                    material.EnableCombinerWatchers();
                }

                var materialOwners = materials.Select(x => x.Owner)
                    .Concat(materials.SelectMany(x => x.GetAllDescendantMaterials()).Select(x => x.Owner)).Distinct();
                foreach (var materialOwner in materialOwners)
                {
                    materialOwner.UpdateReferenceTexturePaths();
                }

                var models = materials.SelectMany(x => x.Referrers)
                    .Concat(materials.SelectMany(x => x.GetAllDescendantMaterials()).SelectMany(x => x.Referrers)).Distinct();
                foreach (var model in models)
                {
                    LoadOrReloadModel.Send(model);
                }
            };

            return commandSet;
        }

        public static EditCommand CreateEditCommand_CombinerOption(GuiObjectGroup targets, IEnumerable<nw3de_CombinerOptionInfo> combinerOptionInfos, bool updateModified, bool reload)
        {
            return new GeneralGroupReferenceEditCommand<nw3de_CombinerOptionInfo>(
                targets,
                GuiObjectID.Material,
                combinerOptionInfos,
                delegate (ref GuiObject target, ref object data, ref object swap)
                {
                    var material = (Material)target;
                    var combinerOptionInfo = (nw3de_CombinerOptionInfo)data;

                    var combiners = material.GetCombiners().Where(x => x.Item1 == material).Select(x => x.Item2).ToArray();
                    if (combiners.Any())
                    {
                        var combinerOptions = combinerOptionInfo?.CombinerOptions?.ToDictionary(x => x.Id, x => x);
                        if (combinerOptions == null)
                        {
                            combinerOptions = new Dictionary<string, CombinerOption>();
                        }

                        var additionalCombinerOptions = new List<CombinerOption>();
                        foreach (var combiner in combiners)
                        {
                            CombinerOption combinerOption;
                            if (combinerOptions.TryGetValue(combiner.id, out combinerOption))
                            {
                                // オプション情報を元にオプション値を更新。
                                combiner.value = !string.IsNullOrEmpty(combinerOption.FileName) ? CombinerShaderConverterManager.ComputeHashUInt32(combinerOption.FileName).ToString() : CombinerShaderConverterManager.ReservedValue.Unset;
                            }
                            else
                            {
                                combinerOption = new CombinerOption()
                                {
                                    Id = combiner.id,
                                    FileName = combiner.value
                                };
                                if (CombinerShaderConverterManager.ReservedValue.IsNullOrEmptyOrUnset(combiner.value))
                                {
                                    // ファイル未指定の場合はファイル名を null にしておく。
                                    combinerOption.FileName = null;
                                }
                                else
                                {
                                    // オプション値をファイル名から生成したハッシュ値に変更する。
                                    combiner.value = CombinerShaderConverterManager.ComputeHashUInt32(combiner.value).ToString();
                                }
                                combinerOptions.Add(combiner.id, combinerOption);
                                additionalCombinerOptions.Add(combinerOption);
                            }
                        }
                        if (additionalCombinerOptions.Any())
                        {
                            if (combinerOptionInfo == null)
                            {
                                combinerOptionInfo = new nw3de_CombinerOptionInfo();
                            }
                            if (combinerOptionInfo.CombinerOptions == null)
                            {
                                combinerOptionInfo.CombinerOptions = new List<CombinerOption>();
                            }
                            combinerOptionInfo.CombinerOptions.AddRange(additionalCombinerOptions);
                        }
                    }

                    swap = material.CombinerOptionInfo;
                    material.CombinerOptionInfo = combinerOptionInfo;
                },
                createEventArgsDelegate: (x, y) => new App.PropertyEdit.DocumentPropertyChangedShaderArgs(x, y),
                postEditDelegate: delegate (System.Collections.ArrayList editTargets, object[] data)
                {
                    // ビューア転送
                    if (reload)
                    {
                        var materials = editTargets.OfType<Material>().ToArray();
                        var models = materials
                            .SelectMany(x => x.Referrers)
                            .Concat(materials.SelectMany(x => x.GetAllDescendantMaterials())
                            .SelectMany(x => x.Referrers)).Distinct().ToArray();
                        foreach (var model in models)
                        {
                            LoadOrReloadModel.Send(model);
                        }
                    }
                },
                updateModified: updateModified
                );
        }

        public static EditCommand CreateEditCommand_UpdateSamplers(GuiObjectGroup targets)
        {
            var materials = targets.Objects.Where(x => x.ObjectID == GuiObjectID.Material).OfType<Material>().Distinct();
            var commandSet = new EditCommandSet();
            foreach (var material in materials)
            {
                material.UpdateSamplers(commandSet);
            }
            return commandSet;
        }
        #endregion

        internal static MaterialReferenceBehaviorItem SetMaterialReferenceBehaviorItemStates(
            MaterialReferenceBehaviorItem referenceBehaviorItem,
            string paramName,
            ShaderItemRestrictionState? childRestrictionState,
            ShaderItemValueState? valueState)
        {
            if (referenceBehaviorItem == null)
            {
                referenceBehaviorItem = new MaterialReferenceBehaviorItem()
                                        {
                                            Id = paramName,
                                            Value = ShaderItemValueState.Refer,
                                            ChildRestriction = ShaderItemRestrictionState.None
                                        };
            }
            if (childRestrictionState.HasValue)
            {
                referenceBehaviorItem.ChildRestriction = childRestrictionState.Value;
            }
            if (valueState.HasValue)
            {
                referenceBehaviorItem.Value = valueState.Value;
            }
            return referenceBehaviorItem;
        }

        internal static void SetReferenceBehaviorToGroup(GuiObjectGroup targets, string groupName, ShaderItemRestrictionState? childRestrictionState, ShaderItemValueState? valueState)
        {
            var commandSet = CreateSetReferenceBehaviorToGroupsCommand(targets, new[] { groupName }, childRestrictionState, valueState);
            TheApp.CommandManager.Add(commandSet.Execute());
            AppContext.OnMaterialReferenceBehaviorSettingChanged();
        }

        public static void SetReferenceBehaviorToGroups(GuiObjectGroup targets, IEnumerable<string> groups, ShaderItemRestrictionState? childRestrictionState, ShaderItemValueState? valueState)
        {
            var commandSet = CreateSetReferenceBehaviorToGroupsCommand(targets, groups, childRestrictionState, valueState);
            TheApp.CommandManager.Add(commandSet.Execute());
            AppContext.OnMaterialReferenceBehaviorSettingChanged();
        }

        private static EditCommandSet CreateSetReferenceBehaviorToGroupsCommand(GuiObjectGroup targets, IEnumerable<string> groups,
            ShaderItemRestrictionState? childRestrictionState, ShaderItemValueState? valueState)
        {
            var commandSet = new EditCommandSet();
            var activeTarget = (Material)targets.Active;
            Debug.Assert(activeTarget != null, "activeTarget != null");

            var shaderAssign = activeTarget.MaterialShaderAssign;
            if (shaderAssign == null)
            {
                return commandSet;
            }
            var materials =
                targets.Objects.OfType<Material>()
                    .Where(
                        x =>
                        x.MaterialShaderAssign?.ShaderDefinitionFileName == shaderAssign.ShaderDefinitionFileName
                        && x.MaterialShaderAssign?.ShaderName == shaderAssign.ShaderName)
                    .ToList();

            var shadingModel = shaderAssign.ShadingModel;
            var paramsInGroupDict = new Dictionary<string, IEnumerable<object>>();
            var groupNames = groups.ToArray();
            foreach (var groupName in groupNames)
            {
                var paramsInGroup =
                    shadingModel.Attributes()
                        .Where(x => x.Group() == groupName)
                        .Cast<object>()
                        .Concat(shadingModel.Options().Where(x => x.Group() == groupName))
                        .Concat(shadingModel.Samplers().Where(x => x.Group() == groupName))
                        .Concat(shadingModel.RenderInfoSlots().Where(x => x.Group() == groupName))
                        .Concat(shadingModel.Uniforms().Where(x => x.Group() == groupName));
                paramsInGroupDict.Add(groupName, paramsInGroup);
            }

            foreach (var material in materials)
            {
                var materialReference = ObjectUtility.Clone(material.MaterialReference)
                                          ?? new nw3de_MaterialReference();
                foreach (var groupName in groupNames)
                {
                    var paramInGroup = paramsInGroupDict[groupName];
                    foreach (var param in paramInGroup)
                    {
                        if (param is attrib_varType)
                        {
                            var id = (param as attrib_varType).id;
                            var behaviorItem =
                                SetMaterialReferenceBehaviorItemStates(
                                    materialReference.AttribAssignBehaviors?.FirstOrDefault(x => x.Id == id),
                                    id,
                                    childRestrictionState,
                                    valueState);
                            AddOrUpdateMaterialReferenceBehaviorItem(
                                behaviorItem,
                                materialReference.AttribAssignBehaviors);
                        }
                        else if (param is option_varType)
                        {
                            var id = (param as option_varType).id;
                            var behaviorItem =
                                SetMaterialReferenceBehaviorItemStates(
                                    materialReference.ShaderOptionBehaviors?.FirstOrDefault(x => x.Id == id),
                                    id,
                                    childRestrictionState,
                                    valueState);
                            AddOrUpdateMaterialReferenceBehaviorItem(
                                behaviorItem,
                                materialReference.ShaderOptionBehaviors);
                        }
                        else if (param is sampler_varType)
                        {
                            var id = (param as sampler_varType).id;
                            var behaviorItem =
                                SetMaterialReferenceBehaviorItemStates(
                                    materialReference.SamplerAssignBehaviors?.FirstOrDefault(x => x.Id == id),
                                    id,
                                    childRestrictionState,
                                    valueState);
                            AddOrUpdateMaterialReferenceBehaviorItem(
                                behaviorItem,
                                materialReference.SamplerAssignBehaviors);
                        }
                        else if (param is render_info_slotType)
                        {
                            var id = (param as render_info_slotType).name;
                            var behaviorItem =
                                SetMaterialReferenceBehaviorItemStates(
                                    materialReference.RenderInfoBehaviors?.FirstOrDefault(x => x.Id == id),
                                    id,
                                    childRestrictionState,
                                    valueState);
                            AddOrUpdateMaterialReferenceBehaviorItem(behaviorItem, materialReference.RenderInfoBehaviors);
                        }
                        else if (param is uniform_varType)
                        {
                            var id = (param as uniform_varType).id;
                            var behaviorItem =
                                SetMaterialReferenceBehaviorItemStates(
                                    materialReference.ShaderParamBehaviors?.FirstOrDefault(x => x.Id == id),
                                    id,
                                    childRestrictionState,
                                    valueState);
                            AddOrUpdateMaterialReferenceBehaviorItem(
                                behaviorItem,
                                materialReference.ShaderParamBehaviors);
                        }
                    }
                }
                commandSet.Add(
                    CreateEditCommand_MaterialReference(new GuiObjectGroup(material), materialReference));
            }
            return commandSet;
        }

        public static EditCommandSet CreateEditCommandSet_ChildSamplerStructureRestriction(IEnumerable<Material> materials,
            ChildSamplerStructureRestrictionState restrictionState)
        {
            var commandSet = new EditCommandSet();
            foreach (var target in materials)
            {
                var samplerBehaviors = ObjectUtility.Clone(target.GetReferenceBehaviorSamplers()) ?? new MaterialReferenceSamplerBehaviors();
                samplerBehaviors.ChildSamplerStructureRestriction = restrictionState;
                var materialReference = ObjectUtility.Clone(target.MaterialReference) ??
                                                new nw3de_MaterialReference();
                materialReference.SamplerBehaviors = samplerBehaviors;
                commandSet.Add(
                    CreateEditCommand_MaterialReference(new GuiObjectGroup(target), materialReference).Execute());
                foreach (var descendantMaterial in target.GetAllDescendantMaterials())
                {
                    descendantMaterial.UpdateSamplers(commandSet);
                }
            }
            commandSet.Reverse();
            return commandSet;
        }

        public static App.Command.ICommand ExecuteEditCommand_SamplerArray(GuiObjectGroup targets, List<samplerType[]> targetSamplers, bool sendMessage)
        {
            using (var propertyBlock = new AppContext.PropertyChangedSuppressBlock())
            using (var viewerBlock = new ViewerDrawSuppressBlock(ViewerDrawSuppressBlock.ReloadModelForShaderParameterMessage))
            {
                return  CreateEditCommand_SamplerArray(targets, targetSamplers, sendMessage).Execute();
            }
        }

        internal static EditCommandSet CreateEditCommand_SamplerArray(GuiObjectGroup targets, List<samplerType[]> targetSamplers, bool sendMessage)
        {
            var commandSet = new EditCommandSet();
            //commandSet.SetViewerDrawSuppressBlockDelegate(ViewerDrawSuppressBlock.ReloadModelForShaderParameterMessage);
            //using (var propertyBlock = new AppContext.PropertyChangedSuppressBlock())
            //using (var viewerBlock = new ViewerDrawSuppressBlock(ViewerDrawSuppressBlock.ReloadModelForShaderParameterMessage))
            {
                var materials = targets.GetObjects(GuiObjectID.Material).OfType<Material>().ToArray();

                // 必要ならサンプラ割り当てを削除
                for (int i = 0; i < materials.Length; i++)
                {
                    foreach (var samplerAssign in materials[i].MaterialShaderAssign.SamplerAssigns)
                    {
                        if (targetSamplers[i].All(x => x.name != samplerAssign.sampler_name))
                        {
                            commandSet.Add(ShaderParamControlGroup.CreateEditCommand_material_shader_assign_sampler_name(
                                new GuiObjectGroup(materials[i]),
                                samplerAssign.id,
                                String.Empty,
                                false
                            ));
                        }
                    }
                }

                commandSet.Add(
                    (new GeneralGroupReferenceEditCommand<samplerType[]>(
                        new GuiObjectGroup(materials),
                        GuiObjectID.Material,
                        targetSamplers,
                        delegate(ref GuiObject target, ref object data, ref object swap)
                        {
                            var material = (target as Material);

                            swap = material.sampler_array.sampler;
                            material.sampler_array.sampler = (samplerType[]) data;
                        },
                        postEditDelegate: (a, o) =>
                        {
                            foreach (var model in materials.SelectMany(x => x.Referrers).Distinct())
                            {
                                if (sendMessage)
                                {
                                    LoadOrReloadModel.Send(model);
                                }
                            }
                        },
                        createEventArgsDelegate: (x, y) => new DocumentPropertyChangedSamplerCountArgs(x, y)
                    ))
                );

                // バインドを更新
                commandSet.Add(
                    DocumentManager.CreateAnimationUpdateBindCommand(
                        targets.Objects.OfType<Material>().SelectMany(x => x.Referrers).Distinct().SelectMany(x => x.AllAnimations).Distinct()
                    )
                );
                var descendantMaterials = materials.SelectMany(x => x.GetAllDescendantMaterials()).Distinct().ToList();
                descendantMaterials = materials.Concat(descendantMaterials).Distinct().ToList();
                commandSet.Add(new LazyCommand(() => CreateEditCommand_UpdateSamplers(new GuiObjectGroup(descendantMaterials))));
                commandSet.OnPostEdit += (sender, args) =>
                {
                    // プレビュー目的で参照外のテクスチャを割り当てることがあるので、ここではテクスチャパスの解決を行わない。

                    if (sendMessage)
                    {
                        foreach (var model in materials.SelectMany(x => x.Referrers))
                        {
                            LoadOrReloadModel.Send(model);

                        }
                    }
                };
            }
            return commandSet;
        }

        public static App.Command.ICommand CreateRemoveReferenceBehaviorSamplerCommand(GuiObjectGroup targets, string samplerName)
        {
            var commandSet = new EditCommandSet();
            foreach (var material in targets.GetObjects(GuiObjectID.Material).Cast<Material>())
            {
                if (material.MaterialReference?.SamplerBehaviors?.SamplerBehaviorItems == null)
                {
                    continue;
                }

                if (material.MaterialReference.SamplerBehaviors.SamplerBehaviorItems.All(x => x.Id != samplerName))
                {
                    continue;
                }

                var materialReference = ObjectUtility.Clone(material.MaterialReference);
                materialReference.SamplerBehaviors.SamplerBehaviorItems =
                    materialReference.SamplerBehaviors.SamplerBehaviorItems.Where(x => x.Id != samplerName).ToList();

                commandSet.Add(CreateEditCommand_MaterialReference(new GuiObjectGroup(material), materialReference));
            }
            return commandSet;
        }
        // 親マテリアル設定が変更されたときにキャッシュをクリア
        private static void OnMaterialReferenceBehaviorSettingChanged()
        {
            lock (ParentMaterialModelPaths)
            {
                ParentMaterialModelPaths.Clear();
            }
        }

        // 親マテリアルパスから検索したパスをキャッシュしておく
        private static readonly Dictionary<string, string> ParentMaterialModelPaths = new Dictionary<string, string>();
        // 親マテリアルパスからモデルのパスを検索する
        public static string SearchParentModelFromParentMaterialPaths(string modelPath)
        {
            lock (ParentMaterialModelPaths)
            {
                var fileName = Path.GetFileName(modelPath);
                if (fileName == null)
                {
                    return null;
                }
                string fullpath;

                if (ParentMaterialModelPaths.TryGetValue(fileName, out fullpath))
                {
                    return fullpath;
                }

                var parentMaterialPaths = ApplicationConfig.FileIo.ParentMaterialPaths;

                // まず親マテリアル参照パスから検索
                foreach (var parentMaterialPath in parentMaterialPaths)
                {
                    var parentSearchPath = parentMaterialPath.path;
                    // 相対パスのみ探す
                    if (string.IsNullOrEmpty(parentSearchPath) || !Path.IsPathRooted(parentSearchPath))
                    {
                        continue;
                    }

                    List<string> matchedFiles;
                    List<string> matchedFolders;
                    MaterialParentDialog.ListUpMatchedFilesAndFolders(parentSearchPath, out matchedFiles, out matchedFolders);

                    // 親マテリアルパスで見つかったファイルと同名ならそれを親マテリアルとして採用
                    fullpath = matchedFiles.FirstOrDefault(x =>
                        Path.GetFileName(x).Equals(fileName, StringComparison.OrdinalIgnoreCase));
                    if (!string.IsNullOrEmpty(fullpath))
                    {
                        ParentMaterialModelPaths.Add(fileName, fullpath);
                        return fullpath;
                    }

                    // 親マテリアルパスで見つかったフォルダからの相対パスでファイルが存在したらそれを親マテリアルとして採用
                    foreach (var folder in matchedFolders)
                    {
                        fullpath = Path.GetFullPath(Path.Combine(folder, fileName));
                        if (File.Exists(fullpath))
                        {
                            ParentMaterialModelPaths.Add(fileName, fullpath);
                            return fullpath;
                        }
                        if (parentMaterialPath.Recursive)
                        {
                            // サブフォルダ以下に同名のファイルが複数存在する場合、
                            // パスを大文字・小文字の区別なしの文字列順でソートした先頭が適用される
                            string[] paths = Directory.EnumerateFiles(folder, fileName, SearchOption.AllDirectories).ToArray();
                            if (paths != null && paths.Length > 0)
                            {
                                Array.Sort(paths, StringComparer.OrdinalIgnoreCase);
                                fullpath = paths.First();

                                if (!string.IsNullOrEmpty(fullpath))
                                {
                                    ParentMaterialModelPaths.Add(fileName, fullpath);
                                    return fullpath;
                                }
                            }
                        }
                    }
                }
            }
            return null;
        }
    }

    public class MaterialShaderAssignPool
    {
        public MaterialShaderAssign this[string shaderArchive, string program]
        {
            get
            {
                var key = new KeyValuePair<string, string>(shaderArchive, program);
                if (assigns.ContainsKey(key))
                {
                    return assigns[key];
                }
                return null;
            }
            set
            {
                var key = new KeyValuePair<string, string>(shaderArchive, program);
                if (assigns.ContainsKey(key))
                {
                    if (value == null)
                    {
                        assigns.Remove(key);
                    }
                    else
                    {
                        assigns[key] = value;
                    }
                }
                else
                {
                    assigns.Add(key, value);
                }
            }
        }

        public IEnumerable<KeyValuePair<string, string>> Keys
        {
            get { return assigns.Keys; }
        }

        private readonly Dictionary<KeyValuePair<string, string>, MaterialShaderAssign> assigns = new Dictionary<KeyValuePair<string, string>, MaterialShaderAssign>();

        public Dictionary<attrib_varType, attrib_assignType> AttribAssigns = new Dictionary<attrib_varType, attrib_assignType>(ShaderTypeUtility.AttribIdTypeComparer.TheComparer);
        public Dictionary<option_varType, shader_optionType> ShaderOptions = new Dictionary<option_varType, shader_optionType>(ShaderTypeUtility.OptionIdChoiceComparer.TheComparer);
        public Dictionary<sampler_varType, sampler_assignType> SamplerAssigns = new Dictionary<sampler_varType, sampler_assignType>(ShaderTypeUtility.SamplerVarIdTypeComparer.TheComparer);
        public Dictionary<uniform_varType, shader_paramType> ShaderParams = new Dictionary<uniform_varType, shader_paramType>(ShaderTypeUtility.UniformIdTypeComparer.TheComparer);
        public Dictionary<render_info_slotType, RenderInfo> RenderInfos = new Dictionary<render_info_slotType, RenderInfo>(ShaderTypeUtility.RenderInfoSlotNameTypeComparer.TheComparer);
    }

    /// <summary>
    /// render_infoType の内部表現
    /// </summary>
    [Serializable]
    public class RenderInfo
    {
        public string name;

        public List<string> values
        {
            get { return internalValues; }
        }

        public List<string> internalValues
        {
            get { return _internalValues; }
            set { _internalValues = value; }
        }

        public List<string> _internalValues;

        public render_info_typeType type;
        public bool recieved = false;

        [NonSerialized] public bool candidateModified = false;

        /// <summary>
        /// ランタイムに転送した要素数。コンストラクタと転送スレッド以外で使用不可!
        /// </summary>
        [NonSerialized] public int HioCount = -1;

        [System.Runtime.Serialization.OnDeserialized]
        private void OnSerializedMethod(System.Runtime.Serialization.StreamingContext context)
        {
            HioCount = -1;
        }

        [NonSerialized] public bool IsDefaultEmpty = false;
    }

    [Serializable]
    public class StringRenderInfoCandidate
    {
        [Serializable]
        public class Item
        {
            public Item(string choiceSrc, byte[] aliasSrc)
            {
                Choice = choiceSrc;

                // Alias を作る
                {
                    if ((aliasSrc != null) && (aliasSrc.Length > 0))
                    {
                        Alias = G3dHioLibProxy.CodePage.HasValue ? Encoding.GetEncoding(G3dHioLibProxy.CodePage.Value).GetString(aliasSrc) : Encoding.Default.GetString(aliasSrc);
                    }
                    else
                    {
                        Alias = Choice;
                    }
                }
            }

            public string Choice { get; private set; }
            public string Alias { get; private set; }
        }

        public string name;

        public List<Item> values;

        public IEnumerable<string> Choices
        {
            get { return values.Select(x => x.Choice); }
        }

        public List<string> defaults;
    }

    [Serializable]
    public class IntRenderInfoCandidate
    {
        public string name;
        public List<int> defaults;
        public int min;
        public int max;
        public bool hasMinMax;
    }

    [Serializable]
    public class FloatRenderInfoCandidate
    {
        public string name;
        public List<float> defaults;
        public float min;
        public float max;
        public bool hasMinMax;
    }

    // Clone しているところはなさそうなので Serializable は外す
    //[Serializable]
    public class RenderInfoPackFromHio
    {
        public RenderInfoPackFromHio()
        {
            IsNull = true;
            StringItems = new StringRenderInfoCandidate[] { };
            IntItems = new IntRenderInfoCandidate[] { };
            FloatItems = new FloatRenderInfoCandidate[] { };
        }

        public RenderInfoPackFromHio(NintendoWare.G3d.Edit.RenderInfoPack src, shading_modelType shadingModel)
        {
            Debug.Assert(src != null);
            var stringRenderInfos = new List<NintendoWare.G3d.Edit.StringRenderInfo>();
            var intRenderInfos = new List<NintendoWare.G3d.Edit.IntRenderInfo>();
            var floatRenderInfos = new List<NintendoWare.G3d.Edit.FloatRenderInfo>();

            if (shadingModel == null)
            {
                // 確認用
                // 来ないはず、(タイミングによってはあり得る？)
                Debug.Assert(false);

                IsNull = true;
            }

            foreach (var renderInfo in src.Items)
            {
                if (renderInfo is NintendoWare.G3d.Edit.StringRenderInfo)
                {
                    stringRenderInfos.Add(renderInfo as NintendoWare.G3d.Edit.StringRenderInfo);
                }
                else if (renderInfo is NintendoWare.G3d.Edit.IntRenderInfo)
                {
                    intRenderInfos.Add(renderInfo as NintendoWare.G3d.Edit.IntRenderInfo);
                }
                else if (renderInfo is NintendoWare.G3d.Edit.FloatRenderInfo)
                {
                    floatRenderInfos.Add(renderInfo as NintendoWare.G3d.Edit.FloatRenderInfo);
                }
            }

            IsNull = false;

            // render_info_slot に choice がない場合のみ使用
            StringItems = (from x in stringRenderInfos
                let slot = shadingModel.RenderInfoSlots().FirstOrDefault(y => y.name == x.Name && string.IsNullOrEmpty(y.choice))
                           where slot != null
                           // 範囲外は "" に置き換え
                let defaults = new List<string>(x.Values.Select(y => x.Items.Any(z => z.Choice == y.DefaultValue) ? y.DefaultValue : ""))
                select new StringRenderInfoCandidate()
                {
                    name = x.Name, values = new List<StringRenderInfoCandidate.Item>(x.Items.Where(y => y != null).Select(z => new StringRenderInfoCandidate.Item(z.Choice, z.Alias)).Distinct()),
                                       // defaults が空なら slot の default を使う
                    defaults = defaults.Any() || string.IsNullOrEmpty(slot.Default()) ? defaults : G3dDataParser.Tokenize(slot.Default()).Select(y => x.Items.Any(z => z.Choice == y) ? y : "").ToList(),
                                   }).ToArray();

            // render_info_slot に choice がない場合のみ使用
            IntItems = (from x in intRenderInfos
                let slot = shadingModel.RenderInfoSlots().FirstOrDefault(y => y.name == x.Name && string.IsNullOrEmpty(y.choice))
                        where slot != null
                        let defaults = new List<int>(x.Values.Select(y => y.DefaultValue))
                select new IntRenderInfoCandidate()
                {
                    name = x.Name, min = x.Min, max = x.Max, defaults = defaults.Any() ? defaults : G3dDataParser.Tokenize(slot.Default()).Select(y =>
                                                    {
                                                        Debug.Assert(x.HasMinMax);
                                                        int result;
                                                        int.TryParse(y, out result);
                                                        // 範囲外のときは min にする
                                                        if (result < x.Min || x.Max < result)
                                                        {
                                                            result = x.Min;
                                                        }
                                                        return result;
                                                    }).ToList(),
                                    hasMinMax = x.HasMinMax,
                                }).ToArray();

            // render_info_slot に choice がない場合のみ使用
            FloatItems = (from x in floatRenderInfos
                let slot = shadingModel.RenderInfoSlots().FirstOrDefault(y => y.name == x.Name && string.IsNullOrEmpty(y.choice))
                          where slot != null
                          let defaults = new List<float>(x.Values.Select(y => y.DefaultValue))
                select new FloatRenderInfoCandidate()
                {
                    name = x.Name, min = x.Min, max = x.Max, defaults = defaults.Any() ? defaults : G3dDataParser.Tokenize(slot.Default()).Select(y =>
                                                            {
                                                                Debug.Assert(x.HasMinMax);
                                                                float result;
                                                                float.TryParse(y, out result);
                                                                // 範囲外のときは min にする
                        if (result < x.Min || x.Max < result)
                                                                {
                                                                    result = x.Min;
                                                                }
                                                                return result;
                                                            }).ToList(),
                                      hasMinMax = x.HasMinMax,
                                  }).ToArray();
        }

        public StringRenderInfoCandidate[] StringItems { get; set; }
        public IntRenderInfoCandidate[] IntItems { get; set; }
        public FloatRenderInfoCandidate[] FloatItems { get; set; }
        //public NintendoWare.G3d.Edit.RenderInfoPack RenderInfoPack { get; private set; }
        public bool IsNull { get; set; }
    }

    /// <summary>
    /// shader_assignType に対応する。シェーダープログラムごとに一つ生成する。
    /// </summary>
    public class MaterialShaderAssign
    {
        public MaterialShaderAssign(string shaderArchive, string shaderName)
        {
            Set(null, false);
            ShaderDefinitionFileName = shaderArchive;
            ShaderName = shaderName;
        }

        public MaterialShaderAssign(shader_assignType shader_assign, bool setEmptyDefaultRenderInfo)
        {
            Set(shader_assign, setEmptyDefaultRenderInfo);
        }

        public string ShaderDefinitionFileName { get; set; }
        public string ShaderName { get; set; }
        public int Revision { get; set; }
        public List<shader_optionType> ShaderOptions;
        public List<sampler_assignType> SamplerAssigns;
        public List<shader_paramType> ShaderParams;
        public List<attrib_assignType> AttribAssigns;
        public List<RenderInfo> RenderInfos;

        /// <summary>
        /// shader_assignType から設定
        /// </summary>
        private void Set(shader_assignType shader_assign, bool setEmptyDefaultRenderInfo)
        {
            if (shader_assign == null)
            {
                ShaderOptions = new List<shader_optionType>();
                SamplerAssigns = new List<sampler_assignType>();
                ShaderParams = new List<shader_paramType>();
                AttribAssigns = new List<attrib_assignType>();
                ShaderDefinitionFileName = string.Empty;
                ShaderName = string.Empty;
                RenderInfos = new List<RenderInfo>();
                Revision = 0;
            }
            else
            {
                ShaderOptions = new List<shader_optionType>(shader_assign.shader_option_array != null ? shader_assign.shader_option_array.shader_option : Enumerable.Empty<shader_optionType>());
                SamplerAssigns = new List<sampler_assignType>(shader_assign.sampler_assign_array != null ? shader_assign.sampler_assign_array.sampler_assign : Enumerable.Empty<sampler_assignType>());
                ShaderParams = new List<shader_paramType>(shader_assign.ShaderParams());
                AttribAssigns = new List<attrib_assignType>(shader_assign.attrib_assign_array != null ? shader_assign.attrib_assign_array.attrib_assign : Enumerable.Empty<attrib_assignType>());
                ShaderDefinitionFileName = shader_assign.shader_archive;
                ShaderName = shader_assign.shading_model;

                if (shader_assign.render_info_array == null)
                {
                    RenderInfos = new List<RenderInfo>();
                }
                else
                {
                    RenderInfos = shader_assign.render_info_array.render_info.Select(x => ConvertToRenderInfo(x, setEmptyDefaultRenderInfo)).ToList();
                }

                Revision = shader_assign.revision;
            }
        }

        private static RenderInfo ConvertToRenderInfo(render_infoType render_info, bool setEmptyDefaultRenderInfo)
        {
            var values = render_info.Value != null ? G3dDataParser.Tokenize(render_info.Value).ToList() : new List<string>();
            return new RenderInfo()
            {
                name = render_info.name,
                internalValues = values,
                type = render_info.type,
                HioCount = values.Count,
                IsDefaultEmpty = setEmptyDefaultRenderInfo,
            };
        }

        /// <summary>
        /// 未割り当て属性を追加する
        /// </summary>
        public void AddNotAssignedAttributes()
        {
            var attribs = ShadingModel.Attributes().ToDictionary(x => x.id, x => x);
            var attribAssigns = AttribAssigns.ToDictionary(x => x.id, x => x);
            var additionalAttribs = ShadingModel.Attributes().Where(x => !attribAssigns.ContainsKey(x.id)).OrderBy(x => x.index).ToList();
            foreach (var additionalAttrib in additionalAttribs)
            {
                var attrib = new attrib_assignType()
                {
                    id = additionalAttrib.id,
                    index = additionalAttrib.index,
                    index_hint = additionalAttrib.index_hint,
                    attrib_name = string.Empty
                };

                // 割り当て順での整合はとれないので、元の割り当て順を保ちつつ、頂点属性順を参考に位置決めする。
                var i = AttribAssigns.FindIndex(x =>
                {
                    attrib_varType attrib_var;
                    return
                        attribs.TryGetValue(x.id, out attrib_var) &&
                        (attrib_var.index > additionalAttrib.index);
                });
                if (i != -1)
                {
                    AttribAssigns.Insert(i, attrib);
                }
                else
                {
                    AttribAssigns.Add(attrib);
                }
            }
        }

        /// <summary>
        /// shader_assignType へ変換
        /// </summary>
        public shader_assignType To_shader_assign(Material owner, bool materialTemplate)
        {
            if (string.IsNullOrEmpty(ShaderDefinitionFileName) || string.IsNullOrEmpty(ShaderName))
            {
                return null;
            }

            var shader_assign = new shader_assignType();
            shader_assign.shader_archive = ShaderDefinitionFileName;
            shader_assign.shading_model = ShaderName;
            shader_assign.shader_option_array = ShaderOptions.Any() ? new shader_option_arrayType() {shader_option = ShaderOptions.ToArray()} : null;
            var samplerAssigns = SamplerAssigns.Where(x => materialTemplate || !string.IsNullOrEmpty(x.sampler_name)).ToArray();
            shader_assign.sampler_assign_array = samplerAssigns.Length > 0 ? new sampler_assign_arrayType() {sampler_assign = samplerAssigns} : null;
            shader_assign.shader_param_array = ShaderParams.Any() ? new shader_param_arrayType() {shader_param = ShaderParams.ToArray()} : null;
            var attribAssigns = AttribAssigns.Where(x => materialTemplate || !string.IsNullOrEmpty(x.attrib_name)).ToArray();
            shader_assign.attrib_assign_array = attribAssigns.Length > 0 ? new attrib_assign_arrayType() {attrib_assign = attribAssigns} : null;
            shader_assign.render_info_array = To_render_info_array(owner, materialTemplate);
            shader_assign.revision = Revision;
            return shader_assign;
        }

        public render_info_arrayType To_render_info_array(Material owner, bool materialTemplate)
        {
            if (RenderInfos.Any())
            {
                return new render_info_arrayType()
                {
                    render_info = RenderInfos.Select(x =>
                    {
                        // Viewer からの情報で出力が変更される!
                        string[] values = new string[0];
                        if (materialTemplate)
                        {
                            values = x.values.ToArray();
                        }
                        else
                        {
                            switch (x.type)
                            {
                                case render_info_typeType.@string:
                                    if (Viewer.Manager.Instance.IsConnected && !owner.RenderInfoPackFromHio.IsNull)
                                    {
                                        // 接続されて状態がとれていれば候補に含まれるもののみ
                                        var candidate = owner.RenderInfoPackFromHio.StringItems.FirstOrDefault(y => y.name == x.name);
                                        values = candidate != null ? x.values.Intersect(candidate.Choices).ToArray() : x.values.ToArray();
                                    }
                                    else
                                    {
                                        values = x.values.ToArray();
                                    }
                                    break;
                                case render_info_typeType.@int:
                                    values = x.values.ToArray();
                                    break;
                                case render_info_typeType.@float:
                                    values = x.values.ToArray();
                                    break;
                                default:
                                    break;
                            }
                        }

                        return ShaderAssignUtility.To_render_info(x, values);
                    }).ToArray()
                };
            }

            return null;
        }

        public ShaderDefinition ShaderDefinition
        {
            get { return DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == ShaderDefinitionFileName); }
        }

        public shading_modelType ShadingModel
        {
            get
            {
                var shaderDefinition = DocumentManager.ShaderDefinitions.FirstOrDefault(x => x.Name == ShaderDefinitionFileName);
                if (shaderDefinition != null)
                {
                    return shaderDefinition.Data.shading_model_array.shading_model.FirstOrDefault(x => x.name == ShaderName);
                }
                return null;
            }
        }

        public Definition Definition
        {
            get
            {
                var shaderDefinition = ShaderDefinition;
                if (shaderDefinition != null)
                {
                    return shaderDefinition.Definitions.FirstOrDefault(x => x.Name == ShaderName);
                }

                return null;
            }
        }

        public bool PageMode
        {
            get
            {
                var shadingModel = ShadingModel;
                return shadingModel != null && shadingModel.page_array != null;
            }
        }

        /// <summary>
        /// グループの情報
        /// </summary>
        public class GroupInfo
        {
            public string groupName;

            /// <summary>
            /// グループ
            /// null の場合もある
            /// </summary>
            public groupType group;

            /// <summary>
            /// ラベルまたは名前の取得
            /// </summary>
            public string Label
            {
                get
                {
                    if (groupName == string.Empty)
                    {
                        return ShaderTypeUtility.RemainsGroupLabel();
                    }

                    return group == null || string.IsNullOrEmpty(group.Label()) ? groupName : group.Label();
                }
            }
        }

        /// <summary>
        /// グループをUI 表示順に取得。condition は考慮していない。
        /// </summary>
        /// <param name="strayGroup">無所属のグループを含めるかどうか</param>
        public IEnumerable<GroupInfo> GetGroups(bool strayGroup)
        {
            if (ShadingModel == null)
            {
                return Enumerable.Empty<GroupInfo>();
            }

            var shadingModel = ShadingModel;
            var groups = from @group in shadingModel.Groups() select @group.name;

            var pageOrders = shadingModel.Pages().Select((item, index) => new {item, index}).ToDictionary(x => x.item.name, x => new Tuple<int, int>(x.item.Order(), x.index));
            return GroupsWithItems().Concat(groups).Where(x => strayGroup || x != string.Empty).Distinct().Select(x => new GroupInfo()
            {
                groupName = x, group = shadingModel.Groups().FirstOrDefault(y => y.name == x),
            }).OrderBy(x =>
            {
                if (x.group == null)
                {
                    if (x.groupName == "")
                    {
                        return new Tuple<Tuple<int, int>, int>(new Tuple<int, int>(int.MinValue, int.MinValue), ShaderTypeUtility.RemainsGroupOrder());
                    }

                    return new Tuple<Tuple<int, int>, int>(new Tuple<int, int>(int.MinValue, int.MinValue), 0);
                }
                else
                {
                    Tuple<int, int> pageOrder;
                    if (!pageOrders.TryGetValue(x.group.page_name, out pageOrder))
                    {
                        pageOrder = new Tuple<int, int>(int.MinValue, int.MinValue);
                    }
                    return new Tuple<Tuple<int, int>, int>(pageOrder, x.group.Order());
                }
            }).ThenBy(x =>
            {
                if (x.group != null)
                {
                    return x.group.index;
                }

                return int.MinValue;
            }).ThenBy(x => x.groupName).ToArray();
        }

        public IEnumerable<string> GroupsWithItems()
        {
            if (ShadingModel == null)
            {
                return Enumerable.Empty<string>();
            }

            var shadingModel = ShadingModel;

            var attribGroups = from attribute in shadingModel.Attributes() from attribAssign in AttribAssigns where attribute.id == attribAssign.id select attribute.Group();

            var optionGroups = from option_var in shadingModel.Options() where option_var.type != option_var_typeType.dynamic where option_var.OptionVisible() from option in ShaderOptions where option.id == option_var.id select option_var.Group();

            var samplerGroups = from sampler_var in shadingModel.Samplers() where sampler_var.SamplerVisible() from samplerAssign in SamplerAssigns where samplerAssign.id == sampler_var.id select sampler_var.Group();

            var paramGroups = from uniform_var in shadingModel.MaterialUniforms() where uniform_var.IsParamVisible() from param in ShaderParams where param.id == uniform_var.id select uniform_var.Group();

            var renderInfoGroups = from slot in shadingModel.RenderInfoSlots() where slot.RenderInfoVisible() from info in RenderInfos where info.name == slot.name select slot.Group();

            return attribGroups.Concat(optionGroups).Concat(samplerGroups).Concat(paramGroups).Concat(renderInfoGroups);
        }
    }
}
