﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class Bone : GuiObject, IHasUserData
    {
        public boneType Data{ get; private set; }
        public Model Owner { get; private set; }
        public override Document OwnerDocument
        {
            get { return Owner; }
        }
        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public Bone(boneType bone, Model owner, int index) : base(GuiObjectID.Bone, index)
        {
            Data = bone;
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            Children = new List<Bone>();

            // オーナー(モデル)を設定
            Owner = owner;
            // 名前を設定
            Name = bone.name;

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, Owner.BinaryStreams);
        }

        #region Data と重複する情報
        public List<Bone> Children { get; private set; }
        public Bone Parent { get; set; }
        #endregion

        // ルートからの深さ
        private int depth_ = -1;
        public int Depth
        {
            get
            {
                if (depth_ == -1)
                {
                    // 作る
                    depth_ = 0;

                    for(var parent = Parent;parent != null;parent = parent.Parent)
                    {
                        ++ depth_;
                    }
                }

                return depth_;
            }
        }

        public void UpdateData()
        {
            Data.tool_data = ToolData.SaveDefault();
            Data.comment = GetComment();

            // ユーザーデータのシリアライズ
            if (UserDataArray == null ||
                UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
                return;
            }

            Data.user_data_array = new user_data_arrayType();
            this.MakeSerializeData(Data.user_data_array, UserDataArray, Owner.BinaryStreams);
        }

        #region SavedData
        private boneType savedData;
        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;
            savedData = ObjectUtility.Clone(Data);
        }

        public void CopySavedData(Bone source)
        {
            CopyGuiObjectSavedData(source);
            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);
            savedData = source.savedData;
        }

        public bool IsValueChanged<T>(Func<boneType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data));
        }

        public bool IsStringChanged(Func<boneType, string> select)
        {
            return select(savedData) != select(Data);
        }

        public bool IsArrayChanged<T>(Func<boneType, T[]> select)
        {
            return !select(savedData).SequenceEqual(select(Data));
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !BoneGeneralPage.IsModified(this) &&
                !UserDataPage.IsModified(this);
        }
        #endregion
    }
}
