﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using App.Command;
using App.Properties;
using App.Utility;
using ConfigCommon;

namespace App.Data
{
    /// <summary>
    /// ＧＵＩオブジェクトクラス。
    /// </summary>
    public abstract class GuiObject
    {
        /// <summary>
        /// オブジェクトＩＤ。
        /// </summary>
        virtual public GuiObjectID ObjectID { get; private set; }

        private Image objectIcon_;
        public virtual Image ObjectIcon
        {
            get
            {
                if (objectIcon_ == null)
                {
                    objectIcon_ = GetObjectIcon(ObjectID);
                    Debug.Assert(objectIcon_ != null);
                }

                return objectIcon_;
            }
        }

        private static readonly Dictionary<GuiObjectID, Image> images = new Dictionary<GuiObjectID, Image>();
        public static Image GetObjectIcon(GuiObjectID ID)
        {
            Image image;
            if (images.TryGetValue(ID, out image))
            {
                return image;
            }

            image = (Image)Resources.ResourceManager.GetObject("GuiObject_" + ID.ToString(), Resources.Culture);
            images.Add(ID, image);
            return image;
        }

        /// <summary>
        /// 名前。
        /// </summary>
        public virtual string Name { get; set; }

        /// <summary>
        /// インデックス。
        /// </summary>
        public virtual int Index { get; set; }

        /// <summary>
        /// コメント。
        /// </summary>
        public string Comment { get; set; }

        /// <summary>
        /// 編集カラー。
        /// </summary>
        public RgbaColor? EditColor { get; set; }

        /// <summary>
        /// ラベル。
        /// </summary>
        public string Label { get; set; }

        public string ToComment(nw.g3d.nw4f_3dif.commentType comment)
        {
            return comment == null ? "" : (comment.text ?? "");
        }

        public RgbaColor? ToColor(nw.g3d.nw4f_3dif.commentType comment)
        {
            if (comment != null && !string.IsNullOrEmpty(comment.color))
            {
                float[] paramArray = nw.g3d.nw4f_3dif.G3dDataParser.ParseFloatArray(comment.color);
                if (paramArray == null || paramArray.Length != 3)
                {
                    return null;
                }
                else
                {
                    return new RgbaColor(paramArray[0], paramArray[1], paramArray[2], 1.0f);
                }
            }
            else
            {
                return null;
            }
        }

        public string ToLabel(nw.g3d.nw4f_3dif.commentType comment)
        {
            return comment == null ? "" : (comment.label ?? "");
        }

        public void MakeComment(nw.g3d.nw4f_3dif.commentType comment)
        {
            if (!Editable)
            {
                return;
            }

            Comment = ToComment(comment);
            EditColor = ToColor(comment);
            Label = ToLabel(comment);
        }

        /// <summary>
        /// コメントと作業用カラーをnw4f_3difに設定する。
        /// </summary>
        public nw.g3d.nw4f_3dif.commentType GetComment()
        {
            nw.g3d.nw4f_3dif.commentType comment = new nw.g3d.nw4f_3dif.commentType();

            if (!Editable)
            {
                return null;
            }

            if (EditColor == null &&
                string.IsNullOrEmpty(Comment) &&
                string.IsNullOrEmpty(Label))
            {
                return null;
            }

            if (EditColor == null)
            {
                comment.color = string.Empty;
            }
            else
            {
                RgbaColor editColor = (RgbaColor)EditColor;
                float[] colorValue = editColor.ToArray(false);
                if (colorValue.Any() && colorValue.Count() == 3)
                {
                    comment.color = string.Format("{0} {1} {2}", colorValue[0], colorValue[1], colorValue[2]);
                }
                else
                {
                    comment.color = string.Empty;
                }
            }


            comment.text = string.IsNullOrEmpty(Comment) ? string.Empty : ObjectUtility.Clone(Comment);

            comment.label = string.IsNullOrEmpty(Label) ? string.Empty : ObjectUtility.Clone(Label);

            return comment;
        }

        /// <summary>
        /// 所属するドキュメント。
        /// ドキュメントに所属するオブジェクトは適切にオーバーライドします。
        /// </summary>
        public abstract Document OwnerDocument { get; }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            return Name;
        }

        protected GuiObject(GuiObjectID objectID, int index = 0)
        {
            ObjectID = objectID;
            Name = "noname";
            Index = index;

            Comment = "";
            ToolData = new ToolData();
        }

        // 状態
        protected enum Diff
        {
            Modified,
            NotModified,
            MaybeModified,
        }

        // 中間ファイルアップデートされたかどうか？
        private bool _isIfUpdated = false;
        public bool IsIfUpdated
        {
            get { return _isIfUpdated; }
            set { _isIfUpdated = value; }
        }

        private Diff diffFromLastSaved = Diff.MaybeModified;

        public virtual bool CheckOwnerDocumentSavedContents() { return true; }

        /// <summary>
        /// 編集状態かどうか。
        /// </summary>
        public bool IsModifiedObject {
            get
            {
                if (CheckOwnerDocumentSavedContents() && OwnerDocument != null)
                {
                    if (!OwnerDocument.SavedContents.Contains(this))
                    {
                        return true;
                    }
                }

                // 中間ファイルアップデートされている
                if (IsIfUpdated)
                {
                    return true;
                }

                switch (diffFromLastSaved)
                {
                    case Diff.MaybeModified:
                        if (EqualsToSavedData())
                        {
                            diffFromLastSaved = Diff.NotModified;
                            return false;
                        }
                        else
                        {
                            diffFromLastSaved = Diff.Modified;
                            return true;
                        }
                    case Diff.Modified:
                        return true;
                }

                return false;
            }
        }

        // 編集時に呼ぶ
        public void SetMaybeModified()
        {
            diffFromLastSaved = Diff.MaybeModified;
        }

        // 保存時に呼ぶ
        public void SetNotModified()
        {
            diffFromLastSaved = Diff.NotModified;

            _isIfUpdated = false;
        }

        /// <summary>
        /// 最初に変更したコマンド
        /// </summary>
        public EditCommand FirstModifiedCommand { get; set; }

        protected ToolData ToolData { get; set; }

        public virtual bool Editable
        {
            get
            {
                return true;
            }
        }

        // 名前変更できるか
        public virtual bool Renamable { get { return Editable; } }

        // 参照されているオブジェクト名変更コマンドを返す
        public virtual ICommand CreateReferenceObjectRenameCommand(Document srcObject, string oldName, string oldDirectory, string newName, string newExt, string newDirectory)
        {
            Debug.Assert(false);
            return null;
        }

        private static readonly HashSet<GuiObjectID> isAnimationHashSet_ = new HashSet<GuiObjectID>()
        {
            GuiObjectID.SkeletalAnimation,
            GuiObjectID.ShapeAnimation,
            GuiObjectID.ShaderParameterAnimation,
            GuiObjectID.ColorAnimation,
            GuiObjectID.TextureSrtAnimation,
            GuiObjectID.MaterialVisibilityAnimation,
            GuiObjectID.BoneVisibilityAnimation,
            GuiObjectID.TexturePatternAnimation,
            GuiObjectID.SceneAnimation,
            GuiObjectID.CameraAnimation,
            GuiObjectID.LightAnimation,
            GuiObjectID.FogAnimation,
            GuiObjectID.MaterialAnimation,
        };

        public static bool IsAnimation(GuiObjectID guiObjectID)
        {
            return isAnimationHashSet_.Contains(guiObjectID);
        }

        public static bool IsModelAnimation(GuiObjectID guiObjectID)
        {
            switch (guiObjectID)
            {
                case GuiObjectID.SkeletalAnimation:
                case GuiObjectID.ShapeAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                case GuiObjectID.MaterialVisibilityAnimation:
                case GuiObjectID.BoneVisibilityAnimation:
                case GuiObjectID.TexturePatternAnimation:
                case GuiObjectID.MaterialAnimation:
                    return true;
            }
            return false;
        }

        public string savedName { get; private set; }
        public string savedComment { get; private set; }
        public string savedLabel { get; private set; }
        public RgbaColor? savedEditColor { get; private set; }

        /// <summary>
        /// 保存済みのデータを更新する。
        /// </summary>
        public virtual void UpdateSavedData()
        {
            savedName = Name;
            savedComment = Comment;
            savedLabel = Label;
            if (EditColor.HasValue)
            {
                savedEditColor = EditColor.Value;
            }
            else
            {
                savedEditColor = null;
            }
        }

        public void CopyGuiObjectSavedData(GuiObject source)
        {
            savedName = source.savedName;
            savedComment = source.savedComment;
            savedLabel = source.savedLabel;
            savedEditColor = source.savedEditColor;
        }


        public virtual bool EqualsToSavedData()
        {
            return savedName == Name &&
                savedComment == Comment &&
                savedLabel == Label &&
                (savedEditColor.HasValue == EditColor.HasValue &&
                (!savedEditColor.HasValue || savedEditColor.Value == EditColor.Value));
        }
    }

    public class DummyObject : GuiObject
    {
        public override Document OwnerDocument
        {
            get
            {
                //throw new NotImplementedException();
                Debug.Assert(false);
                return null;
            }
        }

        public DummyObject() : base(GuiObjectID.DummyObject)
        {
        }

        public static DummyObject TheDummyObject = new DummyObject();
        public static GuiObjectGroup TheDummyObjectGroup = new GuiObjectGroup(TheDummyObject);
    }
}
