﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.Concurrent;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Threading;
using System.Windows.Forms;
using App.ConfigData;
using App.Utility;
using App.res;
using ConfigCommon;
using App.Controls;

namespace App.Data
{
    public abstract class Document : GuiObject, IDisposable, NintendoWare.G3d.Edit.IEditTarget
    {
        public static int DocumentCounterForDebug = 0;
        public int IdForDebug = 0;
        protected Document(GuiObjectID objectID) :
            base(objectID)
        {
            IsAttached = false;
            IsTemporary = false;
            ResFileKey = 0;
            FileLocation = "";
            LastOpenedWriteTime = DateTime.MinValue;
            DocumentAdd += EnableWatcher;
            DocumentRemove += DisableWatcher;

            DocumentSwapAdd += x => EnableWatcher();
            DocumentSwapRemove += DisableWatcher;
            IdForDebug = DocumentCounterForDebug++;
        }

        public void Dispose()
        {
            DisableWatcher();
        }

        /// <summary>
        /// コンテンツを列挙します。
        /// </summary>
        public virtual IEnumerable<GuiObject> ContentObjects { get { yield return this; } }

        public bool IsTemporary { get; set; }

        public override string Name
        {
            get
            {
                return base.Name;
            }
            set
            {
                base.Name = value;
                UpdateWatcher();
            }
        }

        public string fileLocation = "";
        public string FileLocation
        {
            get
            {
                Debug.Assert(fileLocation != null);
                return fileLocation;
            }
            set
            {
                fileLocation = value;
                UpdateWatcher();
                if (!string.IsNullOrEmpty(FilePath) && File.Exists(FilePath))
                {
                    LastOpenedWriteTime = File.GetLastWriteTime(FilePath);
                }
            }
        }

        // ファイル出力できるか？
        public virtual bool IsFileOutputable { get { return true; } }

        // ファイル出力できない時の理由
        public virtual string FileNotOutputErrorMessage
        {
            get
            {
                Debug.Assert(IsFileOutputable == false);
                //throw new NotImplementedException();
                Debug.Assert(false);
                return string.Empty;
            }
        }

        #region リロード
        /// <summary>
        /// Document.DisableWatcher() でのウォッチャー破棄の抑制クラス
        /// インスタンス生成から破棄まで期間において対象 Document のウォッチャー破棄は抑制される。
        /// 破棄が抑制されたウォッチャーは本インスタンス破棄と同時に破棄される。
        /// </summary>
        private class SuppressWatcherDiposing : IDisposable
        {
            private class Key : IEquatable<Key>
            {
                readonly WeakReference<Document> document_;
                readonly int hashCode_;

                public Key(Document document)
                {
                    document_ = new WeakReference<Document>(document);
                    hashCode_ = (document != null) ? document.GetHashCode() : 0;
                }

                public bool Equals(Key key)
                {
                    return
                        document_.TryGetTarget(out var self) &&
                        key.document_.TryGetTarget(out var other) &&
                        (self == other);
                }

                public override int GetHashCode()
                {
                    return document_.TryGetTarget(out var self) ? self.GetHashCode() : hashCode_;
                }
            }

            private class Value
            {
                public long ReferenceCount;
                public ConcurrentQueue<FileSystemWatcher> Watchers;
            }

            private Document document_;
            private Value value_;
            private static ConcurrentDictionary<Key, Value> instances_ = new ConcurrentDictionary<Key, Value>();

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="document">抑制対象のドキュメント</param>
            public SuppressWatcherDiposing(Document document)
            {
                document_ = document;
                value_ = instances_.GetOrAdd(
                    new Key(document),
                    new Value()
                    {
                        ReferenceCount = 0,
                        Watchers = new ConcurrentQueue<FileSystemWatcher>(),
                    });
                Interlocked.Increment(ref value_.ReferenceCount);
                Interlocked.MemoryBarrier();
            }

            /// <summary>
            /// インスタンスの破棄
            /// 抑制したウォッチャーが同時に破棄される。
            /// </summary>
            public void Dispose()
            {
                Interlocked.MemoryBarrier();
                if (Interlocked.Decrement(ref value_.ReferenceCount) == 0)
                {
                    while (value_.Watchers.TryDequeue(out var watcher))
                    {
                        Document.DisposeWatcher(document_, watcher);
                    }
                }
            }

            /// <summary>
            /// document が抑制対象であれば watcher を登録する
            /// ここで登録された watcher は SuppressWatcherDiposing.Dispose() で破棄される。
            /// watcher が document が生成したものでなければ動作不定。
            /// </summary>
            static public bool AddIfSuppressed(Document document, FileSystemWatcher watcher)
            {
                if (instances_.TryGetValue(new Key(document), out var value) && (Interlocked.Read(ref value.ReferenceCount) != 0))
                {
                    value.Watchers.Enqueue(watcher);
                    return true;
                }
                return false;
            }
        }

        public bool NeedsReload { get; set; } = false;

        public FileSystemWatcher watcher;
        public bool IsWatched
        {
            get
            {
                return (watcher != null) && watcher.EnableRaisingEvents;
            }
        }

        public void EnableWatcher()
        {
            if (ConfigData.ApplicationConfig.UserSetting.ReloadOptions.GetOption(ObjectID).Reload)
            {
                // 監視はユーザースレッドではなく、システムスレッドで行う。
                // ユーザースレッドで行うと、通知イベント処理中にブロックされた通知が遅れて届くため。
                watcher = new FileSystemWatcher();
                watcher.NotifyFilter = NotifyFilters.LastWrite | NotifyFilters.CreationTime | NotifyFilters.FileName;
                watcher.SynchronizingObject = null; // null を指定してシステムスレッドで監視する。
                watcher.Changed += watcherChanged;
                watcher.Renamed += watcherChanged;
                watcher.Created += watcherChanged;
                UpdateWatcher();
            }
        }

        public void DisableWatcher()
        {
            if (watcher != null)
            {
                if (!SuppressWatcherDiposing.AddIfSuppressed(this, watcher))
                {
                    DisposeWatcher(this, watcher);
                }
                watcher = null;
            }
        }

        private static void DisposeWatcher(Document owner, FileSystemWatcher watcher)
        {
            if ((owner != null) && (watcher != null))
            {
                watcher.EnableRaisingEvents = false;
                watcher.Changed -= owner.watcherChanged;
                watcher.Renamed -= owner.watcherChanged;
                watcher.Created -= owner.watcherChanged;
                watcher.Dispose();
            }
        }

        public void SuspendWatcher()
        {
            if (watcher != null)
            {
                watcher.EnableRaisingEvents = false;
            }
        }

        public void ResumeWatcher()
        {
            if (watcher != null)
            {
                watcher.EnableRaisingEvents = true;
            }
        }

        private void watcherChanged(object sender, FileSystemEventArgs args)
        {
            Reload(args.FullPath, args.ChangeType);
        }

        public void LazyReload()
        {
            Reload(FilePath, WatcherChangeTypes.All);
        }

        private void Reload(string path, WatcherChangeTypes changeType)
        {
            // ファイル変更通知カウンタをインクリメント。
            if (Interlocked.Increment(ref ExternalModifications) == 1)
            {
                // ユーザースレッドに処理を投げるだけ。
                Action<long> action = null;
                action = (watcherId) =>
                {
                    bool retry = false;
                    do
                    {
                        // n と retry には依存関係があるためメモリバリア不要。

                        // リロード対象かを確認。
                        var n = Interlocked.Read(ref ExternalModifications);
                        if ((n != 0) && (watcher != null) && (watcherId == Interlocked.Read(ref WatcherId)) && FileExsists && (path == FilePath))
                        {
                            // リロード対象。
                            break;
                        }

                        // リロード対象外だったが、確認中にファイル変更通知が届いていれば (カウンタ値が更新されていれば) 再確認。
                        retry = Interlocked.CompareExchange(ref ExternalModifications, 0, n) != n;
                        if (!retry)
                        {
                            // リロード対象外。
                            return;
                        }
                    }
                    while (retry);

                    // 稀にファイルハンドルの競合が起こるので、他のアプリが手放すまで後回し。
                    try
                    {
                        using (var stream = File.Open(path, FileMode.Open, FileAccess.Read))
                        { }
                    }
                    catch (Exception e)
                    {
                        DebugConsole.WriteLine("wacherChanged " + e.Message);

                        // 後回し。
                        // 蓄積した通知カウントを 1 に変更。
                        // ExternalModifications と TheApp.MainFrame.BeginInvoke() には依存関係がないためメモリバリアが必要。
                        Interlocked.Exchange(ref ExternalModifications, 1);
                        Interlocked.MemoryBarrier();
                        TheApp.MainFrame.BeginInvoke(action, watcherId);
                        return;
                    }

                    ReloadAfterCanFocus(path);
                    NeedsReload = false;
                };
                if (TheApp.MainFrame.InvokeRequired)
                {
                    TheApp.MainFrame.BeginInvoke(action, Interlocked.Read(ref WatcherId));
                }
                else
                {
                    action(Interlocked.Read(ref WatcherId));
                }
            }
        }

        private void ReloadAfterActiveForm(string path)
        {
            if ((ApplicationConfig.UserSetting.ReloadOptions.GetOption(ObjectID).ShowDialog || ContentsModified) &&
                System.Windows.Forms.Form.ActiveForm == null)
            {
                // Activeでないとダイアログが角のほうに出る可能性がある。
                TheApp.MainFrame.OnActivatedActions += () => ReloadAfterCanFocus(path);
            }
            else
            {
                ReloadAfterCanFocus(path);
            }
        }

        private void ReloadAfterCanFocus(string path)
        {
            if (!TheApp.MainFrame.CanFocus || ControlUtility.IsAnyCapturing() || App.AppContext.UpdateFromOutSideBlockCounter.CheckBlock)
            {
                EventHandler waitCanFocus = null;
                waitCanFocus = (s, e) =>
                    {
                        if (TheApp.MainFrame.CanFocus && !ControlUtility.IsAnyCapturing() && !App.AppContext.UpdateFromOutSideBlockCounter.CheckBlock)
                        {
                            Application.Idle -= waitCanFocus;
                            ReloadAfterActiveForm(path);
                        }
                    };
                Application.Idle += waitCanFocus;
            }
            else
            {
                // リロード直前のカウンタ値を取得しておく。
                // n と DocumentManager.Reload() には依存関係がないためメモリバリアが必要。
                var n = Interlocked.Read(ref ExternalModifications);
                Interlocked.MemoryBarrier();

                // DocumentManager.Reload() では下記順でウォッチャが操作される。
                // 1. 現在のドキュメントの DisableWatcher() が呼ばれる
                // 2. リロードされた新規ドキュメントで EnableWatcher() が呼ばれる
                // 1 と 2　の間にファイル更新があると更新を取りこぼすため、リロード中はウォッチャ破棄を抑制しておく。
                bool needsReloadAgain;
                using (var swd = new SuppressWatcherDiposing(this))
                {
                    // リロード。
                    DocumentManager.Reload(path, this, (ApplicationConfig.UserSetting.ReloadOptions.GetOption(ObjectID).ShowDialog || ContentsModified));

                    // ファイル変更カウンタがリロード前後で等しければゼロにリセット。
                    // 不一致はリロード中にファイル変更が行われたことを意味するので、変更を取り込むために再リロードが必要。
                    // DocumentManager.Reload() と n には依存関係がないためメモリバリアが必要。
                    Interlocked.MemoryBarrier();
                    needsReloadAgain = Interlocked.CompareExchange(ref ExternalModifications, 0, n) != n;
                }

                if (needsReloadAgain)
                {
                    // 再リロード処理に移るので、現在のドキュメントのカウンタをゼロにリセットする。
                    // ExternalModifications と再リロード処理には依存関係はあるが、この関数内での依存関係はないためメモリバリアが必要。
                    Interlocked.Exchange(ref ExternalModifications, 0);
                    Interlocked.MemoryBarrier();

                    // リロード中の変更を取り込むために再リロードする。
                    if (!DocumentManager.Documents.Contains(this))
                    {
                        // 直前のリロードによって差し替わったドキュメントに変更を取り込む。
                        var document = DocumentManager.Documents.FirstOrDefault(x => x.FilePath == path);
                        if (document != null)
                        {
                            document.Reload(path, WatcherChangeTypes.All);
                        }
                    }
                    else
                    {
                        // 直前のリロードでの失敗等の理由でドキュメントが差し変わらなかったため、現在のドキュメントに変更を取り込む。
                        Reload(path, WatcherChangeTypes.All);
                    }
                }
            }
        }

        public DateTime LastOpenedWriteTime { get; set; }
        private long WatcherId = 0;
        private long ExternalModifications = 0;

        /// <summary>
        /// 更新監視の更新
        /// </summary>
        public void UpdateWatcher()
        {
            bool enable = !string.IsNullOrEmpty(FilePath) && System.IO.File.Exists(FilePath);

            if (watcher != null)
            {
                // 監視対象が変わったら ID も変える。
                var needsUpdate =
                    !string.Equals(watcher.Path, FileLocation, StringComparison.OrdinalIgnoreCase) ||
                    !string.Equals(watcher.Filter, FileName, StringComparison.OrdinalIgnoreCase);
                if (needsUpdate)
                {
                    Interlocked.Increment(ref WatcherId);
                }

                if (enable)
                {
                    watcher.Path = FileLocation;
                    watcher.Filter = FileName;
                    watcher.EnableRaisingEvents = true;
                }
                else
                {
                    watcher.EnableRaisingEvents = false;
                }
            }
        }
        #endregion

        // ファイルが存在するかどうか
        public bool FileExsists
        {
            get
            {
                return !string.IsNullOrEmpty(FilePath) && System.IO.File.Exists(FilePath);
            }
        }

        public string fileDotExt;
        public string FileDotExt
        {
            get
            {
                return fileDotExt;
            }
            set
            {
                fileDotExt = value;
                UpdateWatcher();
            }
        }

        public string FileExt { get { return FileDotExt.Substring(1); } }

        public string FilePath
        {
            get
            {
                if (!string.IsNullOrEmpty(FileLocation))
                {
                    return Path.Combine(FileLocation, FileName);
                }
                else
                {
                    return string.Empty;
                }
            }
        }

        /*
        public string ProjectPath
        {
            get
            {
                string path = string.Empty;
                if (!string.IsNullOrEmpty(FilePath))
                {
                    if (BaseName != null)
                    {
                        string basePath = ConfigData.ApplicationConfig.GetSearchPaths().FirstOrDefault(x => x.Name == BaseName).path + "\\";
                        path = PathUtility.MakeRelativePath(basePath, FilePath);
                    }
                    else if (BasePath != null)
                    {
                        var expanded = Environment.ExpandEnvironmentVariables(BasePath) + "\\";
                        path = PathUtility.MakeRelativePath(expanded, FilePath);
                    }
                    else
                    {
                        if (string.IsNullOrEmpty(DocumentManager.ProjectDocument.FilePath) == false)
                        {
                            path = PathUtility.MakeRelativePath(DocumentManager.ProjectDocument.FilePath, FilePath);
                        }
                    }
                }
                return path;
            }
        }*/

        public string GetProjectPath(out Project.RelativeType relative)
        {
            string path = string.Empty;
            relative = Project.RelativeType.Project;
            if (!string.IsNullOrEmpty(FilePath))
            {
                if (Path.GetFullPath(FilePath).Contains(Path.GetFullPath(DocumentManager.StartupFolderPath)))
                {
                    path = PathUtility.MakeRelativePath(Path.GetFullPath(DocumentManager.StartupFolderPath) + "\\", Path.GetFullPath(FilePath));
                    relative = Project.RelativeType.StartUpFolder;
                }
                else if (BaseName != null)
                {
                    string basePath = ConfigData.ApplicationConfig.GetSearchPaths().FirstOrDefault(x => x.Name == BaseName).path + "\\";
                    path = PathUtility.MakeRelativePath(basePath, FilePath);
                    relative = Project.RelativeType.SearchPath;
                }
                else if (BasePath != null)
                {
                    var expanded = Environment.ExpandEnvironmentVariables(BasePath) + "\\";
                    path = PathUtility.MakeRelativePath(expanded, FilePath);
                    relative = Project.RelativeType.SearchPath;
                }
                else if (string.IsNullOrEmpty(DocumentManager.ProjectDocument.FilePath) == false)
                {
                    path = PathUtility.MakeRelativePath(DocumentManager.ProjectDocument.FilePath, FilePath);
                }
            }

            return path;
        }

        public string ProjectPath
        {
            get
            {
                Project.RelativeType dummy;
                return GetProjectPath(out dummy);
            }
        }

        /// <summary>
        /// 追加参照パスの名前
        /// </summary>
        public string BaseName { get; set; }

        /// <summary>
        /// モデルで設定された追加参照パス
        /// </summary>
        public string BasePath { get; set; }

        public bool OpenedFromStartUp { get; set; }
        public string FileName
        {
            get
            {
                return Name + FileDotExt;
            }
        }

        /// <summary>
        /// 参照先のドキュメント
        /// </summary>
        public virtual IEnumerable<Document> ReferenceDocuments
        {
            get
            {
                return Enumerable.Empty<Document>();
            }
        }

        /// <summary>
        /// 再帰的に参照しているすべてのドキュメント
        /// </summary>
        public IEnumerable<Document> ReferenceDependDocuments
        {
        	get
        	{
                return ReferenceDocuments.SelectMany(x => Enumerable.Repeat(x, 1).Concat(x.ReferenceDependDocuments)).Distinct();
        	}
        }

        public bool ContentsModified { get { return ContentObjects.Any(x => x.IsModifiedObject); } }

        public void SetFileLocation(string directory = "", string baseName = null, string basePath = null)
        {
            if (FileLocation != directory || baseName != null || basePath != null)
            {
                BaseName = baseName;
                BasePath = basePath;
            }
            if (!string.IsNullOrWhiteSpace(directory))
            {
                try
                {
                    FileLocation = Path.GetFullPath(directory);
                }
                catch (Exception)
                {
                    FileLocation = "";
                }
            }
            else
            {
                FileLocation = "";
            }
        }

        public override Document OwnerDocument
        {
            get { return this; }
        }

        public void OnDocumentAdd()
        {
            if (DocumentAdd != null)
            {
                DocumentAdd();
            }
        }
        public event Action DocumentAdd;

        public void OnDocumentRemove()
        {
            if (DocumentRemove != null)
            {
                DocumentRemove();
            }
        }
        public event Action DocumentRemove;

        public virtual void OnDocumentSwapAdd(Document old)
        {
            OpenedFromStartUp = old.OpenedFromStartUp;
            if (DocumentSwapAdd != null)
            {
                DocumentSwapAdd(old);
            }
        }
        public event Action<Document> DocumentSwapAdd;

        public void OnDocumentSwapRemove()
        {
            if (DocumentSwapRemove != null)
            {
                DocumentSwapRemove();
            }
        }
        public event Action DocumentSwapRemove;

        public void SetContentsNotModified()
        {
            SavedContents.Clear();

            // 編集状態を初期化します。
            foreach (GuiObject content in ContentObjects)
            {
                SavedContents.Add(content);
                content.FirstModifiedCommand = null;
                content.SetNotModified();
            }
        }

        public void SetContentsMaybeModified()
        {
            // 編集状態を未確定にします。
            foreach (GuiObject content in ContentObjects)
            {
                content.FirstModifiedCommand = null;
                content.SetMaybeModified();
            }
        }
        // -----------------------------------------------------------------------------------
        // 以下、HIO用。暫定

        private UInt32 key;
        #region ランタイム連携
        /// <summary>
        /// IEditTarget
        /// </summary>
        public UInt32 Key
        {
            get
            {
                return key;
            }
            set
            {
                if (key != value)
                {
                    DebugConsole.WriteLine(string.Format("Key of {0} {1}: {2}", FileName, IdForDebug, value));
                    key = value;
                }
            }
        }

        private uint resFileKey;
        /// <summary>
        /// IEditTarget
        /// </summary>
        public uint ResFileKey {
            get
            {
                return resFileKey;
            }
            set
            {
                if (resFileKey != value)
                {
                    DebugConsole.WriteLine(string.Format("ResFileKey of {0} {1}: {2}", FileName, IdForDebug, value));
                    resFileKey = value;
                }
            }
        }

        /// <summary>
        /// IEditTarget
        /// </summary>
        public virtual bool IsAttached { get; set; }

        /// <summary>
        /// IEditTarget
        /// </summary>
        public bool		IsDeleted	{ get; set; }

        /// <summary>
        /// IEditTarget
        /// </summary>
        public virtual void ResetStatus()
        {
            ResFileKey = 0;
            IsAttached = false;
        }

        public bool HioLoaded;

        /// <summary>
        /// Unload したタイミングで呼ばれる。ResetStatus から分離
        /// </summary>
        public virtual void UnloadedFromHio()
        {
            HioLoaded = false;
        }

#if false
        public uint HashCodeFromFileName()
        {
            return HashCodeFromString(FileName, FileLocation);
        }

        private static readonly Dictionary<Tuple<string, string>, uint> nameUniqueDict_ = new Dictionary<Tuple<string, string>, uint>();
        public static uint HashCodeFromString(string name, string location)
        {
            var key = new Tuple<string, string>(name, location);
            uint hashValue;
            {
                if (nameUniqueDict_.TryGetValue(key, out hashValue) == false)
                {
                    hashValue = (uint)nameUniqueDict_.Count;
                    nameUniqueDict_[key] = hashValue;
                }
            }

            // 下位2bit が 00 にならない値
            return (hashValue << 2) | 1;
        }
#endif
        #endregion
        #region 保存確認
        /// <summary>
        /// 保存の確認をする必要があるか
        /// </summary>
        public virtual bool NeedToAskBeforeSave(string filePath)
        {
            return false;
        }

        /// <summary>
        /// キャンセル時は false
        /// </summary>
        public virtual bool ConfirmBeforeSave()
        {
            return true;
        }

        public virtual bool ConfirmPathBeforeSave(string path)
        {
            return true;
        }

        /// <summary>
        /// キャンセル時は false
        /// </summary>
        public virtual bool ConfirmBeforeBinarySave()
        {
            return true;
        }

        /// <summary>
        /// 不正なデータがあるか？チェックし、不正ならメッセージを表示して、Viewer接続を切断する。
        /// </summary>
        public virtual void CheckAndDisConnect()
        {
        }
        #endregion

        #region savedData
        private string savedFileLocation;
        private string savedFileDotExt;
        private string savedBaseName;
        /// <summary>
        /// 保存済みのデータを更新する。
        /// </summary>
        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedFileLocation = FileLocation;
            savedFileDotExt = fileDotExt;
            savedBaseName = BaseName;
        }

        public HashSet<GuiObject> SavedContents = new HashSet<GuiObject>();

        public void CopyDocumentSavedData(Document source)
        {
            CopyGuiObjectSavedData(source);
            savedFileLocation = source.savedFileLocation;
            savedFileDotExt = source.savedFileDotExt;
            savedBaseName = source.savedBaseName;
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return savedFileLocation == fileLocation &&
                savedFileDotExt == fileDotExt; //&&
                //savedBaseName == BaseName;
        }

        public bool IsFilePathModified()
        {
            return !(savedFileLocation == fileLocation &&
                savedFileDotExt == fileDotExt &&
                savedName == Name);
        }
        #endregion
    }
}
