﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Command;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class MaterialVisibilityAnimation : AnimationDocument, IHasUserData
    {
        public mat_visibility_animType Data{ get; private set; }
        public override object nw4f_3difItem
        {
            get { return Data; }
        }
        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public override process_log_arrayType process_log_array
        {
            get { return Data.process_log_array; }
        }

        public override int FrameCount
        {
            get{ return Data.mat_visibility_anim_info.frame_count; }
            set{ Data.mat_visibility_anim_info.frame_count = value; }
        }

        public override bool Loop
        {
            get { return Data.mat_visibility_anim_info.loop; }
            set { Data.mat_visibility_anim_info.loop = value; }
        }

        public override bool IsFrameCountModified
        {
            get { return savedData.frame_count != Data.mat_visibility_anim_info.frame_count; }
        }

        public override bool IsLoopModified
        {
            get { return savedData.loop != Data.mat_visibility_anim_info.loop; }
        }

        // ファイル出力できるか？
        public override bool IsFileOutputable
        {
            get
            {
                return FileNotOutputErrorMessage == null;
            }
        }

        public override string FileNotOutputErrorMessage
        {
            get
            {
                string error = null;
                Action<MaterialVisibilityMatAnim> errorHandler = delegate(MaterialVisibilityMatAnim materialVisibilityMatAnim)
                {
                    error = string.Format(res.Strings.FileNotOutputError_CurveAnimationDocument, materialVisibilityMatAnim.mat_name);
                };
                foreach (var result in QuantizeAnalyse(errorHandler))
                {
                    if (error != null)
                    {
                        return error;
                    }
                }

                return null;
            }
        }

        public override bool HasAnimation
        {
            get { return MaterialVisibilityMatAnims != null && MaterialVisibilityMatAnims.Any(); }
        }

        public override IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            return QuantizeAnalyse(null);
        }

        private IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse(Action<MaterialVisibilityMatAnim> errorHandler)
        {
            foreach (var materialVisiblityMatAnim in MaterialVisibilityMatAnims)
            {
                var curve = new MaterialVisibilityAnimationCurveTreeNodeInfo(this, materialVisiblityMatAnim.mat_name);

                IfQuantizationAnalysisResult result = null;
                try
                {
                    result = AnimationCurveEditCommand.MakeQuantizationAnalysis(this, ObjectID, curve, false);
                }
                catch (Exception)
                {
                    if (errorHandler != null)
                    {
                        errorHandler(materialVisiblityMatAnim);
                    }
                }
                yield return new Tuple<AnimTarget, IfQuantizationAnalysisResult>(materialVisiblityMatAnim, result);
            }
        }

        public override bool IsEditable
        {
            get
            {
                // 注意：
                // MaterialVisibilityAnimationCurveEditPage の UpdateTreeView() をもとに実装しました
                // 方針：
                // なるべく元のソースをいじらないように実装

                bool IsBound = true;
                var ActiveTarget = this;

                var materials = new HashSet<string>((from model in DocumentManager.Models
                                                     where model.AllAnimations.Contains(ActiveTarget)
                                                     from material in model.Materials
                                                     select material.Name).Distinct());

                foreach (var materialVisibilityMatAnim in ActiveTarget.MaterialVisibilityMatAnims)
                {
                    var materialName = materialVisibilityMatAnim.mat_name;

                    IsBound = materials.Contains(materialName);
                    if (IsBound == false)
                    {
                        return IsBound;
                    }
                }

                Debug.Assert(IsBound);
                return IsBound;
            }
        }

        // マテリアルに関係するか
        public override bool IsMaterialRelated { get { return true; } }

        [Serializable]
        public class MaterialVisibilityMatAnim : AnimTarget
        {
            public MaterialVisibilityMatAnim()
            {
                KeyFrames = new List<KeyFrame>();
                QuantizationInfo	= new QuantizationInfo();
                CurveInterpolationType = InterpolationType.Step;
            }

            public string			mat_name{	get; set; }				// mat_vis_mat_anim.mat_name
        }

        public List<MaterialVisibilityMatAnim>	MaterialVisibilityMatAnims{ get; set; }

        #region 圧縮率
        public bool	IsAllFrameCurveConstant { get { return MaterialVisibilityMatAnims.Any(x => x.KeyFrames.Any()) == false; } }
        public int	FrameCompressDataSize   { get { return GetFrameDataSize(false); } }
        public int	FrameUncompressDataSize { get { return GetFrameDataSize(true);  } }
        #endregion

        // 編集コマンド用
        [Serializable]
        public class QuantizeToleranceData
        {
            public float							QuantizeToleranceFrame{ get; set; }
            public List<MaterialVisibilityMatAnim>	MaterialVisibilityMatAnims{ get; set; }
        }

        public MaterialVisibilityAnimation(mat_visibility_animType animation, List<G3dStream> streams) : base(GuiObjectID.MaterialVisibilityAnimation, streams)
        {
            Data = animation;
            Initialize();
        }

        public void Initialize()
        {
            ToolData.Load(Data.tool_data);
            MakeComment(Data.comment);

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, BinaryStreams);

            MakeMaterialVisibilityMatAnims();
        }

        #region Bridge
        public class MaterialVisibilitySwapData : SwapData
        {
            public ToolData ToolData;
            public List<G3dStream> streams;
            public string Comment;
            public RgbaColor? EditColor;
            public mat_visibility_animType Data;
            public List<MaterialVisibilityMatAnim> MaterialVisibilityMatAnims;
            public UserDataArray userData;
        }

        public override SwapData CreateSwapData(object data, List<G3dStream> streams)
        {
            var current = GetSwapData();
            Data = (mat_visibility_animType)data;
            BinaryStreams = streams;
            Initialize();

            return Swap(current, false);
        }

        public override SwapData GetSwapData()
        {
            return new MaterialVisibilitySwapData()
            {
                Data = Data,
                streams = BinaryStreams,
                Comment = Comment,
                EditColor = EditColor,
                MaterialVisibilityMatAnims = MaterialVisibilityMatAnims,
                userData = UserDataArray,
            };
        }

        public override SwapData Swap(SwapData swapData, bool copyHistory)
        {
            var current = (MaterialVisibilitySwapData)GetSwapData();
            var data = (MaterialVisibilitySwapData)swapData;
            Data = data.Data;
            BinaryStreams = data.streams;
            Comment = data.Comment;
            EditColor = data.EditColor;
            MaterialVisibilityMatAnims = data.MaterialVisibilityMatAnims;
            UserDataArray = data.userData;

            if (copyHistory)
            {
                UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);
                UpdateIsModifiedAnimTargetAll();
            }

            return current;
        }
        #endregion

        public override nw4f_3difType Create_nw4f_3difType(bool viewer)
        {
            lock (this)
            {
                UpdateData(viewer);

                //			nw4f_3difType nw4f_3dif = new nw4f_3difType();
                nw4f_3dif_.Item = Data;
                nw4f_3dif_.file_info = null;

                return nw4f_3dif_;
            }
        }

        public void Optimize()
        {
            // ストリームを更新する。
            UpdateData();
            // <hermite_curve>、<linear_curve>、<step_curve> の frame_type と key_type を
            // none に設定しないと、再分析が行われません。
            if (Data.mat_vis_mat_anim_array != null)
            {
                foreach (var anim in Data.mat_vis_mat_anim_array.mat_vis_mat_anim)
                {
                    if (anim.step_curve != null)
                    {
                        anim.step_curve.frame_type = curve_frame_typeType.none;
                        anim.step_curve.key_type = curve_key_typeType.none;
                    }
                }
            }

            try
            {
                // アニメーションに合わせたAnimQuantizationAnalysisOptimizerを呼び出す。
                IfMatVisibilityAnimQuantizationAnalysisOptimizer optimizer = new IfMatVisibilityAnimQuantizationAnalysisOptimizer();
                optimizer.Optimize(Data, BinaryStreams);
            }
            catch(Exception e)
            {
                DebugConsole.WriteLine("Exception Catch : IfMatVisibilityAnimQuantizationAnalysisOptimizer.Optimize() : {0}", e.Message);
            }

            // 最適化後に、データを再作成する。
            this.MakeUserData(Data.user_data_array, BinaryStreams);
            MakeMaterialVisibilityMatAnims();
        }

        private int GetFrameDataSize(bool isBase)
        {
            int size = 0;
            {
                foreach(var materialVisibilityMatAnim in MaterialVisibilityMatAnims)
                {
                    size += materialVisibilityMatAnim.KeyFrames.Count * (isBase ? 4 : IfUtility.GetByteSizeFromCurveFrameType(materialVisibilityMatAnim.QuantizationInfo.frame_type));
                }
            }
            return size;
        }

        public MaterialVisibilityMatAnim GetParamAnimFromId(string materialName)
        {
            return MaterialVisibilityMatAnims.FirstOrDefault(x => x.mat_name == materialName);
        }

        private void MakeMaterialVisibilityMatAnims()
        {
            MaterialVisibilityMatAnims = new List<MaterialVisibilityMatAnim>();

            if ((Data.mat_vis_mat_anim_array == null) ||
                (Data.mat_vis_mat_anim_array.mat_vis_mat_anim == null))
            {
                return;
            }

            foreach (var matVisMatAnim in Data.mat_vis_mat_anim_array.mat_vis_mat_anim)
            {
                var curve = matVisMatAnim.step_curve;

                bool exist = true;
                if (curve == null)
                {
                    curve = new step_curveType();
                    exist = false;
                }

                var newMaterialVisibilityMatAnim = new MaterialVisibilityMatAnim()
                {
                    mat_name	= matVisMatAnim.mat_name,
                    //
                    QuantizationInfo = new QuantizationInfo()
                    {
                        frame_type	= curve.frame_type,
                        key_type	= curve.key_type,
                        scale		= curve.scale,
                        offset		= curve.offset,
                    },
                    pre_wrap	= curve.pre_wrap,
                    post_wrap	= curve.post_wrap,
                    baked	    = curve.baked
                };

                if(exist)
                {
                    const int dataElementCount = 2;		// frame, value

                    var stream = BinaryStreams[curve.stream_index];
                    var data = stream.FloatData;
                    Debug.Assert(data != null);
                    Debug.Assert(data.Count == curve.count * dataElementCount);

                    for(int i = 0;i != curve.count;++ i)
                    {
                        newMaterialVisibilityMatAnim.KeyFrames.Add(
                            new KeyFrame()
                            {
                                Frame	= data[i * dataElementCount + 0],
                                Value	= data[i * dataElementCount + 1],
                            }
                        );
                    }
                }
                else
                {
                    // コンスタントカーブの場合、中間ファイルにはカーブは存在しないので、
                    // base_value を使用して、0フレームにキーフレームを生成する。
                    newMaterialVisibilityMatAnim.KeyFrames.Add(
                        new KeyFrame()
                        {
                            Frame = 0.0f,
                            Value = matVisMatAnim.base_value
                        }
                    );
                }

                MaterialVisibilityMatAnims.Add(newMaterialVisibilityMatAnim);
            }
        }

        public void UpdateData(bool viewer = false)
        {
            ToolData.Load(Data.tool_data);
            Data.comment = GetComment();

            if (Data.mat_vis_mat_anim_array == null)
            {
                Data.mat_vis_mat_anim_array = new mat_vis_mat_anim_arrayType();
            }

            var	tempMatVisMatAnims	= new List<mat_vis_mat_animType>();
            var	tempStreams			= new List<G3dStream>();

            int targetIndex = 0;    // stream_indexとして使用します。
            foreach (var materialVisibilityMatAnim in MaterialVisibilityMatAnims)
            {
                if (materialVisibilityMatAnim.KeyFrames.Any() == false)
                {
                    // 保存しない
                    continue;
                }

//				if (materialVisibilityMatAnim.KeyFrames.Any())
                {
                    // データ
                    {
                        var matVisMatAnim = new mat_vis_mat_animType()
                        {
                            mat_name	= materialVisibilityMatAnim.mat_name,
                            base_value	= materialVisibilityMatAnim.GetBaseValue(),
                            step_curve	= new step_curveType()
                            {
                                count			= materialVisibilityMatAnim.KeyFrames.Count,
                                frame_type		= materialVisibilityMatAnim.QuantizationInfo.frame_type,
                                key_type		= materialVisibilityMatAnim.QuantizationInfo.key_type,
                                scale			= materialVisibilityMatAnim.QuantizationInfo.scale,
                                offset			= materialVisibilityMatAnim.QuantizationInfo.offset,
                                pre_wrap		= materialVisibilityMatAnim.pre_wrap,
                                post_wrap		= materialVisibilityMatAnim.post_wrap,
                                baked           = materialVisibilityMatAnim.baked,
                                stream_index = targetIndex
                            }
                        };

                        tempMatVisMatAnims.Add(matVisMatAnim);
                    }

                    // ストリーム
                    {
                        // コンスタントカーブかどうかをチェックする
                        {
                            float base_value = 0.0f;
                            if (materialVisibilityMatAnim.KeyFrames.Any() == false)
                            {
                                tempMatVisMatAnims.Last().step_curve = null;
                                continue;
                            }
                            else if (materialVisibilityMatAnim.KeyFrames.IsConstantCurve(out base_value))
                            {
                                if (viewer)
                                {
                                    // ランタイム連携用に水平なカーブを出力する
                                    tempMatVisMatAnims.Last().step_curve = new step_curveType()
                                    {
                                        count = 2,
                                        frame_type = curve_frame_typeType.none,
                                        key_type = curve_key_typeType.none,
                                        scale = 1,
                                        offset = 0,
                                        pre_wrap = curve_wrapType.clamp,
                                        post_wrap = curve_wrapType.clamp,
                                        baked = false,
                                        stream_index = targetIndex,
                                    };
                                    var stream = new G3dStream()
                                    {
                                        type = stream_typeType.@float,
                                        column = 2,
                                    };

                                    stream.FloatData.Add(0); // フレーム
                                    stream.FloatData.Add(base_value);

                                    stream.FloatData.Add(1); // フレーム
                                    stream.FloatData.Add(base_value);
                                    tempStreams.Add(stream);
                                    ++targetIndex;
                                }
                                else
                                {
                                    // コンスタントカーブなのでベース値を保存し、カーブを削除する
                                    tempMatVisMatAnims.Last().step_curve = null;
                                    tempMatVisMatAnims.Last().base_value = base_value;
                                }
                                continue;
                            }
                        }

                        var tempStream = new G3dStream()
                        {
                            type	= stream_typeType.@float,
                            column	= 2
                        };

                        foreach(var keyFrame in materialVisibilityMatAnim.KeyFrames)
                        {
                            tempStream.FloatData.Add(keyFrame.Frame);
                            tempStream.FloatData.Add(keyFrame.Value);
                        }

                        tempStreams.Add(tempStream);
                    }

                    ++ targetIndex;
                }
            }

            //  リストを配列に変換して設定
            {
                Data.mat_vis_mat_anim_array.length = tempMatVisMatAnims.Count;
                Data.mat_vis_mat_anim_array.mat_vis_mat_anim = tempMatVisMatAnims.ToArray();

                if (Data.mat_vis_mat_anim_array.length == 0)
                {
                    Data.mat_vis_mat_anim_array = null;
                }
            }

            // ユーザーデータのシリアライズ
            if (UserDataArray.Data == null ||
                UserDataArray.Data.Count == 0)
            {
                Data.user_data_array = null;
            }
            else
            {
                Data.user_data_array = new user_data_arrayType();
                this.MakeSerializeData(Data.user_data_array, UserDataArray, tempStreams);
            }

            // ストリームの設定
            {
                BinaryStreams = tempStreams;

                // ストリームのソート(や削除等)を行う。
                nw.g3d.iflib.StreamUtility.SortStream(Data, BinaryStreams);

                Data.stream_array = null;
            }
        }

        public override EditCommand CreateUpdateBindCommand()
        {
            bool update = false;
            var newMatVisibilityMatAnims = new List<MaterialVisibilityMatAnim>();

            // バインドされたマテリアルのグループ
            var materials = (from model in DocumentManager.Models
                             where model.AllAnimations.Contains(this)
                             from material in model.Materials
                             select material.Name).ToArray();

            var names = new HashSet<string>(materials);

            var oldMatAnimDictionary = new Dictionary<string, MaterialVisibilityMatAnim>();

            // 既存のもの
            foreach (var anim in MaterialVisibilityMatAnims)
            {
                if (names.Contains(anim.mat_name) || anim.ExportType != CurveExportType.Ignored)
                {
                    oldMatAnimDictionary.Add(anim.mat_name, anim);
                    names.Add(anim.mat_name);
                }
                else
                {
                    update = true;
                }
            }

            // マテリアル名順に並べる
            foreach (var material in names.OrderBy(x => x))
            {
                if (oldMatAnimDictionary.ContainsKey(material))
                {
                    newMatVisibilityMatAnims.Add(oldMatAnimDictionary[material]);
                }
                else
                {
                    newMatVisibilityMatAnims.Add(new MaterialVisibilityMatAnim()
                    {
                        mat_name = material,
                    });
                    update = true;
                }
            }

            if (update)
            {
                // 編集フラグを更新しない
                return new GeneralGroupReferenceEditCommand<DummyObject>(
                    DummyObject.TheDummyObjectGroup,
                    GuiObjectID.DummyObject,
                    Enumerable.Repeat<DummyObject>(DummyObject.TheDummyObject, 1),
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var temp = MaterialVisibilityMatAnims;
                        MaterialVisibilityMatAnims = newMatVisibilityMatAnims;
                        newMatVisibilityMatAnims = temp;

                        // 変更マークを更新する
                        UpdateIsModifiedAnimTargetAll();

                        // シリアライズされるデータに変更はないので Viewer に再転送する必要はない
                    });
            }

            return null;
        }

        #region savedData
        private mat_visibility_anim_infoType savedData;
        public List<MaterialVisibilityMatAnim> savedMaterialVisiblityMatAnims { get; private set; }

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data.mat_visibility_anim_info);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            SavedCurveCount = 0;
            foreach (var animTarget in MaterialVisibilityMatAnims)
            {
                animTarget.IsModified = false;
                if (NotEmpty(animTarget))
                {
                    animTarget.IsSaved = true;
                    SavedCurveCount++;
                }
                else
                {
                    animTarget.IsSaved = false;
                }
            }
            savedMaterialVisiblityMatAnims = ObjectUtility.Clone(MaterialVisibilityMatAnims);
        }

        public void CopySavedData(MaterialVisibilityAnimation source)
        {
            CopyDocumentSavedData(source);
            savedData = source.savedData;

            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !SavedUserDataArray.IsSame(UserDataArray);

            SavedCurveCount = source.SavedCurveCount;
            savedMaterialVisiblityMatAnims = source.savedMaterialVisiblityMatAnims;

            UpdateIsModifiedAnimTargetAll();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !MaterialVisibilityAnimationGeneralPage.IsModified(this) &&
                !IsCurvesModified() &&
                !UserDataPage.IsModified(this);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var MaterialVisibilityMatAnim in MaterialVisibilityMatAnims)
            {
                var saved = savedMaterialVisiblityMatAnims.FirstOrDefault(x => x.mat_name == MaterialVisibilityMatAnim.mat_name);
                UpdateIsModifiedAnimTarget(MaterialVisibilityMatAnim, saved);
            }
        }

        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(MaterialVisibilityMatAnim current, MaterialVisibilityMatAnim saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }

        public bool NotEmpty(MaterialVisibilityMatAnim current)
        {
            return current.KeyFrames.Count > 0;
        }

        public bool IsValueChanged<T>(Func<mat_visibility_anim_infoType, T> select) where T : struct
        {
            return !select(savedData).Equals(select(Data.mat_visibility_anim_info));
        }

        public bool IsStringChanged(Func<mat_visibility_anim_infoType, string> select)
        {
            return select(savedData) != select(Data.mat_visibility_anim_info);
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var animTarget in MaterialVisibilityMatAnims)
            {
                if (animTarget.IsModified)
                {
                    return true;
                }
                if (animTarget.IsSaved)
                {
                    savedCount++;
                }
            }

            return savedCount != SavedCurveCount;
        }
        #endregion

        public override void OnDocumentSwapAdd(Document old)
        {
            CopySavedData((MaterialVisibilityAnimation)old);
            base.OnDocumentSwapAdd(old);
        }
    }
}
