﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;

namespace App.Data
{
    public sealed class FogAnimation : GuiObject, IHasUserData, ISceneAnimationObject, IQuantizeAnalysis
    {
        public fog_animType Data { get; private set; }
        public SceneAnimation Owner { get; private set; }
        public override Document OwnerDocument
        {
            get { return Owner; }
        }

        public UserDataArray UserDataArray { get; set; }
        public UserDataArray SavedUserDataArray { get; set; }
        public bool UserDataArrayChanged { get; set; }

        public bool HasAnimation
        {
            get
            {
                return FogAnimTargets != null && FogAnimTargets.Any();
            }
        }
        // ファイル出力できるか？
        public bool IsFileOutputable
        {
            get
            {
                return FileNotOutputMessage == null;
            }
        }

        public string FileNotOutputMessage
        {
            get
            {
                // 量子化エラー
                string error = null;
                Action<FogAnimationTarget> errorHandler = delegate(FogAnimationTarget animTarget)
                {
                    error = string.Format(res.Strings.FileNotOutputError_FogAnimation, Name, animTarget.targetType.ToString());
                };

                foreach (var tuple in QuantizeAnalyse(errorHandler))
                {
                    if (error != null)
                    {
                        return error;
                    }
                }

                return null;
            }
        }

        public IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse()
        {
            return QuantizeAnalyse(null);
        }

        public IEnumerable<Tuple<AnimTarget, IfQuantizationAnalysisResult>> QuantizeAnalyse(Action<FogAnimationTarget> errorHandler)
        {
            foreach (var animTarget in FogAnimTargets)
            {
                var curve = new FogAnimationCurveTreeNodeInfo(this, animTarget.targetType);

                IfQuantizationAnalysisResult result = null;
                try
                {
                    result = AnimationCurveEditCommand.MakeQuantizationAnalysis(this, ObjectID, curve, false);
                }
                catch (Exception)
                {
                    if (errorHandler != null)
                    {
                        errorHandler(animTarget);
                    }
                }
                yield return new Tuple<AnimTarget, IfQuantizationAnalysisResult>(animTarget, result);
            }
        }

        public FogAnimation(fog_animType fog_anim, SceneAnimation owner)
            : base(GuiObjectID.FogAnimation)
        {
            Name = fog_anim.fog_name;
            Data = fog_anim;
            Owner = owner;
            MakeComment(Data.comment);

            FogAnimTargets = new List<FogAnimationTarget>();
            var targetTypes = Enum.GetValues(typeof(fog_anim_target_targetType)).OfType<fog_anim_target_targetType>().ToArray();
            var fogAnimTargets = fog_anim.fog_anim_target ?? Enumerable.Empty<fog_anim_targetType>();
            // 全てのtargettypeに対してデータにtargetTypeがない場合は作成する
            var allFogTargets =
                targetTypes.Select(
                    targetType =>
                    fogAnimTargets.FirstOrDefault(anim => anim.target == targetType)
                    ?? new fog_anim_targetType() { target = targetType });

            foreach (var fogAnimTarget in allFogTargets.OrderBy(x => x.target))
            {
                var target = new FogAnimationTarget()
                {
                    targetType = fogAnimTarget.target,
                };
                var defaultInterpolationType = InterpolationType.Hermite;

                if (fogAnimTarget.Curve != null)
                {
                    target.QuantizationInfo = new AnimationDocument.QuantizationInfo()
                                                  {
                                                      frame_type = fogAnimTarget.Curve.frame_type,
                                                      key_type = fogAnimTarget.Curve.key_type,
                                                      offset = fogAnimTarget.Curve.offset,
                                                      scale = fogAnimTarget.Curve.scale,
                                                  };
                }

                target.Initialize(fogAnimTarget.Curve, fogAnimTarget.base_value, owner.BinaryStreams, defaultInterpolationType);

                FogAnimTargets.Add(target);
            }

            // ユーザーデータ作成
            this.MakeUserData(Data.user_data_array, Owner.BinaryStreams);

            UpdateSavedData();
        }

        public FogAnimationTarget GetTarget(fog_anim_target_targetType targetType)
        {
            return FogAnimTargets.FirstOrDefault(x => x.targetType == targetType);
        }

        public FogAnimationTarget GetAnimTarget(FogAnimationCurveTreeNodeInfo info)
        {
            return GetTarget((fog_anim_target_targetType)info.ComponentIndex);
        }
        public List<FogAnimationTarget> FogAnimTargets;

        #region savedData
        private fog_animType savedData;
        public List<FogAnimationTarget> savedFogAnimTargets;

        /// <summary>
        /// 空以外の保存されたカーブ
        /// </summary>
        public int SavedCurveCount { get; private set; }

        public override void UpdateSavedData()
        {
            base.UpdateSavedData();
            savedData = ObjectUtility.Clone(Data);
            SavedUserDataArray = ObjectUtility.Clone(UserDataArray);
            UserDataArrayChanged = false;

            SavedCurveCount = 0;
            foreach (var animTarget in FogAnimTargets)
            {
                animTarget.IsModified = false;
                if (NotEmpty(animTarget))
                {
                    animTarget.IsSaved = true;
                    SavedCurveCount++;
                }
                else
                {
                    animTarget.IsSaved = false;
                }
            }
            savedFogAnimTargets = ObjectUtility.Clone(FogAnimTargets);
        }

        public void CopySavedData(FogAnimation source)
        {
            CopyGuiObjectSavedData(source);
            savedData = source.savedData;
            SavedUserDataArray = source.SavedUserDataArray;
            UserDataArrayChanged = !source.SavedUserDataArray.IsSame(UserDataArray);

            SavedCurveCount = source.SavedCurveCount;
            savedFogAnimTargets = source.savedFogAnimTargets;
            UpdateIsModifiedAnimTargetAll();
        }

        public override bool EqualsToSavedData()
        {
            if (!base.EqualsToSavedData())
            {
                return false;
            }

            return !FogAnimationGeneralPage.IsModified(this) &&
                !IsCurvesModified() &&
                !UserDataPage.IsModified(this);
        }

        public void UpdateIsModifiedAnimTargetAll()
        {
            foreach (var animTarget in FogAnimTargets)
            {
                var savedAnimTarget = savedFogAnimTargets.FirstOrDefault(x => x.targetType == animTarget.targetType);
                UpdateIsModifiedAnimTarget(animTarget, savedAnimTarget);
            }
        }

        /// <summary>
        /// カーブの変更フラグを更新する
        /// </summary>
        public void UpdateIsModifiedAnimTarget(FogAnimationTarget current, FogAnimationTarget saved)
        {
            if (saved == null)
            {
                current.IsModified = NotEmpty(current);
                current.IsSaved = false;
            }
            else
            {
                current.IsModified = !current.IsSame(saved);
                current.IsSaved = saved.IsSaved;
            }
        }

        public bool NotEmpty(FogAnimationTarget current)
        {
            return current.KeyFrames.Count > 0;
        }

        public bool IsValueChanged<T>(Func<fog_animType, T> select)
        {
            return !select(savedData).Equals(select(Data));
        }

        public bool IsCurvesModified()
        {
            int savedCount = 0;
            foreach (var animTarget in FogAnimTargets)
            {
                if (animTarget.IsModified)
                {
                    return true;
                }
                if (animTarget.IsSaved)
                {
                    savedCount++;
                }
            }

            return savedCount != SavedCurveCount;
        }
        #endregion
    }


    [Serializable]
    public class FogAnimationTarget : AnimTarget
    {
        public fog_anim_target_targetType targetType;
        public fog_anim_targetType ConvertTo_fog_animType(out G3dStream stream, bool viewer = false)
        {
            IG3dQuantizedCurve curve;
            Convert(out curve, out stream, viewer);
            var target = new fog_anim_targetType()
            {
                base_value = GetBaseValue(),
                Curve = curve,
                target = targetType,
            };
            return target;
        }
    }
}
