﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using nw.g3d.nw4f_3dif;

namespace App.Data
{

    [Serializable]
    public class AnimTarget
    {
        public AnimTarget()
        {
            KeyFrames = new List<KeyFrame>();
            QuantizationInfo = new AnimationDocument.QuantizationInfo();

            IsFileOutputable = true;
        }

        public AnimationDocument.QuantizationInfo	QuantizationInfo { get; set; }
        public curve_wrapType						pre_wrap { get; set; }
        public curve_wrapType						post_wrap { get; set; }
        public bool									baked { get; set; }

        public InterpolationType CurveInterpolationType { get; set; }
        public List<KeyFrame> KeyFrames { get; set; }

        public bool									IsFileOutputable { get; set; }

        [NonSerialized]
        private bool isModified;

        public bool IsModified {
            get { return isModified; }
            set { isModified = value; }
        }

        private bool isSaved;
        public bool IsSaved
        {
            get { return isSaved; }
            set { isSaved = value; }
        }

        public bool HasKeyFrame
        {
            get { return KeyFrames != null && KeyFrames.Any(); }
        }

        public virtual bool IsSame(AnimTarget animTarget)
        {
            if (animTarget == null)
            {
                return false;
            }

            if (KeyFrames.Count == 0 && animTarget.KeyFrames.Count == 0)
            {
                return true;
            }

            float baseValue0 = 0;
            float baseValue1 = 0;
            bool IsConstant0 = KeyFrames.IsConstantCurve(out baseValue0);
            bool IsConstant1 = animTarget.KeyFrames.IsConstantCurve(out baseValue1);
            if (IsConstant0 != IsConstant1)
            {
                return false;
            }

            if (IsConstant0)
            {
                return baseValue0 == baseValue1;
            }

            if ((pre_wrap != animTarget.pre_wrap) ||
                (post_wrap != animTarget.post_wrap) ||
                (baked != animTarget.baked) ||
                (CurveInterpolationType	!= animTarget.CurveInterpolationType) ||
                !QuantizationInfo.IsSame(animTarget.QuantizationInfo) ||
                KeyFrames.Count != animTarget.KeyFrames.Count)
            {
                return false;
            }

            if (CurveInterpolationType == InterpolationType.Hermite)
            {
                return KeyFrames.Zip(animTarget.KeyFrames,
                    (x, y) => x.Frame == y.Frame &&
                        x.Value == y.Value &&
                        x.InSlope == y.InSlope &&
                        x.OutSlope == y.OutSlope).All(x => x);
            }
            else
            {
                return KeyFrames.Zip(animTarget.KeyFrames,
                    (x, y) => x.Frame == y.Frame &&
                        x.Value == y.Value).All(x => x);
            }
        }

        public override bool Equals(object obj)
        {
            // TODO: いずれ削除
            Debug.Assert(false);

            // If parameter is null return false.
            if (obj == null)
            {
                return false;
            }

            // If parameter cannot be cast to Point return false.
            AnimTarget p = obj as AnimTarget;
            if ((object)p == null)
            {
                return false;
            }

            // Return true if the fields match:
            return
                (pre_wrap				== p.pre_wrap) &&
                (post_wrap				== p.post_wrap) &&
                (baked					== p.baked) &&
                (CurveInterpolationType	== p.CurveInterpolationType) &&
                QuantizationInfo.Equals(p.QuantizationInfo) &&
                (KeyFrames.Zip(p.KeyFrames, (x, y) => x.Equals(y)).Any(x => x == false) == false);
        }

        public override int GetHashCode()
        {
            // TODO: いずれ削除
            Debug.Assert(false);
            return base.GetHashCode();
        }

        /// <summary>
        /// 設定
        /// キーフレームは浅いコピー
        /// 量子化情報は何はコピーしない
        /// </summary>
        public void PartialSet(AnimTarget animTarget)
        {
            pre_wrap = animTarget.pre_wrap;
            post_wrap = animTarget.post_wrap;
            baked = animTarget.baked;
            CurveInterpolationType = animTarget.CurveInterpolationType;
            KeyFrames = animTarget.KeyFrames;
        }

        virtual public CurveExportType ExportType
        {
            get
            {
                if (KeyFrames.Any())
                {
                    float base_value = 0;
                    if (!KeyFrames.IsConstantCurve(CurveInterpolationType, out base_value))
                    {
                        return CurveExportType.Curve;
                    }

                    return CurveExportType.Constant;
                }

                return CurveExportType.Ignored;
            }
        }

        public void Initialize(IG3dQuantizedCurve quantizedCurve, float baseValue, List<G3dStream> streams, InterpolationType defaultInterpolationType)
        {
            {
                CurveInterpolationType = (quantizedCurve is hermite_curveType) ? InterpolationType.Hermite :
                                                              (quantizedCurve is linear_curveType) ? InterpolationType.Linear :
                                                              (quantizedCurve is step_curveType) ? InterpolationType.Step :
                                                                                                            defaultInterpolationType;		// null の時
            };

            if (quantizedCurve != null)
            {
                switch (CurveInterpolationType)
                {
                    case InterpolationType.Hermite:
                        {
                            const int dataElementCount = 4;		// frame, value, in-slope, out-slope

                            var curve = quantizedCurve as hermite_curveType;
                            Debug.Assert(curve != null);

                            QuantizationInfo.frame_type = curve.frame_type;
                            QuantizationInfo.key_type = curve.key_type;
                            QuantizationInfo.scale = curve.scale;
                            QuantizationInfo.offset = curve.offset;
                            pre_wrap = curve.pre_wrap;
                            post_wrap = curve.post_wrap;
                            baked = curve.baked;

                            var stream = streams[curve.stream_index];
                            var data = stream.FloatData;
                            Debug.Assert(data != null);
                            Debug.Assert(data.Count == curve.count * dataElementCount);

                            for (int i = 0; i != curve.count; ++i)
                            {
                                KeyFrames.Add(
                                    new KeyFrame()
                                    {
                                        Frame = data[i * dataElementCount + 0],
                                        Value = data[i * dataElementCount + 1],
                                        InSlope = data[i * dataElementCount + 2],
                                        OutSlope = data[i * dataElementCount + 3]
                                    }
                                );
                            }

                            break;
                        }

                    case InterpolationType.Linear:
                    case InterpolationType.Step:
                        {
                            const int dataElementCount = 2;		// frame, value

                            int stream_index = -1;
                            int count = -1;
                            {
                                if (CurveInterpolationType == InterpolationType.Linear)
                                {
                                    var curve = quantizedCurve as linear_curveType;
                                    Debug.Assert(curve != null);

                                    stream_index = curve.stream_index;
                                    count = curve.count;

                                    QuantizationInfo.frame_type = curve.frame_type;
                                    QuantizationInfo.key_type = curve.key_type;
                                    QuantizationInfo.scale = curve.scale;
                                    QuantizationInfo.offset = curve.offset;
                                    pre_wrap = curve.pre_wrap;
                                    post_wrap = curve.post_wrap;
                                    baked = curve.baked;
                                }
                                else
                                    if (CurveInterpolationType == InterpolationType.Step)
                                    {
                                        var curve = quantizedCurve as step_curveType;
                                        Debug.Assert(curve != null);

                                        stream_index = curve.stream_index;
                                        count = curve.count;

                                        QuantizationInfo.frame_type = curve.frame_type;
                                        QuantizationInfo.key_type = curve.key_type;
                                        QuantizationInfo.scale = curve.scale;
                                        QuantizationInfo.offset = curve.offset;
                                        pre_wrap = curve.pre_wrap;
                                        post_wrap = curve.post_wrap;
                                        baked = curve.baked;
                                    }

                                Debug.Assert(stream_index != -1);
                                Debug.Assert(count != -1);
                            }

                            var stream = streams[stream_index];
                            var data = stream.FloatData;
                            Debug.Assert(data != null);
                            Debug.Assert(data.Count == count * dataElementCount);

                            for (int i = 0; i != count; ++i)
                            {
                                KeyFrames.Add(
                                    new KeyFrame()
                                    {
                                        Frame = data[i * dataElementCount + 0],
                                        Value = data[i * dataElementCount + 1]
                                    }
                                );
                            }

                            break;
                        }
                }
            }
            else
            {
                // base_value で初期化します。
                KeyFrames.Add(
                    new KeyFrame()
                    {
                        Frame = 0,
                        Value = baseValue,
                    }
                );
            }
        }

        public void Convert(out IG3dQuantizedCurve curve, out G3dStream stream, bool viewer)
        {
            curve = null;
            stream = null;
            if (!KeyFrames.Any())
            {
                return;
            }

            // コンスタントカーブかどうかをチェックする
            {
                float value = 0.0f;
                if (KeyFrames.IsConstantCurve(CurveInterpolationType, out value))
                {
                    // ランタイム連携用に水平なカーブを出力する
                    if (viewer)
                    {
                        curve = new step_curveType()
                        {
                            count = 2,
                            frame_type = curve_frame_typeType.none,
                            key_type = curve_key_typeType.none,
                            scale = 1,
                            offset = 0,
                            pre_wrap = curve_wrapType.clamp,
                            post_wrap = curve_wrapType.clamp,
                            baked = false,
                        };
                        stream = new G3dStream()
                        {
                            type = stream_typeType.@float,
                            column = 2,
                        };

                        stream.FloatData.Add(0); // フレーム
                        stream.FloatData.Add(value);

                        stream.FloatData.Add(1); // フレーム
                        stream.FloatData.Add(value);
                    }

                    return;
                }
            }

            switch (CurveInterpolationType)
            {
                case InterpolationType.Hermite:
                    {
                        curve = new hermite_curveType();
                        //curve.stream_index = targetIndex;
                        curve.count = KeyFrames.Count();
                        curve.frame_type = QuantizationInfo.frame_type;
                        curve.key_type = QuantizationInfo.key_type;
                        curve.scale = QuantizationInfo.scale;
                        curve.offset = QuantizationInfo.offset;
                        curve.pre_wrap = pre_wrap;
                        curve.post_wrap = post_wrap;
                        ((hermite_curveType)curve).baked = baked;

                        stream = new G3dStream();
                        stream.type = stream_typeType.@float;
                        stream.column = 4;
                        for (int i = 0; i < curve.count; i++)
                        {
                            stream.FloatData.Add(KeyFrames[i].Frame);
                            stream.FloatData.Add(KeyFrames[i].Value);
                            stream.FloatData.Add(KeyFrames[i].InSlope);
                            stream.FloatData.Add(KeyFrames[i].OutSlope);
                        }
                        break;
                    }

                case InterpolationType.Linear:
                    {
                        if (KeyFrames.Count() != 0)
                        {
                            curve = new linear_curveType();
                            //curve.stream_index = targetIndex;
                            curve.count = KeyFrames.Count();
                            curve.frame_type = QuantizationInfo.frame_type;
                            curve.key_type = QuantizationInfo.key_type;
                            curve.scale = QuantizationInfo.scale;
                            curve.offset = QuantizationInfo.offset;
                            curve.pre_wrap = pre_wrap;
                            curve.post_wrap = post_wrap;
                            ((linear_curveType)curve).baked = baked;

                            stream = new G3dStream();
                            stream.type = stream_typeType.@float;
                            stream.column = 2;
                            for (int i = 0; i < curve.count; i++)
                            {
                                stream.FloatData.Add(KeyFrames[i].Frame);
                                stream.FloatData.Add(KeyFrames[i].Value);
                            }
                        }
                        break;
                    }

                case InterpolationType.Step:
                    {
                        if (KeyFrames.Count() != 0)
                        {
                            curve = new step_curveType();
                            //curve.stream_index = targetIndex;
                            curve.count = KeyFrames.Count();
                            curve.frame_type = QuantizationInfo.frame_type;
                            curve.key_type = QuantizationInfo.key_type;
                            curve.scale = QuantizationInfo.scale;
                            curve.offset = QuantizationInfo.offset;
                            curve.pre_wrap = pre_wrap;
                            curve.post_wrap = post_wrap;
                            ((step_curveType)curve).baked = baked;

                            stream = new G3dStream();
                            stream.type = stream_typeType.@float;
                            stream.column = 2;
                            for (int i = 0; i < curve.count; i++)
                            {
                                stream.FloatData.Add(KeyFrames[i].Frame);
                                stream.FloatData.Add(KeyFrames[i].Value);
                            }
                        }
                        break;
                    }

                default:
                    {
                        Debug.Assert(false);
                        break;
                    }
            }
        }

        public void Convert(out step_curveType curve, out G3dStream stream, bool viewer)
        {
            curve = null;
            stream = null;
            if (!KeyFrames.Any())
            {
                return;
            }

            // コンスタントカーブかどうかをチェックする
            {
                float value = 0.0f;
                if (KeyFrames.IsConstantCurve(CurveInterpolationType, out value))
                {
                    // ランタイム連携用に水平なカーブを出力する
                    if (viewer)
                    {
                        curve = new step_curveType()
                        {
                            count = 2,
                            frame_type = curve_frame_typeType.none,
                            key_type = curve_key_typeType.none,
                            scale = 1,
                            offset = 0,
                            pre_wrap = curve_wrapType.clamp,
                            post_wrap = curve_wrapType.clamp,
                            baked = false,
                        };
                        stream = new G3dStream()
                        {
                            type = stream_typeType.@float,
                            column = 2,
                        };

                        stream.FloatData.Add(0); // フレーム
                        stream.FloatData.Add(value);

                        stream.FloatData.Add(1); // フレーム
                        stream.FloatData.Add(value);
                    }

                    return;
                }
            }

            if (KeyFrames.Count() != 0)
            {
                curve = new step_curveType();
                //curve.stream_index = targetIndex;
                curve.count = KeyFrames.Count();
                curve.frame_type = QuantizationInfo.frame_type;
                curve.key_type = QuantizationInfo.key_type;
                curve.scale = QuantizationInfo.scale;
                curve.offset = QuantizationInfo.offset;
                curve.pre_wrap = pre_wrap;
                curve.post_wrap = post_wrap;
                ((step_curveType)curve).baked = baked;

                stream = new G3dStream();
                stream.type = stream_typeType.@float;
                stream.column = 2;
                for (int i = 0; i < curve.count; i++)
                {
                    stream.FloatData.Add(KeyFrames[i].Frame);
                    stream.FloatData.Add(KeyFrames[i].Value);
                }
            }
        }

        public float GetFrameValue(float frame)
        {
            // 念のため、ソートしておきます。
            App.PropertyEdit.KeyFramesUtility.SortByFrame(KeyFrames);
            var keys = KeyFrames;

            // キーフレームが一つの場合は、コンスタントカーブなので、
            // どのフレームでも同じ値になります。
            if (keys.Count == 1)
            {
                return keys[0].Value;
            }

            // キーフレームが無い場合は、カーブが無いので、0.0fで。
            if (!keys.Any())
            {
                Debug.Assert(false);
                return 0.0f;
            }

            int shiftCount;

            // キーフレームを打ってある範囲のフレームが指定されていない場合に、範囲内のフレームにずらす。
            float targetFrame = App.PropertyEdit.AnimationCurveExtensions.calcShiftFrame(frame, keys, pre_wrap, post_wrap, out shiftCount);

            // 指定フレームの値を取得する。
            float value = App.PropertyEdit.AnimationCurveExtensions.calcFrameValue(targetFrame, keys, CurveInterpolationType);

            if ((shiftCount < 0 && pre_wrap == curve_wrapType.relative_repeat) ||
                (shiftCount > 0 && post_wrap == curve_wrapType.relative_repeat))
            {
                value += shiftCount * (keys[keys.Count - 1].Value - keys[0].Value);
            }

            return value;
        }

        public virtual float GetBaseValue()
        {
            if (KeyFrames.Any())
            {
                return GetFrameValue(0);
            }

            return 0;
        }
    }

    public enum CurveExportType
    {
        Ignored,
        Constant,
        Curve,
        Dependent,	// 他にのみ依存する場合
    }
}
