﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;
using System.Diagnostics;

namespace App.Controls
{
    /// <summary>
    /// 値編集パネル用ポップアップスライダークラス。
    /// </summary>
    public abstract partial class ValueEditPopupSlider : UIForm
    {
        // 範囲調節フラグ
        private bool _rangeAdjustable = true;
        // ポップアップ元
        private UIButton _popupSource = null;
        // イベントオブジェクト
        private static readonly object EVENT_SEQUENTIALVALUECHANGED = new object();
        // イベントオブジェクト
        private static readonly object EVENT_SETTINGCHANGED = new object();

        /// <summary>トラックバー。</summary>
        protected readonly UITrackBar _trackBar;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ValueEditPopupSlider(UITrackBar trackBar)
        {
            InitializeComponent();

            // トラックバー
            _trackBar = trackBar;
            _trackBar.Parent = pnlTrackBar;
            _trackBar.Dock = DockStyle.Fill;
            _trackBar.SequentialValueChanged += new SequentialValueChangedEventHandler(Event_trackBar_SequentialValueChanged);

            // 元に戻すボタン
            btnRestore.Click += new EventHandler(Event_btnRestore_Click);

            // 範囲変更ボタン
            btnRangeDown.Click += new EventHandler(Event_btnRangeDown_Click);
            btnRangeUp.Click += new EventHandler(Event_btnRangeUp_Click);
        }

        /// <summary>
        /// 範囲調節フラグ。
        /// </summary>
        public bool RangeAdjustable
        {
            get { return _rangeAdjustable;  }
            set
            {
                _rangeAdjustable = value;

                // 表示切り替え
                btnRangeDown.Visible = value;
                btnRangeUp.Visible   = value;
            }
        }

        /// <summary>
        /// スライダー領域。
        /// </summary>
        public Rectangle SliderBounds
        {
            get { return pnlTrackBar.Bounds; }
        }

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        public event SequentialValueChangedEventHandler SequentialValueChanged
        {
            add { base.Events.AddHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SEQUENTIALVALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            SequentialValueChangedEventHandler handler = (SequentialValueChangedEventHandler)base.Events[EVENT_SEQUENTIALVALUECHANGED];
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// 設定変更イベント。
        /// </summary>
        public event EventHandler SettingChanged
        {
            add { base.Events.AddHandler(EVENT_SETTINGCHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SETTINGCHANGED, value); }
        }

        /// <summary>
        /// 設定変更ハンドラ。
        /// </summary>
        protected virtual void OnSettingChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_SETTINGCHANGED];
            if (handler != null)
            {
                handler(this, e);
            }
        }

        /// <summary>
        /// スライダーを表示。
        /// </summary>
        public void ShowSlider(UIButton popupSource)
        {
            _popupSource = popupSource;
            var form = popupSource.FindForm();
            Show(form);
            BringToFront();
        }

        /// <summary>
        /// 最小値ラベルを設定。
        /// </summary>
        protected void SetMinimumLabel(string text)
        {
            lblMinimum.Text = text;
        }

        /// <summary>
        /// 最大値ラベルを設定。
        /// </summary>
        protected void SetMaximumLabel(string text)
        {
            lblMaximum.Text = text;
        }

        #region 派生クラス実装用
        /// <summary>
        /// 設定を元に戻す。
        /// </summary>
        protected abstract void RestoreSetting();

        /// <summary>
        /// 範囲スケーリング。
        /// </summary>
        protected abstract bool ScaleRange(bool scaleUp);
        #endregion

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams cp = base.CreateParams;
                cp.Style |= Win32.WS.WS_BORDER;
                return cp;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // 既定の処理
            if (base.ProcessCmdKey(ref msg, keyData))
            {
                return true;
            }
            // メインフレームのショートカット処理
            return TheApp.MainFrame.ProcessCommandKey(ref msg, keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // 元に戻す
            if (keyData == Keys.R)
            {
                btnRestore.PerformClick();
                return true;
            }
            // 範囲縮小
            else if (keyData == Keys.Subtract || keyData == Keys.OemMinus)
            {
                if (_rangeAdjustable)
                {
                    btnRangeDown.PerformClick();
                    return true;
                }
            }
            // 範囲拡大
            else if (keyData == Keys.Add || keyData == Keys.Oemplus)
            {
                if (_rangeAdjustable)
                {
                    btnRangeUp.PerformClick();
                    return true;
                }
            }
            // 閉じる
            else if (keyData == Keys.Escape || keyData == Keys.Return)
            {
                // キー入力で閉じる時のみ
                // PostMessage経由で閉じる
                BeginInvoke(new MethodInvoker(Close));
                return true;
            }
            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 元に戻すボタンの再配置
            if (!_rangeAdjustable)
            {
                Point point = btnRestore.Location;
                point.X = (Width - btnRestore.Width) / 2;
                btnRestore.Location = point;
            }
            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDeactivate(EventArgs e)
        {
            // PostMessage経由で閉じる
            if (UIMessageBox.DeactivatedForm != this)
            {
                BeginInvoke(new MethodInvoker(Close));
            }

            base.OnDeactivate(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            // ポップアップ元にフォーカスを戻す
            //if (_restoreFocus)
            {
                _popupSource.Focus();
            }
            base.OnFormClosed(e);
        }
        #endregion

        #region イベントハンドラ
        //---------------------------------------------------------------------------
        // トラックバー
        private void Event_trackBar_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            OnSequentialValueChanged(e);
        }

        //---------------------------------------------------------------------------
        // 元に戻す
        private void Event_btnRestore_Click(object sender, EventArgs e)
        {
            RestoreSetting();
            OnSettingChanged(EventArgs.Empty);
        }

        //---------------------------------------------------------------------------
        // 範囲調節
        private void Event_btnRangeDown_Click(object sender, EventArgs e)
        {
            if (ScaleRange(false))
            {
                OnSettingChanged(EventArgs.Empty);
            }
        }

        private void Event_btnRangeUp_Click(object sender, EventArgs e)
        {
            if (ScaleRange(true))
            {
                OnSettingChanged(EventArgs.Empty);
            }
        }
        #endregion
    }

    /// <summary>
    /// 整数値編集パネル用ポップアップスライダークラス。
    /// </summary>
    public sealed class IntEditPopupSlider : ValueEditPopupSlider
    {
        // 初期設定
        private int _initialMinimum;
        private int _initialMaximum;
        private int _initialSmallChange;
        private int _initialLargeChange;
        private int _initialValue;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntEditPopupSlider() : base(new IntTrackBar())
        {
        }

        protected override void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            UpdateThumbColor();
            base.OnSequentialValueChanged(e);
        }

        private void UpdateThumbColor()
        {
            TrackBar.DrawRedThumb = Value < Minimum || Maximum < Value;
        }

        public void UpdateScaleEnabled()
        {
            // btnRangeDown.Enabled = CanScale(false) を正常実行するための条件。
            // これが守られない場合、CanScale(false) 内の Scale() で Inf, NaN が算出され、
            // 意図した結果にならない。
            Debug.Assert(MinDefault >= MinBound);

            btnRangeDown.Enabled = CanScale(false);
            btnRangeUp.Enabled = CanScale(true);
        }

        protected override void OnSettingChanged(EventArgs e)
        {
            base.OnSettingChanged(e);
            UpdateScaleEnabled();
        }

        /// <summary>
        /// 値。
        /// </summary>
        internal int Value
        {
            get { return TrackBar.Value; }
            set {
                TrackBar.Value = value;
                UpdateThumbColor();
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        internal int Minimum
        {
            get { return TrackBar.Minimum; }
            set
            {
                TrackBar.Minimum = value;
                SetMinimumLabel(value.ToString());
                UpdateThumbColor();
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        internal int Maximum
        {
            get { return TrackBar.Maximum; }
            set
            {
                TrackBar.Maximum = value;
                SetMaximumLabel(value.ToString());
                UpdateThumbColor();
            }
        }

        int _smallChange;
        /// <summary>
        /// 小変化量。
        /// </summary>
        internal int SmallChange
        {
            get { return _smallChange; }
            set {
                _smallChange = value;
                TrackBar.SmallChange = value;
            }
        }

        int _largeChange;
        /// <summary>
        /// 大変化量。
        /// </summary>
        internal int LargeChange
        {
            get { return _largeChange; }
            set {
                _largeChange = value;
                TrackBar.LargeChange = value;
            }
        }

        public int MinBound;

        public int MaxBound;

        public int MinDefault;

        public int MaxDefault;

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 初期設定保存
            _initialMinimum     = Minimum;
            _initialMaximum     = Maximum;
            _initialSmallChange = SmallChange;
            _initialLargeChange = LargeChange;
            _initialValue       = Value;

            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void RestoreSetting()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                Minimum     = _initialMinimum;
                Maximum     = _initialMaximum;
                SmallChange = _initialSmallChange;
                LargeChange = _initialLargeChange;
                Value       = _initialValue;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ScaleRange(bool scaleUp)
        {
            IntRangeProperty range = new IntRangeProperty(Minimum, Maximum, SmallChange, LargeChange, MinBound, MaxBound, MinDefault, MaxDefault);
            if (range.Scale(scaleUp))
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    Minimum     = range.Minimum;
                    Maximum     = range.Maximum;
                    SmallChange = range.ChangeS;
                    LargeChange = range.ChangeL;
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// スケールできるかどうか
        /// </summary>
        private bool CanScale(bool scaleUp)
        {
            IntRangeProperty range = new IntRangeProperty(Minimum, Maximum, SmallChange, LargeChange, MinBound, MaxBound, MinDefault, MaxDefault);
            return range.Scale(scaleUp);
        }

        #region コントロールアクセス
        /// <summary>
        /// トラックバー。
        /// </summary>
        private IntTrackBar TrackBar
        {
            get { return (IntTrackBar)_trackBar; }
        }
        #endregion
    }

    /// <summary>
    /// 符号なし整数値編集パネル用ポップアップスライダークラス。
    /// </summary>
    public sealed class UintEditPopupSlider : ValueEditPopupSlider
    {
        // 初期設定
        private uint _initialMinimum;
        private uint _initialMaximum;
        private uint _initialSmallChange;
        private uint _initialLargeChange;
        private uint _initialValue;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UintEditPopupSlider()
            : base(new UintTrackBar())
        {
        }

        protected override void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            UpdateThumbColor();
            base.OnSequentialValueChanged(e);
        }

        private void UpdateThumbColor()
        {
            TrackBar.DrawRedThumb = Value < Minimum || Maximum < Value;
        }

        public void UpdateScaleEnabled()
        {
            btnRangeDown.Enabled = CanScale(false);
            btnRangeUp.Enabled = CanScale(true);
        }

        protected override void OnSettingChanged(EventArgs e)
        {
            base.OnSettingChanged(e);
            UpdateScaleEnabled();
        }

        /// <summary>
        /// 値。
        /// </summary>
        internal uint Value
        {
            get { return TrackBar.Value; }
            set {
                TrackBar.Value = value;
                UpdateThumbColor();
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        internal uint Minimum
        {
            get { return TrackBar.Minimum; }
            set
            {
                TrackBar.Minimum = value;
                SetMinimumLabel(value.ToString());
                UpdateThumbColor();
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        internal uint Maximum
        {
            get { return TrackBar.Maximum; }
            set
            {
                TrackBar.Maximum = value;
                SetMaximumLabel(value.ToString());
                UpdateThumbColor();
            }
        }

        uint _smallChange;
        /// <summary>
        /// 小変化量。
        /// </summary>
        internal uint SmallChange
        {
            get { return _smallChange; }
            set
            {
                _smallChange = value;
                TrackBar.SmallChange = value;
            }
        }

        uint _largeChange;
        /// <summary>
        /// 大変化量。
        /// </summary>
        internal uint LargeChange
        {
            get { return _largeChange; }
            set
            {
                _largeChange = value;
                TrackBar.LargeChange = value;
            }
        }

        public uint MinBound;

        public uint MaxBound;

        public uint MinDefault;

        public uint MaxDefault;

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 初期設定保存
            _initialMinimum = Minimum;
            _initialMaximum = Maximum;
            _initialSmallChange = SmallChange;
            _initialLargeChange = LargeChange;
            _initialValue = Value;

            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void RestoreSetting()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                Minimum = _initialMinimum;
                Maximum = _initialMaximum;
                SmallChange = _initialSmallChange;
                LargeChange = _initialLargeChange;
                Value = _initialValue;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ScaleRange(bool scaleUp)
        {
            UintRangeProperty range = new UintRangeProperty(Minimum, Maximum, SmallChange, LargeChange, MinBound, MaxBound, MinDefault, MaxDefault);
            if (range.Scale(scaleUp))
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    Minimum = range.Minimum;
                    Maximum = range.Maximum;
                    SmallChange = range.ChangeS;
                    LargeChange = range.ChangeL;
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// スケールできるかどうか
        /// </summary>
        private bool CanScale(bool scaleUp)
        {
            UintRangeProperty range = new UintRangeProperty(Minimum, Maximum, SmallChange, LargeChange, MinBound, MaxBound, MinDefault, MaxDefault);
            return range.Scale(scaleUp);
        }

        #region コントロールアクセス
        /// <summary>
        /// トラックバー。
        /// </summary>
        private UintTrackBar TrackBar
        {
            get { return (UintTrackBar)_trackBar; }
        }
        #endregion
    }

    /// <summary>
    /// 実数値編集パネル用ポップアップスライダークラス。
    /// </summary>
    public sealed class FloatEditPopupSlider : ValueEditPopupSlider
    {
        // 初期設定
        private float _initialMinimum;
        private float _initialMaximum;
        private double _initialSmallChange;
        private double _initialLargeChange;
        private float _initialValue;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatEditPopupSlider() : base(new FloatTrackBar())
        {
        }

        protected override void OnSequentialValueChanged(SequentialValueChangedEventArgs e)
        {
            UpdateThumbColor();
            base.OnSequentialValueChanged(e);
        }

        private void UpdateThumbColor()
        {
            TrackBar.DrawRedThumb = Value < Minimum || Maximum < Value;
        }

        public void UpdateScaleEnabled()
        {
            btnRangeDown.Enabled = CanScale(false);
            btnRangeUp.Enabled = CanScale(true);
        }

        protected override void OnSettingChanged(EventArgs e)
        {
            base.OnSettingChanged(e);
            UpdateScaleEnabled();
        }

        /// <summary>
        /// 値。
        /// </summary>
        internal float Value
        {
            get {
                return TrackBar.Value;
            }
            set {
                TrackBar.Value = value;
                UpdateThumbColor();
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        internal float Minimum
        {
            get { return TrackBar.Minimum; }
            set
            {
                TrackBar.Minimum = value;
                SetMinimumLabel(FormatRangeLabel(value));
                UpdateThumbColor();
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        internal float Maximum
        {
            get { return TrackBar.Maximum; }
            set
            {
                TrackBar.Maximum = value;
                SetMaximumLabel(FormatRangeLabel(value));
                UpdateThumbColor();
            }
        }

        /// <summary>
        /// 小変化量。
        /// </summary>
        internal double SmallChange
        {
            get { return TrackBar.SmallChange; }
            set { TrackBar.SmallChange = value; }
        }

        /// <summary>
        /// 大変化量。
        /// </summary>
        internal double LargeChange
        {
            get { return TrackBar.LargeChange; }
            set { TrackBar.LargeChange = value; }
        }

        public float MinBound;

        public float MaxBound;

        public float MinDefault;

        public float MaxDefault;

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 初期設定保存
            _initialMinimum     = Minimum;
            _initialMaximum     = Maximum;
            _initialSmallChange = SmallChange;
            _initialLargeChange = LargeChange;
            _initialValue       = Value;

            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void RestoreSetting()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                Minimum     = _initialMinimum;
                Maximum     = _initialMaximum;
                SmallChange = _initialSmallChange;
                LargeChange = _initialLargeChange;
                Value       = _initialValue;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ScaleRange(bool scaleUp)
        {
            FloatRangeProperty range = new FloatRangeProperty(Minimum, Maximum, SmallChange, LargeChange, MinBound, MaxBound, MinDefault, MaxDefault);
            if (range.Scale(scaleUp))
            {
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    Minimum     = range.Minimum;
                    Maximum     = range.Maximum;
                    SmallChange = range.ChangeS;
                    LargeChange = range.ChangeL;
                }
                return true;
            }
            return false;
        }

        /// <summary>
        /// スケールできるかどうか
        /// </summary>
        private bool CanScale(bool scaleUp)
        {
            FloatRangeProperty range = new FloatRangeProperty(Minimum, Maximum, SmallChange, LargeChange, MinBound, MaxBound, MinDefault, MaxDefault);
            return range.Scale(scaleUp);
        }

        /// <summary>
        /// 範囲ラベルをフォーマット。
        /// </summary>
        private string FormatRangeLabel(float value)
        {
            return value.ToString("R");
        }

        #region コントロールアクセス
        /// <summary>
        /// トラックバー。
        /// </summary>
        private FloatTrackBar TrackBar
        {
            get { return (FloatTrackBar)_trackBar; }
        }
        #endregion
    }
}
