﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using App.Data;
using App.Utility;
using App.res;

namespace App.Controls
{
    public partial class UserDataDialog : OkCancelDialog
    {
        private readonly UserDataArray userDataArray_ = null;
        private UserDataArray.UserData userData_ = null;
        private readonly UserDataArray.UserData orgUserData_ = null;

        // 16進数モード
        private readonly bool _hexMode = false;

        // 文字列データの正規表現
        private static readonly Regex _stringDataRegex =
            new Regex(@"\p{IsBasicLatin}+", RegexOptions.Compiled);
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UserDataDialog(bool isHexMode, UserDataArray userDataArray)
            : this(isHexMode, userDataArray, null)
        {
            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_property_window_common_page_a_user_data);
        }

        // 追加された型
        public bool addMode = false;
        public static Type LastAddedType = typeof(UserDataArray.UserDataString);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UserDataDialog(bool isHexMode, UserDataArray userDataArray, UserDataArray.UserData userData)
        {
            InitializeComponent();

            _hexMode = isHexMode;

            userDataArray_ = userDataArray;
            chkCanEdit.Checked = userData == null;
            pnlCanEdit.Visible = userData != null;
            lblValueMsg.Text = Strings.UserData_Dlg_VlueMsg_Default;
            addMode = userData == null;

            if (userData == null)
            {
                Height -= pnlCanEdit.Height;
            }
            else if(userData is UserDataArray.UserDataStream)
            {
                // stream型の場合は、編集させない！
                chkCanEdit.Checked = false;
                pnlCanEdit.Visible = false;
                lblValueMsg.Text = Strings.UserData_Dlg_VlueMsg_Stream;
                Height -= pnlCanEdit.Height;
            }

            chkCanEdit_CheckedChanged(null, null);

            if (userData == null)
            {
                if (LastAddedType == typeof(UserDataArray.UserDataInt))
                {
                    userData_ = new UserDataArray.UserDataInt(string.Empty, new List<int>());
                }
                else if (LastAddedType == typeof(UserDataArray.UserDataFloat))
                {
                    userData_ = new UserDataArray.UserDataFloat(string.Empty, new List<float>());
                }
                else if (LastAddedType == typeof(UserDataArray.UserDataWString))
                {
                    userData_ = new UserDataArray.UserDataWString(string.Empty, new List<string>());
                }
                else
                {
                    userData_ = new UserDataArray.UserDataString(string.Empty, new List<string>());
                }
                orgUserData_ = null;
            }
            else
            {
                userData_ = ObjectUtility.Clone(userData);
                orgUserData_ = ObjectUtility.Clone(userData);
            }

            UpdateForm(userData_);
        }

        private void EnableChangeForStream(bool flg)
        {
            rbtStream.Enabled = flg;
            tbxValue.Enabled = !flg;
        }

        private void UpdateForm(UserDataArray.UserData userData)
        {
            // 名前
            {
                tbxName.Text = userData.Name;
            }

            // 型
            {
                if (userData is UserDataArray.UserDataInt)
                {
                    rbtInt.Checked = true;
                    EnableChangeForStream(false);
                }
                else if (userData is UserDataArray.UserDataFloat)
                {
                    rbtFloat.Checked = true;
                    EnableChangeForStream(false);
                }
                else if (userData is UserDataArray.UserDataString)
                {
                    rbtString.Checked = true;
                    EnableChangeForStream(false);
                }
                else if (userData is UserDataArray.UserDataWString)
                {
                    rbtWString.Checked = true;
                    EnableChangeForStream(false);
                }
                else if (userData is UserDataArray.UserDataStream)
                {
                    rbtStream.Checked = true;
                    EnableChangeForStream(true);
                }
            }

            // 値
            {
                if (userData is UserDataArray.UserDataStream)
                {
                    UserDataArray.UserDataStream streamdata = userData as UserDataArray.UserDataStream;
                    List<string> values = new List<string>();
                    {
                        int index = 0;
                        string byteString = string.Empty;
                        foreach (var value in streamdata.Item)
                        {
                            if (index != 15)
                            {
                                byteString += string.Format("{0:x02}", value);
                                index++;
                            }
                            else
                            {
                                values.Add(byteString);
                                index = 0;
                                byteString = string.Empty;
                            }
                        }
                    }
                    tbxValue.Lines = values.ToArray();

                }
                else
                {
                    List<string> values = new List<string>();
                    {
                        foreach (var value in ((dynamic)userData).Item)
                        {
                            if (userData is UserDataArray.UserDataInt)
                            {
                                values.Add(string.Format(_hexMode ? "{0:x08}" : "{0:d}", value));
                            }
                            else if (userData is UserDataArray.UserDataFloat)
                            {
                                values.Add(value.ToString());
                            }
                            else
                            {
                                values.Add(value);
                            }
                        }
                    }
                    tbxValue.Lines = values.ToArray();
                }
            }
        }

        /// <summary>
        /// 結果の取得。
        /// </summary>
        public UserDataArray.UserData GetResult()
        {
            if ((rbtString.Checked && (userData_ is UserDataArray.UserDataString)) ||
                (rbtWString.Checked && (userData_ is UserDataArray.UserDataWString)) ||
                (rbtInt.Checked && (userData_ is UserDataArray.UserDataInt)) ||
                (rbtFloat.Checked && (userData_ is UserDataArray.UserDataFloat)))
            {
                // 形は変わっていない
            }
            else
            {
                // 型変わったのでつくり直す

                string name = userData_.Name;

                if (rbtString.Checked)
                {
                    List<string> stringItem = new List<string>();
                    userData_ = new UserDataArray.UserDataString(name, stringItem);
                }
                else if (rbtWString.Checked)
                {
                    List<string> stringItem = new List<string>();
                    userData_ = new UserDataArray.UserDataWString(name, stringItem);
                }
                else if (rbtInt.Checked)
                {
                    List<int> Item = new List<int>();
                    userData_ = new UserDataArray.UserDataInt(name, Item);
                }
                else if (rbtFloat.Checked)
                {
                    List<float> Item = new List<float>();
                    userData_ = new UserDataArray.UserDataFloat(name, Item);
                }
            }

            if (rbtString.Checked)
            {
                List<string> item = (userData_ as UserDataArray.UserDataString).Item;
                List<string> cloneItem = ObjectUtility.Clone(item);
                item.Clear();

                string errorMessage = null;
                int index = 0;
                foreach (string value in tbxValue.Lines)
                {
                    if (value.Length > 0)
                    {
                        string dst = null;
                        if (ParseStringData(value, ref dst, ref errorMessage))
                        {
                            item.Add(dst);
                        }
                    }
                    index++;
                }
            }
            else if (rbtWString.Checked)
            {
                List<string> item = (userData_ as UserDataArray.UserDataWString).Item;
                List<string> cloneItem = ObjectUtility.Clone(item);
                item.Clear();

                int index = 0;
                foreach (string value in tbxValue.Lines)
                {
                    if (value.Length > 0)
                    {
                        string dst = null;
                        if (ParseWStringData(value, ref dst))
                        {
                            item.Add(dst);
                        }
                    }
                    index++;
                }
            }
            else if (rbtInt.Checked)
            {
                List<int> item = (userData_ as UserDataArray.UserDataInt).Item;
                item.Clear();

                string errorMessage = null;

                foreach (string value in tbxValue.Lines)
                {
                    if (value.Length > 0)
                    {
                        int dst = 0;
                        if (ParseIntData(value, ref dst, ref errorMessage))
                        {
                            item.Add(dst);
                        }
                    }
                }
            }
            else if (rbtFloat.Checked)
            {
                List<float> item = (userData_ as UserDataArray.UserDataFloat).Item;
                item.Clear();

                string errorMessage = null;

                foreach (string value in tbxValue.Lines)
                {
                    if (value.Length > 0)
                    {
                        float dst = 0.0f;
                        if (ParseFloatData(value, ref dst, ref errorMessage))
                        {
                            item.Add(dst);
                        }
                    }
                }
            }

            return userData_;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            // データが変わってなかったらキャンセルとみなす
            if (DataChanged() == false)
            {
                DialogResult = DialogResult.Cancel;
            }

            if (addMode)
            {
                LastAddedType = userData_.GetType();
            }

            return true;
        }

        private bool IsValidData(ref string errorMessage)
        {
            foreach (string value in tbxValue.Lines)
            {
                if (value.Length > 0)
                {
                    if (rbtString.Checked)
                    {
                        string dst = null;
                        if (ParseStringData(value, ref dst, ref errorMessage) == false)
                        {
                            return false;
                        }
                    }
                    else if (rbtWString.Checked)
                    {
                        string dst = null;
                        if (ParseWStringData(value, ref dst) == false)
                        {
                            return false;
                        }
                    }
                    else if (rbtInt.Checked)
                    {
                        int dst = 0;
                        if (ParseIntData(value, ref dst, ref errorMessage) == false)
                        {
                            return false;
                        }
                    }
                    else if (rbtFloat.Checked)
                    {
                        float dst = 0.0f;
                        if (ParseFloatData(value, ref dst, ref errorMessage) == false)
                        {
                            return false;
                        }
                    }
                }
            }

            return true;
        }

        private bool ParseStringData(string src, ref string dst, ref string errorMessage)
        {
            if (!RegexMatch.Check(src, _stringDataRegex))
            {
                errorMessage = Strings.UserData_Input_String_Invalid_Msg;
                return false;
            }
            else
            {
                dst = src;
                return true;
            }
        }

        private bool ParseWStringData(string src, ref string dst)
        {
            dst = src;
            return true;
        }

        private bool ParseIntData(string src, ref int dst, ref string errorMessage)
        {
            try
            {
                if (_hexMode)
                {
                    if (src.StartsWith("0x") || src.StartsWith("0X"))
                    {
                        dst = Int32.Parse(src.Substring(2), NumberStyles.HexNumber);
                    }
                    else
                    {
                        dst = Int32.Parse(src, NumberStyles.HexNumber);
                    }
                }
                else
                {
                    dst = Convert.ToInt32(src);
                }
                return true;
            }
            catch (FormatException)
            {
                errorMessage = Strings.UserData_Input_Int_Invalid_Msg;
                return false;
            }
            catch (OverflowException)
            {
                errorMessage = Strings.UserData_Input_Int_Overflow_Msg;
                return false;
            }
        }

        private bool ParseFloatData(string src, ref float dst, ref string errorMessage)
        {
            try
            {
                dst = Convert.ToSingle(src);
                return true;
            }
            catch (FormatException)
            {
                errorMessage = Strings.UserData_Input_Float_Invalid_Msg;
                return false;
            }
            catch (OverflowException)
            {
                errorMessage = Strings.UserData_Input_Float_Overflow_Msg;
                return false;
            }
        }

        /// <summary>
        /// データ変更チェック
        /// </summary>
        private bool DataChanged()
        {
            return GetResult().Equals(orgUserData_) == false;
        }

        private void tbxName_TextChanged(object sender, EventArgs e)
        {
            GetResult().Name = tbxName.Text;
        }

        private void UserDataDialog_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                UserDataArray.UserData result = GetResult();

                // 名無しチェック
                {
                    if (result.Name == string.Empty)
                    {
                        UIMessageBox.Warning(Strings.UserData_Input_Name_Invalid_Msg);
                        e.Cancel = true;
                        return;
                    }
                }

                // 無効な文字を含んでいないかをチェック
                {
                    if (!StringUtility.CheckValidName(result.Name, StringUtility.ValidNameRegExpType.NumberAlphabetMoreThan1))
                    {
                        UIMessageBox.Warning(Strings.UserData_Input_Name_Invalid_Msg);
                        e.Cancel = true;
                        return;
                    }
                }

                // 重複名チェック
                {
                    if (userDataArray_ != null)
                    {
                        // 新規の時のみチェックする
                        foreach (UserDataArray.UserData userData in userDataArray_.Data)
                        {
                            // 変更時で元々の名前であれば重複とは扱わない。
                            if ((orgUserData_ != null) && orgUserData_.Name == userData.Name)
                            {
                                continue;
                            }

                            if (result.Name == userData.Name)
                            {
                                UIMessageBox.Warning(Strings.UserData_Input_Name_Duplicate_Msg);
                                e.Cancel = true;
                                return;
                            }
                        }
                    }
                }

                // 正しいデータかのチェック
                {
                    string errorMessage = null;
                    if (IsValidData(ref errorMessage) == false)
                    {
                        UIMessageBox.Warning(errorMessage);
                        e.Cancel = true;
                        return;
                    }
                }
            }
        }

        private void chkCanEdit_CheckedChanged(object sender, EventArgs e)
        {
            lblName.Enabled = chkCanEdit.Checked;
            tbxName.Enabled = chkCanEdit.Checked;
            gbxType.Enabled = chkCanEdit.Checked;
        }
    }
}
