﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Windows.Forms.Design;
using App.Utility;

namespace App.Controls
{
    //-------------------------------------------------------------------------
    // クラス階層
    //
    // TextBox
    // └UITextBox
    //   └ValueInputBox (abstract)
    //     ├NumericInputBox (abstract)
    //     │├IntInputBox				：整数値入力用
    //     │└FloatInputBox			：実数値入力用
    //     └StringInputBox				：文字列値入力用
    //
    // ToolStripControlHost
    // └ToolStripValueInputBox (abstract)
    //   ├ToolStripIntInputBox			：整数値入力用
    //   ├ToolStripFloatInputBox		：実数値入力用
    //   └ToolStripStringInputBox		：文字列値入力用
    //
    //-------------------------------------------------------------------------

    /// <summary>
    /// ＵＩテキストボックスクラス。
    /// </summary>
    public class UITextBox : TextBox, KeyEditableControl
    {
        // ツールストリップ用オーナー
        private ToolStripItem _toolStripOwner = null;
        // マウスオーバーフラグ
        private bool _mouseIsOver = false;

        private UIToolTip _ttpHint = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UITextBox()
        {
        }

        public bool CanKeyEdit		// interface KeyEditableControl
        {
            get
            {
                return ReadOnly == false;
            }
        }

        private Point padding;

        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        [DefaultValue(typeof(Point))]
        public new Point Padding
        {
            get
            {
                return padding;
            }
            set
            {
                padding = value;
                SetPaddingRect();
            }
        }

        private bool _AutoHint = false;
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public bool AutoHint
        {
            get
            {
                return _AutoHint;
            }
            set
            {
                _AutoHint = value;
                UpdateHint();
            }
        }

        private bool _OverrideContextMenu = false;
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Visible)]
        public bool OverrideContextMenu
        {
            get
            {
                return _OverrideContextMenu;
            }
            set
            {
                _OverrideContextMenu = value;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public override string Text
        {
            get
            {
                return base.Text;
            }
            set
            {
                base.Text = value;
                UpdateHint();
            }
        }

        public override Color ForeColor
        {
            get
            {
                return base.ForeColor;
            }
            set
            {
                // バックカラーを設定しないとフォアカラーが反映されないことがある。http://bytes.com/topic/c-sharp/answers/233961-read-only-textbox
                BackColor = BackColor;
                base.ForeColor = value;
            }
        }

        /// <summary>
        /// ツールストリップ用オーナー。
        /// </summary>
        internal ToolStripItem ToolStripOwner
        {
            get { return _toolStripOwner; }
            set { _toolStripOwner = value; }
        }

        /// <summary>
        /// マウスオーバーフラグ。
        /// </summary>
        private bool MouseIsOver
        {
            set
            {
                if (_mouseIsOver != value)
                {
                    _mouseIsOver = value;
                    if (!Focused)
                    {
                        InvalidateNonClient();
                    }
                }
            }
        }

        /// <summary>
        /// 絶対クライアント領域。
        /// </summary>
        private Win32.RECT AbsoluteClientRect
        {
            get
            {
                Win32.RECT rect = new App.Win32.RECT();

                CreateParams cp = CreateParams;
                Win32.NativeMethods.AdjustWindowRectEx(ref rect, cp.Style, false, cp.ExStyle);
                int offsetH = -rect.left;
                int offsetV = -rect.top;

                Win32.NativeMethods.GetClientRect(Handle, ref rect);
                rect.left   += offsetH;
                rect.top    += offsetV;
                rect.right  += offsetH;
                rect.bottom += offsetV;

                return rect;
            }
        }

        /// <summary>
        /// 非クライアント領域を無効化。
        /// </summary>
        private void InvalidateNonClient()
        {
            if (_toolStripOwner != null)
            {
                Win32.RECT rcClient = AbsoluteClientRect;
                IntPtr rgnResult = IntPtr.Zero;
                IntPtr rgnClient = IntPtr.Zero;
                IntPtr rgnEntire = IntPtr.Zero;
                try
                {
                    rgnEntire = Win32.NativeMethods.CreateRectRgn(0, 0, base.Width, base.Height);
                    rgnClient = Win32.NativeMethods.CreateRectRgn(rcClient.left, rcClient.top, rcClient.right, rcClient.bottom);
                    rgnResult = Win32.NativeMethods.CreateRectRgn(0, 0, 0, 0);
                    Win32.NativeMethods.CombineRgn(rgnResult, rgnEntire, rgnClient, Win32.RGN.RGN_XOR);

                    // 0x705 =
                    // Win32.RDW.RDW_INVALIDATE |
                    // Win32.RDW.RDW_ERASE      |
                    // Win32.RDW.RDW_UPDATENOW  |
                    // Win32.RDW.RDW_ERASENOW   |
                    // Win32.RDW.RDW_FRAME
                    Win32.RECT rect = new App.Win32.RECT();
                    Win32.NativeMethods.RedrawWindow(base.Handle, ref rect, rgnResult, 0x705);
                }
                finally
                {
                    // REGON ハンドルを開放
                    try
                    {
                        if (rgnResult != IntPtr.Zero) { Win32.NativeMethods.DeleteObject(rgnResult); }
                    }
                    finally
                    {
                        try
                        {
                            if (rgnClient != IntPtr.Zero) { Win32.NativeMethods.DeleteObject(rgnClient); }
                        }
                        finally
                        {
                            if (rgnEntire != IntPtr.Zero) { Win32.NativeMethods.DeleteObject(rgnEntire); }
                        }
                    }
                }
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // 非クライアント領域描画
                case Win32.WM.WM_NCPAINT:
                    //WmNCPaint(ref m);
                    //return;
                    break;

                default:
                    break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (_OverrideContextMenu)
            {
                if (e.Button == MouseButtons.Right)
                {
                    if (e.X < 0 || e.Y < 0 || ClientSize.Width >= e.X || ClientSize.Height >= e.Y)
                    {
                        Capture = false;
                    }
                }
            }

            base.OnMouseMove(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            base.OnGotFocus(e);
            InvalidateNonClient();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            base.OnLostFocus(e);
            InvalidateNonClient();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseEnter(EventArgs e)
        {
            base.OnMouseEnter(e);
            MouseIsOver = true;

            ForceShowHint();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);
            MouseIsOver = false;

            ForceHideHint();
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            SetPaddingRect();
        }

        protected override void OnVisibleChanged(EventArgs e)
        {
            base.OnVisibleChanged(e);

            SetPaddingRect();
        }

        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public sealed override RightToLeft RightToLeft
        {
            get { return base.RightToLeft;  }
            set { base.RightToLeft = value; }
        }
        #endregion

        private void SetPaddingRect()
        {
            if (Multiline)
            {
                var rect = new App.Win32.NativeMethods.Rect()
                {
                    Left = Padding.X,
                    Top = Padding.Y,
                    Right = ClientSize.Width - Padding.X,
                    Bottom = ClientSize.Height - Padding.Y,
                };

                const int EM_SETRECT = 0x00B3;

                App.Win32.NativeMethods.SendMessage(Handle, EM_SETRECT, IntPtr.Zero, ref rect);
            }
        }

        private void UpdateHint()
        {
            if (AutoHint)
            {
                if (_ttpHint == null)
                {
                    _ttpHint = new UIToolTip();
                }

                _ttpHint.SetToolTip(this, Text);
            }
            else
            {
                if (_ttpHint != null)
                {
                    _ttpHint.RemoveAll();
                    _ttpHint = null;
                }
            }
        }

        private void ForceShowHint()
        {
            if (AutoHint)
            {
                if (_ttpHint == null)
                {
                    _ttpHint = new UIToolTip();
                }

                _ttpHint.SetToolTip(this, Text);
            }
        }

        private void ForceHideHint()
        {
            if (_ttpHint != null)
            {
                _ttpHint.RemoveAll();
                _ttpHint = null;
            }
        }

        private void InitializeComponent()
        {
            this.SuspendLayout();
            this.ResumeLayout(false);

        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (Multiline && ModifierKeys == Keys.Control && (keyData & Keys.KeyCode) == Keys.A)
            {
                SelectAll();
                return true;
            }
            return base.ProcessCmdKey(ref msg, keyData);
        }
    }

    //----------------------------------------------------------------------------
    // 値編集用
    #region ValueInputBox
    /// <summary>
    /// 値入力ボックスクラス。
    /// </summary>
    [DefaultEvent("ValueChanged")]
    public abstract class ValueInputBox : UITextBox
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ValueInputBox()
        {
        }

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_VALUECHANGED = new object();
        private static readonly object EVENT_RETURNKEYDOWN = new object();

        /// <summary>
        /// 値変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("値が変更された時に発生します。")]
        public event EventHandler ValueChanged
        {
            add { base.Events.AddHandler(EVENT_VALUECHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_VALUECHANGED, value); }
        }

        /// <summary>
        /// 値変更ハンドラ。
        /// </summary>
        protected virtual void OnValueChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_VALUECHANGED];
            if (handler != null)
            {
                DebugConsole.WriteLine("ValueInputBox: OnValueChanged");
                UIControlHelper.EventMessage("ValueChanged", this, e);
                handler(this, e);
            }
        }

        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("リターンキーが押された時に発生します。")]
        public event EventHandler ReturnKeyDown
        {
            add { base.Events.AddHandler(EVENT_RETURNKEYDOWN, value); }
            remove { base.Events.RemoveHandler(EVENT_RETURNKEYDOWN, value); }
        }

        #endregion

        /// <summary>
        /// テキストを値に反映。
        /// </summary>
        protected abstract bool UpdateValue();

        /// <summary>
        /// 値をテキストに反映。
        /// </summary>
        protected abstract void UpdateText();

        /// <summary>
        /// テキストのパース処理。
        /// </summary>
        private void ParseText(bool withSelectAll)
        {
            // 値を更新して再度テキストに反映
            bool valueChanged = UpdateValue();
            UpdateText();

            // 全選択状態にする
            if (withSelectAll)
            {
                SelectAll();
            }

            // 最後にイベントを発行
            if (valueChanged)
            {
                OnValueChanged(EventArgs.Empty);
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            get { return new Size(80, 21); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessDialogKey(Keys keyData)
        {
            // リターン入力処理
            if (keyData == Keys.Return)
            {
                ParseText(true);

                var handler = (EventHandler)base.Events[EVENT_RETURNKEYDOWN];
                if (handler != null)
                {
                    handler(this, EventArgs.Empty);
                }

                return true;
            }
            return base.ProcessDialogKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            ParseText(false);
            base.OnLostFocus(e);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new AutoCompleteStringCollection AutoCompleteCustomSource
        {
            get { return base.AutoCompleteCustomSource; }
            set { base.AutoCompleteCustomSource = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new AutoCompleteMode AutoCompleteMode
        {
            get { return base.AutoCompleteMode; }
            set { base.AutoCompleteMode = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new AutoCompleteSource AutoCompleteSource
        {
            get { return base.AutoCompleteSource; }
            set { base.AutoCompleteSource = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool AcceptsReturn
        {
            get { return base.AcceptsReturn; }
            set { base.AcceptsReturn = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool AcceptsTab
        {
            get { return base.AcceptsTab; }
            set { base.AcceptsTab = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new CharacterCasing CharacterCasing
        {
            get { return base.CharacterCasing; }
            set { base.CharacterCasing = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new string[] Lines
        {
            get { return base.Lines; }
            set { base.Lines = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public override bool Multiline
        {
            get { return base.Multiline; }
            set { base.Multiline = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new ScrollBars ScrollBars
        {
            get { return base.ScrollBars; }
            set { base.ScrollBars = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public override string Text
        {
            get
            {
                return base.Text;
            }
            set
            {
                base.Text = value;
            }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseSystemPasswordChar
        {
            get { return base.UseSystemPasswordChar; }
            set { base.UseSystemPasswordChar = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool WordWrap
        {
            get { return base.WordWrap; }
            set { base.WordWrap = value; }
        }
        #endregion
    }
    #endregion

    #region NumericInputBox
    /// <summary>
    /// 数値入力ボックスクラス。
    /// </summary>
    public abstract class NumericInputBox : ValueInputBox
    {
        // 範囲有効フラグ
        private bool _enableRange = true;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public NumericInputBox()
        {
            TextAlign = HorizontalAlignment.Right;
        }

        /// <summary>
        /// 範囲有効フラグ。
        /// </summary>
        [DefaultValue(true)]
        [Description("範囲指定を有効にするかを示します。")]
        public bool EnableRange
        {
            get { return _enableRange; }
            set { _enableRange = value; }
        }

        /// <summary>
        /// 入力可能な文字か。
        /// </summary>
        protected virtual bool IsEnableInputChar(char ch)
        {
            // マイナス
            if (ch == '-')
            {
                return true;
            }
            // 数字か編集キー（VK_BACK=0x08）
            else if (('0' <= ch && ch <= '9')
                || ch == 0x08 //backspace
                || ch == 0x01 // ctrl+a
                || ch == 0x03 // ctrl+c
                || ch == 0x16 // ctrl+v
                || ch == 0x18 // ctrl+x
                )
            {
                return true;
            }
            return false;
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // 文字入力
                case Win32.WM.WM_CHAR:
                    {
                        char ch = (char)m.WParam;
                        if (!IsEnableInputChar(ch))
                        {
                            return;
                        }
                    }
                    break;

                default:
                    break;
            }
            base.WndProc(ref m);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DefaultValue(HorizontalAlignment.Right)]
        public new HorizontalAlignment TextAlign
        {
            get { return base.TextAlign; }
            set { base.TextAlign = value; }
        }
        #endregion
    }
    #endregion

    #region IntInputBox
    /// <summary>
    /// 整数値入力ボックスクラス。
    /// </summary>
    public sealed class IntInputBox : NumericInputBox
    {
        // 値
        private int _value = 0;
        // 最小値
        private int _minimum = 0;
        // 最大値
        private int _maximum = 100;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public IntInputBox()
        {
            UpdateText();
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0)]
        [Description("値。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public int Value
        {
            get { return _value; }
            set
            {
                if (_value != value)
                {
                    _value = value;
                    UpdateText();
                    OnValueChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0)]
        [Description("最小値。")]
        public int Minimum
        {
            get { return _minimum; }
            set
            {
                if (value < _maximum)
                {
                    _minimum = value;
                }
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100)]
        [Description("最大値。")]
        public int Maximum
        {
            get { return _maximum; }
            set
            {
                if (value > _minimum)
                {
                    _maximum = value;
                }
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool UpdateValue()
        {
            int prev = _value;
            if (StringUtility.TrySafeIntParse(Text, out _value))
            {
                if (EnableRange)
                {
                    if (_value < _minimum) { _value = _minimum; }
                    else if (_value > _maximum) { _value = _maximum; }
                }
            }
            else
            {
                _value = prev;
                return false;
            }

            return _value != prev;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateText()
        {
            Text = _value.ToString();
        }
        #endregion
    }
    #endregion

    #region FloatInputBox
    /// <summary>
    /// 実数値入力ボックスクラス。
    /// </summary>
    public sealed class FloatInputBox : NumericInputBox
    {
        // 値
        private float _value = 0.0f;
        // 最小値
        private float _minimum = 0.0f;
        // 最大値
        private float _maximum = 100.0f;
        // 表示形式
        private FloatValueDisplayStyle _valueDisplayStyle = FloatValueDisplayStyle.AutoDecimal;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FloatInputBox()
        {
            UpdateText();
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("値。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public float Value
        {
            get { return _value; }
            set
            {
                if (_value != value || empty)
                {
                    empty = false;
                    _value = value;
                    UpdateText();
                    OnValueChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 値を表示するかどうか
        /// </summary>
        private bool empty;

        /// <summary>
        /// 値を表示しない
        /// </summary>
        public void ClearText()
        {
            if (!empty)
            {
                empty = true;
                UpdateText();
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("最小値。")]
        public float Minimum
        {
            get { return _minimum; }
            set
            {
                if (value < _maximum)
                {
                    _minimum = value;
                }
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100.0f)]
        [Description("最大値。")]
        public float Maximum
        {
            get { return _maximum; }
            set
            {
                if (value > _minimum)
                {
                    _maximum = value;
                }
            }
        }

        /// <summary>
        /// 値の表示形式。
        /// </summary>
        [DefaultValue(FloatValueDisplayStyle.AutoDecimal)]
        [Description("値の表示形式。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public FloatValueDisplayStyle ValueDisplayStyle
        {
            get { return _valueDisplayStyle; }
            set
            {
                if (_valueDisplayStyle != value)
                {
                    _valueDisplayStyle = value;
                    UpdateText();
                }
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool UpdateValue()
        {
            float prev = _value;

            if (string.IsNullOrWhiteSpace(Text))
            {
                return false;
            }

            if (StringUtility.TrySafeFloatParse(Text, out _value))
            {
                if (EnableRange)
                {
                    if (_value < _minimum) { _value = _minimum; }
                    else if (_value > _maximum) { _value = _maximum; }
                }
            }
            else
            {
                _value = prev;
                return false;
            }

            return _value != prev || empty;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateText()
        {
            if (empty)
            {
                Text = "";
            }
            else
            {
                Text = UIControlHelper.GetFloatValueDisplayString(_value, _valueDisplayStyle);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool IsEnableInputChar(char ch)
        {
            // 小数点
            if (ch == '.')
            {
                return true;
            }
            return base.IsEnableInputChar(ch);
        }
        #endregion
    }
    #endregion

    #region DoubleInputBox
    /// <summary>
    /// 実数値入力ボックスクラス。
    /// </summary>
    public sealed class DoubleInputBox : NumericInputBox
    {
        // 値
        private double _value = 0.0;
        // 最小値
        private double _minimum = 0.0;
        // 最大値
        private double _maximum = 100.0;
        // 表示形式
        private FloatValueDisplayStyle _valueDisplayStyle = FloatValueDisplayStyle.AutoDecimal;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DoubleInputBox()
        {
            UpdateText();
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0.0)]
        [Description("値。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public double Value
        {
            get { return _value; }
            set
            {
                if (_value != value || empty)
                {
                    empty = false;
                    _value = value;
                    UpdateText();
                    OnValueChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 値を表示するかどうか
        /// </summary>
        private bool empty;

        /// <summary>
        /// 値を表示しない
        /// </summary>
        public void ClearText()
        {
            if (!empty)
            {
                empty = true;
                UpdateText();
            }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0.0)]
        [Description("最小値。")]
        public double Minimum
        {
            get { return _minimum; }
            set
            {
                if (value < _maximum)
                {
                    _minimum = value;
                }
            }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100.0)]
        [Description("最大値。")]
        public double Maximum
        {
            get { return _maximum; }
            set
            {
                if (value > _minimum)
                {
                    _maximum = value;
                }
            }
        }

        /// <summary>
        /// 値の表示形式。
        /// </summary>
        [DefaultValue(FloatValueDisplayStyle.AutoDecimal)]
        [Description("値の表示形式。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public FloatValueDisplayStyle ValueDisplayStyle
        {
            get { return _valueDisplayStyle; }
            set
            {
                if (_valueDisplayStyle != value)
                {
                    _valueDisplayStyle = value;
                    UpdateText();
                }
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool UpdateValue()
        {
            double prev = _value;

            if (string.IsNullOrWhiteSpace(Text))
            {
                return false;
            }

            if (double.TryParse(Text, out _value))
            {
                if (EnableRange)
                {
                    if (_value < _minimum) { _value = _minimum; }
                    else if (_value > _maximum) { _value = _maximum; }
                }
            }
            else
            {
                _value = prev;
                return false;
            }

            return _value != prev || empty;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateText()
        {
            if (empty)
            {
                Text = "";
            }
            else
            {
                Text = UIControlHelper.GetFloatValueDisplayString(_value, _valueDisplayStyle);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool IsEnableInputChar(char ch)
        {
            // 小数点
            if (ch == '.')
            {
                return true;
            }
            return base.IsEnableInputChar(ch);
        }
        #endregion
    }
    #endregion

    #region StringInputBox
    /// <summary>
    /// 文字列値入力ボックスクラス。
    /// </summary>
    public sealed class StringInputBox : ValueInputBox
    {
        // 値
        private string _value = string.Empty;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public StringInputBox()
        {
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue("")]
        [Description("値。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public string Value
        {
            get { return _value; }
            set
            {
                if (_value != (value ?? ""))
                {
                    _value = value ?? "";
                    UpdateText();
                    OnValueChanged(EventArgs.Empty);
                }
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool UpdateValue()
        {
            string prev = _value;
            _value = Text;
            return _value != prev;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateText()
        {
            Text = _value;
        }
        #endregion
    }
    #endregion

    //----------------------------------------------------------------------------
    // ツールストリップ用
    #region ToolStripValueInputBox
    /// <summary>
    /// ツールストリップ項目用値入力ボックスクラス。
    /// </summary>
    [DefaultEvent("ValueChanged")]
    [ToolStripItemDesignerAvailability(ToolStripItemDesignerAvailability.ToolStrip)]
    public abstract class ToolStripValueInputBox : ToolStripControlHost
    {
        /// <summary>値変更イベント。</summary>
        public event EventHandler ValueChanged = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ToolStripValueInputBox(ValueInputBox control)
            : base(control)
        {
            base.AutoSize = false;

            // ツールストリップ用に設定
            control.AutoSize = false;
            control.Height = 19;
            control.ToolStripOwner = this;
        }

        /// <summary>
        /// ValueChanged ハンドラ。
        /// </summary>
        private void HandleValueChanged(object sender, EventArgs e)
        {
            // base.Events[key] で取得する必要はない
            if (ValueChanged != null)
            {
                ValueChanged(this, e);
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Padding DefaultMargin
        {
            // ToolStripTextBox と同じ
            get
            {
                if (base.IsOnDropDown)
                {
                    return new Padding(1);
                }
                return new Padding(1, 0, 1, 0);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override Size DefaultSize
        {
            // 高さは ToolStripTextBox と同じ
            get { return new Size(80, 0x16); }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSubscribeControlEvents(Control control)
        {
            ValueInputBox target = control as ValueInputBox;
            if (target != null)
            {
                target.ValueChanged += new EventHandler(HandleValueChanged);
            }
            base.OnSubscribeControlEvents(control);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnUnsubscribeControlEvents(Control control)
        {
            ValueInputBox target = control as ValueInputBox;
            if (target != null)
            {
                target.ValueChanged -= new EventHandler(HandleValueChanged);
            }
            base.OnUnsubscribeControlEvents(control);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        public new bool AutoSize
        {
            get { return base.AutoSize; }
            set { base.AutoSize = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public override string Text
        {
            get { return base.Text; }
            set { base.Text = value; }
        }
        #endregion
    }
    #endregion

    #region ToolStripIntInputBox
    /// <summary>
    /// ツールストリップ項目用整数値入力ボックスクラス。
    /// </summary>
    public sealed class ToolStripIntInputBox : ToolStripValueInputBox
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ToolStripIntInputBox()
            : base(CreateControlInstance())
        {
        }

        /// <summary>
        /// ホストしているコントロール。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IntInputBox HostControl
        {
            get { return (IntInputBox)base.Control; }
        }

        /// <summary>
        /// 範囲有効フラグ。
        /// </summary>
        [DefaultValue(true)]
        [Description("範囲指定を有効にするかを示します。")]
        public bool EnableRange
        {
            get { return HostControl.EnableRange; }
            set { HostControl.EnableRange = value; }
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0)]
        [Description("値。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public int Value
        {
            get { return HostControl.Value; }
            set { HostControl.Value = value; }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0)]
        [Description("最小値。")]
        public int Minimum
        {
            get { return HostControl.Minimum; }
            set { HostControl.Minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100)]
        [Description("最大値。")]
        public int Maximum
        {
            get { return HostControl.Maximum; }
            set { HostControl.Maximum = value; }
        }

        /// <summary>
        /// コントロールインスタンス作成。
        /// </summary>
        private static ValueInputBox CreateControlInstance()
        {
            IntInputBox control = new IntInputBox();
            return control;
        }
    }
    #endregion

    #region ToolStripFloatInputBox
    /// <summary>
    /// ツールストリップ項目用実数値入力ボックスクラス。
    /// </summary>
    public sealed class ToolStripFloatInputBox : ToolStripValueInputBox
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ToolStripFloatInputBox()
            : base(CreateControlInstance())
        {
        }

        /// <summary>
        /// ホストしているコントロール。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public FloatInputBox HostControl
        {
            get { return (FloatInputBox)base.Control; }
        }

        /// <summary>
        /// 範囲有効フラグ。
        /// </summary>
        [DefaultValue(true)]
        [Description("範囲指定を有効にするかを示します。")]
        public bool EnableRange
        {
            get { return HostControl.EnableRange; }
            set { HostControl.EnableRange = value; }
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("値。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public float Value
        {
            get { return HostControl.Value; }
            set { HostControl.Value = value; }
        }

        /// <summary>
        /// 最小値。
        /// </summary>
        [DefaultValue(0.0f)]
        [Description("最小値。")]
        public float Minimum
        {
            get { return HostControl.Minimum; }
            set { HostControl.Minimum = value; }
        }

        /// <summary>
        /// 最大値。
        /// </summary>
        [DefaultValue(100.0f)]
        [Description("最大値。")]
        public float Maximum
        {
            get { return HostControl.Maximum; }
            set { HostControl.Maximum = value; }
        }

        /// <summary>
        /// 値の表示形式。
        /// </summary>
        [DefaultValue(FloatValueDisplayStyle.AutoDecimal)]
        [Description("値の表示形式。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public FloatValueDisplayStyle ValueDisplayStyle
        {
            get { return HostControl.ValueDisplayStyle; }
            set { HostControl.ValueDisplayStyle = value; }
        }

        /// <summary>
        /// コントロールインスタンス作成。
        /// </summary>
        private static ValueInputBox CreateControlInstance()
        {
            FloatInputBox control = new FloatInputBox();
            return control;
        }

        /// <summary>
        /// 値を表示しない
        /// </summary>
        public void ClearText()
        {
            HostControl.ClearText();
        }
    }
    #endregion

    #region ToolStripStringInputBox
    /// <summary>
    /// ツールストリップ項目用文字列値入力ボックスクラス。
    /// </summary>
    public sealed class ToolStripStringInputBox : ToolStripValueInputBox
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ToolStripStringInputBox()
            : base(CreateControlInstance())
        {
        }

        /// <summary>
        /// ホストしているコントロール。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public StringInputBox HostControl
        {
            get { return (StringInputBox)base.Control; }
        }

        /// <summary>
        /// 値。
        /// </summary>
        [DefaultValue("")]
        [Description("値。")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public string Value
        {
            get { return HostControl.Value; }
            set { HostControl.Value = value; }
        }

        /// <summary>
        /// コントロールインスタンス作成。
        /// </summary>
        private static ValueInputBox CreateControlInstance()
        {
            StringInputBox control = new StringInputBox();
            return control;
        }
    }
    #endregion
}
