﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// ＵＩリストビュークラス。
    /// </summary>
    [DefaultEvent("SelectionChanged")]
    public class UIListView : ListView
    {
        // カスタム描画フラグ（詳細表示用）
        private bool _customDraw = false;
        // カスタム描画時の状態保存値
        private readonly CustomDrawState _customDrawState = new CustomDrawState();
        // 選択項目を常にハイライトする
        private bool _highlightSelection = false;
        // 列サイズ変更をロックする
        private bool _lockColumnResize = false;
        // ヘッダークリック時にソートを行うか
        private bool _sortOnColumnClick = true;
        // 列ヘッダ（詳細表示用）
        private HeaderControl _header = null;
        // 項目比較オブジェクト
        private readonly ItemComparer _itemComparer = null;
        // 選択変更タイマー
        private readonly Timer _tmrSelectionChanged;
        // コンポーネントリスト
        private readonly IContainer _components = new Container();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIListView()
        {
            // 標準は最も頻繁に使用する詳細ビューにする
            base.View = View.Details;
            base.FullRowSelect = true;
            base.HideSelection = false;
            base.MultiSelect = false;
            base.DoubleBuffered = true;

            // 項目比較用
            _itemComparer = new ItemComparer(OnCompareItem);

            // 選択変更タイマー
            _tmrSelectionChanged = new Timer(_components);
            _tmrSelectionChanged.Interval = 25;
            _tmrSelectionChanged.Tick += new EventHandler(tmrSelectionChanged_Tick);

            //
            SelectionChangedEnable = true;

            ControlRemoved += (sender, e) =>
            {
                if (_HintTimer != null)
                {
                    _HintTimer.Dispose();
                    _HintTimer = null;
                }
            };
        }

        #region プロパティ
        /// <summary>
        /// カスタム描画フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("カスタム描画を行うかどうかを示します。OwnerDrawプロパティがtrueの場合のみ有効です。")]
        public bool CustomDraw
        {
            get { return _customDraw; }
            set { _customDraw = value; }
        }

        /// <summary>
        /// 選択項目を常にハイライト表示フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("非フォーカスでも選択項目をハイライト表示するかどうかを示します。カスタム描画時にのみ反映されます。")]
        public bool HighlightSelection
        {
            get { return _highlightSelection; }
            set { _highlightSelection = value; }
        }

        /// <summary>
        /// 選択変更イベントに用いるタイマー間隔。
        /// </summary>
        [DefaultValue(25)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("選択変更イベントに用いるタイマーの間隔。")]
        public int SelectionChangedInterval
        {
            get { return _tmrSelectionChanged.Interval; }
            set { _tmrSelectionChanged.Interval = value; }
        }

        /// <summary>
        /// 選択変更イベントを有効にするか
        /// </summary>
        [DefaultValue(true)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("選択変更イベントを有効にするか。")]
        public bool SelectionChangedEnable { get; set; }

        /// <summary>
        /// 列サイズ変更のロックフラグ。
        /// </summary>
        [DefaultValue(false)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("列サイズの変更をロックするかどうかを示します。")]
        public bool LockColumnResize
        {
            get { return _lockColumnResize; }
            set { _lockColumnResize = value; }
        }

        /// <summary>
        /// ヘッダークリック時にソートを行うか
        /// </summary>
        [DefaultValue(true)]
        [Category(UIControlHelper.OriginalPropertyCategoryName)]
        [Description("ヘッダークリック時にソートを行うか。")]
        public bool SortOnColumnClick
        {
            get { return _sortOnColumnClick; }
            set { _sortOnColumnClick = value; }
        }

        /// <summary>
        /// 選択項目データ。
        /// 各項目は UIListControlItem 型である必要があります。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object SelectedItemData
        {
            get
            {
                Debug.Assert(DesignMode == true || MultiSelect == false);

                return
                    (SelectedItems.Count == 0) ?
                        null :
                        SelectedItems[0].Tag;
            }

            set
            {
                Debug.Assert(MultiSelect == false);

                // データが null の項目を探す
                if (value == null)
                {
                    foreach(ListViewItem item in Items)
                    {
                        item.Selected = item.Tag == null;
                    }
                }
                // データが等しい項目を探す
                else
                {
                    foreach(ListViewItem item in Items)
                    {
                        item.Selected = (item.Tag != null) && item.Tag.Equals(value);
                    }
                }
            }
        }

        public bool StripeDraw{ get; set; }

        public int SelectedIndex{	get{	return (SelectedIndices.Count >= 1) ? SelectedIndices[0] : -1; } }

        #endregion

        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_SELECTIONCHANGED = new object();

        /// <summary>
        /// 選択変更イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("選択項目が変更された後で発生します。SelectionChangedInterval で設定された間隔を待つため SelectedIndexChanged のような不要な選択変更を受け取りません。")]
        public event EventHandler SelectionChanged
        {
            add { base.Events.AddHandler(EVENT_SELECTIONCHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_SELECTIONCHANGED, value); }
        }

        /// <summary>
        /// 選択変更ハンドラ。
        /// </summary>
        protected virtual void OnSelectionChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_SELECTIONCHANGED];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_CUSTOMDRAWITEM = new object();

        /// <summary>
        /// 項目カスタム描画イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("項目をカスタム描画する時に発生します。")]
        public event CustomDrawListViewItemEventHandler CustomDrawItem
        {
            add { base.Events.AddHandler(EVENT_CUSTOMDRAWITEM, value); }
            remove { base.Events.RemoveHandler(EVENT_CUSTOMDRAWITEM, value); }
        }

        /// <summary>
        /// 項目カスタム描画ハンドラ。
        /// </summary>
        protected virtual void OnCustomDrawItem(CustomDrawListViewItemEventArgs e)
        {
            CustomDrawListViewItemEventHandler handler = (CustomDrawListViewItemEventHandler)base.Events[EVENT_CUSTOMDRAWITEM];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_COMPAREITEM = new object();

        /// <summary>
        /// 項目比較イベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("列項目をクリックして項目を並べ替える時に発生します。")]
        public event CompareListViewItemEventHandler CompareItem
        {
            add { base.Events.AddHandler(EVENT_COMPAREITEM, value); }
            remove { base.Events.RemoveHandler(EVENT_COMPAREITEM, value); }
        }

        /// <summary>
        /// 項目比較ハンドラ。
        /// </summary>
        protected virtual int OnCompareItem(CompareListViewItemEventArgs e)
        {
            CompareListViewItemEventHandler handler = (CompareListViewItemEventHandler)base.Events[EVENT_COMPAREITEM];
            if (handler != null) { return handler(this, e); }

            // ハンドラがなければデフォルト比較
            return e.CompareDefault();
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_COLUMNDIVIDERDOUBLECLICK = new object();

        /// <summary>
        /// 列分割線ダブルクリックイベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("列項目の分割線をダブルクリックした時に発生します。")]
        public event ColumnDividerDoubleClickEventHandler ColumnDividerDoubleClick
        {
            add { base.Events.AddHandler(EVENT_COLUMNDIVIDERDOUBLECLICK, value); }
            remove { base.Events.RemoveHandler(EVENT_COLUMNDIVIDERDOUBLECLICK, value); }
        }

        /// <summary>
        /// 列分割線ダブルクリックハンドラ。
        /// </summary>
        protected virtual void OnColumnDividerDoubleClick(ColumnDividerDoubleClickEventArgs e)
        {
            ColumnDividerDoubleClickEventHandler handler = (ColumnDividerDoubleClickEventHandler)base.Events[EVENT_COLUMNDIVIDERDOUBLECLICK];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_COLUMNCONTEXTMENUPOPUP = new object();
        private static readonly object EVENT_COLUMNMOUSEENTER = new object();
        private static readonly object EVENT_COLUMNMOUSEMOVE = new object();
        private static readonly object EVENT_COLUMNMOUSELEAVE = new object();

        /// <summary>
        /// 列項目コンテキストメニューポップアップイベント。
        /// </summary>
        [Category(UIControlHelper.OriginalEventCategoryName)]
        [Description("列項目上でコンテキストメニューを表示する時に発生します。")]
        public event ColumnMouseEventHandler ColumnContextMenuPopup
        {
            add { base.Events.AddHandler(EVENT_COLUMNCONTEXTMENUPOPUP, value); }
            remove { base.Events.RemoveHandler(EVENT_COLUMNCONTEXTMENUPOPUP, value); }
        }

        [Category(UIControlHelper.OriginalEventCategoryName)]
        public event ColumnMouseEventHandler ColumnMouseEnter
        {
            add { base.Events.AddHandler(EVENT_COLUMNMOUSEENTER, value); }
            remove { base.Events.RemoveHandler(EVENT_COLUMNMOUSEENTER, value); }
        }

        [Category(UIControlHelper.OriginalEventCategoryName)]
        public event ColumnMouseEventHandler ColumnMouseMove
        {
            add { base.Events.AddHandler(EVENT_COLUMNMOUSEMOVE, value); }
            remove { base.Events.RemoveHandler(EVENT_COLUMNMOUSEMOVE, value); }
        }

        [Category(UIControlHelper.OriginalEventCategoryName)]
        public event ColumnMouseEventHandler ColumnMouseLeave
        {
            add { base.Events.AddHandler(EVENT_COLUMNMOUSELEAVE, value); }
            remove { base.Events.RemoveHandler(EVENT_COLUMNMOUSELEAVE, value); }
        }

        /// <summary>
        /// 列項目コンテキストメニューポップアップハンドラ。
        /// </summary>
        protected virtual void OnColumnContextMenuPopup(ColumnMouseEventArgs e)
        {
            ColumnMouseEventHandler handler = (ColumnMouseEventHandler)base.Events[EVENT_COLUMNCONTEXTMENUPOPUP];
            if (handler != null) { handler(this, e); }
        }

        /// <summary>
        /// OnColumnMouseEnter
        /// </summary>
        protected virtual void OnColumnMouseEnter(ColumnMouseEventArgs e)
        {
            ColumnMouseEventHandler handler = (ColumnMouseEventHandler)base.Events[EVENT_COLUMNMOUSEENTER];
            if (handler != null) { handler(this, e); }
        }

        private UIToolTip _ttpHint = null;
        private string oldHintText = string.Empty;
        private Timer _HintTimer = null;

        private void ShowToolTip(string text, int delay = 500)
        {
            oldHintText = text;

            if (_HintTimer != null)
            {
                _HintTimer.Dispose();
                _HintTimer = null;
            }

            if (_HintTimer == null)
            {
                _HintTimer = new Timer()
                {
                    Interval = delay
                };
            }

            _HintTimer.Tick += (sender, te) =>
            {
                try
                {
                    _HintTimer.Dispose();
                    _HintTimer = null;

                    if (_ttpHint == null)
                    {
                        _ttpHint = new UIToolTip();
                        _ttpHint.UseFading = false;
                    }
                    else
                    {
                        _ttpHint.Hide(this);
                        _ttpHint.RemoveAll();
                    }

                    Point pos = PointToClient(System.Windows.Forms.Cursor.Position);
                    int x = pos.X + 0;
                    int y = pos.Y + 24;

                    _ttpHint.Show(oldHintText, this, x, y);
                }
                catch (Exception /*exception*/)
                {
                    // 握りつぶす
                    // DebugConsole.WriteLine("{0}", exception.ToString());
                }
            };

            _HintTimer.Start();
        }

        /// <summary>
        /// OnColumnMouseMove
        /// </summary>
        protected virtual void OnColumnMouseMove(ColumnMouseEventArgs e)
        {
            ColumnMouseEventHandler handler = (ColumnMouseEventHandler)base.Events[EVENT_COLUMNMOUSEMOVE];
            if (handler != null) { handler(this, e); }

            if (DesignMode == false)
            {
                if (e.Header != null){
                    if (oldHintText != e.Header.Text)
                    {
                        ShowToolTip(e.Header.Text);
                    }
                }
            }
        }


        /// <summary>
        /// OnColumnMouseLeave
        /// </summary>
        protected virtual void OnColumnMouseLeave(ColumnMouseEventArgs e)
        {
            ColumnMouseEventHandler handler = (ColumnMouseEventHandler)base.Events[EVENT_COLUMNMOUSELEAVE];
            if (handler != null) { handler(this, e); }

            if (DesignMode == false)
            {
                HideToolTip();
            }
        }

        /// <summary>
        /// 列項目移動中イベント。
        /// </summary>
        /// <param name="e"></param>
        protected virtual void OnColumnReordering(ColumnReorderingEventArgs e)
        {
            if ((e.NewLocation.X > e.OldLocation.X) && ((e.NewLocation.X + SystemInformation.VerticalScrollBarWidth) > ClientRectangle.Width))
            {
                // 右にスクロール。(リストは左に流れる)
                var scrollInfo = new Win32.NativeMethods.SCROLLINFO();
                scrollInfo.cbSize = (uint)System.Runtime.InteropServices.Marshal.SizeOf(scrollInfo);
                scrollInfo.fMask = (int)Win32.NativeMethods.ScrollInfoMask.SIF_POS | (int)Win32.NativeMethods.ScrollInfoMask.SIF_RANGE;
                Win32.NativeMethods.GetScrollInfo(Handle, (int)Win32.NativeMethods.ScrollBarDirection.SB_HORZ, ref scrollInfo);
                if (scrollInfo.nPos < scrollInfo.nMax)
                {
                    // SetScrollInfo と SetScrollPos はリストビューと同期しないので、ここでは使えない。
                    // SB_THUMBPOSITION での位置指定も働かない。
                    // 唯一効果のある SB_LINERIGHT でスクロールさせる。
                    int nPos = Win32.SB.SB_LINERIGHT;
                    Win32.NativeMethods.SendMessage(Handle, Win32.WM.WM_HSCROLL, (IntPtr)nPos, IntPtr.Zero);
                    Refresh();
                }
            }
            else if ((e.NewLocation.X < e.OldLocation.X) && ((e.NewLocation.X - SystemInformation.VerticalScrollBarWidth) < ClientRectangle.X))
            {
                // 左にスクロール。(リストは右に流れる)
                var scrollInfo = new Win32.NativeMethods.SCROLLINFO();
                scrollInfo.cbSize = (uint)System.Runtime.InteropServices.Marshal.SizeOf(scrollInfo);
                scrollInfo.fMask = (int)Win32.NativeMethods.ScrollInfoMask.SIF_POS | (int)Win32.NativeMethods.ScrollInfoMask.SIF_RANGE;
                Win32.NativeMethods.GetScrollInfo(Handle, (int)Win32.NativeMethods.ScrollBarDirection.SB_HORZ, ref scrollInfo);
                if (scrollInfo.nPos > scrollInfo.nMin)
                {
                    // SetScrollInfo と SetScrollPos はリストビューと同期しないので、ここでは使えない。
                    // SB_THUMBPOSITION での位置指定も働かない。
                    // 唯一効果のある SB_LINELEFT でスクロールさせる。
                    int nPos = Win32.SB.SB_LINELEFT;
                    Win32.NativeMethods.SendMessage(Handle, Win32.WM.WM_HSCROLL, (IntPtr)nPos, IntPtr.Zero);
                    Refresh();
                }
            }
        }

        protected override void OnItemMouseHover(ListViewItemMouseHoverEventArgs e)
        {
            if (e.Item != null)
            {
                ShowToolTip(e.Item.ToolTipText, 200);
            }

            base.OnItemMouseHover(e);
        }

        protected override void OnMouseLeave(EventArgs e)
        {
            HideToolTip();
            base.OnMouseLeave(e);
        }

        #endregion

        //---------------------------------------------------------------------
        // 項目の移動
        /// <summary>
        /// 選択されている項目を手前に移動（単一選択モードのみ）。
        /// イベントは発生しません。
        /// </summary>
        public bool MoveSelectedItemPrev()
        {
            if (!MultiSelect && SelectedItems.Count > 0)
            {
                ListViewItem item = SelectedItems[0];
                int itemIndex = item.Index;
                if (itemIndex > 0)
                {
                    using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                    {
                        BeginUpdate();
                        {
                            // 手前に移動
                            Items.Remove(item);
                            Items.Insert(itemIndex - 1, item);

                            // 選択項目変更
                            Items[itemIndex - 1].Focused = true;
                            Items[itemIndex - 1].Selected = true;
                        }
                        EndUpdate();
                    }
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 選択されている項目を奥に移動（単一選択モードのみ）。
        /// イベントは発生しません。
        /// </summary>
        public bool MoveSelectedItemNext()
        {
            if (!MultiSelect && SelectedItems.Count > 0)
            {
                ListViewItem item = SelectedItems[0];
                int itemIndex = item.Index;
                if (itemIndex < (Items.Count - 1))
                {
                    using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                    {
                        BeginUpdate();
                        {
                            // 手前に移動
                            Items.Remove(item);
                            Items.Insert(itemIndex + 1, item);

                            // 選択項目変更
                            Items[itemIndex + 1].Focused = true;
                            Items[itemIndex + 1].Selected = true;
                        }
                        EndUpdate();
                    }
                    return true;
                }
            }
            return false;
        }

        //---------------------------------------------------------------------
        // 項目の並べ替え
        /// <summary>
        /// 項目の並べ替え。
        /// メイン項目のみで並べ替えます。
        /// </summary>
        public void Sort(bool ascent)
        {
            // メイン項目で並べ替える
            Sort(0, ascent);
        }

        /// <summary>
        /// 項目の並べ替え。
        /// サブ項目（列項目）を指定して並べ替えます。
        /// サブ項目がない場合は状態は変更されません。
        /// </summary>
        public void Sort(int column, bool ascent)
        {
            // 比較条件設定
            _itemComparer.ColumnIndex = column;
            _itemComparer.ColumnHeader = Columns[column];
            _itemComparer.Ascent = ascent;

            // ソート開始
            if (ListViewItemSorter != _itemComparer)
            {
                // 内部で Sort() を呼んでくれる
                ListViewItemSorter = _itemComparer;
            }
            else
            {
                // 手動で開始させる
                Sort();
            }
        }

        //---------------------------------------------------------------------
        // 項目の状態設定
        /// <summary>
        /// 選択項目を設定（単一選択モード用）。
        /// </summary>
        public void SetSelectedItem(int itemIndex)
        {
            if (!MultiSelect && 0 <= itemIndex && itemIndex < Items.Count)
            {
                SetItemSelectedFocusedState(Items[itemIndex], true);
            }
        }

        /// <summary>
        /// 項目の選択状態を設定。
        /// </summary>
        public static void SetItemSelectedState(ListViewItem item, bool selected)
        {
            // 異なる場合のみ設定する
            // ListViewItem.Selected は値が同じでも処理をするのでその対策
            if (item.Selected != selected)
            {
                item.Selected = selected;
            }
        }

        /// <summary>
        /// 項目のフォーカス状態を設定。
        /// </summary>
        public static void SetItemFocusedState(ListViewItem item, bool focused)
        {
            // 異なる場合のみ設定する
            // ListViewItem.Focused は値が同じでも処理をするのでその対策
            if (item.Focused != focused)
            {
                item.Focused = focused;
            }
        }

        /// <summary>
        /// 項目の選択状態とフォーカス状態を設定。
        /// </summary>
        public static void SetItemSelectedFocusedState(ListViewItem item, bool selected)
        {
            SetItemSelectedState(item, selected);
            SetItemFocusedState(item, selected);
        }

        //---------------------------------------------------------------------
        // 項目の情報取得
        /// <summary>
        /// サブ項目領域を取得。
        /// </summary>
        private Rectangle GetSubItemRect(int item, int subItem)
        {
            // 領域取得
            Rectangle rcItem = ListViewUtility.GetSubItemRect(this, item, subItem, Win32.LVIR.LVIR_BOUNDS);

            // メイン項目の領域補正
            // item == 0 の時は行全体が取得される為
            if (subItem == 0)
            {
                // 右位置補正
                Rectangle rcLabel = ListViewUtility.GetSubItemRect(this, item, subItem, Win32.LVIR.LVIR_LABEL);
                RectangleUtility.SetRight(ref rcItem, rcLabel.Right);

                // 左端じゃなければ左位置補正
                int[] order = ListViewUtility.GetColumnOrderArray(this);
                if (order[0] != 0)
                {
                    // 列番号
                    int column = 1;
                    for (int i = 1; i < order.Length; i++)
                    {
                        if (order[i] == 0)
                        {
                            column = i;
                            break;
                        }
                    }

                    // 左位置を左隣列の右位置に合わせる
                    Rectangle rcLeft = ListViewUtility.GetSubItemRect(this, item, order[column - 1], Win32.LVIR.LVIR_BOUNDS);
                    RectangleUtility.SetLeft(ref rcItem, rcLeft.Right);
                }
            }
            return rcItem;
        }

        /// <summary>
        /// アイテム数を設定する。
        /// </summary>
        public void SetItemCount(int count)
        {
            while (Items.Count < count)
            {
                ListViewItem item = new ListViewItem();
                for(int i = 0;i != Columns.Count;++ i)
                {
                    item.SubItems.Add(string.Empty);
                }
                Items.Add(item);
            }

            while (Items.Count > count)
            {
                Items.RemoveAt(Items.Count - 1);
            }
        }

        //---------------------------------------------------------------------
        // イベントハンドラ
        private void tmrSelectionChanged_Tick(object sender, EventArgs e)
        {
            _tmrSelectionChanged.Stop();
            OnSelectionChanged(EventArgs.Empty);
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                _components.Dispose();
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // ヘッダ通知メッセージ
                case Win32.WM.WM_NOTIFY:
                    {
                        Win32.NMHDR nmhdr = (Win32.NMHDR)m.GetLParam(typeof(Win32.NMHDR));
                        switch (nmhdr.code)
                        {
                            // 分割線トラッキング開始
                            case Win32.HDN.HDN_BEGINTRACKA:
                            case Win32.HDN.HDN_BEGINTRACKW:
                                if (_lockColumnResize)
                                {
                                    // Returns FALSE to allow tracking of the divider, or TRUE to prevent tracking.
                                    m.Result = (IntPtr)1;
                                    return;
                                }
                                break;

                            // 分割線ダブルクリック
                            case Win32.HDN.HDN_DIVIDERDBLCLICKA:
                            case Win32.HDN.HDN_DIVIDERDBLCLICKW:
                                {
                                    if (_lockColumnResize) { return; }

                                    // 列番号
                                    Win32.NMHEADER nmheader = (Win32.NMHEADER)m.GetLParam(typeof(Win32.NMHEADER));
                                    int column = nmheader.iItem;

                                    // イベント発行
                                    ColumnDividerDoubleClickEventArgs e = new ColumnDividerDoubleClickEventArgs(Columns[column]);
                                    OnColumnDividerDoubleClick(e);
                                    if (e.Handled) { return; }
                                }
                                break;
                            default:
                                break;
                        }
                    }
                    break;

                // マウスダウン
                case Win32.WM.WM_LBUTTONDOWN:
                case Win32.WM.WM_MBUTTONDOWN:
                case Win32.WM.WM_RBUTTONDOWN:
                    // アクティベート
                    if (!Focused)
                    {
                        Focus();
                    }
                    break;

                default:
                    break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnHandleCreated(EventArgs e)
        {
            // 列ヘッダ作成（詳細表示時のみ）
            if (View == View.Details)
            {
                // ハンドル再作成もあるので常に再作成する
                _header = new HeaderControl(this);
            }
            base.OnHandleCreated(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnItemCheck(ItemCheckEventArgs e)
        {
            // CanRaiseEvents 経由で発行されないのでここで調整する
            // 基本クラスメソッドはイベントを発行するだけなのは確認済み
            if (CanRaiseEvents)
            {
                base.OnItemCheck(e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnItemChecked(ItemCheckedEventArgs e)
        {
            // CanRaiseEvents 経由で発行されないのでここで調整する
            // 基本クラスメソッドはイベントを発行するだけなのは確認済み
            if (CanRaiseEvents)
            {
                base.OnItemChecked(e);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnSelectedIndexChanged(EventArgs e)
        {
            if (SelectionChangedEnable)
            {
                _tmrSelectionChanged.Start();
            }

            base.OnSelectedIndexChanged(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDrawItem(DrawListViewItemEventArgs e)
        {
            // 無効な状態フラグの時は何もしない
            if (e.State == 0) { return; }

            // カスタム描画（詳細表示のみ）
            if (_customDraw && View == View.Details)
            {
                // 領域補正
                Rectangle bounds = e.Bounds;
                if (GridLines)
                {
                    bounds.Height -= 1;
                }

                // 選択背景
                if (e.Item.Selected)
                {
                    if (_highlightSelection || Focused)
                    {
                        e.Graphics.FillRectangle(SystemBrushes.Highlight, bounds);
                    }
                    else
                    {
                        e.Graphics.FillRectangle(SystemBrushes.InactiveBorder, bounds);
                    }
                }
                else
                {
                    if (StripeDraw)
                    {
                        if ((e.ItemIndex & 1) == 1)
                        {
                            e.Graphics.FillRectangle(Brushes.WhiteSmoke, bounds);
                        }
                    }
                }

                // フォーカス枠
                if ((e.State & ListViewItemStates.Focused) != 0)
                {
                    GraphicsUtility.DrawFocusRectangle(e.Graphics, bounds);
                }

                // 状態をサブ項目描画用に保持しておく
                // 選択状態：e.State の ListViewItemStates.Selected は常に立っているので使えない（バグ？）
                // 列０の左：常に行の左端として取得されるため、そのままではサブ項目描画時に使えない
                _customDrawState.Selected    = e.Item.Selected;
                _customDrawState.Column0Left = GetSubItemRect(e.ItemIndex, 0).Left;
            }
            base.OnDrawItem(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDrawSubItem(DrawListViewSubItemEventArgs e)
        {
            // 無効な状態フラグの時は何もしない
            if (e.ItemState == 0) { return; }

            // カスタム描画（詳細表示のみ）
            if (_customDraw && View == View.Details)
            {
                // 領域補正
                Rectangle bounds = e.Bounds;
                {
                    // 列０は順序変更された時でも左位置が０のままになってしまう
                    if (e.ColumnIndex == 0)
                    {
                        bounds.X = _customDrawState.Column0Left;
                    }
                }
                if (GridLines)
                {
                    bounds.Height -= 1;
                }

                // イベント発行
                CustomDrawListViewItemEventArgs cde = new CustomDrawListViewItemEventArgs(this, e, bounds, _customDrawState.Selected);
                OnCustomDrawItem(cde);
                if (!cde.Handled)
                {
                    cde.DrawDefault();
                }
            }
            base.OnDrawSubItem(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDrawColumnHeader(DrawListViewColumnHeaderEventArgs e)
        {
            // OwnerDrawを有効にすると列ヘッダもオーナー描画になる
            e.DrawDefault = true;
            base.OnDrawColumnHeader(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnColumnClick(ColumnClickEventArgs e)
        {
            if (SortOnColumnClick)
            {
                // 項目があれば並べ替えを行う
                if (Items.Count > 1)
                {
                    // 比較条件
                    bool ascent = false;
                    if (_itemComparer.ColumnIndex == e.Column)
                    {
                        // 同じ列なら順序を逆にする
                        ascent = !_itemComparer.Ascent;
                    }

                    // 並べ替え開始
                    Sort(e.Column, ascent);
                }
            }
            base.OnColumnClick(e);
        }
        #endregion

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new ItemActivation Activation
        {
            get { return base.Activation; }
            set { base.Activation = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new ListViewAlignment Alignment
        {
            get { return base.Alignment; }
            set { base.Alignment = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(true)]
        [ReadOnly(true)]
        public new bool FullRowSelect
        {
            get { return base.FullRowSelect; }
            set { base.FullRowSelect = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        [ReadOnly(true)]
        public new bool HideSelection
        {
            get { return base.HideSelection; }
            set { base.HideSelection = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(false)]
        public new bool MultiSelect
        {
            get { return base.MultiSelect; }
            set { base.MultiSelect = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public sealed override RightToLeft RightToLeft
        {
            get { return base.RightToLeft; }
            set { base.RightToLeft = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public sealed override bool RightToLeftLayout
        {
            get { return base.RightToLeftLayout;  }
            set { base.RightToLeftLayout = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [DefaultValue(View.Details)]
        public new View View
        {
            get { return base.View; }
            set { base.View = value; }
        }

        #endregion

        public void HideToolTip()
        {
            if (_ttpHint != null)
            {
                _ttpHint.Hide(this);
                _ttpHint.RemoveAll();
                _ttpHint = null;
            }
            oldHintText = string.Empty;
        }

        /// <summary>
        /// すべて選択/非選択切り替え用のチェックボックスを設定する。
        /// </summary>
        public void SetupCheckAllCheckBox(UICheckBox checkBox)
        {
            checkBox.ThreeState = true;

            // チェック時のイベント
            ItemChecked += (s, e) =>
            {
                int checkedCount = CheckedItems.Count;
                checkBox.CheckState = checkedCount == 0 ? CheckState.Unchecked :
                    checkedCount < Items.Count ? CheckState.Indeterminate :
                    CheckState.Checked;
            };

            // チェックボックス変更時のイベント
            checkBox.CheckedChanged += (s, e) =>
            {
                ListViewItem updatedItem = null;
                using (var block = new UIControlEventSuppressBlock())
                {
                    switch (checkBox.CheckState)
                    {
                        case CheckState.Checked:
                            foreach (ListViewItem item in Items)
                            {
                                if (!item.Checked)
                                {
                                    item.Checked = true;
                                    updatedItem = item;
                                }
                            }
                            checkBox.ThreeState = false;
                            break;
                        case CheckState.Unchecked:
                            foreach (ListViewItem item in Items)
                            {
                                item.Checked = false;
                                updatedItem = item;
                            }
                            checkBox.ThreeState = false;
                            break;
                        case CheckState.Indeterminate:
                            checkBox.ThreeState = true;
                            break;
                    }
                }

                if (updatedItem != null)
                {
                    OnItemChecked(new ItemCheckedEventArgs(updatedItem));
                }
            };
        }

        // コンテンツ幅とヘッダー幅の両方を考慮したAutoResizeColumns
        public void AutoResizeColumnsByHeaderAndContents()
        {
            AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);
            for (var i = 0; i < Columns.Count; i++)
            {
                var colWidth = TextRenderer.MeasureText(Columns[i].Text, Font).Width + 10;
                if (colWidth > Columns[i].Width)
                {
                    Columns[i].Width = colWidth;
                }
            }
        }

        //---------------------------------------------------------------------
        // 内部クラス
        //---------------------------------------------------------------------
        #region ItemComparer
        /// <summary>項目比較ハンドラデリゲート。</summary>
        private delegate int CompareItemHandler(CompareListViewItemEventArgs e);

        /// <summary>
        /// 項目比較クラス。
        /// </summary>
        private sealed class ItemComparer : System.Collections.IComparer
        {
            // 列番号
            private int _columnIndex = 0;
            // 列ヘッダ
            private ColumnHeader _columnHeader = null;
            // 昇順比較フラグ
            private bool _ascent = false;
            // 項目比較ハンドラ
            private readonly CompareItemHandler _handler = null;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public ItemComparer(CompareItemHandler handler)
            {
                _handler = handler;
            }

            /// <summary>
            /// 列番号。
            /// </summary>
            public int ColumnIndex
            {
                get { return _columnIndex; }
                set { _columnIndex = value; }
            }

            /// <summary>
            /// 列ヘッダ。
            /// </summary>
            public ColumnHeader ColumnHeader
            {
                set { _columnHeader = value; }
            }

            /// <summary>
            /// 昇順比較か。
            /// </summary>
            public bool Ascent
            {
                get { return _ascent; }
                set { _ascent = value; }
            }

            /// <summary>
            /// 比較。
            /// </summary>
            public int Compare(object x, object y)
            {
                ListViewItem item1 = (ListViewItem)x;
                ListViewItem item2 = (ListViewItem)y;

                // 昇順比較
                if (_ascent)
                {
                    return _handler(new CompareListViewItemEventArgs(_columnIndex, _columnHeader, item2, item1));
                }
                // 降順比較
                else
                {
                    return _handler(new CompareListViewItemEventArgs(_columnIndex, _columnHeader, item1, item2));
                }
            }
        }
        #endregion

        #region HeaderControl
        /// <summary>
        /// ヘッダコントロールクラス。
        /// </summary>
        private sealed class HeaderControl : NativeWindow
        {
            // オーナー
            private readonly UIListView _owner;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public HeaderControl(UIListView owner)
            {
                _owner = owner;

                // ヘッダのハンドルを割り当てる
                IntPtr handle = ListViewUtility.GetHeader(owner);
                AssignHandle(handle);

                // ヘッダスタイル調整
                int style = Win32.NativeMethods.GetWindowLong(handle, Win32.GWL.GWL_STYLE);
                style |= Win32.HDS.HDS_HOTTRACK;
                Win32.NativeMethods.SetWindowLong(handle, Win32.GWL.GWL_STYLE, style);

                tmrColumnReordering = new Timer(components);
                tmrColumnReordering.Interval = 10;
                tmrColumnReordering.Tick += new EventHandler(tmrColumnReordering_Tick);
            }

            private bool inMouse = false;
            private ColumnHeader draggingColumnHeader = null;
            private Point startDraggingPosition = new Point();
            private Stopwatch swColumnReordering = new Stopwatch();
            private readonly Timer tmrColumnReordering = null;
            private readonly IContainer components = new Container();

            /// <summary>
            /// オーバーライド。
            /// </summary>
            protected override void WndProc(ref Message m)
            {
                switch (m.Msg)
                {
                    // サイズ変更のロック
                    case Win32.WM.WM_SETCURSOR:
                        if (_owner.LockColumnResize)
                        {
                            // If an application processes this message,
                            // it should return TRUE to halt further processing or FALSE to continue.
                            m.Result = (IntPtr)1;
                            return;
                        }
                        break;

                    // コンテキストメニュー
                    // ディバイダモード時には何もしない
                    case Win32.WM.WM_RBUTTONUP:
                        if (Cursor.Current == Cursors.Default)
                        {
                            _owner.OnColumnContextMenuPopup(CreateColumnContextMenuPopupEventArgs(ref m));
                        }
                        break;

                    case Win32.WM.WM_MOUSEMOVE:
                        if (inMouse == false)
                        {
                            //DebugConsole.WriteLine("MOUSE ENTER");
                            _owner.OnColumnMouseEnter(CreateColumnContextMenuPopupEventArgs(ref m));
                            inMouse = true;
                        }
                        else
                        {
                            //DebugConsole.WriteLine("MOUSE MOVE");
                            _owner.OnColumnMouseMove(CreateColumnContextMenuPopupEventArgs(ref m));
                        }
                        break;

                    case Win32.WM.WM_MOUSELEAVE:
                        //DebugConsole.WriteLine("MOUSE  LEAVE");
                        _owner.OnColumnMouseLeave(CreateColumnContextMenuPopupEventArgs(ref m));
                        inMouse = false;
                        break;

                    case Win32.WM.WM_LBUTTONDOWN:
                        if ((Cursor.Current == Cursors.Default) && _owner.AllowColumnReorder && inMouse)
                        {
                            tmrColumnReordering.Stop();

                            // ヘッダ上でのクリック位置
                            Point point = Win32.Utility.LParamToSignedPoint(m.LParam);

                            // 対応する列項目
                            draggingColumnHeader = null;
                            foreach (ColumnHeader column in _owner.Columns)
                            {
                                Rectangle rcColumn = HeaderUtility.GetItemRect(Handle, column.Index);
                                if (rcColumn.Contains(point))
                                {
                                    draggingColumnHeader = column;
                                    break;
                                }
                            }

                            if (draggingColumnHeader != null)
                            {
                                startDraggingPosition = point;

                                // 自分のスクリーン座標
                                Win32.RECT rcHeader = new Win32.RECT();
                                Win32.NativeMethods.GetWindowRect(Handle, ref rcHeader);

                                // 親のスクリーン座標から相対座標に変換
                                Rectangle rcOwner = _owner.RectangleToScreen(_owner.ClientRectangle);
                                var offset = rcOwner.X - rcHeader.left;
                                startDraggingPosition.X -= offset;

                                swColumnReordering.Reset();
                                tmrColumnReordering.Start();
                            }
                        }
                        break;

                    case Win32.WM.WM_LBUTTONUP:
                        tmrColumnReordering.Stop();
                        swColumnReordering.Stop();
                        draggingColumnHeader = null;
                        break;

                    default:
                        break;
                }
                base.WndProc(ref m);
            }

            private void tmrColumnReordering_Tick(object sender, EventArgs e)
            {
                _owner.OnColumnReordering(CreateColumnReorderingEventArgs());
            }

            private ColumnMouseEventArgs CreateColumnContextMenuPopupEventArgs(ref Message m)
            {
                // ヘッダ上でのクリック位置
                Point point = Win32.Utility.LParamToPoint(m.LParam);

                // 対応する列項目
                ColumnHeader columnHeader = null;
                foreach (ColumnHeader column in _owner.Columns)
                {
                    Rectangle rcColumn = HeaderUtility.GetItemRect(Handle, column.Index);
                    if (rcColumn.Contains(point))
                    {
                        columnHeader = column;
                        break;
                    }
                }

                // 自分のスクリーン座標
                Win32.RECT rcHeader = new Win32.RECT();
                Win32.NativeMethods.GetWindowRect(Handle, ref rcHeader);

                // 親のスクリーン座標から相対座標に変換してイベント発行
                Rectangle rcOwner = _owner.RectangleToScreen(_owner.ClientRectangle);
                point.X -= (rcOwner.X - rcHeader.left);

                return new ColumnMouseEventArgs(point, columnHeader);
            }

            private ColumnReorderingEventArgs CreateColumnReorderingEventArgs()
            {
                // 親のクライアント座標系でのマウス移動開始位置。
                Point oldPoint = new Point(startDraggingPosition.X, startDraggingPosition.Y);

                // 親のクライアント座標系でのマウス位置。
                Point newPoint = _owner.PointToClient(Cursor.Position);

                var deltaTime = swColumnReordering.Elapsed;
                swColumnReordering.Restart();

                return new ColumnReorderingEventArgs(oldPoint, newPoint, draggingColumnHeader, deltaTime);
            }
        }

        #endregion

        #region CustomDrawState
        /// <summary>
        /// カスタム描画状態保持クラス。
        /// </summary>
        private sealed class CustomDrawState
        {
            /// <summary>選択状態。</summary>
            public bool Selected = false;
            /// <summary>列項目０の左位置。</summary>
            public int Column0Left = 0;

            /// <summary>
            /// コンストラクタ。
            /// </summary>
            public CustomDrawState()
            {
            }
        }
        #endregion

    }
}
