﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

namespace App.Controls
{
    /// <summary>
    /// ＵＩラベルクラス。
    /// </summary>
    public class UILabel : Label
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UILabel()
        {
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new FlatStyle FlatStyle
        {
            get { return base.FlatStyle;  }
            set { base.FlatStyle = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseCompatibleTextRendering
        {
            get { return base.UseCompatibleTextRendering; }
            set { base.UseCompatibleTextRendering = value; }
        }
        #endregion
    }

    /// <summary>
    /// 編集されたかどうかを表示できるラベル
    /// </summary>
    public class UIModifiedMarkLabel : UILabel
    {
        /// <summary>
        /// 修正されたかどうか
        /// </summary>
        public bool IsModified
        {
            get
            {
                return modified;
            }
            set
            {
                if (modified != value)
                {
                    modified = value;
                    Invalidate();
                }
            }
        }
        private bool modified = false;

        public bool IsMarkShow
        {
            get
            {
                return isMarkShow_;
            }
            set
            {
                if (isMarkShow_ != value)
                {
                    isMarkShow_ = value;
                    Invalidate();
                }
            }
        }
        private bool isMarkShow_ = true;

        public override bool AutoSize
        {
            get { return false; }
            set { ; }		// 無視
        }

        protected override void OnTextChanged(EventArgs e)
        {
            ResetSize();
        }

        protected override void OnFontChanged(EventArgs e)
        {
            ResetSize();
        }

        protected virtual void ResetSize()
        {
            using (var g = CreateGraphics())
            {
                // サイズ、位置を計算し直す
                var textStringSize = TextRenderer.MeasureText(g, Text, Font);

                MarkPosX = textStringSize.Width;

                Size = new Size(textStringSize.Width + ModifiedMark.StarMark.Width, textStringSize.Height);
            }

            Invalidate();
        }

        protected int MarkPosX = 0;

        protected override void OnPaint(PaintEventArgs e)
        {
            if (IsMarkShow)
            {
                if (IsModified || DesignMode)
                {
                    e.Graphics.DrawImage(ModifiedMark.StarMark, MarkPosX, 0);
                }
            }
            base.OnPaint(e);
        }
    }

    public class UIModifiedMarkAndImageLabel : UIModifiedMarkLabel
    {
        // ラベル右に表示される画像を保持
        // ラベル、画像、変更マークの順で表示される
        private List<Bitmap> _tailImages = new List<Bitmap>();

        public List<Bitmap> TailImages
        {
            get { return _tailImages; }
            set
            {
                _tailImages = value;
                ResetSize();
            }
        }

        private int _imagePosX = 0;

        protected override void ResetSize()
        {
            using (var g = CreateGraphics())
            {
                // サイズ、位置を計算し直す
                var textStringSize = TextRenderer.MeasureText(g, Text, Font);

                var rightPos = textStringSize.Width;
                var imageHeight = 0;
                if (_tailImages != null && _tailImages.Any())
                {
                    _imagePosX = rightPos;
                    imageHeight = _tailImages.Max(x => x.Height);
                    rightPos = _imagePosX + _tailImages.Sum(x => x.Width) + (_tailImages.Count - 1) * 2;
                }

                MarkPosX = rightPos;

                Size = new Size(rightPos + ModifiedMark.StarMark.Width, Math.Max(textStringSize.Height, imageHeight));
            }

            Invalidate();
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            if (_tailImages != null && _tailImages.Any())
            {
                var x = _imagePosX;
                foreach (var image in _tailImages)
                {
                    e.Graphics.DrawImage(image, x, 0, image.Width, image.Height);
                    x += image.Width + 2;
                }
            }
        }
    }

    /// <summary>
    /// ＵＩリンクラベルクラス。
    /// </summary>
    public class UILinkLabel : LinkLabel
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UILinkLabel()
        {
        }

        /// <summary>
        /// リンク範囲を文字列で指定します。
        /// LinkArea プロパティよりは柔軟に設定可能です。
        /// </summary>
        public void SetLinkArea(string linkString)
        {
            // 大文字小文字の違いは無視して検索する
            int startIndex = Text.IndexOf(linkString, StringComparison.CurrentCultureIgnoreCase);
            if (startIndex != -1)
            {
                LinkArea = new LinkArea(startIndex, linkString.Length);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [ReadOnly(true)]
        public new bool UseCompatibleTextRendering
        {
            get { return base.UseCompatibleTextRendering;  }
            set { base.UseCompatibleTextRendering = value; }
        }
        #endregion
    }
}
