﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using App.Utility;

namespace App.Controls
{
    /// <summary>
    /// ＵＩコントロールクラス。
    /// </summary>
    public abstract class UIControl : Control
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIControl()
        {
            Font = TheApp.GuiFont;
        }
    }

    //-------------------------------------------------------------------------
    // UIControl 派生クラス
    //-------------------------------------------------------------------------
    #region UnselectableControl
    /// <summary>
    /// 選択不可コントロールクラス。
    /// </summary>
    public abstract class UnselectableControl : UIControl
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UnselectableControl()
        {
            base.SetStyle(ControlStyles.Opaque, true);
            base.SetStyle(ControlStyles.ResizeRedraw, true);
            base.SetStyle(ControlStyles.Selectable, false);
            base.DoubleBuffered = true;
            base.TabStop = false;
        }

        #region デザイナ制御
        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new int TabIndex
        {
            get { return base.TabIndex; }
            set { base.TabIndex = value; }
        }

        /// <summary>
        /// 再定義。
        /// </summary>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new bool TabStop
        {
            get { return base.TabStop; }
            set { base.TabStop = value; }
        }
        #endregion
    }
    #endregion

    #region MouseOperatableControl
    /// <summary>
    /// マウス操作制御コントロールクラス。
    /// </summary>
    [ToolboxItem(false)]
    public class MouseOperatableControl : ScrollableControl
    {
        // フォーカス変更時再描画フラグ
        private bool _focusRedraw = false;
        // Altキー入力有効フラグ
        private bool _enableAltInput = false;
        // Alt修飾有効キーリスト
        private readonly List<Keys> _altModifiedKeys = new List<Keys>();
        // 押下キーリスト
        private readonly List<Keys> _pressKeys = new List<Keys>();
        // マウスドラッグ
        private readonly MouseDrag _mouseDrag;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MouseOperatableControl()
        {
            _mouseDrag = new MouseDrag(this);

            // 独自描画を前提にする
            DoubleBuffered = true;
            ResizeRedraw = true;
        }

        //---------------------------------------------------------------------
        // プロパティ
        //---------------------------------------------------------------------
        #region プロパティ
        /// <summary>
        /// フォーカス変更時再描画フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Description("フォーカス切り替わり時に再描画が必要かどうかを示します。")]
        public bool FocusRedraw
        {
            get { return _focusRedraw; }
            set { _focusRedraw = value; }
        }

        /// <summary>
        /// Altキー入力有効フラグ。
        /// </summary>
        [DefaultValue(false)]
        [Description("Altキー入力を有効にするかどうかを示します。")]
        public bool EnableAltInput
        {
            get { return _enableAltInput; }
            set { _enableAltInput = value; }
        }

        /// <summary>
        /// 押下キーリスト。
        /// </summary>
        [Browsable(false)]
        public ReadOnlyList<Keys> PressKeys
        {
            get { return new ReadOnlyList<Keys>(_pressKeys); }
        }

        /// <summary>
        /// マウスドラッグボタン。
        /// </summary>
        [Browsable(false)]
        public MouseButtons MouseDragButton
        {
            get { return _mouseDrag.Button; }
        }

        /// <summary>
        /// マウスドラッグ位置。
        /// </summary>
        [Browsable(false)]
        public Point MouseDragPoint
        {
            get { return _mouseDrag.CurrentPt; }
        }

        /// <summary>
        /// マウスドラッグ開始位置。
        /// </summary>
        [Browsable(false)]
        public Point MouseDragStartPoint
        {
            get { return _mouseDrag.StartPt; }
        }

        /// <summary>
        /// マウスドラッグ中継位置。
        /// </summary>
        [Browsable(false)]
        public Point MouseDragRelayPoint
        {
            get { return _mouseDrag.RelayPt; }
        }

        /// <summary>
        /// マウスドラッグ領域。
        /// </summary>
        [Browsable(false)]
        public Rectangle MouseDragBounds
        {
            get { return _mouseDrag.DragBounds; }
        }

        /// <summary>
        /// マウスドラッグ増分領域。
        /// </summary>
        [Browsable(false)]
        public Rectangle MouseDragDeltaBounds
        {
            get { return _mouseDrag.DeltaBounds; }
        }

        /// <summary>
        /// マウスドラッグサイズ。
        /// </summary>
        [Browsable(false)]
        public Size MouseDragSize
        {
            get { return _mouseDrag.DragSize; }
        }

        /// <summary>
        /// マウスドラッグ増分サイズ。
        /// </summary>
        [Browsable(false)]
        public Size MouseDragDeltaSize
        {
            get { return _mouseDrag.DeltaSize; }
        }
        #endregion

        //---------------------------------------------------------------------
        // イベント
        //---------------------------------------------------------------------
        #region イベント
        //---------------------------------------------------------------------
        private static readonly object EVENT_BEGINMOUSEDRAG = new object();

        /// <summary>
        /// マウスドラッグ開始イベント。
        /// </summary>
        public event MouseBeginDragEventHandler BeginMouseDrag
        {
            add { base.Events.AddHandler(EVENT_BEGINMOUSEDRAG, value); }
            remove { base.Events.RemoveHandler(EVENT_BEGINMOUSEDRAG, value); }
        }

        /// <summary>
        /// マウスドラッグ開始ハンドラ。
        /// </summary>
        protected virtual void OnBeginMouseDrag(MouseBeginDragEventArgs e)
        {
            MouseBeginDragEventHandler handler = (MouseBeginDragEventHandler)base.Events[EVENT_BEGINMOUSEDRAG];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_MOUSEDRAG = new object();

        /// <summary>
        /// マウスドラッグイベント。
        /// </summary>
        public event MouseEventHandler MouseDrag
        {
            add { base.Events.AddHandler(EVENT_MOUSEDRAG, value); }
            remove { base.Events.RemoveHandler(EVENT_MOUSEDRAG, value); }
        }

        /// <summary>
        /// マウスドラッグハンドラ。
        /// </summary>
        protected virtual void OnMouseDrag(MouseEventArgs e)
        {
            MouseEventHandler handler = (MouseEventHandler)base.Events[EVENT_MOUSEDRAG];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_ENDMOUSEDRAG = new object();

        /// <summary>
        /// マウスドラッグ終了イベント。
        /// </summary>
        public event MouseEventHandler EndMouseDrag
        {
            add { base.Events.AddHandler(EVENT_ENDMOUSEDRAG, value); }
            remove { base.Events.RemoveHandler(EVENT_ENDMOUSEDRAG, value); }
        }

        /// <summary>
        /// マウスドラッグ終了ハンドラ。
        /// </summary>
        protected virtual void OnEndMouseDrag(MouseEventArgs e)
        {
            MouseEventHandler handler = (MouseEventHandler)base.Events[EVENT_ENDMOUSEDRAG];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_PRESSKEYSCHANGED = new object();

        /// <summary>
        /// 押下キー変更イベント。
        /// </summary>
        public event EventHandler PressKeysChanged
        {
            add { base.Events.AddHandler(EVENT_PRESSKEYSCHANGED, value); }
            remove { base.Events.RemoveHandler(EVENT_PRESSKEYSCHANGED, value); }
        }

        /// <summary>
        /// 押下キー変更ハンドラ。
        /// </summary>
        protected virtual void OnPressKeysChanged(EventArgs e)
        {
            EventHandler handler = (EventHandler)base.Events[EVENT_PRESSKEYSCHANGED];
            if (handler != null) { handler(this, e); }
        }

        //---------------------------------------------------------------------
        private static readonly object EVENT_CONTEXTMENUPOPUP = new object();

        /// <summary>
        /// コンテキストメニューポップアップイベント。
        /// </summary>
        public event ContextMenuPopupEventHandler ContextMenuPopup
        {
            add { base.Events.AddHandler(EVENT_CONTEXTMENUPOPUP, value); }
            remove { base.Events.RemoveHandler(EVENT_CONTEXTMENUPOPUP, value); }
        }

        /// <summary>
        /// コンテキストメニューポップアップハンドラ。
        /// </summary>
        protected virtual void OnContextMenuPopup(ContextMenuPopupEventArgs e)
        {
            ContextMenuPopupEventHandler handler = (ContextMenuPopupEventHandler)base.Events[EVENT_CONTEXTMENUPOPUP];
            if (handler != null) { handler(this, e); }
        }
        #endregion

        //---------------------------------------------------------------------
        // メソッド
        //---------------------------------------------------------------------
        /// <summary>
        /// Alt修飾有効キーを登録します。
        /// Alt修飾＋登録キーでのオーナーウィンドウによるコマンド処理が無効になり、通常のキー入力処理を行うことができます。
        /// </summary>
        public void RegisterAltModifiedKeys(Keys keyCode)
        {
            // 修飾キーは除外
            if ((keyCode & (Keys.Alt | Keys.Control | Keys.Shift)) != Keys.None)
            {
                return;
            }
            // 定義済み値のみ有効
            if (Enum.IsDefined(typeof(Keys), keyCode))
            {
                if (!_altModifiedKeys.Contains(keyCode))
                {
                    _altModifiedKeys.Add(keyCode);
                }
            }
        }

        /// <summary>
        /// マウスドラッグ中か。
        /// </summary>
        public bool IsMouseDragging()
        {
            return _mouseDrag.IsDragging();
        }

        /// <summary>
        /// マウスドラッグ中か。
        /// </summary>
        public bool IsMouseDragging(MouseButtons button)
        {
            return _mouseDrag.IsDragging(button);
        }

        /// <summary>
        /// マウスドラッグ量が指定ピクセル数を超えているかどうかの検証。
        /// 超えた場合はドラッグ中継点がリセットされます。
        /// </summary>
        public bool TestMouseDragSizeOver(int pixel)
        {
            Size size = MouseDragSize;
            if (Math.Abs(size.Width) > pixel || Math.Abs(size.Height) > pixel)
            {
                _mouseDrag.RelayPt = _mouseDrag.StartPt;
                return true;
            }
            return false;
        }

        /// <summary>
        /// マウスドラッグを強制的に解除。
        /// </summary>
        public void CancelMouseDragging()
        {
            if (_mouseDrag.IsDragging())
            {
                OnEndMouseDrag(new MouseEventArgs(_mouseDrag.Button, 1, _mouseDrag.CurrentPt.X, _mouseDrag.CurrentPt.Y, 1));
                _mouseDrag.End();
            }
        }

        /// <summary>
        /// キー押下されているか。
        /// </summary>
        public bool IsPressKeys(params Keys[] keyCodes)
        {
            // 押下キーリストと一致するかどうか
            if (keyCodes.Length > 0)
            {
                if (keyCodes.Length == _pressKeys.Count)
                {
                    List<Keys> keyList = new List<Keys>(keyCodes);
                    for (int i = 0; i < keyList.Count; i++)
                    {
                        if (keyList[i] != _pressKeys[i])
                        {
                            return false;
                        }
                    }
                    return true;
                }
            }
            return false;
        }

        #region オーバーライド
        //---------------------------------------------------------------------
        // プロパティ
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected sealed override bool CanRaiseEvents
        {
            get
            {
                if (UIControlEventSuppressBlock.Enabled)
                {
                    return false;
                }
                return base.CanRaiseEvents;
            }
        }

        //---------------------------------------------------------------------
        // プリプロセス
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            switch (m.Msg)
            {
                // アクティベート
                // OnMouseDown()より前に行う
                case Win32.WM.WM_LBUTTONDOWN:
                case Win32.WM.WM_MBUTTONDOWN:
                case Win32.WM.WM_RBUTTONDOWN:
                    if (!Focused)
                    {
                        Focus();
                    }
                    break;

                default: break;
            }
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // Alt入力有効時
            if (_enableAltInput)
            {
                // Altキー単体入力
                if (keyData == (Keys.Alt | Keys.Menu))
                {
                    // コマンドキー処理をしない
                    OnKeyDown(new KeyEventArgs(keyData));
                    return true;
                }
                // Alt修飾＋有効キー入力
                else if ((keyData & Keys.Alt) != Keys.None)
                {
                    foreach (Keys key in _altModifiedKeys)
                    {
                        if (keyData == (Keys.Alt | key))
                        {
                            // コマンドキー処理をしない
                            OnKeyDown(new KeyEventArgs(keyData));
                            return true;
                        }
                    }
                }
            }
            return base.ProcessCmdKey(ref msg, keyData);
        }

        //---------------------------------------------------------------------
        // フォーカス
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            // フォーカス切り替わり時の再描画
            if (_focusRedraw)
            {
                Invalidate();
            }
            base.OnGotFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            // マウスドラッグ解除
            CancelMouseDragging();

            // 押下キー解除
            if (_pressKeys.Count > 0)
            {
                _pressKeys.Clear();
                OnPressKeysChanged(EventArgs.Empty);
            }

            // フォーカス切り替わり時の再描画
            if (_focusRedraw)
            {
                Invalidate();
            }
            base.OnLostFocus(e);
        }

        //---------------------------------------------------------------------
        // マウス
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            // マウスドラッグ開始
            if (!_mouseDrag.IsDragging())
            {
                _mouseDrag.Start(e.Location, e.Button);

                // キャンセルされたらドラッグ中断
                MouseBeginDragEventArgs bde = new MouseBeginDragEventArgs(e);
                OnBeginMouseDrag(bde);
                if (bde.Cancel)
                {
                    _mouseDrag.End();
                }
            }
            base.OnMouseDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            // マウスドラッグ中
            if (_mouseDrag.IsDragging())
            {
                if (e.Button == MouseButtons.None)
                {
                    _mouseDrag.End();
                }
                else
                {
                    _mouseDrag.Drag(e.Location);
                    OnMouseDrag(new MouseEventArgs(_mouseDrag.Button, e.Clicks, e.X, e.Y, e.Delta));
                }
            }
            base.OnMouseMove(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            // マウスドラッグ終了
            if (_mouseDrag.IsDragging(e.Button))
            {
                OnEndMouseDrag(e);
                _mouseDrag.End();

                // コンテキストメニュー
                if (e.Button == MouseButtons.Right)
                {
                    OnContextMenuPopup(new ContextMenuPopupEventArgs(e.Location, false));
                }
            }
            base.OnMouseUp(e);
        }

        //---------------------------------------------------------------------
        // キー入力
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            // 押下キー登録
            if (!_pressKeys.Contains(e.KeyCode))
            {
                _pressKeys.Add(e.KeyCode);
                _pressKeys.Sort();
                OnPressKeysChanged(EventArgs.Empty);
            }
            base.OnKeyDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            // 押下キー削除
            if (_pressKeys.Contains(e.KeyCode))
            {
                _pressKeys.Remove(e.KeyCode);
                OnPressKeysChanged(EventArgs.Empty);
            }
            // コンテキストメニュー
            if (e.KeyCode == Keys.Apps)
            {
                // カーソル位置に表示
                OnContextMenuPopup(new ContextMenuPopupEventArgs(PointToClient(Cursor.Position), true));
            }
            base.OnKeyUp(e);
        }
        #endregion
    }
    #endregion

    //-------------------------------------------------------------------------
    // UIControl ユーティリティクラス
    //-------------------------------------------------------------------------
    #region UIControlEventSuppressBlock
    /// <summary>
    /// ＵＩコントロールイベント抑制ブロッククラス。
    /// </summary>
    public sealed class UIControlEventSuppressBlock : IDisposable
    {
        // インスタンスリスト
        private static readonly List<UIControlEventSuppressBlock> _instances = new List<UIControlEventSuppressBlock>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UIControlEventSuppressBlock()
        {
            // リストに追加
            _instances.Add(this);
        }

        /// <summary>
        /// 開放処理。
        /// </summary>
        public void Dispose()
        {
            // リストから削除
            _instances.Remove(this);
        }

        /// <summary>
        /// 抑制中かどうか。
        /// </summary>
        public static bool Enabled
        {
            get
            {
                // インスタンスがあれば抑制中
                return _instances.Count > 0;
            }
        }
    }
    #endregion

    #region UIControlHelper
    /// <summary>
    /// ＵＩコントロールヘルパクラス。
    /// </summary>
    public static class UIControlHelper
    {
        //---------------------------------------------------------------------------
        // 汎用カテゴリ名
        /// <summary>オリジナルプロパティカテゴリ名。</summary>
        public const string OriginalPropertyCategoryName = "[独自プロパティ]";
        /// <summary>オリジナルイベントカテゴリ名。</summary>
        public const string OriginalEventCategoryName = "[独自イベント]";

        //---------------------------------------------------------------------------
        // 汎用プロパティ説明（DescriptionAttribute 用）
        /// <summary>拡張ボーダースタイルプロパティ説明。</summary>
        public const string PropertyDesc_ExtendedBorderStyle = "拡張ボーダースタイル（WS_EX_STATICEDGE、WS_EX_DLGMODALFRAME）を示します。";
        /// <summary>ダブルバッファリングプロパティ説明。</summary>
        public const string PropertyDesc_DoubleBufferring = "ダブルバッファリングするかどうかを示します。";

        //---------------------------------------------------------------------------
        // ユーティリティメソッド
        /// <summary>
        /// 拡張ボーダースタイル用に CreateParams を設定。
        /// </summary>
        public static void SetExtendedBorderStyleCreateParams(CreateParams cp, ExtendedBorderStyle style)
        {
            switch (style)
            {
                case ExtendedBorderStyle.StaticEdge:
                    cp.ExStyle |= Win32.WS_EX.WS_EX_STATICEDGE;
                    break;
                case ExtendedBorderStyle.DlgModalFrame:
                    cp.ExStyle |= Win32.WS_EX.WS_EX_DLGMODALFRAME;
                    break;
                default:
                    break;
            }
        }

        /// <summary>
        /// 浮動小数値の表示用文字列を取得。
        /// </summary>
        public static string GetFloatValueDisplayString(float value, FloatValueDisplayStyle style)
        {
            return GetFloatValueDisplayString<float>(value, style);
        }

        /// <summary>
        /// 浮動小数値の表示用文字列を取得。
        /// </summary>
        public static string GetFloatValueDisplayString(double value, FloatValueDisplayStyle style)
        {
            return GetFloatValueDisplayString<double>(value, style);
        }

        /// <summary>
        /// 浮動小数値の表示用文字列を取得。
        /// </summary>
        private static string GetFloatValueDisplayString<Type>(Type value, FloatValueDisplayStyle style) where Type : IFormattable
        {
            switch (style)
            {
                case FloatValueDisplayStyle.AutoDecimal:
                    return GetFloatValueAutoDecimalString(value);
                case FloatValueDisplayStyle.Standard:
                    return value.ToString();
                case FloatValueDisplayStyle.Fixed1:
                    return value.ToString("f1", System.Globalization.CultureInfo.CurrentCulture);
                case FloatValueDisplayStyle.Fixed2:
                    return value.ToString("f2", System.Globalization.CultureInfo.CurrentCulture);
                case FloatValueDisplayStyle.Fixed3:
                    return value.ToString("f3", System.Globalization.CultureInfo.CurrentCulture);
                case FloatValueDisplayStyle.Fixed4:
                    return value.ToString("f4", System.Globalization.CultureInfo.CurrentCulture);
                case FloatValueDisplayStyle.Fixed5:
                    return value.ToString("f5", System.Globalization.CultureInfo.CurrentCulture);
                case FloatValueDisplayStyle.Fixed6:
                    return value.ToString("f6", System.Globalization.CultureInfo.CurrentCulture);
                default:
                    Debug.Assert(false);
                    break;
            }
            return value.ToString();
        }

        /// <summary>
        /// 浮動小数値の自動小数形式文字列を取得。
        /// </summary>
        private static string GetFloatValueAutoDecimalString<Type>(Type value) where Type : IFormattable
        {
            string valueString = value.ToString("f6", System.Globalization.CultureInfo.CurrentCulture);

            // 小数点位置
            int periodIndex = valueString.LastIndexOf('.');
            if (periodIndex == -1)
            {
                return valueString + ".0";
            }

            // 小数部文字列
            string decimalString = valueString.Substring(periodIndex + 1);
            if (decimalString.Length <= 1)
            {
                return valueString;
            }

            // 右側から「０」の文字数を計測
            int rightZeroCount = 0;
            for (int i = decimalString.Length - 1; i >= 0; i--)
            {
                if (decimalString[i] == '0')
                {
                    rightZeroCount++;
                }
                else
                {
                    break;
                }
            }
            // 最低１桁は表示する
            if (rightZeroCount == decimalString.Length)
            {
                rightZeroCount--;
            }

            return valueString.Substring(0, valueString.Length - rightZeroCount);
        }

        /// <summary>
        /// イベントメッセージ出力。
        /// </summary>
        [Conditional("DEBUG")]
        public static void EventMessage(string eventName, Control sender, object arg)
        {
            DebugConsole.WriteLine("[ControlEvent] {0}::{1} {2} - {3}",
                sender.GetType().Name,
                eventName,
                sender.Name,
                arg.ToString()
            );
        }
    }
    #endregion

    //-------------------------------------------------------------------------
    // イベント定義
    //-------------------------------------------------------------------------
    #region MouseBeginDragEvent
    /// <summary>
    /// マウスドラッグ開始イベントハンドラデリゲート。
    /// </summary>
    public delegate void MouseBeginDragEventHandler(object sender, MouseBeginDragEventArgs e);

    /// <summary>
    /// マウスドラッグ開始イベントデータクラス。
    /// </summary>
    public sealed class MouseBeginDragEventArgs : MouseEventArgs
    {
        // キャンセルフラグ
        private bool _cancel = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MouseBeginDragEventArgs(MouseEventArgs e)
            : base(e.Button, e.Clicks, e.X, e.Y, e.Delta)
        {
        }

        /// <summary>
        /// ドラッグ開始をキャンセルするかどうか。
        /// </summary>
        public bool Cancel
        {
            get { return _cancel; }
            set { _cancel = value; }
        }
    }
    #endregion

    #region ContextMenuPopupEvent
    /// <summary>
    /// コンテキストメニューポップアップイベントハンドラデリゲート。
    /// </summary>
    public delegate void ContextMenuPopupEventHandler(object sender, ContextMenuPopupEventArgs e);

    /// <summary>
    /// コンテキストメニューポップアップイベントデータクラス。
    /// </summary>
    public class ContextMenuPopupEventArgs : EventArgs
    {
        // 位置
        private readonly Point _location;
        // キー入力からかどうか
        private readonly bool _fromAppKey;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ContextMenuPopupEventArgs(Point location, bool fromAppKey)
        {
            _location = location;
            _fromAppKey = fromAppKey;
        }

        /// <summary>
        /// 位置。
        /// </summary>
        public Point Location
        {
            get { return _location; }
        }

        /// <summary>
        /// 位置Ｘ。
        /// </summary>
        public int X
        {
            get { return _location.X; }
        }

        /// <summary>
        /// 位置Ｙ。
        /// </summary>
        public int Y
        {
            get { return _location.Y; }
        }

        /// <summary>
        /// キー入力からかどうか。
        /// </summary>
        public bool FromAppKey
        {
            get { return _fromAppKey; }
        }
    }
    #endregion

    #region ColumnContextMenuPopupEvent
    /// <summary>
    /// 列項目コンテキストメニューポップアップイベントハンドラデリゲート。
    /// </summary>
    public delegate void ColumnMouseEventHandler(object sender, ColumnMouseEventArgs e);

    /// <summary>
    /// 列項目コンテキストメニューポップアップイベントデータクラス。
    /// </summary>
    public sealed class ColumnMouseEventArgs : ContextMenuPopupEventArgs
    {
        // 列項目
        private readonly ColumnHeader _header;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColumnMouseEventArgs(Point location, ColumnHeader header)
            : base(location, false)
        {
            _header = header;
        }

        /// <summary>
        /// 列項目。
        /// </summary>
        public ColumnHeader Header
        {
            get { return _header; }
        }
    }
    #endregion

    #region ColumnDividerDoubleClickEvent
    /// <summary>
    /// 列分割線ダブルクリックイベントハンドラデリゲート。
    /// </summary>
    public delegate void ColumnDividerDoubleClickEventHandler(object sender, ColumnDividerDoubleClickEventArgs e);

    /// <summary>
    /// 列分割線ダブルクリックイベントデータクラス。
    /// </summary>
    public sealed class ColumnDividerDoubleClickEventArgs : HandledEventArgs
    {
        // 列項目
        private readonly ColumnHeader _header;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColumnDividerDoubleClickEventArgs(ColumnHeader header)
        {
            _header = header;
        }

        /// <summary>
        /// 列項目。
        /// </summary>
        public ColumnHeader Header
        {
            get { return _header; }
        }
    }
    #endregion

    #region ColumnReorderingEvent
    /// <summary>
    /// 列順の変更中イベントハンドラデリゲート。
    /// </summary>
    public delegate void ColumnReorderingEventHandler(object sender, ColumnReorderingEventArgs e);

    /// <summary>
    /// 列順の変更中イベントデータクラス。
    /// </summary>
    public sealed class ColumnReorderingEventArgs : HandledEventArgs
    {
        // マウスがクリックされた位置
        private readonly Point _oldLocation;
        // 現在のマウスカーソル位置
        private readonly Point _newLocation;
        // 列項目
        private readonly ColumnHeader _header;
        // ドラッグ中に連続発生しているイベントの、直前のイベント呼び出しからの経過時間
        private readonly TimeSpan _deltaTime;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColumnReorderingEventArgs(Point oldLocation, Point newLocation, ColumnHeader header, TimeSpan deltaTime)
        {
            _oldLocation = oldLocation;
            _newLocation = newLocation;
            _header = header;
            _deltaTime = deltaTime;
        }

        /// <summary>
        /// 列項目。
        /// </summary>
        public ColumnHeader Header
        {
            get { return _header; }
        }

        /// <summary>
        /// マウスがクリックされた位置。
        /// </summary>
        public Point OldLocation
        {
            get { return _oldLocation; }
        }

        /// <summary>
        /// 現在のマウスカーソル位置。
        /// </summary>
        public Point NewLocation
        {
            get { return _newLocation; }
        }

        /// <summary>
        /// ドラッグ中に連続発生しているイベントの、直前のイベント呼び出しからの経過時間。
        /// </summary>
        public TimeSpan DeltaTime
        {
            get { return _deltaTime; }
        }
    }
    #endregion

    #region CustomDrawItemEventArgs
    /// <summary>
    /// 項目カスタム描画イベントデータクラス。
    /// </summary>
    public class CustomDrawItemEventArgs : HandledEventArgs
    {
        //---------------------------------------------------------------------
        // ユーザー指定項目（既定描画用）
        //---------------------------------------------------------------------
        // 背景色
        private Color _backColor = Color.Empty;
        // 前景色
        private Color _foreColor = Color.Empty;
        // テキスト
        private string _text = null;
        // フォント
        private Font _font = null;
        // イメージ番号
        private int _imageIndex = -1;
        // イメージキー
        private string _imageKey = null;
        // イメージ
        private Image _image;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CustomDrawItemEventArgs()
        {
        }

        /// <summary>
        /// 背景色の指定。
        /// </summary>
        public Color SpecificBackColor
        {
            get { return _backColor; }
            set { _backColor = value; }
        }

        /// <summary>
        /// 前景色の指定。
        /// </summary>
        public Color SpecificForeColor
        {
            get { return _foreColor; }
            set { _foreColor = value; }
        }

        /// <summary>
        /// テキストの指定。
        /// </summary>
        public string SpecificText
        {
            get { return _text; }
            set { _text = value; }
        }

        /// <summary>
        /// フォントの指定。
        /// </summary>
        public Font SpecificFont
        {
            get { return _font; }
            set { _font = value; }
        }

        /// <summary>
        /// イメージ番号の指定。
        /// </summary>
        public int SpecificImageIndex
        {
            get { return _imageIndex; }
            set { _imageIndex = value; }
        }

        /// <summary>
        /// イメージキーの指定。
        /// </summary>
        public string SpecificImageKey
        {
            get { return _imageKey; }
            set { _imageKey = value; }
        }

        /// <summary>
        /// イメージの指定。
        /// </summary>
        public Image SpecificImage
        {
            get { return _image; }
            set { _image = value; }
        }

        /// <summary>
        /// 枠線描画用のペン。
        /// </summary>
        public Pen FramePen{ get; set; }

        /// <summary>
        /// イメージを取得。
        /// </summary>
        protected Image GetImage(ImageList imageList, int index, string key)
        {
            if (imageList != null)
            {
                if (index >= 0 && index < imageList.Images.Count)
                {
                    return imageList.Images[index];
                }
                else if (key != null && imageList.Images.ContainsKey(key))
                {
                    return imageList.Images[key];
                }
            }
            return null;
        }
    }
    #endregion

    #region CustomDrawListViewItemEvent
    /// <summary>
    /// リストビュー項目カスタム描画イベントハンドラデリゲート。
    /// </summary>
    public delegate void CustomDrawListViewItemEventHandler(object sender, CustomDrawListViewItemEventArgs e);

    /// <summary>
    /// リストビュー項目カスタム描画イベントデータクラス。
    /// </summary>
    public sealed class CustomDrawListViewItemEventArgs : CustomDrawItemEventArgs
    {
        // リストビュー
        private readonly UIListView _listView;
        // 元イベントデータ
        private readonly DrawListViewSubItemEventArgs _eventArgs;
        // 領域
        private readonly Rectangle _bounds;
        // 選択状態
        private readonly bool _selected;
        // 既定のイメージ描画を無視
        private bool _ignoreDefaultImage = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CustomDrawListViewItemEventArgs(UIListView listView, DrawListViewSubItemEventArgs eventArgs, Rectangle bounds, bool selected)
        {
            // eventArgs.Bounds は列順序入れ替え時に
            // 正しい値にならないので参照しない
            _listView  = listView;
            _eventArgs = eventArgs;
            _bounds    = bounds;
            _selected  = selected;
        }

        /// <summary>
        /// グラフィクス。
        /// </summary>
        public Graphics Graphics
        {
            get { return _eventArgs.Graphics; }
        }

        /// <summary>
        /// 領域。
        /// </summary>
        public Rectangle Bounds
        {
            get { return _bounds; }
        }

        /// <summary>
        /// 項目背景色。
        /// </summary>
        public Color ItemBackColor
        {
            get { return _eventArgs.SubItem.BackColor; }
        }

        /// <summary>
        /// 項目前景色。
        /// </summary>
        public Color ItemForeColor
        {
            get { return _eventArgs.SubItem.ForeColor; }
        }

        /// <summary>
        /// 項目テキスト。
        /// </summary>
        public string ItemText
        {
            get { return _eventArgs.SubItem.Text; }
        }

        /// <summary>
        /// 項目フォント。
        /// </summary>
        public Font ItemFont
        {
            get { return _eventArgs.SubItem.Font; }
        }

        /// <summary>
        /// 既定描画時の背景カラー。
        /// </summary>
        public Color DefaultDrawBackColor
        {
            get
            {
                if (IsSelected)
                {
                    if (_listView.Focused || _listView.HighlightSelection)
                    {
                        return SystemColors.Highlight;
                    }
                    else
                    {
                        return SystemColors.InactiveBorder;
                    }
                }
                else
                {
                    return SystemColors.Window;
                }
            }
        }

        /// <summary>
        /// 既定描画時の前景カラー。
        /// </summary>
        public Color DefaultDrawForeColor
        {
            get
            {
                if (IsSelected && (_listView.Focused || _listView.HighlightSelection))
                {
                    return SystemColors.HighlightText;
                }
                else
                {
                    return SystemColors.WindowText;
                }
            }
        }

        /// <summary>
        /// リストビュー。
        /// </summary>
        public UIListView ListView
        {
            get { return _listView; }
        }

        /// <summary>
        /// 項目。
        /// </summary>
        public ListViewItem Item
        {
            get { return _eventArgs.Item; }
        }

        /// <summary>
        /// 項目番号。
        /// </summary>
        public int ItemIndex
        {
            get { return _eventArgs.ItemIndex; }
        }

        /// <summary>
        /// サブ項目。
        /// </summary>
        public ListViewItem.ListViewSubItem SubItem
        {
            get { return _eventArgs.SubItem; }
        }

        /// <summary>
        /// 列番号。
        /// </summary>
        public int ColumnIndex
        {
            get { return _eventArgs.ColumnIndex; }
        }

        /// <summary>
        /// 列ヘッダ。
        /// </summary>
        public ColumnHeader ColumnHeader
        {
            get { return _eventArgs.Header; }
        }

        /// <summary>
        /// 既定のイメージ描画を無視。
        /// </summary>
        public bool IgnoreDefaultImage
        {
            get { return _ignoreDefaultImage; }
            set { _ignoreDefaultImage = value; }
        }

        /// <summary>
        /// 選択状態か。
        /// </summary>
        public bool IsSelected
        {
            get { return _selected; }
        }

        /// <summary>
        /// フォーカス状態か。
        /// </summary>
        public bool IsFocused
        {
            get { return (_eventArgs.ItemState & ListViewItemStates.Focused) != 0; }
        }

        /// <summary>
        /// チェック状態か。
        /// </summary>
        public bool IsChecked
        {
            get { return (_eventArgs.ItemState & ListViewItemStates.Checked) != 0; }
        }

        /// <summary>
        /// 中間状態か（３状態チェック時）。
        /// </summary>
        public bool IsIndeterminate
        {
            get { return (_eventArgs.ItemState & ListViewItemStates.Indeterminate) != 0; }
        }

        /// <summary>
        /// 既定の描画処理。
        /// </summary>
        public void DrawDefault()
        {
            //-----------------------------------------------------------------
            // 背景
            if (SpecificBackColor != Color.Empty)
            {
                // 一回り小さい領域にする
                Rectangle rect = Bounds;
                rect.Inflate(-1, -1);

                // 選択時は選択色とブレンド
                Color color = SpecificBackColor;
                if (IsSelected)
                {
                    if (_listView.Focused || _listView.HighlightSelection)
                    {
                        color = ColorUtility.Blend(SpecificBackColor, SystemColors.Highlight);
                    }
                    else
                    {
                        color = ColorUtility.Blend(SpecificBackColor, SystemColors.InactiveBorder);
                    }
                }
                using (Brush brush = new SolidBrush(color))
                {
                    Graphics.FillRectangle(brush, rect);
                }
            }

            //-----------------------------------------------------------------
            // 前景色
            Color foreColor = ItemForeColor;
            if (SpecificForeColor != Color.Empty)
            {
                foreColor = SpecificForeColor;
            }
            else if (IsSelected)
            {
                if (_listView.Focused || _listView.HighlightSelection)
                {
                    foreColor = SystemColors.HighlightText;
                }
            }

            //-----------------------------------------------------------------
            // テキスト
            string text = ItemText;
            if (SpecificText != null)
            {
                text = SpecificText;
            }

            //-----------------------------------------------------------------
            // フォント
            Font font = ItemFont;
            if (SpecificFont != null)
            {
                font = SpecificFont;
            }

            //-----------------------------------------------------------------
            // イメージ
            Image image = null;
            if (SpecificImage != null)
            {
                image = SpecificImage;
            }
            else
            {
                // 指定イメージ番号＆イメージキーで検索
                image = GetImage(_listView.SmallImageList, SpecificImageIndex, SpecificImageKey);
                if (image == null)
                {
                    // 項目０の既定イメージ
                    if (_eventArgs.ColumnIndex == 0 && !_ignoreDefaultImage)
                    {
                        ListViewItem item = _eventArgs.Item;
                        image = GetImage(_listView.SmallImageList, item.ImageIndex, item.ImageKey);
                    }
                }
            }

            //-----------------------------------------------------------------
            // ラベル描画
            Rectangle rcLabel = Bounds;
            rcLabel.Inflate(-1, 0);
            GraphicsUtility.DrawText(
                Graphics,
                text,
                font,
                foreColor,
                rcLabel,
                _eventArgs.Header.TextAlign,
                image
            );

            //-----------------------------------------------------------------
            // 枠線描画
            if (FramePen != null)
            {
                Rectangle rcFrame = Bounds;
                rcFrame.Inflate(-1, 0);

                GraphicsUtility.DrawRectangle(Graphics, FramePen, rcFrame);
            }
        }
    }
    #endregion

    #region CompareListViewItemEvent
    /// <summary>
    /// リストビュー項目比較イベントハンドラデリゲート。
    /// </summary>
    public delegate int CompareListViewItemEventHandler(object sender, CompareListViewItemEventArgs e);

    /// <summary>
    /// リストビュー項目比較イベントデータクラス。
    /// </summary>
    public sealed class CompareListViewItemEventArgs : EventArgs
    {
        // 列番号
        private readonly int _columnIndex;
        // 列ヘッダ
        private readonly ColumnHeader _columnHeader;
        // 比較項目１
        private readonly ListViewItem _item1;
        // 比較項目２
        private readonly ListViewItem _item2;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public CompareListViewItemEventArgs(int columnIndex, ColumnHeader columnHeader, ListViewItem item1, ListViewItem item2)
        {
            _columnIndex  = columnIndex;
            _columnHeader = columnHeader;
            _item1        = item1;
            _item2        = item2;
        }

        /// <summary>
        /// 列番号。
        /// </summary>
        public int ColumnIndex
        {
            get { return _columnIndex; }
        }

        /// <summary>
        /// 列ヘッダ。
        /// </summary>
        public ColumnHeader ColumnHeader
        {
            get { return _columnHeader; }
        }

        /// <summary>
        /// 比較項目１。
        /// </summary>
        public ListViewItem Item1
        {
            get { return _item1; }
        }

        /// <summary>
        /// 比較項目２。
        /// </summary>
        public ListViewItem Item2
        {
            get { return _item2; }
        }

        #region 比較
        /// <summary>
        /// 整数値比較。
        /// </summary>
        public short CompareShort(short value1, short value2)
        {
            if (value1 < value2) { return -1; }
            else if (value1 > value2) { return 1; }
            return 0;
        }

        /// <summary>
        /// 整数値比較。
        /// </summary>
        public short CompareShort(short? value1, short? value2)
        {
            if ((value1 == null) && (value2 == null)) { return 0; }
            else if ((value1 != null) && (value2 == null)) { return +1; }
            else if ((value1 == null) && (value2 != null)) { return -1; }
            else
            {
                return CompareShort((short)value1, (short)value2);
            }
        }

        /// <summary>
        /// 整数値比較。
        /// </summary>
        public int CompareInt(int value1, int value2)
        {
            if (value1 < value2) { return -1; }
            else if (value1 > value2) { return 1; }
            return 0;
        }

        /// <summary>
        /// 整数値比較。
        /// </summary>
        public int CompareInt(int? value1, int? value2)
        {
            if ((value1 == null) && (value2 == null)) { return 0; }
            else if ((value1 != null) && (value2 == null)) { return +1; }
            else if ((value1 == null) && (value2 != null)) { return -1; }
            else
            {
                return CompareInt((int)value1, (int)value2);
            }
        }

        /// <summary>
        /// 実数値比較。
        /// </summary>
        public int CompareFloat(float value1, float value2)
        {
            if      (value1 < value2) { return -1; }
            else if (value1 > value2) { return  1; }
            return 0;
        }

        /// <summary>
        /// 実数値比較。
        /// </summary>
        public int CompareFloat(float? value1, float? value2)
        {
                 if ((value1 == null) && (value2 == null)){	return 0;	}
            else if ((value1 != null) && (value2 == null)){	return +1;	}
            else if ((value1 == null) && (value2 != null)){	return -1;	}
            else
            {
                return CompareFloat((float)value1, (float)value2);
            }
        }

        /// <summary>
        /// 真偽値比較。
        /// </summary>
        public int CompareBool(bool value1, bool value2)
        {
            if      (!value1 &&  value2) { return -1; }
            else if ( value1 && !value2) { return  1; }
            return 0;
        }

        /// <summary>
        /// 真偽値比較。
        /// </summary>
        public int CompareBool(bool? value1, bool? value2)
        {
                 if ((value1 == null) && (value2 == null)){	return 0;	}
            else if ((value1 != null) && (value2 == null)){	return +1;	}
            else if ((value1 == null) && (value2 != null)){	return -1;	}
            else
            {
                return CompareBool((bool)value1, (bool)value2);
            }
        }

        /// <summary>
        /// 文字列比較。
        /// </summary>
        public int CompareString(string value1, string value2)
        {
                 if ((value1 == null) && (value2 == null)){	return 0;	}
            else if ((value1 != null) && (value2 == null)){	return +1;	}
            else if ((value1 == null) && (value2 != null)){	return -1;	}
            else
            {
                return string.Compare(value1, value2);
            }
        }

        /// <summary>
        /// カラー比較（ＲＧＢ成分）。
        /// </summary>
        public int CompareColorRGB(Color value1, Color value2)
        {
            // 明度で比較
            int intensity1 = value1.R + value1.G + value1.B;
            int intensity2 = value2.R + value2.G + value2.B;

            if      (intensity1 < intensity2) { return -1; }
            else if (intensity1 > intensity2) { return  1; }

            // 成分で比較
            int component1 = value1.B << 16 | value1.G << 8 | value1.R;
            int component2 = value2.B << 16 | value2.G << 8 | value2.R;

            if      (component1 < component2) { return -1; }
            else if (component1 > component2) { return  1; }
            return 0;
        }

        /// <summary>
        /// カラー比較（ＲＧＢ成分）。
        /// </summary>
        public int CompareColorRGB(Color? value1, Color? value2)
        {
                 if ((value1 == null) && (value2 == null)){	return 0;	}
            else if ((value1 != null) && (value2 == null)){	return +1;	}
            else if ((value1 == null) && (value2 != null)){	return -1;	}
            else
            {
                return CompareColorRGB((Color)value1, (Color)value2);
            }
        }

        /// <summary>
        /// カラー比較（ＲＧＢＡ成分）。
        /// </summary>
        public int CompareColorRGBA(Color value1, Color value2)
        {
            // 明度で比較
            int intensity1 = value1.R + value1.G + value1.B + value1.A;
            int intensity2 = value2.R + value2.G + value2.B + value2.A;

            if      (intensity1 < intensity2) { return -1; }
            else if (intensity1 > intensity2) { return  1; }

            // 成分で比較
            int component1 = value1.A << 24 | value1.B << 16 | value1.G << 8 | value1.R;
            int component2 = value2.A << 24 | value2.B << 16 | value2.G << 8 | value2.R;

            if      (component1 < component2) { return -1; }
            else if (component1 > component2) { return  1; }
            return 0;
        }

        /// <summary>
        /// カラー比較（ＲＧＢＡ成分）。
        /// </summary>
        public int CompareColorRGBA(Color? value1, Color? value2)
        {
                 if ((value1 == null) && (value2 == null)){	return 0;	}
            else if ((value1 != null) && (value2 == null)){	return +1;	}
            else if ((value1 == null) && (value2 != null)){	return -1;	}
            else
            {
                return CompareColorRGBA((Color)value1, (Color)value2);
            }
        }

        /// <summary>
        /// デフォルト比較。
        /// </summary>
        public int CompareDefault()
        {
            // メイン項目
            if (_columnIndex == 0)
            {
                return CompareString(_item1.Text, _item2.Text);
            }
            // サブ項目
            else
            {
                int subItemIndex = _columnIndex;

                // 項目１
                ListViewItem.ListViewSubItem subItem1 = null;
                if (subItemIndex < _item1.SubItems.Count)
                {
                    subItem1 = _item1.SubItems[subItemIndex];
                }
                // 項目２
                ListViewItem.ListViewSubItem subItem2 = null;
                if (subItemIndex < _item2.SubItems.Count)
                {
                    subItem2 = _item2.SubItems[subItemIndex];
                }

                // 条件を分けて比較する
                if      (subItem1 == null && subItem2 == null) { return  0; }
                else if (subItem1 != null && subItem2 == null) { return -1; }
                else if (subItem1 == null && subItem2 != null) { return  1; }
                else
                {
                    return CompareString(subItem1.Text, subItem2.Text);
                }
            }
        }
        #endregion
    }
    #endregion

    //-------------------------------------------------------------------------
    // 列挙型定義
    //-------------------------------------------------------------------------
    #region ExtendedBorderStyle
    /// <summary>
    /// 拡張ボーダースタイル。
    /// </summary>
    public enum ExtendedBorderStyle
    {
        /// <summary>なし。</summary>
        None = 0,
        /// <summary>スタティックエッジ（WS_EX_STATICEDGE）。</summary>
        StaticEdge,
        /// <summary>ダイアログモーダルフレーム（WS_EX_DLGMODALFRAME）。</summary>
        DlgModalFrame,
    }
    #endregion

    #region FloatValueDisplayStyle
    /// <summary>
    /// 実数値表示スタイル。
    /// </summary>
    public enum FloatValueDisplayStyle
    {
        /// <summary>自動小数部桁数。</summary>
        AutoDecimal,
        /// <summary>標準（書式指定無し）。</summary>
        Standard,
        /// <summary>小数部固定１桁（{f1}）。</summary>
        Fixed1,
        /// <summary>小数部固定２桁（{f2}）。</summary>
        Fixed2,
        /// <summary>小数部固定３桁（{f3}）。</summary>
        Fixed3,
        /// <summary>小数部固定４桁（{f4}）。</summary>
        Fixed4,
        /// <summary>小数部固定５桁（{f5}）。</summary>
        Fixed5,
        /// <summary>小数部固定６桁（{f6}）。</summary>
        Fixed6,
    }
    #endregion
}
