﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Data;
using App.Properties;
using App.Utility;
using App.res;
using ConfigCommon;

using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    public sealed partial class TextureDetailsDialog : ModelessDialog
    {
        // インスタンスリスト
        private static readonly List<TextureDetailsDialog> instances_ = new List<TextureDetailsDialog>();

        public static void ShowWindow(Texture target, bool initialShowAlpha)
        {
            ShowWindow(target, initialShowAlpha ? ChannelFlags.A : ChannelFlags.RGB);
        }

        public static void ShowWindow(Texture target, ChannelFlags channel)
        {
            using (WaitCursor wait = new WaitCursor())
            {
                // ウィンドウ作成
                TextureDetailsDialog dialog = new TextureDetailsDialog(target, channel);
                {
                    // 表示位置
                    Size szWindow  = dialog.Size;
                    Size szDesktop = SystemInformation.WorkingArea.Size;
                    if (instances_.Count == 0)
                    {
                        // １個目のウィンドウはデスクトップ中央
                        dialog.DesktopLocation = new Point(
                            Math.Max(0, (szDesktop.Width  - szWindow.Width)  / 2),
                            Math.Max(0, (szDesktop.Height - szWindow.Height) / 2)
                        );
                    }
                    else
                    {
                        // ２個目以降は前のウィンドウの右下
                        TextureDetailsDialog previous = instances_[instances_.Count - 1];
                        Point ptWindow = new Point(
                            previous.DesktopLocation.X + TheApp.WindowCascadingMargin,
                            previous.DesktopLocation.Y + TheApp.WindowCascadingMargin
                        );
                        if (ptWindow.X >= szDesktop.Width || ptWindow.Y >= szDesktop.Height)
                        {
                            ptWindow.X = 0;
                            ptWindow.Y = 0;
                        }
                        dialog.DesktopLocation = ptWindow;
                    }

                    // インスタンス登録して表示
                    instances_.Add(dialog);
                    dialog.Show(TheApp.MainFrame);
                }
            }
        }

        // イメージビュー
        private TextureImageView imageView_;

        // 標準ウィンドウサイズ
        private Size standardSize_ = Size.Empty;

        private Texture target_;
        private readonly ChannelFlags channel_;

        private TextureDetailsDialog(Texture target, ChannelFlags channel)
        {
            InitializeComponent();

            DialogUtility.SetHelp(this, HelpUtility.PageKey.p_texture_property_window_general_page_a_detail);

            // 標準ウィンドウサイズを一旦保持
            standardSize_ = Size;

            target_ = target;
            channel_ = channel;

            UpdateImage();

            //-----------------------------------------------------------------
            // その他の初期化
            //-----------------------------------------------------------------
            // 外部イベント登録
            DocumentAddOrRemovedEventHandler addedOrRemoved = (s, a, r, swaped, reloaded) =>
                {
                    GuiObject texture = null;
                    if (swaped.TryGetValue(target_, out texture))
                    {
                        Debug.Assert(texture is Texture);
                        target_ = (Texture)texture;
                    }
                    else if (r.Any(x => x == target_))
                    {
                        Close();
                    }
                };

            DocumentPropertyChangedEventHandler propertyChanged = (s, e) =>
                {
                foreach (GuiObject guiObject in e.OfType<FileView.FileTreeView.TextureImageChangedArgs>().Select(x => x.Target))
                {
                    if (guiObject.ObjectID == GuiObjectID.Texture)
                    {
                        if (guiObject.Name == target.Name)
                        {
                            target_ = (Texture)guiObject;
                            UpdateImage();
                        }
                    }
                }
                };

            App.AppContext.DocumentAddedOrRemoved += addedOrRemoved;
            App.AppContext.PropertyChanged += propertyChanged;
            Disposed += (s, e) =>
                {
                    App.AppContext.DocumentAddedOrRemoved -= addedOrRemoved;
                    App.AppContext.PropertyChanged -= propertyChanged;
                };

            // 表示深さコンボボックス
            {
                cbxDepth.Items.Clear();

                if ((target_ == null) || (target_.IsArray == false))
                {
                    cbxDepth.Enabled = false;
                }
                else
                {
                    cbxDepth.Items.Add(Strings.TextureImageViewPanel_AllView);

                    var maxDepth = target_.IsCube ? target_.DepthCount / 6 : target_.DepthCount;
                    for(int i = 0;i != maxDepth;++ i)
                    {
                        cbxDepth.Items.Add(i.ToString());
                    }

                    cbxDepth.SelectedIndex = 0;

                    cbxDepth.Enabled = true;
                }
            }
        }

        private void UpdateImage()
        {
            // ウィンドウタイトル
            Text = string.Format(Text, target_.Name);

            if (imageView_ != null)
            {
                imageView_.Dispose();
            }

            imageView_ =
                new TextureImageView(
                    this,
                    target_,
                    target_.HasAlpha ?
                        channel_ :
                        ChannelFlags.RGB,
                    (cbxDepth == null) ?
                        -1 :
                        Math.Max(0, cbxDepth.SelectedIndex) - 1
                )
                {
                    Dock = DockStyle.Fill,
                    Parent = pnlImageView,
                };

            imageView_.MouseClick     += imageView_MouseClick;
            imageView_.MagnifyChanged += imageView_MagnifyChanged;
            imageView_.ChannelChanged += imageView_ChannelChanged;
            imageView_.PixelHint      += imageView_PixelHint;

            {
                // 描画イメージサイズ
                Size szRendering = imageView_.RenderingSize;

                // ツールバー、ステータスバー以外のクライアントサイズ
                Size szClient = ClientSize;
                szClient.Height -= (tspMain.Height + sspMain.Height);

                // 初期ウィンドウサイズ計算
                Size szPadding = new Size(48, 48);
                Size szWindow = Size;
                Size szNCDelta = new Size(szWindow.Width - szClient.Width, szWindow.Height - szClient.Height);
                Size szInitial = new Size(
                    szRendering.Width + szPadding.Width + szNCDelta.Width,
                    szRendering.Height + szPadding.Height + szNCDelta.Height
                );

                // 初期状態では標準サイズ以下にはしない
                if (szInitial.Width < standardSize_.Width) { szInitial.Width = standardSize_.Width; }
                if (szInitial.Height < standardSize_.Height) { szInitial.Height = standardSize_.Height; }
                Size = szInitial;
            }

            //-----------------------------------------------------------------
            // ツールバー初期設定
            //-----------------------------------------------------------------
            // 表示倍率選択項目初期設定
            foreach (UIToolStripMenuItem item in tsiMagnifyOptions.DropDownItems)
            {
                item.Click += tsiMagnifyOption_Click;
            }

            // 表示倍率状態
            UpdateMagnifyState();

            // チャンネル状態
            tsiChannelRGB.Enabled = true;
            tsiChannelR.Enabled = true;
            tsiChannelG.Enabled = true;
            tsiChannelB.Enabled = true;
            tsiChannelA.Enabled = target_.HasAlpha;

            UpdateChannelState();

            // ミップマップ表示切替
            if (target_.MipLevel == 1)
            {
                tsiShowMipmap.Enabled = false;
                tsiMipmapOrientation.Enabled = false;
            }
            else
            {
                tsiShowMipmap.Checked = true;
                tsiMipmapOrientation.Enabled = true;
            }
            imageView_.ShowMipmap = tsiShowMipmap.Checked;

            imageView_.MipmapOrientation = ApplicationConfig.Setting.TextureDetailsDialog.MipmapOrientation;
            UpdateMipmapOrientation();

            //-----------------------------------------------------------------
            // ステータスバー初期設定
            //-----------------------------------------------------------------
            {
                tsiInfo0.Text = UIText.EnumValue(target_.Data.texture_info.quantize_type); ;
                tsiInfo1.Text = string.Format("{0} x {1}", target_.Data.texture_info.width, target_.Data.texture_info.height);
                tsiInfo2.Text = UIText.MipLevel(target_.Data.texture_info.mip_level);
                tsiInfo3.Text = DataSize.XBytesText(target_.Data.texture_info.size);
                tsiInfo4.Text = target_.ConverterName;
            }

            // ピクセルヒント
            lblPixelHint0.Text = string.Empty;
            lblPixelHint1.Text = string.Empty;
        }

        #region オーバーライド
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // 倍率テキストボックス確定
            if ((keyData == Keys.Return || keyData == Keys.Enter) && tsiMagnify.Focused)
            {
                OnEnterMagnifyInput();
                return true;
            }
            // 拡大
            else if (keyData == (Keys.Control | Keys.Add))
            {
                imageView_.ZoomIn();
                return true;
            }
            // 縮小
            else if (keyData == (Keys.Control | Keys.Subtract))
            {
                imageView_.ZoomOut();
                return true;
            }
            // ＲＧＢチャンネル
            else if (keyData == (Keys.Control | Keys.D1))
            {
                if (tsiChannelRGB.Enabled)
                {
                    imageView_.Channel = ChannelFlags.RGB;
                }
                return true;
            }
            // Ｒチャンネル
            else if (keyData == (Keys.Control | Keys.D2))
            {
                if (tsiChannelR.Enabled)
                {
                    imageView_.Channel = ChannelFlags.R;
                }
                return true;
            }
            // Ｇチャンネル
            else if (keyData == (Keys.Control | Keys.D3))
            {
                if (tsiChannelG.Enabled)
                {
                    imageView_.Channel = ChannelFlags.G;
                }
                return true;
            }
            // Ｂチャンネル
            else if (keyData == (Keys.Control | Keys.D4))
            {
                if (tsiChannelB.Enabled)
                {
                    imageView_.Channel = ChannelFlags.B;
                }
                return true;
            }
            // Ａチャンネル
            else if (keyData == (Keys.Control | Keys.D5))
            {
                if (tsiChannelA.Enabled)
                {
                    imageView_.Channel = ChannelFlags.A;
                }
                return true;
            }
            // ミップマップ表示
            else if (keyData == (Keys.Control | Keys.M))
            {
                if (tsiShowMipmap.Enabled)
                {
                    imageView_.ShowMipmap = !imageView_.ShowMipmap;
                    tsiShowMipmap.Checked = imageView_.ShowMipmap;
                }
                return true;
            }
            // ミップマップ方向
            else if (keyData == (Keys.Control | Keys.D))
            {
                if (tsiMipmapOrientation.Enabled)
                {
                    imageView_.MipmapOrientation = (imageView_.MipmapOrientation == ConfigCommon.MipmapOrientationType.Horizontal) ?
                                                        ConfigCommon.MipmapOrientationType.Vertical :
                                                        ConfigCommon.MipmapOrientationType.Horizontal;
                    UpdateMipmapOrientation();

                    // 即座に保存する
                    ApplicationConfig.Setting.TextureDetailsDialog.MipmapOrientation = imageView_.MipmapOrientation;
                }
                return true;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            // インスタンスリストから削除
            instances_.Remove(this);

            base.OnFormClosed(e);
        }
        #endregion

        //---------------------------------------------------------------------
        // イメージビュー
        private void imageView_MouseClick(object sender, MouseEventArgs e)
        {
            // 右クリックで閉じる
            if (e.Button == MouseButtons.Right)
            {
                Close();
            }
        }

        //---------------------------------------------------------------------
        // 表示倍率
        private void tsiMagnify_Leave(object sender, EventArgs e)
        {
            OnEnterMagnifyInput();
        }

        private void tsiMagnifyOption_Click(object sender, EventArgs e)
        {
            UIToolStripMenuItem item = (UIToolStripMenuItem)sender;
            imageView_.ChangeMagnify(float.Parse((string)item.Tag));
        }

        private void tsiMagnifyPlus_Click(object sender, EventArgs e)
        {
            imageView_.ZoomIn();
        }

        private void tsiMagnifyMinus_Click(object sender, EventArgs e)
        {
            imageView_.ZoomOut();
        }

        //---------------------------------------------------------------------
        // チャンネル
        private void tsiChannelRGB_Click(object sender, EventArgs e)
        {
            imageView_.ChangeChannel(ChannelFlags.RGB);
        }

        private void tsiChannelR_Click(object sender, EventArgs e)
        {
            imageView_.ChangeChannel(ChannelFlags.R);
        }

        private void tsiChannelG_Click(object sender, EventArgs e)
        {
            imageView_.ChangeChannel(ChannelFlags.G);
        }

        private void tsiChannelB_Click(object sender, EventArgs e)
        {
            imageView_.ChangeChannel(ChannelFlags.B);
        }

        private void tsiChannelA_Click(object sender, EventArgs e)
        {
            imageView_.ChangeChannel(ChannelFlags.A);
        }

        //---------------------------------------------------------------------
        // 表示深さ
        private void cbxDepth_SelectedIndexChanged(object sender, EventArgs e)
        {
            imageView_.DepthIndex = Math.Max(0, cbxDepth.SelectedIndex) - 1;
        }

        //---------------------------------------------------------------------
        // ミップマップ表示
        private void tsiShowMipmap_CheckedChanged(object sender, EventArgs e)
        {
            imageView_.ShowMipmap = tsiShowMipmap.Checked;
        }

        private void tsiMipmapOrientation_Click(object sender, EventArgs e)
        {
            imageView_.MipmapOrientation = (imageView_.MipmapOrientation == ConfigCommon.MipmapOrientationType.Horizontal) ?
                                                ConfigCommon.MipmapOrientationType.Vertical :
                                                ConfigCommon.MipmapOrientationType.Horizontal;

            // 即座に保存する
            ApplicationConfig.Setting.TextureDetailsDialog.MipmapOrientation = imageView_.MipmapOrientation;

            UpdateMipmapOrientation();
        }

        private void imageView_MagnifyChanged(object sender, EventArgs e)
        {
            UpdateMagnifyState();
        }

        private void imageView_ChannelChanged(object sender, EventArgs e)
        {
            UpdateChannelState();
        }

        private void imageView_PixelHint(object sender, TPBitmapImagePixelHintEventArgs e)
        {
            // ヒントに応じて表示内容を変える
            switch (e.PixelHint)
            {
                case TPBitmapImagePixelHint.PixelColor:
                {
                    SetPixelLocationHint(e.PixelLocation);
                    SetPixelColorHint(e.Color);
                    break;
                }

                default:
                {
                    lblPixelHint0.Text = string.Empty;
                    lblPixelHint1.Text = string.Empty;
                    break;
                }
            }
        }

        /// <summary>
        /// 倍率入力の確定処理。
        /// </summary>
        private void OnEnterMagnifyInput()
        {
            // 文字列から'%'を省く
            string text = tsiMagnify.Text;
            text = text.TrimEnd('%');

            // 倍率に変換
            float magnify = 1.0f;
            try
            {
                magnify = float.Parse(text) / 100.0f;
            }
            catch (Exception)
            {
                magnify = imageView_.Magnify;
            }

            // 倍率変更
            // 変更されない場合もあるので常に UpdateMagnifyState() を呼んでおく
            imageView_.ChangeMagnify(magnify);
            UpdateMagnifyState();

            // フォーカスをイメージビューに移す
            if (tsiMagnify.Focused)
            {
                // Leave イベントを発生させない
                using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
                {
                    imageView_.Focus();
                }
            }
        }

        private void SetPixelLocationHint(Point point)
        {
            lblPixelHint0.Text = string.Format("X: {0,4}  Y: {1,4}", point.X, point.Y);
        }

        private static byte SrgbByteToLinearByte(byte b)
        {
            if (b == 0)
            {
                return 0;
            }
            var v = b / 255.0;
            var r = v <= 0.04045 ? v / 12.92 : Math.Pow((v + 0.055) / 1.055, 2.4);
            r = Math.Min(Math.Max(0.0, r), 1.0);
            return (byte)(255 * r);
        }


        private void SetPixelColorHint(Color color)
        {
            var s = "";
            var linear = target_.Data.texture_info.linear;
            var comp = target_.Data.texture_info.comp_sel;

            s = string.Format("R: {0,3}  G: {1,3}  B: {2,3}  A: {3,3}", color.R, color.G, color.B, color.A);
            if (linear.Any(x => x))
            {
                // コンポーネントセレクタに応じてリニアフラグを判定
                var isLinear = new bool[4];
                for (var i = 0; i < 4; i++)
                {
                    switch (comp[i])
                    {
                        case texture_info_comp_selValue.r:
                            isLinear[i] = linear[0];
                            break;
                        case texture_info_comp_selValue.g:
                            isLinear[i] = linear[1];
                            break;
                        case texture_info_comp_selValue.b:
                            isLinear[i] = linear[2];
                            break;
                        case texture_info_comp_selValue.a:
                            isLinear[i] = linear[3];
                            break;
                        default:
                            isLinear[i] = false;
                            break;
                    }

                }
                s += string.Format(
                    " (Linear {0,3},{1,3},{2,3},{3,3})",
                    isLinear[0] ? SrgbByteToLinearByte(color.R).ToString() : "---",
                    isLinear[1] ? SrgbByteToLinearByte(color.G).ToString() : "---",
                    isLinear[2] ? SrgbByteToLinearByte(color.B).ToString() : "---",
                    isLinear[3] ? SrgbByteToLinearByte(color.A).ToString() : "---");
            }
            lblPixelHint1.Text = s;
        }

        // チャンネル状態更新。
        private void UpdateChannelState()
        {
            tsiChannelRGB.Checked = imageView_.Contains(ChannelFlags.RGB);
            tsiChannelR.Checked   = imageView_.Contains(ChannelFlags.R);
            tsiChannelG.Checked   = imageView_.Contains(ChannelFlags.G);
            tsiChannelB.Checked   = imageView_.Contains(ChannelFlags.B);
            tsiChannelA.Checked   = imageView_.Contains(ChannelFlags.A);
        }

        // 表示倍率状態更新。
        private void UpdateMagnifyState()
        {
            using (UIControlEventSuppressBlock block = new UIControlEventSuppressBlock())
            {
                tsiMagnify.Text = string.Format("{0}%", imageView_.Magnify * 100);
            }

            // 倍率選択メニュー
            foreach (UIToolStripMenuItem item in tsiMagnifyOptions.DropDownItems)
            {
                float magnify = float.Parse((string)item.Tag);
                item.Checked = (magnify == imageView_.Magnify);
            }
        }

        private void UpdateMipmapOrientation()
        {
            tsiMipmapOrientation.Image = (imageView_.MipmapOrientation == ConfigCommon.MipmapOrientationType.Horizontal) ?
                Resources.TextureDetailsDialog_MipmapOrientationH :
                Resources.TextureDetailsDialog_MipmapOrientationV;
        }

        private void tsiShowHelp_Click(object sender, EventArgs e)
        {
            DialogUtility.EmulatePushF1();
        }
    }

    /// <summary>
    /// チャンネルフラグ。
    /// </summary>
    [Flags]
    public enum ChannelFlags
    {
        R = 0x01,
        G = 0x02,
        B = 0x04,
        RGB = 0x07,
        A = 0x08
    }
}
