﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace App.Controls
{
    public partial class PresetLightAnimDialog : OkCancelDialog
    {
        /// <summary>
        /// Tag に設定するデータ
        /// </summary>
        public class TagData
        {
            public TeamConfig.Preset.LightAnimPreset LightAnimPreset;
            public HashSet<string> AssignedLightAnimPresetLabels;
            public HashSet<string> AssignedLightAnimPresetTypes;

            public TagData(TeamConfig.Preset.LightAnimPreset lightAnimPreset, IReadOnlyCollection<TeamConfig.Preset.LightAnimPreset> assignedLightAnimPresets)
            {
                LightAnimPreset = lightAnimPreset;
                AssignedLightAnimPresetLabels = new HashSet<string>(assignedLightAnimPresets.Select(x => x.Label));
                AssignedLightAnimPresetTypes = new HashSet<string>(assignedLightAnimPresets.Select(x => x.Type));
            }
        }

        const int EM_SETCUEBANNER = 0x1500 + 1;

        private List<Func<bool>> acceptable = new List<Func<bool>>();

        private TeamConfig.Preset.LightAnimPreset.LightAnimTargetType[] LightAnimTargetItems
        {
            get
            {
                return lvwLightAnimTarget.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.LightAnimPreset.LightAnimTargetType)x.Tag).ToArray();
            }
        }

        public PresetLightAnimDialog()
        {
            InitializeComponent();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            // 入力必須項目に設定。
            if (tbxLightAnimType.IsHandleCreated)
            {
                var ptr = System.Runtime.InteropServices.Marshal.StringToBSTR(App.res.Strings.MandatoryField);
                Win32.NativeMethods.SendMessage(tbxLightAnimType.Handle, EM_SETCUEBANNER, IntPtr.Zero, ptr);
                acceptable.Add(() => { return !string.IsNullOrEmpty(tbxLightAnimType.Text); });
            }

            var data = (TagData)Tag;
            tbxLightAnimType.Text = data.LightAnimPreset.Type;
            tbxLightAnimLabel.Text = data.LightAnimPreset.Label;

            lvwLightAnimTarget.SetItemCount(data.LightAnimPreset.LightAnimTargets.Count);
            for (int i = 0; i < data.LightAnimPreset.LightAnimTargets.Count; i++)
            {
                // PresetLightAnimTargetDialog 側での直接編集を防ぐために、ここで複製したインスタンスを渡す。
                // PresetLightAnimDialog の側での確定時に PresetLightAnimTargetDialog の変更を取り込む。
                var target = (TeamConfig.Preset.LightAnimPreset.LightAnimTargetType)Utility.ObjectUtility.Clone(data.LightAnimPreset.LightAnimTargets[i]);
                var item = lvwLightAnimTarget.Items[i];
                item.Tag = target;
                item.SubItems[0].Text = target.Target.ToString();
                item.SubItems[1].Text = target.HasDefault() ? target.DefaultValue : string.Empty;
            }
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            base.OnClosing(e);

            // e.Cancel 操作は base.OnClosing() 後に行う必要がある。
            if (DialogResult == DialogResult.OK)
            {
                if (!acceptable.All(x => x()))
                {
                    e.Cancel = true;
                }

                var data = (TagData)Tag;

                // 同名のラベルは許可しない。
                if (data.AssignedLightAnimPresetLabels.Contains(tbxLightAnimLabel.Text))
                {
                    UIMessageBox.Error(res.Strings.Preset_Load_Duplicated_Type_Or_Label, tbxLightAnimLabel.Text);
                    e.Cancel = true;
                    return;
                }

                // 同名のタイプは許可しない。
                if (data.AssignedLightAnimPresetTypes.Contains(tbxLightAnimType.Text))
                {
                    UIMessageBox.Error(res.Strings.Preset_Load_Duplicated_Type_Or_Label, tbxLightAnimType.Text);
                    e.Cancel = true;
                    return;
                }
            }
        }

        protected override void OnClosed(EventArgs e)
        {
            base.OnClosed(e);

            // 変更を適用。
            if (DialogResult == DialogResult.OK)
            {
                // null 文字から空文字への変更を回避。
                Func<string, string, string> CorrectString = (oldValue, newValue) =>
                {
                    return ((oldValue == null) && string.IsNullOrEmpty(newValue)) ? null : newValue;
                };

                // null リストから空リストへの変換を回避。
                Func<List<TeamConfig.Preset.LightAnimPreset.LightAnimTargetType>, List<TeamConfig.Preset.LightAnimPreset.LightAnimTargetType>, List<TeamConfig.Preset.LightAnimPreset.LightAnimTargetType>> CorrectLightAnimTargets = (oldValue, newValue) =>
                {
                    return ((oldValue == null) && ((newValue == null) || !newValue.Any())) ? null : newValue;
                };

                var data = (TagData)Tag;
                data.LightAnimPreset.Type = CorrectString(data.LightAnimPreset.Type, tbxLightAnimType.Text);
                data.LightAnimPreset.Label = CorrectString(data.LightAnimPreset.Label, tbxLightAnimLabel.Text);
                data.LightAnimPreset.LightAnimTargets = CorrectLightAnimTargets(data.LightAnimPreset.LightAnimTargets, lvwLightAnimTarget.Items.Cast<ListViewItem>().Select(x => (TeamConfig.Preset.LightAnimPreset.LightAnimTargetType)x.Tag).ToList());
            }
        }

        private void lvweditLightAnimTarget_EditButtonClicked(object sender, EventArgs e)
        {
            if (lvwLightAnimTarget.SelectedItems.Count > 0)
            {
                var item = lvwLightAnimTarget.SelectedItems[0];

                using (var dialog = new PresetLightAnimTargetDialog()
                {
                    Tag = new PresetLightAnimTargetDialog.TagData((TeamConfig.Preset.LightAnimPreset.LightAnimTargetType)item.Tag, LightAnimTargetItems)
                })
                {
                    if (dialog.ShowDialog(this) == DialogResult.OK)
                    {
                        var data = (PresetLightAnimTargetDialog.TagData)dialog.Tag;
                        item.SubItems[0].Text = data.LightAnimTargetType.Target.ToString();
                        item.SubItems[1].Text = data.LightAnimTargetType.HasDefault() ? data.LightAnimTargetType.DefaultValue : string.Empty;
                    }
                }
            }
        }

        private void lvweditLightAnimTarget_AddButtonClicked(object sender, EventArgs e)
        {
            using (var dialog = new PresetLightAnimTargetDialog()
            {
                Tag = new PresetLightAnimTargetDialog.TagData(new TeamConfig.Preset.LightAnimPreset.LightAnimTargetType(), LightAnimTargetItems)
            })
            {
                if (dialog.ShowDialog(this) == DialogResult.OK)
                {
                    var n = lvwLightAnimTarget.Items.Count;
                    lvwLightAnimTarget.SetItemCount(n + 1);
                    var item = lvwLightAnimTarget.Items[n];

                    var data = (PresetLightAnimTargetDialog.TagData)dialog.Tag;
                    item.SubItems[0].Text = data.LightAnimTargetType.Target.ToString();
                    item.SubItems[1].Text = data.LightAnimTargetType.HasDefault() ? data.LightAnimTargetType.DefaultValue : string.Empty;

                    item.Tag = data.LightAnimTargetType;
                }
            }
        }

        private void lvweditLightAnimTarget_RemoveButtonClicked(object sender, EventArgs e)
        {
            if (lvwLightAnimTarget.SelectedItems.Count == 1)
            {
                var item = lvwLightAnimTarget.SelectedItems[0];
                lvwLightAnimTarget.Items.Remove(item);
            }
        }
    }
}
