﻿using System;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;
using App.Controls;

namespace ColorPicker
{
    /// <summary>
    /// カラーパレットコントロールクラス。
    /// </summary>
//	public class PalettePanel : System.Windows.Forms.Control
    public class PalettePanel : UIUserControl
    {
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        private int		m_colorVNum	= 8;
        private int		m_colorHNum	= 2;
        private RGB[]	m_palette	= new RGB[16];
        private int		m_select	= 0;
        public bool IsLinear
        {
            private get;
            set;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PalettePanel()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            m_palette[0]	= new RGB(Color.FromArgb( 255, 255, 255 ));
            m_palette[1]	= new RGB(Color.FromArgb( 255,   0,   0 ));
            m_palette[2]	= new RGB(Color.FromArgb( 255, 255,   0 ));
            m_palette[3]	= new RGB(Color.FromArgb(   0, 255,   0 ));
            m_palette[4]	= new RGB(Color.FromArgb(   0, 255, 255 ));
            m_palette[5]	= new RGB(Color.FromArgb(   0,   0, 255 ));
            m_palette[6]	= new RGB(Color.FromArgb( 255,   0, 255 ));
            m_palette[7]	= new RGB(Color.FromArgb(   0,   0,   0 ));
            m_palette[8]	= new RGB(Color.FromArgb( 128, 128, 128 ));
            m_palette[9]	= new RGB(Color.FromArgb( 128,   0,   0 ));
            m_palette[10]	= new RGB(Color.FromArgb( 128, 128,   0 ));
            m_palette[11]	= new RGB(Color.FromArgb(   0, 128,   0 ));
            m_palette[12]	= new RGB(Color.FromArgb(   0, 128, 128 ));
            m_palette[13]	= new RGB(Color.FromArgb(   0,   0, 128 ));
            m_palette[14]	= new RGB(Color.FromArgb( 128,   0, 128 ));
            m_palette[15]	= new RGB(Color.FromArgb(   0,   0,   0 ));

            //	ちらつきの防止用
            SetStyle( ControlStyles.UserPaint | ControlStyles.AllPaintingInWmPaint | ControlStyles.DoubleBuffer, true );
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            components = new System.ComponentModel.Container();
        }
        #endregion

        #region イベント

        internal delegate void ColorSelectedHandler( object sender, ColorChangedEventArgs e );
        internal event ColorSelectedHandler ColorSelected;

        #endregion

        #region プロパティ

        /// <summary>
        /// 横の色数
        /// </summary>
        public int	ColorVNumber
        {
            get	{	return m_colorVNum;		}
            set	{	m_colorVNum	= value;	}
        }

        /// <summary>
        /// 縦の色数
        /// </summary>
        public int	ColorHNumber
        {
            get	{	return m_colorHNum;		}
            set	{	m_colorHNum	= value;	}
        }

        /// <summary>
        /// 選択番号
        /// </summary>
        public int	 Selection
        {
            get	{	return m_select;		}
        }

        #endregion

        #region オーバーライド

        /// <summary>
        ///	OnMouseDownのオーバーライド（左と右ボタンで選択処理）
        /// </summary>
        /// <param name="e"></param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown (e);

            //	中ボタンは使わない
            if( e.Button != MouseButtons.Left && e.Button != MouseButtons.Right )
                return;

            int		w = (Size.Width-4)/m_colorHNum, h = (Size.Height-8)/m_colorVNum;
            if( w < 5 )
                w	= 5;
            if( h < 5 )
                h	= 5;
            int		x	= ( e.X - 1 ) / w;
            int		y	= ( e.Y - 1 ) / h;
            if( x >= 0 && y >= 0 && x < m_colorHNum && y < m_colorVNum )
            {
                m_select	= x + y * m_colorHNum;

                if( e.Button == MouseButtons.Left && ColorSelected != null )
                    ColorSelected( this, new ColorChangedEventArgs( m_palette[m_select], true, IsLinear ) );
                Invalidate();
            }
        }

        /// <summary>
        /// OnPaintのオーバーライド
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPaint(PaintEventArgs e)
        {
            if( DesignMode )
                return;

            if( Enabled )
            {
                Graphics g = e.Graphics;
                int		total	= 0, j, x, y, w = (Size.Width-4)/m_colorHNum, h = (Size.Height-8)/m_colorVNum;
                if( w < 5 )
                    w	= 5;
                if( h < 5 )
                    h	= 5;
                for( int i = 0; i < m_colorVNum; i ++ )
                {
                    for( j = 0; j < m_colorHNum; j ++, total ++ )
                    {
                        x	= j * w;
                        y	= i * h;
                        Color color;
                        if (IsLinear)
                        {
                            color = m_palette[total].ToSrgbFromLinear().GetColor();
                        }
                        else
                        {
                            color = m_palette[total].GetColor();
                        }

                        using( SolidBrush brush = new SolidBrush(color))
                        {
                            g.FillRectangle( brush, x + 4, y + 4, w-4, h-4 );
                        }
                        Point[]	pp	= new Point[3];
                        pp[0]	= new Point( x + 3, y + 3 + h-2 );
                        pp[1]	= new Point( x + 3, y + 3 );
                        pp[2]	= new Point( x + 3 + w-2, y + 3 );
                        g.DrawLines( Pens.Gray, pp );
                        pp[1]	= new Point( x + 3 + w-2, y + 3 + h-2 );
                        g.DrawLines( Pens.White, pp );
                        if( total == m_select )
                            g.DrawRectangle( Pens.Black, x + 2, y + 2, w-1, h-1 );
                    }
                }
            }
            else
                base.OnPaint(e);
        }

        /// <summary>
        /// デザインモード時は親を再描画
        /// </summary>
        /// <param name="e"></param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (DesignMode)
            {
                if (Parent != null)
                {
                    Parent.Invalidate();
                }
            }
            base.OnSizeChanged(e);
        }

        #endregion

        internal RGB[]	GetPalette()
        {
            return m_palette;
        }

        internal void		SetPalette( RGB[] color )
        {
            m_palette	= color;
        }

        /// <summary>
        /// パレットカラーの更新
        /// </summary>
        /// <param name="index">更新先インデックス</param>
        /// <param name="color">変更カラー</param>
        public void		UpdatePalette( int index, RGB color )
        {
            if( !DesignMode )
                Debug.Assert( index < m_palette.Length );
            m_palette[index]	= new RGB(1f, color);
            Invalidate();
        }
    }
}
