﻿using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace ColorPicker
{
    /// <summary>
    /// カラーピッカーコントロールクラス。
    /// </summary>
    /// <remarks>
    /// カラー選択を行う為の汎用的なコンポーネントクラスです。
    /// System.Color 型を扱い、ＲＧＢ、ＲＧＢＡ、ＨＳＶ、ＨＳＶＡの各成分でカラー選択が行えます。
    /// フォームデザイナで特定のフォームに貼り付けて使用して下さい。
    /// <para>
    /// フォームデザイナで使用するには、プロジェクトの参照設定の「参照の追加...」と、
    /// ツールボックスの「アイテムの追加と削除...」で dll ファイルを追加して下さい。
    /// </para>
    /// </remarks>
    /// <example>
    /// <code>
    /// using ColorPicker;
    ///
    /// public class Form1() : System.Windows.Forms.Form
    /// {
    ///   // 〜省略〜
    ///   private ColorPickerControl _colorPicker;
    ///
    ///   public Form1()
    ///   {
    ///     InitializeComponent();
    ///
    ///     // カラー成分と表示方式を設定します。
    ///     // フォームの表示後に動的に切り替えても構いません。
    ///     _colorPicker.ColorMode = ColorMode.RGB;
    ///     _colorPicker.ValueMode = ValueMode.Float;
    ///
    ///     // カラー変更イベントハンドラを登録します。
    ///     _colorPicker.ColorChanged += new ColorChangedEventHandler(colorPicker_ColorChanged)
    ///   }
    ///
    ///   private void colorPicker_ColorChanged(object sender, ColorChangedEventArgs e)
    ///   {
    ///     // カラー変更時のイベントを処理します。
    ///     System.Color color = e.Color;
    ///
    ///     // 編集中（スライダをドラッグしている最中）か編集確定かを判断して処理することができます。
    ///     if (e.EditFixed)
    ///     {
    ///       // 編集確定時の処理
    ///     }
    ///     else
    ///     {
    ///       // 編集中の処理
    ///     }
    ///   }
    /// }
    /// </code>
    /// </example>
    [ToolboxBitmap(typeof(UserControl))]
    public class ColorPickerControl : System.Windows.Forms.UserControl
    {
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private readonly System.ComponentModel.Container components = null;

        private RGB		m_SrgbColor	= new RGB(0,0,0);
        private RGB m_LinearColor = new RGB(0, 0, 0);
        /// <summary>SV選択バー</summary>
        protected ColorPicker.SaturationValueBar	m_gradCtrl;
        /// <summary>H選択バー</summary>
        protected ColorPicker.HueBar	m_hBar;
        /// <summary>パレットパネル</summary>
        protected ColorPicker.PalettePanel m_pltPanel;
        /// <summary>カラー選択パネル</summary>
        protected ColorPicker.ComponentPanel m_componentPanel;
        /// <summary>パレットにカラーを転送するボタン</summary>
        protected System.Windows.Forms.Button m_colSetButton;
//		private System.Windows.Forms.Button m_colPickButton;
        /// <summary>現在カラー表示パネル</summary>
        protected ColorPicker.ColorBox m_colorPanel;

        //	装飾用
        private Rectangle m_GradBoxWakuOut;
        private Rectangle m_GradBoxWakuIn;
        private Rectangle m_hBarWakuOut;
        private Rectangle m_hBarWakuIn;
        public bool IsDefaultLinear
        {
            get
            {
                return m_componentPanel.IsDefaultLinear;
            }
            set
            {
                m_pltPanel.IsLinear = value;
                m_componentPanel.IsDefaultLinear = value;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorPickerControl()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();
//			hookNotify = RegisterWindowMessage("MOUSEHOOK_MSG");

            //	イベント、プロパティ設定
            m_gradCtrl.ColorSelected	+= new SaturationValueBar.ColorSelectedHandler( GradCtrl_ColorSelected );
            m_hBar.ColorSelected		+= new ColorSlider.ColorSelectedHandler( HueBar_ColorSelected );
            m_pltPanel.ColorSelected	+= new PalettePanel.ColorSelectedHandler( PalettePanel_ColorSelected );
            m_componentPanel.ColorChanged	+= new ColorChangedEventHandler( ComponentPanel_ColorSelected );

            //	スポイトボタンを無効化中はボタンをずらす
            m_colSetButton.Location	= new Point( m_colSetButton.Location.X, m_colSetButton.Location.Y + m_colSetButton.Size.Height / 2 );

            DoubleBuffered = true;
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if( components != null )
                    components.Dispose();
            }
            base.Dispose( disposing );
        }

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード］エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            m_colSetButton = new System.Windows.Forms.Button();
            m_componentPanel = new ColorPicker.ComponentPanel();
            m_pltPanel = new ColorPicker.PalettePanel();
            m_colorPanel = new ColorPicker.ColorBox();
            m_hBar = new ColorPicker.HueBar();
            m_gradCtrl = new ColorPicker.SaturationValueBar();
            SuspendLayout();
            //
            // m_colSetButton
            //
            m_colSetButton.FlatStyle = System.Windows.Forms.FlatStyle.System;
            m_colSetButton.Location = new System.Drawing.Point(112, 0);
            m_colSetButton.Name = "m_colSetButton";
            m_colSetButton.Size = new System.Drawing.Size(24, 23);
            m_colSetButton.TabIndex = 1;
            m_colSetButton.Text = ">";
            m_colSetButton.MouseDown += new System.Windows.Forms.MouseEventHandler(colSetButton_MouseDown);
            //
            // m_componentPanel
            //
            m_componentPanel.Location = new System.Drawing.Point(0, 336);
            m_componentPanel.Name = "m_componentPanel";
            m_componentPanel.Size = new System.Drawing.Size(336, 120);
            m_componentPanel.TabIndex = 6;
            //
            // m_pltPanel
            //
            m_pltPanel.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
            m_pltPanel.ColorHNumber = 8;
            m_pltPanel.ColorVNumber = 2;
            m_pltPanel.Location = new System.Drawing.Point(160, 0);
            m_pltPanel.Name = "m_pltPanel";
            m_pltPanel.Size = new System.Drawing.Size(168, 46);
            m_pltPanel.TabIndex = 3;
            m_pltPanel.TabStop = false;
            //
            // m_colorPanel
            //
            m_colorPanel.BorderStyle = System.Windows.Forms.BorderStyle.Fixed3D;
            m_colorPanel.Location = new System.Drawing.Point(18, 2);
            m_colorPanel.Name = "m_colorPanel";
            m_colorPanel.Size = new System.Drawing.Size(68, 44);
            m_colorPanel.TabIndex = 0;
            //
            // m_hBar
            //
            m_hBar.ColorValue = 0;
            m_hBar.Location = new System.Drawing.Point(32, 64);
            m_hBar.MaxValue = 255;
            m_hBar.Name = "m_hBar";
            m_hBar.Size = new System.Drawing.Size(24, 256);
            m_hBar.TabIndex = 4;
            m_hBar.TabStop = false;
            //
            // m_gradCtrl
            //
            m_gradCtrl.Location = new System.Drawing.Point(72, 64);
            m_gradCtrl.Name = "m_gradCtrl";
            m_gradCtrl.Size = new System.Drawing.Size(256, 256);
            m_gradCtrl.TabIndex = 5;
            m_gradCtrl.TabStop = false;
            //
            // ColorPickerControl
            //
            Controls.Add(m_componentPanel);
            Controls.Add(m_pltPanel);
            Controls.Add(m_colSetButton);
            Controls.Add(m_colorPanel);
            Controls.Add(m_hBar);
            Controls.Add(m_gradCtrl);
            Name = "ColorPickerControl";
            Size = new System.Drawing.Size(336, 456);
            ResumeLayout(false);

        }
        #endregion

        #region イベント

        /// <summary>
        /// カラーが変更されたときに発生します。
        /// </summary>
        [Category("カスタム")]
        [Description("カラーが変更されたときに発生します。")]
        public event ColorChangedEventHandler ColorChanged;

        /// <summary>
        /// デフォルトリニアが変更されたときに発生します。
        /// </summary>
        [Category("カスタム")]
        [Description("デフォルトリニアが変更されたときに発生します。")]
        public event EventHandler IsDefaultLinearChanged
        {
            add
            {
                m_componentPanel.IsDefaultLinearChanaged += value;
            }
            remove
            {
                m_componentPanel.IsDefaultLinearChanaged -= value;
            }
        }

        [Category("カスタム")]
        [Description("カラーモードが変更されたときに発生します。")]
        public event ColorModeChangedEventHandler ColorModeChanged;

        /// <summary>
        /// パレットカラーが挿入されたときに発生します。
        /// </summary>
        [Category("カスタム")]
        [Description("パレットカラーが挿入されたときに発生します。")]
        public event PaletteColorInsertedEventHandler PaletteColorInserted;
        #endregion

        #region プロパティ

        /// <summary>
        /// 選択カラー。
        /// </summary>
        [Category("カスタム")]
        [Description("選択カラー。")]
        [DefaultValue(typeof(Color), "Black")]
        public RGB SrgbColor
        {
            get {	return m_SrgbColor;		}
            set {
                m_componentPanel.SrgbColor	= value;
                m_SrgbColor	= value;
            }
        }

        public RGB LinearColor
        {
            get { return m_LinearColor; }
            set
            {
                m_componentPanel.LinearColor = value;
                m_LinearColor = value;
            }
        }

        /// <summary>
        /// カラー値の表示方式。
        /// </summary>
        [Category("カスタム")]
        [Description("カラー値の表示方式。")]
        [DefaultValue(ConfigCommon.ValueMode.Float)]
        public ConfigCommon.ValueMode	ValueMode
        {
            get	{	return m_componentPanel.ValueMode;		}
            set	{	m_componentPanel.ValueMode	= value;	}
        }

        /// <summary>
        /// カラー成分のモード。
        /// </summary>
        [Category("カスタム")]
        [Description("カラー成分のモード。")]
        [DefaultValue(ConfigCommon.ColorMode.RGBA)]
        public ConfigCommon.ColorMode	ColorMode
        {
            get
            {
                return m_componentPanel.ColorMode;
            }
            set
            {
                m_componentPanel.ColorMode = value;

                if (ColorModeChanged != null)
                {
                    ColorModeChanged(this, new ColorModeChangedEventArgs(m_componentPanel.ColorMode, m_componentPanel.IsLinear()));
                }
            }
        }

        /// <summary>
        /// カラーパレット（１６色）。
        /// </summary>
        [Category("カスタム")]
        [Description("カラーパレット（16色）。")]
        public RGB[]	CustomColors
        {
            get	{	return m_pltPanel.GetPalette();		}
        }

        /// <summary>
        /// テキストボックスの表示位置を右側に設定する。
        /// </summary>
        [Category("カスタム")]
        [Description("テキストボックスの表示位置を右側に設定する。")]
        [DefaultValue(false)]
        public bool		TextBoxIsRight
        {
            get	{	return m_componentPanel.TextBoxIsRight;		}
            set	{	m_componentPanel.TextBoxIsRight	= value;	}
        }

        /// <summary>
        /// HDR の上限値を設定する。
        /// </summary>
        public float HDRUpperBound
        {
            set
            {
                m_componentPanel.HDRUpperBound = value;
            }
        }

        /// <summary>
        /// カラースケールを設定する。
        /// </summary>
        public float ColorScale
        {
            set
            {
                m_componentPanel.ColorScale = value;
            }
        }
        #endregion

        #region オーバーライド

        /// <summary>
        /// OnLoadのオーバーライド（プロパティ値からコントロールの設定を初期化する）。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad (e);

            //	プロパティ値からコントロールの設定を初期化
            InitializeControl();
        }

        /// <summary>
        /// OnPaintのオーバーライド（ボーダーの装飾）。
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint (e);

            Graphics g = e.Graphics;
            if( DesignMode )
                InitializeLine();

            Pen		pen	= Enabled ? Pens.Black : Pens.DarkGray;
            if( m_gradCtrl.Visible )
            {
                ControlPaint.DrawBorder3D( g, m_GradBoxWakuOut );
                g.DrawRectangle( pen, m_GradBoxWakuIn );
            }
            if( m_hBar.Visible )
            {
                ControlPaint.DrawBorder3D( g, m_hBarWakuOut );
                g.DrawRectangle( pen, m_hBarWakuIn );
            }
        }

        #endregion

        /// <summary>
        /// イベント発行せずに、現在のカラーを変更する。
        /// </summary>
        public void SetColor(RGB color)
        {
            if (IsDefaultLinear)
            {
                if (m_LinearColor != color)
                {
                    m_LinearColor = color;
                    m_componentPanel.SetLinearColor(color);
                    m_SrgbColor = color.ToSrgbFromLinear();
                    m_colorPanel.BackColor = m_SrgbColor.GetColor();
                    HSV hsb = HSV.FromRGB(m_SrgbColor);
                    if (m_gradCtrl.UpdateColor(hsb, true))
                    {
                        //int		hh	= (int)Math.Round( (double)hsb.H*255/360 );
                        m_hBar.ColorValue = (hsb.H * 255 / 360) % 255;//( hh & 0xFF );
                    }
                    Invalidate();
                }
            }
            else
            {
                if (m_SrgbColor != color)
                {
                    m_componentPanel.SetSrgbColor(color);
                    m_SrgbColor = color;
                    m_LinearColor = color.ToLinearFromSrgb();
                    m_colorPanel.BackColor = m_SrgbColor.GetColor();
                    HSV hsb = HSV.FromRGB(m_SrgbColor);
                    if (m_gradCtrl.UpdateColor(hsb, true))
                    {
                        //int		hh	= (int)Math.Round( (double)hsb.H*255/360 );
                        m_hBar.ColorValue = (hsb.H * 255 / 360) % 255;//( hh & 0xFF );
                    }
                    Invalidate();
                }
            }
        }
        #region カラー変更イベント処理

        /// <summary>
        /// グラデーションピッカーでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void GradCtrl_ColorSelected( object sender, ColorChangedEventArgs e )
        {
            //	アルファ以外が変更されたか
            if( e.Color.R != m_SrgbColor.R || e.Color.G != m_SrgbColor.G || e.Color.B != m_SrgbColor.B )
            {
                var	select	= new RGB( m_SrgbColor.A, e.Color );
                if( m_componentPanel.Visible )
                    m_componentPanel.UpdateSrgbColor( select, true );
                ChangeColor( new ColorChangedEventArgs( select, e.EditFixed, true, true, true, false, false ));
            }
            else if( e.EditFixed )
                ChangeColor( new ColorChangedEventArgs( m_SrgbColor, e.EditFixed, true, true, true, false, false ) );
            ( (Control)sender ).Invalidate();
        }

        /// <summary>
        /// Hueスライダーでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void HueBar_ColorSelected( object sender, ColorChangedEventArgs e )
        {
            if( m_gradCtrl.UpdateColor( new HSV( (float)( (double)( m_hBar.ColorValue * 360 / 255 ) ), 100, 100 ), false ) )
            {
                var	select	= new RGB( m_SrgbColor.A, m_gradCtrl.GetSelectPointColor() );
                if( m_componentPanel.Visible )
                    m_componentPanel.UpdateSrgbColor( select, true );
                ChangeColor( new ColorChangedEventArgs( select, e.EditFixed, true, true, true, false, false ) );
            }
            else if( e.EditFixed )
                ChangeColor( new ColorChangedEventArgs( m_SrgbColor, e.EditFixed, true, true, true, false, false ) );
            ( (Control)sender ).Invalidate();
        }

        /// <summary>
        /// パレットでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PalettePanel_ColorSelected( object sender, ColorChangedEventArgs e )
        {
            //	更新
            if (IsDefaultLinear)
            {
                HSV hsb = HSV.FromRGB(e.Color.ToSrgbFromLinear());
                m_gradCtrl.UpdateColor(hsb, true);
                {
                    var color = new RGB(m_LinearColor.A, e.Color);
                    if (m_componentPanel.Visible)
                        m_componentPanel.UpdateLinearColor(color, true);
                    float hh = (float)((double)hsb.H * 255 / 360);
                    if (m_hBar.ColorValue != hh)
                    {
                        m_hBar.ColorValue = hh % 255;
                        m_hBar.Invalidate();
                    }
                    ChangeColor(new ColorChangedEventArgs(color, e.EditFixed, true, true, true, false, true));
                }
            }
            else
            {
                HSV hsb = HSV.FromRGB(e.Color);
                m_gradCtrl.UpdateColor(hsb, true);
                {
                    var color = new RGB(m_SrgbColor.A, e.Color);
                    if (m_componentPanel.Visible)
                        m_componentPanel.UpdateSrgbColor(color, true);
                    float hh = (float)((double)hsb.H * 255 / 360);
                    if (m_hBar.ColorValue != hh)
                    {
                        m_hBar.ColorValue = hh % 255;
                        m_hBar.Invalidate();
                    }
                    ChangeColor(new ColorChangedEventArgs(color, e.EditFixed, true, true, true, false, false));
                }
            }
        }

        /// <summary>
        /// コンポーネントパネルでカラー選択されたときに実行される
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ComponentPanel_ColorSelected( object sender, ColorChangedEventArgs e )
        {
            //	更新
            HSV hsb = (m_componentPanel.ColorMode == ConfigCommon.ColorMode.HSVA || m_componentPanel.ColorMode == ConfigCommon.ColorMode.HSV) ?
                m_componentPanel.HSV : m_componentPanel.IsLinear() ? HSV.FromRGB(e.Color.ToSrgbFromLinear()) : HSV.FromRGB( e.Color );
            if (m_gradCtrl.UpdateColor( hsb, true ))
            {
                float		hh	= (float)((double)hsb.H*255/360 );
                if( m_hBar.ColorValue != hh )
                {
                    m_hBar.ColorValue	= hh % 255;
                    m_hBar.Invalidate();
                }
                ChangeColor( e );
            }
            else if( m_SrgbColor.A != e.Color.A )
            {
                ChangeAlpha( e );
            }
            else if( e.EditFixed )
                ChangeColor( e );
        }

        /// <summary>
        /// RGBカラーが変更されたときに呼ばれる
        /// </summary>
        /// <param name="e"></param>
        private void ChangeColor( ColorChangedEventArgs e )
        {
            if (e.IsLinear)
            {
                m_LinearColor = e.Color;
                m_SrgbColor = m_LinearColor.ToSrgbFromLinear();
            }
            else
            {
                m_SrgbColor = e.Color;
                m_LinearColor = m_SrgbColor.ToLinearFromSrgb();
            }

            m_colorPanel.BackColor = m_SrgbColor.GetColor();

            if( ColorChanged != null )
                ColorChanged( this, e );
        }

        /// <summary>
        /// Alphaカラーが変更されたときに呼ばれる
        /// </summary>
        /// <param name="e"></param>
        private void ChangeAlpha( ColorChangedEventArgs e )
        {
            if (e.IsLinear)
            {
                m_LinearColor = e.Color;
                m_SrgbColor = m_LinearColor.ToSrgbFromLinear();
            }
            else
            {
                m_SrgbColor = e.Color;
                m_LinearColor = m_SrgbColor.ToLinearFromSrgb();
            }
            if( ColorChanged != null )
                ColorChanged( this, e );
        }

        #endregion

        /// <summary>
        /// 選択色をパレットにコピーするボタンが押された
        /// </summary>
        private void colSetButton_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            // パレットには正規化した色を設定する。
            var color = IsDefaultLinear ? m_LinearColor : m_SrgbColor;
            var m = Math.Max(Math.Max(Math.Max(color.R, color.G), color.B), 1.0f);
            var ncolor = new RGB(color.A, color.R / m, color.G / m, color.B / m);
            m_pltPanel.UpdatePalette(m_pltPanel.Selection, ncolor);

            if (PaletteColorInserted != null)
            {
                PaletteColorInserted(this, new PaletteColorInsertedEventArgs(m_pltPanel.Selection, ncolor));
            }
        }

        /// <summary>
        /// プロパティ値からコントロールの設定を初期化
        /// </summary>
        private void	InitializeControl()
        {
            InitializeLine();
            HSV		hsb	= HSV.FromRGB( m_SrgbColor );
            m_gradCtrl.CreateColorBitmap( hsb );
            m_hBar.ColorValue	= (float)((double)hsb.H*255/360);
            m_hBar.CreateColorBitmap();

            m_colorPanel.BackColor = m_SrgbColor.GetColor();
        }

        /// <summary>
        /// ラインの再設定
        /// </summary>
        private void	InitializeLine()
        {
            int		x, y, w, h;
            x	= m_gradCtrl.Location.X;
            y	= m_gradCtrl.Location.Y;
            w	= m_gradCtrl.Size.Width;
            h	= m_gradCtrl.Size.Height;
            m_GradBoxWakuOut	= new Rectangle( x-3, y-3, w+7, h+7 );
            m_GradBoxWakuIn		= new Rectangle( x-1, y-1, w+1, h+1 );
            x	= m_hBar.Location.X;
            y	= m_hBar.Location.Y;
            w	= m_hBar.Size.Width;
            h	= m_hBar.Size.Height;
            m_hBarWakuOut		= new Rectangle( x-3, y-3, w+7, h+7 );
            m_hBarWakuIn		= new Rectangle( x-1, y-1, w+1, h+1 );
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス前景色を設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="color">前景色</param>
        public void SetTextBoxForeColor(ColorChannel ch, Color color)
        {
            m_componentPanel.SetTextBoxForeColor(ch, color);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス前景色を取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>前景色</returns>
        public Color GetTextBoxForeColor(ColorChannel ch)
        {
            return m_componentPanel.GetTextBoxForeColor(ch);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス背景色を設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="color">背景色</param>
        public void SetTextBoxBackColor(ColorChannel ch, Color color)
        {
            m_componentPanel.SetTextBoxBackColor(ch, color);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックス背景色を取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>背景色</returns>
        public Color GetTextBoxBackColor(ColorChannel ch)
        {
            return m_componentPanel.GetTextBoxBackColor(ch);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックスフォントを設定する
        /// </summary>
        /// <param name="ch">設定するチャンネル</param>
        /// <param name="font">フォント</param>
        public void SetTextBoxFont(ColorChannel ch, Font font)
        {
            m_componentPanel.SetTextBoxFont(ch, font);
        }

        /// <summary>
        /// 指定したチャンネルのテキストボックスフォントを取得する
        /// </summary>
        /// <param name="ch">取得するチャンネル</param>
        /// <returns>フォント</returns>
        public Font GetTextBoxFont(ColorChannel ch)
        {
            return m_componentPanel.GetTextBoxFont(ch);
        }

        public static bool HasAlpha(ConfigCommon.ColorMode ColorMode)
        {
            switch (ColorMode)
            {
                case ConfigCommon.ColorMode.RGBA:
                case ConfigCommon.ColorMode.LinearRgba:
                case ConfigCommon.ColorMode.HSVA:
                case ConfigCommon.ColorMode.A:
                    return true;
                default:
                    return false;
            }
        }
    }

    /// <summary>
    /// カラーチャンネル種別
    /// </summary>
    public enum ColorChannel : int
    {
        ///	<summary>R</summary>
        R,
        ///	<summary>G</summary>
        G,
        ///	<summary>B</summary>
        B,
        ///	<summary>A</summary>
        A
    }

    /// <summary>
    /// カラー変更イベントを処理するメソッドを表します。
    /// </summary>
    public delegate void ColorChangedEventHandler(object sender, ColorChangedEventArgs e);

    /// <summary>
    /// カラーモード変更イベントを処理するメソッドを表します。
    /// </summary>
    public delegate void ColorModeChangedEventHandler(object sender, ColorModeChangedEventArgs e);

    /// <summary>
    /// カラースケール変更イベントを処理するメソッドを表します。
    /// </summary>
    public delegate void ColorScaleChangedEventHandler(object sender, ColorScaleChangedEventArgs e);

    /// <summary>
    /// パレットカラー挿入イベントを処理するメソッドを表します。
    /// </summary>
    public delegate void PaletteColorInsertedEventHandler(object sender, PaletteColorInsertedEventArgs e);

    /// <summary>
    /// カラー最大値変更イベントを処理するメソッドを表します。
    /// </summary>
    internal delegate void SliderMaxValueChangedEventHandler(object sender, SliderMaxValueChangedEventArgs e);

    /// <summary>
    /// カラー変更イベントデータ。
    /// </summary>
    public class ColorChangedEventArgs : EventArgs
    {
        private readonly RGB m_color;
        private readonly bool m_editFixed;

        /// <summary>
        /// 選択カラー。
        /// </summary>
        public RGB Color
        {
            get { return m_color; }
        }
        /// <summary>
        /// エディット終了か否か。
        /// </summary>
        public bool EditFixed
        {
            get { return m_editFixed; }
        }

        public bool RedEdited{   get; private set; }
        public bool GreenEdited{ get; private set; }
        public bool BlueEdited{  get; private set; }
        public bool AlphaEdited{ get; private set; }
        public bool IsLinear { get; set; }

        public ColorChangedEventArgs( RGB color, bool editfixed, bool redEdited, bool greenEdited, bool blueEdited, bool alphaEdited, bool isLinear)
        {
            m_color = color;
            m_editFixed	= editfixed;

            RedEdited	= redEdited;
            GreenEdited	= greenEdited;
            BlueEdited	= blueEdited;
            AlphaEdited	= alphaEdited;
            IsLinear = isLinear;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="color">選択カラー。</param>
        /// <param name="editfixed">エディット終了か否か。</param>
        public ColorChangedEventArgs( RGB color, bool editfixed, bool isLinear)
            : this(color, editfixed, true, true, true, true, isLinear)
        {
        }
    }

    /// <summary>
    /// カラーモード変更イベントデータ。
    /// </summary>
    public class ColorModeChangedEventArgs : EventArgs
    {
        private readonly ConfigCommon.ColorMode m_colorMode;
        private readonly bool m_isLinear;

        public ConfigCommon.ColorMode ColorMode
        {
            get { return m_colorMode; }
        }

        public bool IsLinear
        {
            get { return m_isLinear; }
        }

        public ColorModeChangedEventArgs(ConfigCommon.ColorMode colorMode, bool isLinear)
        {
            m_colorMode = colorMode;
            m_isLinear = isLinear;
        }
    }

    /// <summary>
    /// カラースケール変更イベントデータ。
    /// </summary>
    public class ColorScaleChangedEventArgs : EventArgs
    {
        private readonly float m_colorScale;
        private readonly bool m_editFixed;

        /// <summary>
        /// エディット終了か否か。
        /// </summary>
        public float ColorScale
        {
            get { return m_colorScale; }
        }

        /// <summary>
        /// エディット終了か否か。
        /// </summary>
        public bool EditFixed
        {
            get { return m_editFixed; }
        }

        public ColorScaleChangedEventArgs(float colorScale, bool editFixed)
        {
            m_colorScale = colorScale;
            m_editFixed = editFixed;
        }
    }

    /// <summary>
    /// パレットカラー挿入イベントデータ。
    /// </summary>
    public class PaletteColorInsertedEventArgs : EventArgs
    {
        private readonly int m_selection;
        private readonly RGB m_color;

        public int Selection
        {
            get { return m_selection; }
        }

        public RGB Color
        {
            get { return m_color; }
        }

        public PaletteColorInsertedEventArgs(int selection, RGB color)
        {
            m_selection = selection;
            m_color = color;
        }
    }

    /// <summary>
    /// カラー最大値変更イベントデータ。
    /// </summary>
    internal class SliderMaxValueChangedEventArgs : EventArgs
    {
        private readonly ColorType m_type;
        private readonly float m_value;

        internal ColorType ComponentType
        {
            get { return m_type; }
        }

        internal float MaxValue
        {
            get { return m_value; }
        }

        internal SliderMaxValueChangedEventArgs(ColorType type, float value)
        {
            m_type = type;
            m_value = value;
        }
    }
}
