﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using App.Command;
using App.ConfigData;
using App.Controls;
using App.Data;
using App.PropertyEdit;
using App.Utility;
using ConfigCommon;
using nw.g3d.nw4f_3dif;

namespace App.Controls
{
    partial class GeneralAnimationSettingPopup : AnimationSettingPopupBase
    {
        private readonly Form _parent;

        private GuiObjectGroup TargetGroup;
        private int _prevFrameCount = 0;

        public GeneralAnimationSettingPopup(Form parent)
        {
            InitializeComponent();

            _parent = parent;
            TargetGroup = parent.Tag as GuiObjectGroup;
            Debug.Assert(TargetGroup != null);


            if (TargetGroup.GetObjects(GuiObjectID.ShapeAnimation).Any())
            {
                cbxAutoScale.Enabled = lblFrameCount.Enabled = iepFrameSize.Enabled = false;
            }

            iepFrameSize.SetContextMenuStrip(uiContextMenuTotalFrames);
            iepFrameSize.SliderCloseHandler = (sender, ev) =>
            {
                // http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_3de&action=view_report&id=1810
                // スライダが閉じられた時に、ポップアップがアクティブでない場合は
                // 閉じる処理を入れる
                var flag = _parent == Form.ActiveForm;
                if (flag) return;
                if (TheApp.MainFrame.CanFocus) return;

                EventHandler waitCanFocus = null;
                waitCanFocus = (s, e) =>
                {
                    if (TheApp.MainFrame.CanFocus)
                    {
                        Application.Idle -= waitCanFocus;
                        _parent.Close();
                    }
                };
                Application.Idle += waitCanFocus;
            };

            DocumentPropertyChangedEventHandler propertyChanged = (s, e) =>
            {
                // 表示中なら
                if (Visible)
                {
                    UpdateForm();
                }
            };

            App.AppContext.PropertyChanged += propertyChanged;
            Disposed += (s, e) =>
            {
                App.AppContext.PropertyChanged -= propertyChanged;
            };

            UpdateForm();
        }

        public override bool IsEditing()
        {
            return iepFrameSize.SliderShowing;
        }

        private void UpdateForm()
        {
            using (var block = new UIControlEventSuppressBlock())
            {
                cbxAutoScale.Checked = AnimationSettingPopupBase.AutoScale;

                foreach (var obj in TargetGroup.Objects)
                {
                    if (obj is AnimationDocument)
                    {
                        var anim = obj as AnimationDocument;
                        lblFrameCount.IsModified = anim.IsFrameCountModified;
                        iepFrameSize.Value = anim.FrameCount;
                        _prevFrameCount = anim.FrameCount;
                        cbxLoop.IsModified = anim.IsLoopModified;
                        cbxLoop.Checked = anim.Loop;
                        break;
                    }
                    else if (obj != null)
                    {
                        var objId = obj.ObjectID;
                        bool? isloop = null;
                        var frameCount = 0;
                        switch (objId)
                        {
                            case GuiObjectID.CameraAnimation:
                                {
                                    var anim = obj as CameraAnimation;
                                    Debug.Assert(anim != null, "anim != null");
                                    isloop = anim.Data.loop;
                                    frameCount = anim.Data.frame_count;
                                    lblFrameCount.IsModified = anim.IsValueChanged(x => x.frame_count);
                                    cbxLoop.IsModified = anim.IsValueChanged(x => x.loop);
                                    break;
                                }
                            case GuiObjectID.FogAnimation:
                                {
                                    var anim = obj as FogAnimation;
                                    Debug.Assert(anim != null, "anim != null");
                                    isloop = anim.Data.loop;
                                    frameCount = anim.Data.frame_count;
                                    lblFrameCount.IsModified = anim.IsValueChanged(x => x.frame_count);
                                    cbxLoop.IsModified = anim.IsValueChanged(x => x.loop);
                                    break;
                                }
                            case GuiObjectID.LightAnimation:
                                {
                                    var anim = obj as LightAnimation;
                                    Debug.Assert(anim != null, "anim != null");
                                    isloop = anim.Data.loop;
                                    frameCount = anim.Data.frame_count;
                                    lblFrameCount.IsModified = anim.IsValueChanged(x => x.frame_count);
                                    cbxLoop.IsModified = anim.IsValueChanged(x => x.loop);
                                    break;
                                }
                        }

                        if (isloop == null) continue;

                        cbxLoop.Checked = isloop == true;
                        iepFrameSize.Value = frameCount;
                        _prevFrameCount = frameCount;
                        break;
                    }
                }
            }
        }


        #region コマンド

        private static GroupEditCommand CreateEditCommand_frame_count(GuiObjectGroup targets, int frameCount)
        {
            return
                new GeneralGroupValueEditCommand<int>(
                    targets,
                    targets.Active.ObjectID,
                    frameCount,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var targetId = target.ObjectID;

                        if (targetId != GuiObjectID.SceneAnimation
                            && targetId != GuiObjectID.SkeletalAnimation
                            && target is AnimationDocument
                            )
                        {
                            var animDoc = target as AnimationDocument;
                            swap = animDoc.FrameCount;
                            animDoc.FrameCount = (int)data;
                            AnimationDocument.NotifyFrameCountChanged(animDoc, EventArgs.Empty);
                        }
                        else if (targetId == GuiObjectID.CameraAnimation)
                        {
                            var anim = target as CameraAnimation;
                            swap = anim.Data.frame_count;
                            anim.Data.frame_count = (int)data;
                            AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);
                        }
                        else if (targetId == GuiObjectID.FogAnimation)
                        {
                            var anim = target as FogAnimation;
                            swap = anim.Data.frame_count;
                            anim.Data.frame_count = (int)data;
                            AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);
                        }
                        else if (targetId == GuiObjectID.LightAnimation)
                        {
                            var anim = target as LightAnimation;
                            swap = anim.Data.frame_count;
                            anim.Data.frame_count = (int)data;
                            AnimationDocument.NotifyFrameCountChanged(anim, EventArgs.Empty);
                        }
                    }
                );
        }

        private static GroupEditCommand CreateEditCommand_loop(GuiObjectGroup targets, bool value)
        {
            return
                new GeneralGroupValueEditCommand<bool>(
                    targets,
                    targets.Active.ObjectID,
                    value,
                    delegate(ref GuiObject target, ref object data, ref object swap)
                    {
                        var targetId = target.ObjectID;

                        if (targetId != GuiObjectID.SceneAnimation
                            && targetId != GuiObjectID.SkeletalAnimation
                            && target is AnimationDocument
                            )
                        {
                            var anim = target as AnimationDocument;
                            swap = anim.Loop;
                            anim.Loop = (bool)data;
                            Viewer.LoadOrReloadAnimation.Send(anim);
                        }
                        else if (targetId == GuiObjectID.CameraAnimation)
                        {
                            var anim = target as CameraAnimation;
                            swap = anim.Data.loop;
                            anim.Data.loop = (bool)data;
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }
                        else if (targetId == GuiObjectID.FogAnimation)
                        {
                            var anim = target as FogAnimation;
                            swap = anim.Data.loop;
                            anim.Data.loop = (bool)data;
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }
                        else if (targetId == GuiObjectID.LightAnimation)
                        {
                            var anim = target as LightAnimation;
                            swap = anim.Data.loop;
                            anim.Data.loop = (bool)data;
                            Viewer.LoadOrReloadAnimation.Send(anim.Owner);
                        }
                    }
                );
        }
        /**/
        public static ICommand CreateScaleAnimationEditCommand(GuiObjectGroup targetGroup, float scaleValue)
        {
            var slopeScaleValue = (Math.Abs(scaleValue) > Single.Epsilon) ? (1.0f / scaleValue) : 1.0f;
            var animTargets = new List<AnimTarget>();

            foreach (var obj in targetGroup.Objects)
            {
                animTargets = GetAnimationTargets(obj);

                if (animTargets != null && animTargets.Count > 0)
                {
                    break;
                }
            }

            Debug.Assert(animTargets != null, "animTargets != null");
            var newAnimList = animTargets.Select(ObjectUtility.Clone).ToList();
            foreach (var t in newAnimList)
            {
                foreach (var k in t.KeyFrames)
                {
                    k.Frame *= scaleValue;
                    k.InSlope *= slopeScaleValue;
                    k.OutSlope *= slopeScaleValue;
                }
            }

            EditCommandSet commandSet = new EditCommandSet();
            commandSet.Add(new GeneralGroupValueEditCommand<int>(
                targetGroup,
                null,
                0,
                delegate(ref GuiObject target, ref object data, ref object swap)
                {
                    var origAnimList = GetAnimationTargets(target);

                    Debug.Assert(newAnimList != null, "newAnimList != null");
                    Debug.Assert(newAnimList.Count == origAnimList.Count);

                    for (var i = 0; i < newAnimList.Count; i++)
                    {
                        var orgAnim = origAnimList[i];
                        var newAnim = newAnimList[i];
                        var tmpkey = orgAnim.KeyFrames;
                        orgAnim.KeyFrames = newAnim.KeyFrames;
                        newAnim.KeyFrames = tmpkey;
                    }

                    AnimationCurveEditCommand.UpdateIsModifiedAnimTargetAll(target);
                }
                ));
            var activeTarget = targetGroup.Active;
            commandSet.Add(new LazyCommand(() => AnimationCurveEditCommand.CreateQuantizeAnalyseAllCommand(activeTarget, false)));
            commandSet.OnPostEdit += (s, e) =>
            {
                if (activeTarget.OwnerDocument is AnimationDocument)
                {
                    Viewer.LoadOrReloadAnimation.Send((AnimationDocument)activeTarget.OwnerDocument);
                }
            };
            return commandSet;
        }


        private static List<AnimTarget> GetAnimationTargets(App.Data.GuiObject obj)
        {
            List<AnimTarget> animTargets = null;

            var objId = obj.ObjectID;
            switch (objId)
            {
                case GuiObjectID.CameraAnimation:
                    {
                        var anim = obj as CameraAnimation;
                        if (anim != null) animTargets = new List<AnimTarget>(anim.CameraAnimTargets);
                        break;
                    }
                case GuiObjectID.FogAnimation:
                    {
                        var anim = obj as FogAnimation;
                        if (anim != null) animTargets = new List<AnimTarget>(anim.FogAnimTargets);
                        break;
                    }
                case GuiObjectID.LightAnimation:
                    {
                        var anim = obj as LightAnimation;
                        if (anim != null) animTargets = new List<AnimTarget>(anim.LightAnimTargets);
                        break;
                    }

                case GuiObjectID.BoneVisibilityAnimation:
                    {
                        var anim = obj as BoneVisibilityAnimation;
                        if (anim != null) animTargets = new List<AnimTarget>(anim.BoneVisibilityBoneAnims);
                        break;
                    }
                case GuiObjectID.MaterialVisibilityAnimation:
                    {
                        var anim = obj as MaterialVisibilityAnimation;
                        if (anim != null) animTargets = new List<AnimTarget>(anim.MaterialVisibilityMatAnims);
                        break;
                    }
                case GuiObjectID.TexturePatternAnimation:
                    {
                        var anim = obj as TexturePatternAnimation;
                        if (anim != null)
                            animTargets =
                                new List<AnimTarget>(anim.TexPatternMatAnims.SelectMany(x => x.PatternAnimTargets));
                        break;
                    }
                case GuiObjectID.ShapeAnimation:
                    {
                        var anim = obj as ShapeAnimation;
                        if (anim != null)
                            animTargets = new List<AnimTarget>(anim.VertexShapeAnims.SelectMany(x => x.ShapeAnims));
                        break;
                    }
                case GuiObjectID.ColorAnimation:
                case GuiObjectID.TextureSrtAnimation:
                case GuiObjectID.ShaderParameterAnimation:
                    {
                        var anim = obj as ShaderParameterAnimation;

                        if (anim != null)
                        {
                            var shaderAnims = anim.ShaderParamAnims;
                            Debug.Assert(shaderAnims != null, "shaderAnims != null");
                            var slist = shaderAnims.Select((x, y) => x).ToList();
                            var llist = slist.SelectMany((x, y) => x.ParamAnims).SelectMany(x => x.ParamAnimTargets);
                            animTargets = new List<AnimTarget>(llist);
                        }
                        break;
                    }
                case GuiObjectID.MaterialAnimation:
                    {
                        var anim = obj as MaterialAnimation;

                        if (anim != null)
                        {
                            animTargets = anim.PerMaterialAnims.SelectMany(x =>
                                x.ParamAnims.SelectMany(y => y.ParamAnimTargets.OfType<AnimTarget>())
                                .Concat(x.PatternAnimTargets)
                                .Concat(Enumerable.Repeat(x.MaterialVisibilityMatAnim, 1))).ToList();

                        }
                        break;
                    }
            }

            return animTargets;
        }

        #endregion

        #region イベント

        private void iepFrameSize_SequentialValueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (e.Changing)
            {

            }
            else
            {
                var commandSet = new EditCommandSet();
                commandSet.Add(CreateEditCommand_frame_count(TargetGroup, iepFrameSize.Value));
                if (cbxAutoScale.Checked && _prevFrameCount > 0)
                {
                    var scale = (float)iepFrameSize.Value / (float)_prevFrameCount;
                    commandSet.Add(CreateScaleAnimationEditCommand(TargetGroup,  scale));
                }
                var target = TargetGroup.Active;
                var targetId = target.ObjectID;
                AnimationDocument animDoc = null;

                if (targetId != GuiObjectID.SceneAnimation
                    && targetId != GuiObjectID.SkeletalAnimation
                    && target is AnimationDocument
                    )
                {
                    animDoc = target as AnimationDocument;
                }
                else if (targetId == GuiObjectID.CameraAnimation)
                {
                    var anim = target as CameraAnimation;
                    if (anim != null) animDoc = anim.Owner;
                }
                else if (targetId == GuiObjectID.FogAnimation)
                {
                    var anim = target as FogAnimation;
                    if (anim != null) animDoc = anim.Owner;
                }
                else if (targetId == GuiObjectID.LightAnimation)
                {
                    var anim = target as LightAnimation;
                    if (anim != null) animDoc = anim.Owner;
                }
                Debug.Assert(animDoc != null, "animDoc != null");
                commandSet.OnPostEdit += (s, args) =>
                {
                    Viewer.LoadOrReloadAnimation.Send(animDoc);
                };

                TheApp.CommandManager.Add(commandSet.Execute());
            }

        }

        private void cbxLoop_CheckedChanged(object sender, EventArgs e)
        {
            TheApp.CommandManager.Execute(CreateEditCommand_loop(TargetGroup, cbxLoop.Checked));
        }


        #endregion

        private void cbxAutoScale_CheckedChanged(object sender, EventArgs e)
        {
            AnimationSettingPopupBase.AutoScale = cbxAutoScale.Checked;
        }

        private void cmiCopy_Click(object sender, EventArgs e)
        {
            App.Utility.ClipboardUtility.SetDataObject(iepFrameSize.Value.ToString());
        }

        private void cmiPaste_Click(object sender, EventArgs e)
        {
            var iData = App.Utility.ClipboardUtility.GetDataObject();
            if (iData == null || !iData.GetDataPresent(DataFormats.Text)) return;
            int val;
            if (StringUtility.TrySafeIntParse((String)iData.GetData(DataFormats.Text), out val))
            {
                iepFrameSize.Value = val;
            }
        }

        private void cmiMaxValue_Click(object sender, EventArgs e)
        {
            iepFrameSize.Value = iepFrameSize.MaxBound;
        }

    }
}
