﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Windows.Forms;
using App.ConfigData;
using App.Data;
using App.Utility;
using App.res;

namespace App.Controls
{
    /// <summary>
    /// アニメーション作成ダイアログクラス。
    /// </summary>
    public sealed partial class LightAnimationCreateDialog : OkCancelDialog
    {
        //private readonly GuiObjectID id_;
        readonly SceneAnimation parent;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public LightAnimationCreateDialog(SceneAnimation parent)
        {
            InitializeComponent();

            this.parent = parent;

            var name = parent.Name;
            var i = 1;

            while (parent.LightAnims.Any(x => x.Name == name))
            {
                name = parent.Name + i;
                i++;
            }

            cmbName.Items.Add(name);

            if (ApplicationConfig.Preset.LightAnimPresets.Count <= 0) return;

            foreach (var preset in ApplicationConfig.Preset.LightAnimPresets)
            {
                cmbType.AddItem(preset.Label, preset.Type);
            }
            cmbType.SelectedIndex = 0;
        }


        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool OnResultOk()
        {
            // アニメーション名の整合確認
            string name = AnimationName;

            // 空文字
            if (name == string.Empty)
            {
                UIMessageBox.Warning(Strings.AnimationCreateDialog_NameEmpty);
                return false;
            }

            foreach(var doc in parent.LightAnims)
            {
                if (doc.Name == name)
                {
                    UIMessageBox.Warning(string.Format(Strings.IO_Name_Collision, name));
                    return false;
                }
            }

            // 正規表現
            if (RegexMatch.Check(name, "[0-9A-Za-z\\-\\._]+") == false)
            {
                UIMessageBox.Warning(string.Format(Strings.IO_Name_Invalid, name));
                return false;
            }

            // タイプ名
            if (RegexMatch.Check(Type, "[0-9A-Za-z\\-\\._]*") == false)
            {
                UIMessageBox.Warning(string.Format(Strings.IO_Char_Invalid, name));
                return false;
            }

            return true;
        }

        /// <summary>
        /// アニメーション名。
        /// </summary>
        public string AnimationName { get { return cmbName.Text; } }

        /// <summary>
        /// フレーム数。
        /// </summary>
        public int FrameCount { get { return tbxFrameCount.Value; } }

        /// <summary>
        /// ループ。
        /// </summary>
        public bool IsLoop { get { return rbtLoopOn.Checked; } }

        private void ReturnKeyDown(object sender, EventArgs e)
        {
            DialogResult = DialogResult.OK;
        }

        public string Type
        {
            get
            {
                return (cmbType.SelectedItemData == null) ? string.Empty : cmbType.SelectedItemData.ToString();
            }
        }
        private void cmiCopy_Click(object sender, EventArgs e)
        {
            App.Utility.ClipboardUtility.SetDataObject(tbxFrameCount.Value.ToString());
        }

        private void cmiPaste_Click(object sender, EventArgs e)
        {
            var iData = App.Utility.ClipboardUtility.GetDataObject();
            if (iData == null || !iData.GetDataPresent(DataFormats.Text)) return;
            int val;
            if (StringUtility.TrySafeIntParse((String)iData.GetData(DataFormats.Text), out val))
            {
                tbxFrameCount.Value = val;
            }
        }

        private void cmiMaxValue_Click(object sender, EventArgs e)
        {
            tbxFrameCount.Value = tbxFrameCount.Maximum;
        }
    }
}

