﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;
using App.Data;
using App.Utility;

namespace App.Controls
{
    public sealed partial class AddMultiTexureToTexturePatternDialog : OkCancelDialog
    {
        // テクスチャパターンのインデックス
        private int selectedIndex_ = -1;
        public int SelectedIndex
        {
            get
            {
                return selectedIndex_;
            }

            set
            {
                selectedIndex_ = Math.Max(0, Math.Min(value, _texPatterns().Count - 1));
            }
        }

        public List<Texture> SelectedTextures;

        // ３２サイズイメージリスト
        private static readonly ImageList iml32Size_ = new ImageList()
        {
            ImageSize = new Size(32+2, 32+2),
            ColorDepth = ColorDepth.Depth8Bit,
            TransparentColor = Color.Magenta,
        };

        // ６４サイズイメージリスト
        private static readonly ImageList Iml64Size = new ImageList()
        {
            ImageSize = new Size(64+2, 64+2),
            ColorDepth = ColorDepth.Depth8Bit,
            TransparentColor = Color.Magenta,
        };

        private readonly Func<List<TexturePatternAnimation.TexPattern>> _texPatterns;

        public AddMultiTexureToTexturePatternDialog(Func<List<TexturePatternAnimation.TexPattern>> texPatterns)
        {
            Debug.Assert(texPatterns != null);
            SelectedTextures = new List<Texture>();
            _texPatterns = texPatterns;

            InitializeComponent();

            // 行の高さを変更
            lvwTexList.SmallImageList = iml32Size_;
            lvwTexList.LargeImageList = Iml64Size;
            lvwTexList.CheckBoxes = true;

            UpdateList();

            int depth = 0;
            lvwTexList.ItemChecked += (s, e) =>
            {
                depth++;
                if (e.Item.Checked)
                {
                    foreach (ListViewItem item in lvwTexList.Items)
                    {
                        if (e.Item != item && ((Texture)e.Item.Tag).Name == ((Texture)item.Tag).Name)
                        {
                            Invoke((Action)(() => { item.Checked = false; }));
                        }
                    }
                }
                depth--;
                if (depth == 0)
                {
                    btnOK.Enabled = lvwTexList.Items.Cast<ListViewItem>().Any(x => x != null && x.Checked);
                }
            };
        }

        protected override bool OnResultOk()
        {
            SelectedTextures = lvwTexList.Items.Cast<ListViewItem>().Where(item => item.Checked).Select(x => (Texture)x.Tag).ToList();
            return true;
        }

        private void UpdateList()
        {
            using (var block = new UpdateBlock(lvwTexList))
            {
                var patterns = new HashSet<string>(_texPatterns().Select(x => x.tex_name));
                foreach (var tex in DocumentManager.SortedTextures.Where(x => !patterns.Contains(x.Name)))
                {
                    var name = tex.FileName;
                    var indexText = DocumentManager.GetSameNameIndexText(tex, false);
                    if (!string.IsNullOrEmpty(indexText))
                    {
                        name += " " + indexText;
                    }
                    var item = new ListViewItem(name)
                    {
                        Tag = tex,
                        Checked = false
                    };
                    item.SubItems.Add(string.Empty);
                    lvwTexList.Items.Add(item);
                }
            }
        }

        private void lvwTexList_CustomDrawItem(object sender, CustomDrawListViewItemEventArgs e)
        {
            if (e.Item != null)
            {
                var texName = e.Item.Text;
                var texture = e.Item.Tag as Texture;

                if (e.ColumnIndex == clhTexImage.Index)
                {
                    if (texture != null)
                    {
                        GraphicsUtility.DrawTextureImage(texture, e.Graphics, e.Bounds);
                    }

                    e.Handled = true;
                }
                else
                // 文字
                {
                    e.SpecificForeColor = (texture == null) ? Const.DisreferenceColor : SystemColors.WindowText;

                    var paddingX = 4;
                    var state = e.Item.Checked ? CheckBoxState.CheckedNormal : CheckBoxState.UncheckedNormal;
                    var chksize = CheckBoxRenderer.GetGlyphSize(e.Graphics, state);
                    chksize.Width += paddingX;
                    var loc = new Point(e.Bounds.Location.X + paddingX, e.Bounds.Location.Y + (e.Bounds.Height - chksize.Height) / 2);
                    //var rect = new Rectangle(loc.X + chksize.Width, loc.Y, e.Bounds.Width - chksize.Width, e.Bounds.Height);
                    var rect = new Rectangle(loc.X + chksize.Width, loc.Y
                        , e.Bounds.Width - chksize.Width - paddingX
                        , e.Bounds.Height - (e.Bounds.Height - chksize.Height) / 2);
                    CheckBoxRenderer.DrawCheckBox(e.Graphics, loc, rect, texName, e.ListView.Font
                        , TextFormatFlags.Left | TextFormatFlags.Top | TextFormatFlags.LeftAndRightPadding | TextFormatFlags.WordEllipsis
                        , false, state);
                    //e.DrawDefault();
                    e.Handled = true;
                }
            }
        }
    }
}
