﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using App.Data;
using App.Utility;

namespace App.Command
{
    /// <summary>
    /// コマンドインターフェース
    /// </summary>
    public interface ICommand
    {
        /// <summary>
        /// 実行
        /// </summary>
        ICommand Execute();

        /// <summary>
        /// コマンドが実行されたか
        /// </summary>
        bool IsExecuted { get; }

        /// <summary>
        /// コマンドを Undo/Redo スタックに積んでよいか
        /// </summary>
        /// <remarks>
        /// コマンド実行後に検証すること
        /// </remarks>
        bool IsValid();

        /// <summary>
        /// スタックに積まれたコマンドを Undo できるか
        /// </summary>
        bool CanUndo();

        /// <summary>
        /// スタックに積まれたコマンドを Redo できるか
        /// </summary>
        bool CanRedo();
    }

    /// <summary>
    /// コマンドセットインターフェース
    /// </summary>
    public interface ICommandSet : ICommand
    {
        /// <summary>
        /// 追加
        /// </summary>
        void Add(ICommand command);

        /// <summary>
        /// 反転
        /// </summary>
        void Reverse();
    }

    /// <summary>
    /// コマンドインターフェース
    /// </summary>
    public interface IHasDocumentComponentChangedInfo
    {
        GuiObjectGroup ChangeSource{ get; }
    }

    /// <summary>
    /// コマンドデリゲート
    /// </summary>
    public delegate void CommandDelegate(ICommand command);

    //-------------------------------------------------------------------------
    /// <summary>
    /// コマンドマネージャ
    /// </summary>
    public sealed class CommandManager
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommandManager()
        {
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 追加
        /// </summary>
        public void Add(ICommand command)
        {
            Debug.Assert(command.IsExecuted, "Command is not executed.");

            if (clearNext)
            {
                Clear();
                clearNext = false;
                return;
            }

            if (!command.IsValid())
            {
                return;
            }

            lock(undoStack)
            {
                redoStack.Clear();

                undoStack.Add(command);

                while (undoStack.Count > ConfigData.ApplicationConfig.UserSetting.EditHistory.MaximumItemCount)
                {
                    undoStack.RemoveAt(0);
                    DebugConsole.WriteLine("oldest undostack item cleared");
                }
            }
        }

        /// <summary>
        /// 実行
        /// </summary>
        public void Execute(ICommand command)
        {
            command = command.Execute();
            if (!command.IsValid())
            {
                return;
            }

            if (commandSetBlock == null)
            {
                Add(command);
            }
            else
            {
                commandSetBlock.Add(command);
            }
        }

        private bool clearNext = false;

        /// <summary>
        /// 次にコマンドが積まれるタイミングでクリア
        /// </summary>
        public void ClearNext()
        {
            clearNext = true;
        }

        /// <summary>
        /// クリア
        /// </summary>
        public void Clear()
        {
            lock(undoStack)
            {
                undoStack.Clear();
                redoStack.Clear();
                DebugConsole.WriteLine("undoStack cleared");
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// アンドゥが可能か
        /// </summary>
        public bool CanUndo()
        {
            lock(undoStack) { return undoStack.Count != 0; }
        }

        /// <summary>
        /// リドゥが可能か
        /// </summary>
        public bool CanRedo()
        {
            lock(undoStack) { return redoStack.Count != 0; }
        }

        // 直接呼ばず CommandSetBlock で使うようにしてください。
        public void BeginCommandSetBlock()
        {
            if (commandSetBlockDepth == 0)
            {
                Debug.Assert(commandSetBlock == null);
                commandSetBlock = new EditCommandSet();
            }

            ++ commandSetBlockDepth;
        }

        // 直接呼ばず CommandSetBlock で使うようにしてください。
        public void EndCommandSetBlock()
        {
            Debug.Assert(commandSetBlockDepth > 0);

            -- commandSetBlockDepth;

            if (commandSetBlockDepth == 0)
            {
                Debug.Assert(commandSetBlock != null);

                EditCommandSet cmdSet = commandSetBlock;
                commandSetBlock = null;

                Execute(cmdSet);
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// アンドゥ
        /// </summary>
        public void Undo()
        {
            Debug.Assert(CanUndo());

            using (var wc = new WaitCursor())
            {
                lock(undoStack)
                {
                    int lastIndex = undoStack.Count - 1;
                    ICommand command = undoStack[lastIndex];

                    if (!command.CanUndo())
                    {
                        return;
                    }

                    undoStack.RemoveAt(lastIndex);
                    redoStack.Add(command.Execute());
                }
            }
        }

        /// <summary>
        ///	リドゥ
        /// </summary>
        public void Redo()
        {
            Debug.Assert(CanRedo());

            using (var wc = new WaitCursor())
            {
                lock(undoStack)
                {
                    int lastIndex = redoStack.Count - 1;
                    ICommand command = redoStack[lastIndex];

                    if (!command.CanRedo())
                    {
                        return;
                    }

                    redoStack.RemoveAt(lastIndex);
                    undoStack.Add(command.Execute());
                }
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        ///	文字列化
        /// </summary>
        public override string ToString()
        {
            return String.Format("Undo {0}  Redo {1}", undoStack.Count, redoStack.Count);
        }

        //---------------------------------------------------------------------
        #region コマンドハンドラ
        /// <summary>
        /// コマンドハンドラ。
        /// </summary>
        public void Command_EditUndo(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                if (!CanUndo())
                {

                }
                args.CommandUI.Enabled = CanUndo();
                //DebugConsole.WriteLine(args.CommandUI.Enabled.ToString());
            }
            // 実行時
            else
            {
                if (!CanUndo()) { return; }
                {
                    Undo();
                }
                DebugConsole.WriteLine("Undo [ " + ToString() + " ]");
            }
        }

        /// <summary>
        ///	コマンドハンドラ。
        /// </summary>
        public void Command_EditRedo(MenuCommandArgs args)
        {
            // 更新時
            if (args.RequireUpdate)
            {
                args.CommandUI.Enabled = CanRedo();
            }
            // 実行時
            else
            {
                if (!CanRedo()) { return; }
                {
                    TheApp.CommandManager.Redo();
                }
                DebugConsole.WriteLine("Redo [ " + ToString() + " ]");
            }
        }
        #endregion

        // コマンドスタック
        private readonly List<ICommand> undoStack = new List<ICommand>();

        // リドゥスタック
        private readonly List<ICommand> redoStack = new List<ICommand>();

        // コマンドセットブロック
        private int commandSetBlockDepth = 0;
        private EditCommandSet commandSetBlock = null;
    }

    public sealed class CommandSetBlock : IDisposable
    {
        private readonly CommandManager commandManager_;

        public CommandSetBlock(CommandManager commandManager)
        {
            Debug.Assert(commandManager != null);
            DebugConsole.WriteLine("CommandSetBlock start");
            commandManager_ = commandManager;
            commandManager_.BeginCommandSetBlock();
        }

        public void Dispose()
        {
            DebugConsole.WriteLine("CommandSetBlock end");
            commandManager_.EndCommandSetBlock();
        }
    }

    /// <summary>
    ///  実行時までコマンドの生成を遅延するためのコマンドのラッパ
    /// </summary>
    public class LazyCommand : EditCommand
    {
        public LazyCommand(Func<EditCommand> generator)
        {
            CommandGenerator = generator;
        }

        // コマンドを生成する関数
        private Func<EditCommand> CommandGenerator;

        // 実行時に生成されたコマンド
        private EditCommand editCommand;

        public override ICommand Edit(DocumentPropertyChangedEventArgs eventArgs)
        {
            if (editCommand == null && CommandGenerator != null)
            {
                editCommand = CommandGenerator();
            }

            if (editCommand == null)
            {
                CommandGenerator = null;
            }
            else
            {
                editCommand.Edit(eventArgs);
            }

            return this;
        }

        public override bool IsValid()
        {
            return editCommand != null && editCommand.IsValid();
        }
    }
}
