﻿using System.Windows;
using System.Windows.Interactivity;

namespace Nintendo.InGameEditing.WPF
{
    /// <summary>
    /// FrameworkElement へのビヘイビアの初期化・終了を安全に行えるようにします。
    /// </summary>
    internal class SafeBehavior<T> : Behavior<T>
        where T : FrameworkElement
    {
        /// <summary>
        /// クリーンアップ済みかどうかを取得します。
        /// </summary>
        protected bool IsCleanedUp { get; private set; } = true;

        /// <summary>
        /// AssociatedObject へのイベント登録など、セットアップ処理を記述します。
        /// </summary>
        protected virtual void OnSetup()
        {
        }

        /// <summary>
        /// AssociatedObject へのイベント登録解除など、クリーンアップ処理を記述します。
        /// クリーンアップ後、再セットアップで同じ状態に戻るよう実装する必要があります。
        /// </summary>
        protected virtual void OnCleanup()
        {
        }

        /// <summary>
        /// ビヘイビアのセットアップ処理を行います。
        /// 通常、AssociatedObject.Loaded イベントで呼び出されます。
        /// </summary>
        protected void Setup()
        {
            if (!this.IsCleanedUp) return;
            this.IsCleanedUp = false;

            OnSetup();
        }

        /// <summary>
        /// ビヘイビアのクリーンアップ処理を行います。
        /// 通常、ビヘイビアのデタッチ時または AssociatedObject.Unloaded イベントで呼び出されます。
        /// </summary>
        protected void Cleanup()
        {
            if (this.IsCleanedUp) return;
            this.IsCleanedUp = true;

            OnCleanup();
        }

        /// <summary>
        /// ビヘイビアのアタッチ時の処理を行います。
        /// </summary>
        /// <remarks>
        /// アタッチ時のみの処理が必要な場合、代わりに OnAttachedImpl() をオーバーライドしてください。
        /// </remarks>
        protected sealed override void OnAttached()
        {
            base.OnAttached();
            OnAttachedImpl();

            if (AssociatedObject.IsLoaded)
            {
                Setup();
            }

            AssociatedObject.Loaded -= AssociatedObject_Loaded;
            AssociatedObject.Loaded += AssociatedObject_Loaded;

            AssociatedObject.Unloaded -= AssociatedObject_Unloaded;
            AssociatedObject.Unloaded += AssociatedObject_Unloaded;
        }

        /// <summary>
        /// ビヘイビアのアタッチ時のみの処理を記述します。
        /// </summary>
        protected virtual void OnAttachedImpl()
        {
        }

        /// <summary>
        /// ビヘイビアのデタッチ時の処理を行います。
        /// </summary>
        /// <remarks>
        /// このメソッドは呼び出されない場合があるため、必要な後処理は OnCleanup に記述してください。
        /// デタッチ時のみの後処理が必要な場合、代わりに OnDetachingImpl() をオーバーライドしてください。
        /// </remarks>
        protected sealed override void OnDetaching()
        {
            AssociatedObject.Unloaded -= AssociatedObject_Unloaded;
            AssociatedObject.Loaded -= AssociatedObject_Loaded;

            Cleanup();

            OnDetachingImpl();
            base.OnDetaching();
        }

        /// <summary>
        /// ビヘイビアのデタッチ時のみの処理を記述します。
        /// </summary>
        protected virtual void OnDetachingImpl()
        {
        }

        private void AssociatedObject_Loaded(object sender, RoutedEventArgs e) => Setup();

        private void AssociatedObject_Unloaded(object sender, RoutedEventArgs e) => Cleanup();
    }
}
